;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;
;;;                Centre for Speech Technology Research                  ;;
;;;                     University of Edinburgh, UK                       ;;
;;;                       Copyright (c) 1996,1997                         ;;
;;;                        All Rights Reserved.                           ;;
;;;                                                                       ;;
;;;  Permission to use, copy, modify, distribute this software and its    ;;
;;;  documentation for research, educational and individual use only, is  ;;
;;;  hereby granted without fee, subject to the following conditions:     ;;
;;;   1. The code must retain the above copyright notice, this list of    ;;
;;;      conditions and the following disclaimer.                         ;;
;;;   2. Any modifications must be clearly marked as such.                ;;
;;;   3. Original authors' names are not deleted.                         ;;
;;;  This software may not be used for commercial purposes without        ;;
;;;  specific prior written permission from the authors.                  ;;
;;;                                                                       ;;
;;;  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        ;;
;;;  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      ;;
;;;  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   ;;
;;;  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     ;;
;;;  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    ;;
;;;  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   ;;
;;;  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          ;;
;;;  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       ;;
;;;  THIS SOFTWARE.                                                       ;;
;;;                                                                       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                Author: Robert A. J. Clark and Alan W Black
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Generate F0 points from tobi labels using rules given in:
;;;  `Jilka, Mohler & Dogil 1997'
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Still to do:
;;;     Complete the accent inventory.  
;;;     (Or deny the existence of the other accents!)
;;;
;;;
;;;  Features:
;;;      Can't currently use voicing intervals which cross syllable boundaries,
;;;      so pre/post-nuclear tones are currently places 0.2s before/after the 
;;;      nuclear tone even if no voicing occurs. Failing this they default a
;;;      percentage of the voicing for that syllable. 
;;; 
;;;      Don't know about target points ahead of the current syllable.
;;;      (As you need to know what comes before them to calculate them)
;;;      So: post accent tones aren't placed in the next syllable, and
;;;          when testing for H ... H condition we assume the target for 
;;;          the second H is at the begining of the syllable. 
;;;
;;;  Bugs:
;;;        Very short utterances may fail.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  To use this in a voice 
;;;     (require 'tobi_f0)
;;;  And in the voice call
;;;     (setup_tobi_f0_method)
;;;  Set the following for your speaker's F0 range
;;;  (Parameter.set 'Default_Topline 110)
;;;  (Parameter.set 'Default_Start_Baseline 87)
;;;  (Parameter.set 'Default_End_Baseline 83)
;;;  (Parameter.set 'Current_Topline (Parameter.get 'Default_Topline))
;;;  (Parameter.set 'Valley_Dip 75)

(define (setup_tobi_f0_method)
  "(setup_tobi_f0_method)
Set up parameters for current voice to use the implementaion
of ToBI labels to F0 targets by rule."
  (Parameter.set 'Int_Method Intonation_Tree)
  (Parameter.set 'Int_Target_Method Int_Targets_General)
  (set! int_accent_cart_tree no_int_cart_tree)
  (set! int_tone_cart_tree no_int_cart_tree)
  (set! int_general_params
	(list 
	 (list 'targ_func tobi_f0_targets)))
  (Parameter.set 'Phrase_Method 'cart_tree)
  (set! phrase_cart_tree tobi_label_phrase_cart_tree)
  t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;
;;;;;; Define and set the new f0 rules
;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Set global parameters
;;; You may want to reset these for differen speakers

(Parameter.set 'Default_Topline 110)
(Parameter.set 'Default_Start_Baseline 87)
(Parameter.set 'Default_End_Baseline 83)
(Parameter.set 'Current_Topline (Parameter.get 'Default_Topline))
(Parameter.set 'Valley_Dip 75)
;;; function to add target points on a given syllable and fill in 
;;; targets where necessary

(define (tobi_f0_targets utt syl)
  "(tobi_f0_targets UTT STREAMITEM)
   Returns a list of targets for the given syllable."

  ; get current label. This assumes that there is only one accent and
  ; one endtone on a syllable. Although there can be one of each.

  (let ((voicing  (ttt_get_voice_times utt syl))            ; voicing interval
	(pvoicing (ttt_get_voice_times utt
				       (streamitem.prev syl)))  ; previous voicing
	(nvoicing (ttt_get_voice_times utt 
				       (streamitem.next syl)))) ; next voicing

    ; get last target. Set to begining of utterance if first syl.
    (if (eq (utt.streamitem.feat utt syl 'start) 0)
	(Parameter.set 'Last_Target '(0 0))
	(if (utt.stream utt 'Target)
	    (Parameter.set
	     'Last_Target 
	     (list (utt.streamitem.feat utt (car (last (utt.stream utt 'Target)))
				    'end)
		   (parse-number (utt.streamitem.feat utt 
						  (car (last 
							(utt.stream utt 
								    'Target)))
						  'name))))))

    
    ; if first syl of phrase set Phrase_Start and Phrase_End parameters 
    ;  and reset downstep (currently does so on big and little breaks.)
    (if (eqv? (utt.streamitem.feat utt syl 'Word.Phrase.start)
	      (utt.streamitem.feat utt syl 'start))
	(progn
	 (Parameter.set 'Phrase_Start (utt.streamitem.feat utt syl 'Word.Phrase.start))
	 (Parameter.set 'Phrase_End (utt.streamitem.feat utt syl 'Word.Phrase.end))
	 (Parameter.set 'Current_Topline (Parameter.get 'Default_Topline))))



    ; do stuff
    (append (ttt_to_targets utt syl (wagon utt syl ttt_starttone_tree)
			voicing
			pvoicing
			nvoicing
			'Starttones)
    	    (ttt_to_targets utt syl (wagon utt syl ttt_accent_tree)
			voicing 
			pvoicing 
			nvoicing 
			'Accents)
	    (ttt_to_targets utt syl (wagon utt syl ttt_endtone_tree)
			voicing
			pvoicing
			nvoicing
			'Endtones))))


;;; CART tree to specify no accents

(set! no_int_cart_tree
'
((NONE)))

;;;
;;; Relate phrasing to boundary tones.
;;;

(set! tobi_label_phrase_cart_tree
'
((tone in ("L-" "H-"))
 ((B))
 ((tone in ("H-H%" "H-L%" "L-L%" "L-H%"))
  ((BB))
  ((NB)))))

;;;
;;;  The other functions
;;;

;;; process a list of relative targets and convert to actual targets

(define (ttt_to_targets utt syl rlist voicing pvoicing nvoicing type)
  "Takes a list of target sets and returns a list of targets."
;;; (DEBUG): print
;  (print "in ttt_to_targets with:")
;  (pprintf rlist)
;  (print voicing)
;  (print pvoicing)
;  (print nvoicing)
;   (print type)
(cond 
 ;; nowt
 ((eq (length rlist) 0) ())
 ;; a target set
 ((atom (car (car rlist)))
  (cond
   ((eq type 'Accents)
    (ttt_accent_set_to_targets utt syl rlist voicing pvoicing nvoicing))
   ((eq type 'Starttones)
    (ttt_bound_set_to_targets rlist voicing pvoicing))
   ((eq type 'Endtones)
    (ttt_bound_set_to_targets rlist voicing pvoicing))
   (t (error "unknown target set encountered in ttt_to_targets"))))
 ;; list of target sets
 ((atom (car (car (car rlist))))
  (append (ttt_to_targets utt syl (cdr rlist) voicing pvoicing nvoicing type)
	  (ttt_to_targets utt syl (car rlist) voicing pvoicing nvoicing type)))
 ;; error
 (t (error "something strange has happened in accents_to_targets"))))


;; process a starttone/endtone target set.

(define (ttt_bound_set_to_targets tset voicing pvoicing)
  "takes a start/endtone target set and returns a list of target points."
;;; (DEBUG): print
;  (print "in ttt_bound_set_to_targets with:")
;  (print tset)
;  (print voicing)
;  (print pvoicing)
  (cond
   ;; usually target given is NONE. (also ignore unknown!)
   ((or (eq (car (car tset)) 'NONE)
	(eq (car (car tset)) 'UNKNOWN))
    nil)
   ;; a pair of target pairs
   ((eq (length tset) 2)
    (list (ttt_get_target (car tset) voicing) 
	  (ttt_get_target (car (cdr tset)) voicing)))
   ;; single target pair
   ((eq (length tset) 1)
    (cond
     ;; an actual target pair
     ((not (null (cdr (car tset))))
      (list (ttt_get_target (car tset) voicing)))
     ;; a TAKEOVER marker
     ((eq (car (car tset)) 'TAKEOVER)
      (list (list (ttt_interval_percent voicing 0) 
		   (car (cdr (Parameter.get 'Last_Target))))))
     (t (error "unknown target pair in ttt_bound_set_to_targets"))))
   (t (error "unknown target set type in ttt_bound_set_to_targets"))))


;; process an accent target set.

(define (ttt_accent_set_to_targets utt syl tset voicing pvoicing nvoicing)
  "takes a accent target set and returns a list of target points."
;;; (DEBUG): print
;  (print "in ttt_accent_set_to_targets with:")
;  (pprintf tset)
  (cond
   ;; single target in set
   ((null (cdr tset)) 
    (cond
     ; target given is NONE.
     ((or (eq (car (car tset)) 'NONE)
	  (eq (car (car tset)) 'UNKNOWN)) nil) 
     ; V1 marker
     ((eq (car (car tset)) 'V1)
      (let ((target_time (+ (/ (- (utt.streamitem.feat utt syl 'lisp_next_accent_start)
				  (ttt_last_target utt syl))
			       2.0)
			    (ttt_last_target utt syl))))
	(list (list target_time (ttt_accent_pitch (Parameter.get 'Valley_Dip) target_time)))))
     ; V2 marker
     ((eq (car (car tset)) 'V2)
      (let ((target_time (+ (ttt_last_target utt syl) 0.25)))
	(list (list target_time (ttt_accent_pitch (Parameter.get 'Valley_Dip) target_time)))))
     ; V3 marker
     ((eq (car (car tset)) 'V3)
      (let ((target_time (- (utt.streamitem.feat utt syl 'lisp_next_accent_start) 0.25)))
	(list (list target_time (ttt_accent_pitch (Parameter.get 'Valley_Dip) target_time)))))     
     ; single target pair
     (t (list (ttt_get_target (car tset) voicing)))))
   ;; a pair of targets
   ((length tset 2)
    (cond
     ;; a *ed tone with PRE type tone (as in L+H*)
     ((eq (car (car tset)) 'PRE)
      (let ((star_target (ttt_get_target (car (cdr tset)) voicing)))
	(cond
	 ; normal 0.2s case (UNTESTED!) % currently doesn't check for voicing.
	 ((> (- (car star_target) 0.2) (car (Parameter.get 'Last_Target)))
	  (list  (list (- (car star_target) 0.2)
	   	       (ttt_accent_pitch (car (cdr (car tset)))
					 (- (car star_target) 0.2))) ; the time
	   	 star_target))
	 ; 90% prev voiced if not before last target
	    ; ((> (ttt_interval_percent pvoicing 90) 
	    ;     (car (Parameter.get 'Last_Target)))
	    ;  (list (list (ttt_interval_percent pvoicing 90)
	    ;	       (ttt_accent_pitch (car (cdr (car tset)))
	    ;				(ttt_interval_percent pvoicing 90)))
	    ; 	star_target))

	 ;  otherwise (UNTESTED) [NOTE: Voicing for this syllable only]
	 (t (list (list (ttt_interval_percent voicing 20)
	 		(ttt_accent_pitch (car (cdr (car tset)))
	 				  (ttt_interval_percent voicing 20)))
	 	  star_target)))))
     ; a *ed tone with POST type tone (as L*+H)
     ((eq (car (cdr tset)) 'POST)
      (let ((star_target (ttt_get_target (car (cdr tset)) voicing))
	    (next_target (nil) )) ; interesting problem
	(cond
	 ; normal 0.2s case (UNTESTED)
	 ((< (+ (car star_target) 0.2) (utt.streamitem.feat utt syl "end"))
	  (list star_target 
		(list (+ (car star_target) 0.2) 
		      (ttt_accent_pitch (car (cdr (car (cdr tset))))
				    (+ (car star_target) 0.2) ))))
	 ; 20% next voiced (BUG: Can't do this as the next target hasn't been
	 ;                                                     calculated yet!)
	 (nil nil)
	 ;otherwise (UNTESTED)
	 (t (list star_target
		  (list (ttt_interval_percent voicing 90)
			(ttt_accent_pitch (car (cdr (car (cdr tset))))
					  (ttt_interval_percent voicing 90) )))))))
     
     ((t (error "Unknown pair of targets.")))))
   
   ;; something else...
   (t (error "unknown accent set in ttt_accent_set_to_targets"))))



(define (ttt_get_target pair voicing)
  "Returns actual target pair, usually for a stared tone."
  (list (ttt_interval_percent voicing (car pair))
	(ttt_accent_pitch (car (cdr pair))
			  (ttt_interval_percent voicing (car pair)))))

(define (ttt_accent_pitch value time)
  "Converts a accent pitch entry to a pitch value."
  (cond
   ;; a real value
   ((number? value) 
    (ttt_interval_percent (list (ttt_get_current_baseline time)
				(Parameter.get 'Current_Topline))
			  value))
   ;; Downstep then Topline
   ((eq value 'DHIGH)
    (progn
     (Parameter.set 'Current_Topline (+ (ttt_get_current_baseline time)
					(* 0.8
					   (- (Parameter.get 'Current_Topline)
					      (ttt_get_current_baseline time)))))
     (ttt_interval_percent (list (ttt_get_current_baseline time)
				 (Parameter.get 'Current_Topline))
			   100)))
     
   ;; Unknown
   (t  (error "Unknown accent pitch value encountered"))))


(define (ttt_get_current_baseline v)
  "Returns the current declined baseline at time v."
  (let ((h (Parameter.get 'Default_Start_Baseline))
	(l (Parameter.get 'Default_End_Baseline))
	(e (Parameter.get 'Phrase_End))
	(s (Parameter.get 'Phrase_Start)))
    (- h (* (/ (- h l) (- e s)) (- v s)))))

;;; find the time n% through an inteval

(define (ttt_interval_percent pair percent)
  "Returns the time that is percent percent thought the pair."
  (cond
   ; no pair given: just return nil
   ((null pair) nil)
   ; otherwise do the calculation
   (t (let ((start (car pair))
	    (end (car(cdr pair))))
	(+ start (* (- end start) (/ percent 100)))))))


;;;  Getting start and end voicing times in a syllable

(define (ttt_get_voice_times utt syl_item)
  "Returns a pair of start time of first voiced phone in syllable and
end of last voiced phone in syllable, or nil if syllable is nil"
  (cond
   ((null syl_item) nil)
   (t (let ((segs (utt.streamitem.rel utt syl_item "Segment")))
	(list
	 (utt.streamitem.feat utt (ttt_first_voiced utt segs) "start")
	 (utt.streamitem.feat utt (ttt_first_voiced utt (reverse segs)) "end"))))))

(define (ttt_first_voiced utt segs)
  "Returns first segment that is voiced (vowel or voiced consonant)
returns last segment if all are unvoiced."
  (cond
   ((null (cdr segs))
    (car segs))  ;; last possibility
   ((equal? "+" (utt.streamitem.feat utt (car segs) "ph_vc"))
    (car segs))
   ((equal? "+" (utt.streamitem.feat utt (car segs) "ph_cvox"))
    (car segs))
   (t
    (ttt_first_voiced utt (cdr segs)))))

(define (ttt_last_target utt syl)
  "Returns the end of the most recent previous target (not in this syllable)
in the utterance or nil if there is not one present
"
  (cond
   ((null (streamitem.prev syl)) nil)
   ((ttt_last_target_segs 
     utt 
     (reverse (utt.streamitem.rel utt (streamitem.prev syl) "Segment"))))
   (t (ttt_last_target utt (streamitem.prev syl)))))

(define (ttt_last_target_segs utt segs)
  "Returns the end of the first target in a list of segments,
or nil if there is not one
"
  (cond
   ((null segs) nil)
   ((and  (> (parse-number (utt.streamitem.feat utt (car segs)  "Target.name")) 0)
	  (eq 0 (utt.streamitem.feat utt (car segs) "Syllable.lisp_lh_condition"))
	  (eq 0 (utt.streamitem.feat utt (car segs) "Syllable.lisp_hl_condition"))
	  (eq 0 (utt.streamitem.feat utt (car segs) "Syllable.lisp_valley_condition")))
    (utt.streamitem.feat utt (car segs) "Target.end"))
   
   (t (ttt_last_target_segs utt (cdr segs)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;
;;;;;; CART TREES                           (ttt - tobi to target)
;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;
;;; Return a list of target lists. A target list comprises of a list
;;; of related targets (ie for the L and H in L+H*), just to confuse
;;; matters each target is also a list! (pos pitch)
;;;


(set! ttt_endtone_tree  ; BUG: does it check the current syl for last accent?
      '
      ((tobi_endtone is NONE)        ; ususally none
       ((((NONE))))
       ((tobi_endtone is "H-H%")     ; H-H%
	((((100 120))))
	((tobi_endtone is "L-L%")    ; L-L%
	 ((((100 -20))))
	 ((tobi_endtone is "L-H%")   ; L-H%
	  ((lisp_last_accent > 2)
	   ((lisp_last_accent_type is "L*")
	    ((((0 25) (100 80))))
	    ((((0 0) (100 80)))))
	   ((lisp_last_accent_type is "L*")
	    ((((100 80))))
	    ((((50 0) (100 80))))))
	  ((tobi_endtone is "H-L%")  ; H-L%
	   ((lisp_last_accent_type is "L*")
	    ((tobi_accent is"L*")
	     ((((50 100) (100 100))))
	     ((((0 100) (100 100)))))
	    ((((100 100)))))
	   ((tobi_endtone is "H-")
	    ((((100 100))))
	    ((tobi_endtone is "L-")
	     ((((100 0))))
	     ((((UNKNOWN))))))))))))
	  



(set! ttt_starttone_tree
      '
      ((tobi_endtone is NONE)   ; usually NONE; 
       ((syl_in = 0)
	((p.tobi_endtone in ("H-" "!H-" "L-"))
	 ((((TAKEOVER))))       ; takeover case
	 ((lisp_next_accent > 2)     ; default cases  
	  ((tobi_accent is NONE) ; 
	   ((lisp_next_accent_type is "L*")
	    ((((0 50)(100 25))))
	    ((((0 50)(100 75)))))
	   ((lisp_next_accent_type is "L*")
	    ((((0 30))))
	    ((((0 70))))))
	  ((lisp_next_accent_type is "L*") ; (repeated)
	   ((((0 30))))
	   ((((0 70)))))))
	((((NONE)))))           ; otherwise (and usually) nothing.  
       ((tobi_endtone is "%H")  ; or %H
	((((0 100))))
	((((UNKNOWN)))))))


(set! ttt_accent_tree
      '
      ((tobi_accent is "H*" )    ; H*
       ((syl_in = 0) 
	((syl_out = 0)
	 ((((50 100))))
	 ((((85 100)))))
	((syl_out = 0)
	 ((((25 100))))
	 ((((60 100))))))
       ((tobi_accent is "!H*" )    ; !H*
	((syl_in = 0) 
	 ((syl_out = 0)
	  ((((50 DHIGH))))
	  ((((85 DHIGH)))))
	 ((syl_out = 0)
	  ((((25 DHIGH))))
	  ((((60 DHIGH))))))
	((tobi_accent is "L*" )    ; L*
	 ((syl_in = 0) 
	  ((syl_out = 0)
	   ((((50 0))))
	   ((((85 0)))))
	  ((syl_out = 0)
	   ((((25 0))))
	   ((((60 0))))))
	 ((tobi_accent is "L+H*")   ; L+H*
	  ((syl_in = 0) 
	   ((syl_out = 0)
	    ((((PRE 20) (70 100))))
	    ((((PRE 20) (90 100)))))
	   ((syl_out = 0)
	    ((((PRE 20) (25 100))))
	    ((((PRE 20) (75 100))))))
	  ((tobi_accent is "L*+H")   ; L*+H
	   ((syl_in = 0) 
	    ((syl_out = 0)
	     ((((35 0) (POST 100))))
	     ((((55 0) (POST 100)))))
	    ((syl_out = 0)
	     ((((15 0) (POST 100))))
	     ((((40 0) (POST 100))))))
	   ((tobi_accent is "H+!H*")    ; H+!H* 
	    ((syl_in = 0)
	     ((syl_out = 0)
	      ((((PRE 20) (60 DHIGH))))
	      ((((PRE 20) (90 DHIGH)))))
	     ((syl_out = 0)
	      ((((PRE 20) (20 DHIGH))))
	      ((((PRE 20) (60 DHIGH))))))
	    ((lisp_lh_condition = 1) 
	     ((((100 75))))
	     ((lisp_lh_condition = 2)
	      ((((0 90))))    ; is 90 correct?
	      ((lisp_hl_condition = 1)
	       ((((100 25))))
	       ((lisp_valley_condition = 1)
		((((V1 85))))
		((lisp_valley_condition = 2)
		 ((((V2 70))))
		 ((lisp_valley_condition = 3)
		  ((((V3 70))))
		  ((tobi_accent is NONE)   ; usually we find no accent
		   ((((NONE))))
		   ((((UNKNOWN))))))))))))))))))     ; UNKNOWN TARGET FOUND
      
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;
;;;;;;   Lisp Feature functions.
;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define (valley_condition utt syl)
"(valley_condition utt syl)
Function to determine if a lowered target between two high target points
is needed in this syllable.
Returns:  0 - no target required
          1 - the single target case
          2 - the first of the two target case
          3 - the second of the two target case
"
(cond
 ((and (eq 0 (utt.streamitem.feat utt syl 'accented))
       (string-matches (utt.streamitem.feat utt syl 'lisp_next_accent_type)
		       "\\(H\\*\\|H\\-\\|H\\-L\\%\\|H\\-H\\%\\)")
       (string-matches (utt.streamitem.feat utt syl 'lisp_last_accent_type)
		       "\\(H\\*\\|L\\+H\\*\\|L\\*\\+H\\|\\%H\\)"))
  (let ((nas (utt.streamitem.feat utt syl 'lisp_next_accent_start))
	(syls (utt.streamitem.feat utt syl 'start))
	(syle (utt.streamitem.feat utt syl 'end))
	(las (ttt_last_target utt syl)))
    (cond
     ((and (< (- nas las) 0.5)
	   (> (- nas las) 0.25)
	   (< syls (+ (/ (- nas las) 2.0) (ttt_last_target utt syl)))
	   (> syle (+ (/ (- nas las) 2.0) (ttt_last_target utt syl)))) 1)
     ((and (> (- nas las) 0.5)
	   (< syls (+ (ttt_last_target utt syl) 0.25))
	   (> syle (+ (ttt_last_target utt syl) 0.25))) 2)
     ((and (> (- nas las) 0.5)
	   (< syls (- nas 0.25))
	   (> syle (- nas 0.25))) 3)
     (t 0))))
 (t 0))) 
   
       

(define (lh_condition utt syl)
"(lh_condition utt syl)
Function to determine the need for extra target points between an L and an H
Returns: 1 - first extra target required
         2 - second extra target required
         0 - no target required.
"
(cond
 ((and (eq 0 (utt.streamitem.feat utt syl 'accented))
       (string-matches (utt.streamitem.feat utt syl 'lisp_last_accent_type)
		       "\\(L\\*\\)")
       (string-matches (utt.streamitem.feat utt syl 'lisp_next_accent_type)
		       "\\(H\\*\\|H\\-\\|H\\-L\\%\\|H\\-H\\%\\)"))
  (cond
   ((and (eq 1 (utt.streamitem.feat utt syl 'lisp_last_accent))
	 (< 2 (utt.streamitem.feat utt syl 'lisp_next_accent))) 1)
   ((and (< 2 (utt.streamitem.feat utt syl 'lisp_last_accent))
	 (eq 1 (utt.streamitem.feat utt syl 'lisp_next_accent))) 2)
   (t 0)))
 (t 0)))

(define (hl_condition utt syl)
"(lh_condition utt syl)
Function to determine the need for extra target points between an H and an L
Returns: 1 - extra target required
         0 - no target required.
"
(cond
 ((and (eq 0 (utt.streamitem.feat utt syl 'accented))
       (string-matches (utt.streamitem.feat utt syl 'lisp_next_accent_type)
		       "\\(L\\*\\|L\\+H\\*\\|L\\*\\+H\\|L\\-\\|L\\-L\\%\\|L-H\\%\\)")
       (string-matches (utt.streamitem.feat utt syl 'lisp_last_accent_type)
		       "\\(H\\*\\|L\\+H\\*\\|L\\*\\+H\\|\\%H\\)")
       (eq 1 (utt.streamitem.feat utt syl 'lisp_last_accent))
       (< 2 (utt.streamitem.feat utt syl 'lisp_next_accent))) 1)
 (t 0)))

(define (next_accent utt syl)
"(next_accent utt syl)
Wrapper for c++ func ff_next_accent.
Returns the number of the syllables to the next accent in the following format.
0 - no next accent
1 - next syllable
2 - next next syllable
etc..."
(cond
 ((eq 0 (utt.streamitem.feat utt syl 'lisp_next_accent_type)) 0)
 (t (+ (utt.streamitem.feat utt syl 'next_accent) 1))))


(define (last_accent utt syl)
"(last_accent utt syl)
Wrapper for c++ func ff_last_accent.
Returns the number of the syllables to the previous accent in the following format.
0 - no prev accent
1 - prev syllable
2 - prev to prev syllable
etc..."
(cond
 ((eq 0 (utt.streamitem.feat utt syl 'lisp_last_accent_type)) 0)
 (t (+  (utt.streamitem.feat utt syl 'last_accent) 1))))

(define (next_accent_type utt syl)
"(next_accent_type utt syl)
Returns the type of the next accent."
(cond 
 ((not (eq 0 (utt.streamitem.feat utt syl 'n.IntEvent.name)))
  (utt.streamitem.feat utt syl 'n.IntEvent.name))
 ((eq 0 (utt.streamitem.feat utt syl 'syl_out)) 0)
 (t (next_accent_type utt (streamitem.next syl)))))


(define (last_accent_type utt syl)
"(last_accent_type utt syl)
Returns the type of the last (previous)  accent."
(cond
  ((not (equal? "NONE"  (utt.streamitem.feat utt syl 'p.tobi_endtone)))
   (utt.streamitem.feat utt syl 'p.tobi_endtone))
  ((not (equal? "NONE"  (utt.streamitem.feat utt syl 'p.tobi_accent)))
   (utt.streamitem.feat utt syl 'p.tobi_accent))
  ((eq 0 (utt.streamitem.feat utt syl 'syl_in)) 0)
  (t (last_accent_type utt (streamitem.prev syl)))))

(define (next_accent_start utt syl)
"(next_accent_start utt syl)
Returns the start time  of the next accented syllable"
(cond 
 ((not (eq 0 (utt.streamitem.feat utt syl 'n.IntEvent.name)))
  (utt.streamitem.feat utt syl 'n.start))
 ((eq 0 (utt.streamitem.feat utt syl 'syl_out)) 0)
 (t (next_accent_start utt (streamitem.next syl)))))

(provide 'tobi_f0)
