 /*************************************************************************/
 /*                                                                       */
 /*                Centre for Speech Technology Research                  */
 /*                     University of Edinburgh, UK                       */
 /*                       Copyright (c) 1996,1997                         */
 /*                        All Rights Reserved.                           */
 /*  Permission to use, copy, modify, distribute this software and its    */
 /*  documentation for research, educational and individual use only, is  */
 /*  hereby granted without fee, subject to the following conditions:     */
 /*   1. The code must retain the above copyright notice, this list of    */
 /*      conditions and the following disclaimer.                         */
 /*   2. Any modifications must be clearly marked as such.                */
 /*   3. Original authors' names are not deleted.                         */
 /*  This software may not be used for commercial purposes without        */
 /*  specific prior written permission from the authors.                  */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
 /*  THIS SOFTWARE.                                                       */
 /*                                                                       */
 /*************************************************************************/
 /*                                                                       */
 /*                  Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Wed Mar 19 1997                               */
 /* --------------------------------------------------------------------- */
 /* A Unit type for units which have content which consists of frames     */
 /* each representing a pitch period containing LPCs, residual,           */
 /* original signal or whatever.                                          */
 /*                                                                       */
 /*************************************************************************/

#include "FramesUnit.h"
#include "../Joining/FramesJoin.h"
#include "EST_ContourType.h"

FramesUnit::FramesUnit(void)
{
  p_signal = p_residual = NULL;
  p_lpc = NULL;
  p_pitchmarks = NULL;
  p_segment_ends = NULL;
  p_n_mods = 0;
  for(int i=0; i< MAX_UNITS_IN_CHUNK; i++)
      p_modifications[i] = NULL;
}

FramesUnit::~FramesUnit(void)
{
  if (p_signal)
    delete p_signal;
  if (p_residual)
    delete p_residual;
  if (p_lpc)
    delete p_lpc;
  if (p_pitchmarks)
    delete p_pitchmarks;
  if (p_segment_ends)
    delete p_segment_ends;
  for(int i=0; i< MAX_UNITS_IN_CHUNK; i++)
      if (p_modifications[i])
	  delete p_modifications[i];
}

Unit *FramesUnit::create(void) 
{
  FramesUnit *unit = new FramesUnit;

  return unit;
}

ValueType FramesUnit::property(EST_String name) const
{
  if (name == "frames_subtype")
    return c_string_as_value("yes");

  return Unit::property(name);
}


int FramesUnit::length(void)
{
  EST_Track *pms = pitchmarks();

  return pms->num_frames();
}

int FramesUnit::add_modifications(EST_Track *mods)
{
  if (p_n_mods >= MAX_UNITS_IN_CHUNK)
    err("Too many modification tracks", name());

  // cout << "add mod " << name() << " " << p_n_mods << "\n";

  p_modifications[p_n_mods++] = mods;

  return p_n_mods;
}


EST_TVector<int> *FramesUnit::segment_ends(void)
{
  if (underlying())
    return underlying()->segment_ends();

  if (!p_segment_ends && database())
    {
      EST_Track *pms = pitchmarks();

      EST_TVector<float> *segments = database()->get_segments(*this);

      if (!segments)
	return NULL;

      p_segment_ends = new EST_TVector<int>(segments->num_points());

      for(int ii=0; ii< segments->num_points(); ii++)
	  (*p_segment_ends)(ii) = pms->index((*segments)(ii));
      for(int i=0; i+2< segments->num_points(); i++)
	if ((*p_segment_ends)(i) < (*p_segment_ends)(i+1) &&
	    (*p_segment_ends)(i+1) == (*p_segment_ends)(i+2))
	  (*p_segment_ends)(i+1) -= 1;
	else if ((*p_segment_ends)(i) == (*p_segment_ends)(i+1) &&
	    (*p_segment_ends)(i+1) < (*p_segment_ends)(i+2))
	  (*p_segment_ends)(i+1) += 1;
    }

  return p_segment_ends;
}

EST_Wave *FramesUnit::signal(void)
{
  if (underlying())
    return ((FramesUnit *)underlying())->signal();
  if (!p_signal && database())
    p_signal = database()->get_wave(UnitDatabase::ct_signal, *this);

  return p_signal;
}

EST_Wave *FramesUnit::residual(void)
{
  if (underlying())
    return ((FramesUnit *)underlying())->residual();

  if (!p_residual && database())
    p_residual = database()->get_wave(UnitDatabase::ct_residual, *this);

  return p_residual;
}

EST_Track *FramesUnit::lpc(void)
{
  if (underlying())
    return ((FramesUnit *)underlying())->lpc();

  if (!p_lpc && database())
    p_lpc = database()->get_coefficients(UnitDatabase::ct_analysis, 
					 EST_ContourType::LPC, 
					 *this);

  return p_lpc;
}

EST_Track *FramesUnit::pitchmarks(void)
{
  if (underlying())
    return ((FramesUnit *)underlying())->pitchmarks();

  if (!p_pitchmarks && database())
    p_pitchmarks = database()->get_coefficients(UnitDatabase::ct_analysis, 
					 EST_ContourType::PITCHMARKS, 
					 *this);

  return p_pitchmarks;
}

EST_Track *FramesUnit::modifications(int i) const
{
  //  cout << "get mod " << i << "\n";
  return p_modifications[i];
}

int FramesUnit::n_modifications(void) const
{
  return p_n_mods;
}

int FramesUnit::sample_rate(void)
{
  EST_Wave *w = residual();
  return w?w->sample_rate():0;
}


static void extend_space(FramesUnit::Chunk * &chunks, int &chunks_space, int add=50, FramesUnit::Chunk **ind=NULL)
{
  FramesUnit::Chunk *newchunks = new FramesUnit::Chunk[chunks_space + add];

  if (chunks)
    {
      memcpy(newchunks, chunks, chunks_space * sizeof(FramesUnit::Chunk));
      delete[] chunks;
      if (ind)
	*ind = newchunks + (*ind- chunks);
    }
  chunks = newchunks;
  chunks_space += add;
}



FramesUnit::Chunk *FramesUnit::chunk_utterance(EST_Utterance *utterance,
					 const EST_String unit_stream_name,
					 const EST_String join_stream_name)
{
  if(!utterance->stream_present(unit_stream_name))
    return NULL;
  if(!utterance->stream_present(join_stream_name))
    return NULL;

  EST_Stream * unit_stream = &(utterance->stream(unit_stream_name));
  EST_Stream * join_stream = &(utterance->stream(join_stream_name));

  return FramesUnit::chunk_utterance(utterance, unit_stream, join_stream);
}

FramesUnit::Chunk *FramesUnit::chunk_utterance(EST_Utterance *utt,
			     EST_Stream *unit_stream,
			     EST_Stream *join_stream)
{
  (void)utt;
  static int chunks_space=0;
  static Chunk *chunks = NULL;
  
  int nchunks=0;

  EST_Stream_Item *un;

  if (!chunks)
    extend_space(chunks, chunks_space, 100);

  Chunk *chunk = chunks+nchunks;
  chunk->n=0;

  for(un=unit_stream->head(); un; un=next(un))
    {
      FramesUnit *unit = (FramesUnit *)(un->contents());
      EST_Track *pms = unit->pitchmarks();
      

      FramesJoin *start_join=NULL, *end_join=NULL;
      EST_TVector<int> *segment_ends = unit->segment_ends();

      if (join_stream)
	{
	  // Look to see if we have joins at the end of this unit
	  EST_Stream_Item *join_item1, *join_item2;

	  un->get_linked(join_stream, join_item1, join_item2);


	  if (join_item1)
	    {
	      FramesJoin *j = (FramesJoin *)join_item1->contents();

	      if (j && j->after() == unit)
		start_join = j;
	      else if (j && j->before() == unit)
		end_join = j;
	      else
		cerr << "join not to here " << join_item1->name() << "\n";
	    }
	  if (join_item2)
	    {
	      FramesJoin *j = (FramesJoin *)join_item2->contents();

	      if (j && j->after() == unit)
		start_join = j;
	      else if (j && j->before() == unit)
		end_join = j;
	      else
		cerr << "join not to here " << join_item2->name() << "\n";
	    }
	}

      int f=start_join?start_join->start_index():(*segment_ends)(0);
      int end = end_join?end_join->end_index():pms->num_frames();

      if (f<0)
	f=0;
      if (end>unit->length())
	end = unit->length();

      // stope through the segment ends (0 is the start of the first segment
      // so skip it.
      for(int seg=1; f < end && seg < segment_ends->num_points(); seg++)
	{
	  int s_end = (*segment_ends)(seg);
	  if ( s_end < f)
	    continue;

	  if (s_end > end)
	    s_end=end;

	  chunk->bit[chunk->n].unit = unit;
	  chunk->bit[chunk->n].start_frame = f;
	  chunk->bit[chunk->n].end_frame = s_end;
	      
	  chunk->n++;
	  
	  if (s_end < end)
	    {
	      // if this segment ends inside the region move on to a new
	      // chunk
	      nchunks++;
	      if (nchunks >= chunks_space)
		extend_space(chunks, chunks_space, 100, &chunk);

	      chunk++;
	      chunk->n = 0;
	    }

	  f = s_end;
	}
    }
  if (chunks[nchunks].n!=0)
    chunks[nchunks+1].n = 0;
  return chunks;
}

void FramesUnit::dump_chunks(ostream &stream, const Chunk *chunks)
{
  stream << "{ CHUNKS\n";
  while (chunks->n > 0)
    {
      stream << "      [ ";
      for(int i=0; i<chunks->n;i++)
	{
	  if (i>0)
	    stream << "\t| ";
	  stream << chunks->bit[i].unit->name() 
		 << "\t" << chunks->bit[i].start_frame 
		 << " " << chunks->bit[i].end_frame;
	}
      stream << "\t]\n";
      chunks++;
    }
  stream << "}\n";
}
