/* This file Copyright 1993 by Clifford A. Adams */
/* sconfig.c
 *
 * Easy menu-based configuration.
 */

#include "EXTERN.h"
#include "common.h"
#include "intrp.h"
#include "term.h"
#include "util.h"
#include "scan.h"
#include "scmd.h"
#include "INTERN.h"
#include "sconfig.h"

static int scf_num_ents;
static int scf_num_ents_alloc;
static char **scf_ents;
static char *scf_config_fname;

/* system-wide data */
static int scf_s_num_ents;
static int scf_s_num_ents_alloc;
static char **scf_s_ents;
static char *scf_s_config_fname;

static bool scf_changed;

static char scf_buf[LBUFLEN];

void
scf_new_file()
{
    FILE *fp;

    if (!(fp = fopen(scf_config_fname,"w"))) {
	return;		/* strange, but not a real problem. */
    }
    fprintf(fp,"### Strn configuration file (not for human editing)\n");

/* Consider: should strn write default values?
 * Advantage: only one spot for all default values
 * Disadvantage: if defaults change, written-defaults will be static
 *
 */
#if 0
    /* set the starting commands */
    fprintf(fp,"GROUPDEFAULT %s\n",getval("GROUPDEFAULT","+;nq"));
#endif
    fclose(fp);
}

/* returns either the value of the variable or Nullch */
char *
scf_getval(name,type)
char *name;
int type;	/* 1: system, 2: user, 3: both */
{
    int i;
    int len;

    len = strlen(name);
    if (type & 2) {	/* user */
	for (i=0;i<scf_num_ents;i++)
	    if (strnEQ(scf_ents[i],name,len))
		return(scf_ents[i]+len+1);   /* +1 to point past the space */
    }
    if (type & 1) {	/* system */
	for (i=0;i<scf_s_num_ents;i++)
	    if (strnEQ(scf_s_ents[i],name,len))
		return(scf_s_ents[i]+len+1); /* +1 to point past the space */
    }
    return(Nullch);
}

/* change the user's configuration */
void
scf_newval(name,val)
char *name;
char *val;
{
    int i;
    int len;

    len = strlen(name);
    for (i=0;i<scf_num_ents;i++) {
	if (strnEQ(scf_ents[i],name,len)) {
	    if (strEQ(val,scf_ents[i]+len+1))
		return;		/* same as old value */
	    scf_changed = TRUE;
	    free(scf_ents[i]);
	    sprintf(scf_buf,"%s %s",name,val);
	    scf_ents[i] = savestr(scf_buf);
	    return;
	}
    }
    /* not found, so add the line to the end. */
    scf_changed = TRUE;
    sprintf(scf_buf,"%s %s",name,val);
    if (scf_num_ents==scf_num_ents_alloc) {
	scf_num_ents_alloc += 100;
	scf_ents = (char**)saferealloc((char*)scf_ents,
			sizeof(char**)*scf_num_ents_alloc);
    }
    scf_ents[scf_num_ents++] = savestr(scf_buf);
}

void
scf_init()
{
    FILE *fp;

    scf_num_ents = scf_num_ents_alloc = 0;
    scf_ents = (char**)NULL;
/* consider adding an environment variable here later */
    scf_config_fname = savestr(filexp("%./.strnconfig"));
    scf_changed = FALSE;

    if (!(fp = fopen(scf_config_fname,"r"))) {
	scf_new_file();
	/* try re-opening. */
	fp = fopen(scf_config_fname,"r");
    }
    if (fp) {
	while (fgets(scf_buf,LBUFLEN-4,fp)) {
	    if (*scf_buf == '#')
		continue;
	    if (scf_num_ents==scf_num_ents_alloc) {
		scf_num_ents_alloc += 100;
		scf_ents = (char**)saferealloc((char*)scf_ents,
			sizeof(char**)*scf_num_ents_alloc);
	    }
	    scf_buf[strlen(scf_buf)-1] = '\0';	/* delete the '\n' */
	    scf_ents[scf_num_ents++] = savestr(scf_buf);
	}
	fclose(fp);
    }

    /* load in the system-wide .strnrc values */
/* later: maybe allow different global .strnconfig names? */
    scf_s_num_ents = scf_s_num_ents_alloc = 0;
    scf_s_ents = (char**)NULL;

    scf_s_config_fname = savestr(filexp("%X/strnconfig"));
    if (!(fp = fopen(scf_s_config_fname,"r"))) {
	free(scf_s_config_fname);
	scf_s_config_fname = savestr(filexp("%X/.strnconfig"));
	fp = fopen(scf_s_config_fname,"r");
    }
    if (fp) {
	while (fgets(scf_buf,LBUFLEN-4,fp)) {
	    if (*scf_buf == '#')
		continue;
	    if (scf_s_num_ents==scf_s_num_ents_alloc) {
		scf_s_num_ents_alloc += 100;
		scf_s_ents = (char**)saferealloc((char*)scf_s_ents,
			sizeof(char**)*scf_s_num_ents_alloc);
	    }
	    scf_buf[strlen(scf_buf)-1] = '\0';	/* delete the '\n' */
	    scf_s_ents[scf_s_num_ents++] = savestr(scf_buf);
	}
	fclose(fp);
    }
}

void
scf_cleanup()
{
    FILE *fp;
    int i;

    if (!scf_changed)
	return;
    sprintf(scf_buf,"%s.tmp",scf_config_fname);
    fp = fopen(scf_buf,"w");
    if (!fp)
	return;		/* print an error message later? */
    waiting = TRUE;	/* don't interrupt */
    fprintf(fp,"### Strn configuration file (not for human editing)\n");
    for (i=0;i<scf_num_ents;i++) {
	fprintf(fp,"%s\n",scf_ents[i]);
	if (ferror(fp)) {
	    fclose(fp);
	    UNLINK(scf_buf);
	    waiting = FALSE;
	    return;
	}
    }
#ifdef RENAME
    rename(scf_buf,scf_config_fname);
#else
    UNLINK(scf_config_fname);
    link(scf_buf,scf_config_fname);
    UNLINK(scf_buf);
#endif
    waiting = FALSE;
}

void scf_groupscan _((void));
void scf_artscan _((void));
void scf_virtscan _((void));
void scf_score _((void));
void scf_misc _((void));

/* return "ON" or "OFF": used extensively in menus */
char *
scf_on(flag)
bool_int flag;
{
    if (flag)
	return("ON");
    else
	return("OFF");
}

void
scf_menu()
{
    char ch;
    bool q_done;

    q_done = FALSE;
    while (!q_done) {
	printf("\nStrn configuration menu:\n") FLUSH;
	printf("0) Exit.\n");
	printf("1) Group scan mode.\n");
	printf("2) Article scan mode.\n");
	printf("3) Virtual scan mode.\n");
	printf("4) Scoring.\n");
	printf("5) Miscellaneous.\n");
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		scf_groupscan();
		break;
	    case '2':
		scf_artscan();
		break;
	    case '3':
		scf_virtscan();
		break;
	    case '4':
		scf_score();
		break;
	    case '5':
		scf_misc();
		break;
	    default:
		break;
	}
    }
}

void
scf_groupscan()
{
    char ch;
    bool q_done;
    char *s;

    q_done = FALSE;
    while (!q_done) {
	printf("\nGroup scan configuration menu:\n") FLUSH;
	printf("0) Exit.\n");
	s = scf_getval("STARTCMD",3);
	if (s && (*s == ':'))
	    s = "ON";
	else
	    s = "OFF";
	printf("1) Enter group scan mode when strn starts.\n");
	printf("   (currently %s)\n",s) FLUSH;
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		if (strEQ(s,"OFF"))
		    scf_newval("STARTCMD",":");
		else
		    scf_newval("STARTCMD","^");
	    break;
	    default:
		break;
	}
    }
}

void
scf_sa_display()
{
    char ch;
    bool q_done;
    char *s;
    long value;
    int i,sp;
    char lbuf[32];	/* big enough for a long */

    q_done = FALSE;
    while (!q_done) {
	s = scf_getval("SADISPLAY",3);
	if (!s)
#ifdef SCORE
	    s = "26";
#else
	    s = "24";
#endif
	value = atoi(s);
	printf("\nArticle scan display:\n") FLUSH;
	printf("0) Exit.\n");
	printf("1) Article number (%s)\n",scf_on(value&1));
	printf("2) Score (%s)\n",scf_on(value&2));
	printf("3) Number of articles with same subject (%s)\n",
	       scf_on(value&4));
	printf("4) Author (%s)\n",scf_on(value&8));
	printf("5) Summary (%s)\n",scf_on(value&32));
	printf("6) Keywords (%s)\n",scf_on(value&64));
/*	printf("7) Subject (%s)\n",scf_on(value&16)); /* */
 
	if ((value & 32) || (value & 64))
	  printf("[Note: Summary or Keywords lines will slow down strn.]\n");
	printf("An article will look like this:\n");
	printf("+....  ");
	sp = 7;
	if (value&1) {
	    printf(" 25652 ");
	    sp += 7;
	}
	if (value&2) {
	    printf("[  26] ");
	    sp += 7;
	}
	if (value&4) {
	    printf("(17) ");
	    sp += 5;
	}
	if (value&8)
	    printf("John Q. Public   ");
	if (value&16)
	    printf("Subject of the article");
	printf("\n");
	if (value&32) {
	    for (i=0;i<sp;i++)
		printf(" ");
	    printf("Summary: This is just a test article.\n");
	}
	if (value&64) {
	    for (i=0;i<sp;i++)
		printf(" ");
	    printf("Keys: foo, bar, misc, strn\n");
	}
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		value = value ^ 1;
		break;
	    case '2':
		value = value ^ 2;
		break;
	    case '3':
		value = value ^ 4;
		break;
	    case '4':
		value = value ^ 8;
		break;
	    case '5':
		value = value ^ 32;
		break;
	    case '6':
		value = value ^ 64;
		break;
	    default:
		break;
	}
	sprintf(lbuf,"%ld",value);
	scf_newval("SADISPLAY",lbuf);
    } /* while */
}

void
scf_sa_misc()
{
    char ch;
    bool q_done;
    char *s;
    long value;
    char lbuf[32];	/* big enough for a long */

    q_done = FALSE;
    while (!q_done) {
	s = scf_getval("SAMODE",3);
	if (!s)
#ifdef SCORE
	    s = "9";
#else
	    s = "8";
#endif
	value = atoi(s);
	printf("\nArticle scan misc:\n") FLUSH;
	printf("0) Exit.\n");
	printf("1) Show articles in score order. (%s)\n",scf_on(value&1));
	printf("2) In score ordering, show newer articles first\n");
	printf("   when two articles have the same score. (%s)\n",
	       scf_on(value&2));
	printf("3) Show articles that have been read before. (%s)\n",
	       scf_on(value&4));
	printf("4) Follow threads when reading. (%s)\n",scf_on(value&8));
	printf("5) Fold subjects (hide followups with same subject). (%s)\n",
	       scf_on(value&16));
	/* 32: zoom mode not user-settable */
	/* 64: old "vi" mode flag will be set in misc. */
	printf("6) Don't move pointer after mark/select commands. (%s)\n",
	       scf_on(value&128));
	/* 256: currently reserved for compatability */
	printf("7) When un-zooming, re-Fold the subjects. (%s)\n",
	       scf_on(value&512));
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		value = value ^ 1;
		break;
	    case '2':
		value = value ^ 2;
		break;
	    case '3':
		value = value ^ 4;
		break;
	    case '4':
		value = value ^ 8;
		break;
	    case '5':
		value = value ^ 16;
		break;
	    case '6':
		value = value ^ 128;
		break;
	    case '7':
		value = value ^ 512;
		break;
	    default:
		break;
	}
	sprintf(lbuf,"%ld",value);
	scf_newval("SAMODE",lbuf);
    } /* while */
}

void
scf_artscan()
{
    char ch;
    bool q_done;

    q_done = FALSE;
    while (!q_done) {
	printf("\nArticle scan configuration menu:\n") FLUSH;
	printf("0) Exit.\n");
	printf("1) Change the displayed fields.\n");
	printf("   (Author, threadcount, summary, etc...)\n");
	printf("2) Change ordering and misc. flags.\n");
	printf("   (score ordering, fold, follow, etc...)\n");
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		scf_sa_display();
		break;
	    case '2':
		scf_sa_misc();
		break;
	    default:
		break;
	}
    }
}

void
scf_sv_display()
{
    char ch;
    bool q_done;
    char *s;
    long value;
    int sp;
    char lbuf[32];	/* big enough for a long */

    q_done = FALSE;
    while (!q_done) {
	s = scf_getval("SVDISPLAY",3);
	if (!s)
#ifdef SCORE
	    s = "15";
#else
	    s = "14";
#endif
	value = atoi(s);
	printf("\nVirtual scan display:\n") FLUSH;
	printf("0) Exit.\n");
	printf("1) Score (%s)\n",scf_on(value&1));
	printf("2) Author (%s)\n",scf_on(value&2));
	printf("3) Subject (%s)\n",scf_on(value&4));
 	printf("4) Newsgroup (%s)\n",scf_on(value&8));
	printf("5) True subject (%s)\n",scf_on(value&16));
	if (value&16)
	    printf("   (Display the article's actual Subject: line.)\n");
	else
	    printf("   (Display virtual subject if available.)\n");
	printf("An article will look like this:\n");
	printf("+....  ");
	sp = 7;
	if (value&1)
	    printf("[-18] ");
	if (value&2)
	    printf("John Q. Public   ");
	if (value&4) {
	    if (value&16)
		printf("Subject of the article ");
	    else
		printf("Subject or virtual subject ");
	}
	if (value&8)
	    printf("<soc.singles>");
	printf("\n");
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		value = value ^ 1;
		break;
	    case '2':
		value = value ^ 2;
		break;
	    case '3':
		value = value ^ 4;
		break;
	    case '4':
		value = value ^ 8;
		break;
	    case '5':
		value = value ^ 16;
		break;
	    default:
		break;
	}
	sprintf(lbuf,"%ld",value);
	scf_newval("SVDISPLAY",lbuf);
    } /* while */
}

void
scf_sv_misc()
{
    char ch;
    bool q_done;
    char *s;
    long value;
    char lbuf[32];	/* big enough for a long */

    q_done = FALSE;
    while (!q_done) {
	s = scf_getval("SVMODE",3);
	if (!s)
#ifdef SCORE
	    s = "5";
#else
	    s = "4";
#endif
	value = atoi(s);
	printf("\nVirtual scan misc:\n") FLUSH;
	printf("0) Exit.\n");
	/* note out-of-order bits */
	printf("1) Show articles in score order. (%s)\n",scf_on(value&4));
	printf("2) Follow threads when reading. (%s)\n",scf_on(value&1));
	printf("3) Add previously read articles to virtual groups. (%s)\n",
	       scf_on(value&2));
	printf("4) Display previously read articles in virtual groups. (%s)\n",
	       scf_on(value&8));
	printf("[Options 3 and 4 may be changed within a virtual group.]\n");
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		value = value ^ 4;
		break;
	    case '2':
		value = value ^ 1;
		break;
	    case '3':
		value = value ^ 2;
		break;
	    case '4':
		value = value ^ 8;
		break;
	    default:
		break;
	}
	sprintf(lbuf,"%ld",value);
	scf_newval("SVMODE",lbuf);
    } /* while */
}

void
scf_virtscan()
{
    char ch;
    bool q_done;

    q_done = FALSE;
    while (!q_done) {
	printf("\nVirtual scan configuration menu:\n") FLUSH;
	printf("0) Exit.\n");
	printf("1) Change the displayed fields.\n");
	printf("   (Author, true subject,  etc...)\n");
	printf("2) Change ordering and misc. flags.\n");
	printf("   (score ordering, follow, eligibility, etc...)\n");
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		scf_sv_display();
		break;
	    case '2':
		scf_sv_misc();
		break;
	    default:
		break;
	}
    }
}

void
scf_score()
{
    char ch;
    bool q_done;
    char *s;
    long value;
    char lbuf[32];	/* big enough for a long */

    q_done = FALSE;
    while (!q_done) {
	s = scf_getval("SCOREMODE",3);
	if (!s)
	    s = "1";
	value = atoi(s);
	printf("\nScoring configuration menu:\n") FLUSH;
	printf("0) Exit.\n");
	printf("1) Score in the background (%s)\n",scf_on(value&1));
	printf("2) Verbose scorefile reading (%s)\n",scf_on(value&2));
	printf("3) Verbose score loading (%s)\n",scf_on(value&4));
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		value = value ^ 1;
		break;
	    case '2':
		value = value ^ 2;
		break;
	    case '3':
		value = value ^ 4;
		break;
	    default:
		break;
	}
	sprintf(lbuf,"%ld",value);
	scf_newval("SCOREMODE",lbuf);
    }
}

void
scf_misc()
{
    char ch;
    bool q_done;
    char *s;
    int vi_value;
    int arrow_value;
    int mouse_value;
    char lbuf[32];	/* big enough for a long */

    q_done = FALSE;
    while (!q_done) {
	s = scf_getval("SCANVI",3);
	if (!s)
	    s = "0";
	vi_value = atoi(s);
	s = scf_getval("STRNARROWLEVEL",3);
	if (!s)
	    s = "0";
	arrow_value = atoi(s);
	s = scf_getval("STRNXMOUSE",3);
	if (!s)
	    s = "0";
	mouse_value = atoi(s);

	printf("\nMisc. configuration menu:\n") FLUSH;
	printf("0) Exit.\n");
	printf("1) In scan modes, allow VI-like movement (%s)\n",
	       scf_on(vi_value&1));
	printf("   (In VI-like movement 'j' moves down and 'k' moves up)\n");
	printf("2) Left/Right arrow key behavior: (%s).\n",
	       (arrow_value&1)?"quit/enter level":"previous/next page");
	switch (mouse_value) {
	    case 0:
		s = "when running under an xterm";
		break;
	    case 1:
		s = "always";
		break;
	    case 2:
		s = "never";
		break;
	}
	printf("3) Strn/Xterm mouse tracking: (%s)\n",s);
	ch = menu_get_char();
	switch (ch) {
	    case '0':
		q_done = TRUE;
		break;
	    case '1':
		vi_value = vi_value ^ 1;
		sprintf(lbuf,"%d",vi_value);
		scf_newval("SCANVI",lbuf);
		s_mode_vi = 0;	/* set it back to "unknown" status" */
		break;
	    case '2':
		arrow_value = arrow_value ^ 1;
		sprintf(lbuf,"%d",arrow_value);
		scf_newval("STRNARROWLEVEL",lbuf);
		arrow_macros(scf_buf);		/* reset the macros */
		break;
	    case '3':
		mouse_value = (mouse_value + 1)%3;
		sprintf(lbuf,"%d",mouse_value);
		scf_newval("STRNXMOUSE",lbuf);
		printf("\
(Mouse behavior changes will occur next time strn is run.)\n");
		break;
	    default:
		break;
	}
    }
}
