
/*
 * setSCCserial.c: set clock frequencies of the Atari's SCC
 *
 * Copyright 1995 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 *
 * Usage: setSCCserial [-v] device [rtxc=...] [trxc=...] [pclk=...]
 *                     {subst baud1=baud2} [default]
 *
 *   rtxc=...: set clock frequency connected to the SCC's RTxC pin (in Hz)
 *   trxc=...: set clock frequency connected to the SCC's TRxC pin (in Hz)
 *   pclk=...: set clock frequency connected to the SCC's PCLK pin (in Hz)
 *             This affects both channels, since there is only one PCLK!
 *             (All numbers can be given in kHz and MHz, too, by appending 'k'
 *             or 'M'.)
 *   subst baud1=baud2: Use baud2 in place of baud1 (which must be one of the
 *             standard rates). This is used if one of the standard rates
 *             cannot be generated by the SCC, as it is the case for standard
 *             Modem2: 57600bps isn't possible there and 78400bps is used
 *             instead.
 *   default:  Reset default settings (not to be used with any other argument
 *			   except device and/or -v)
 *   -v:	   show how the baud rates will be generated
 *   -vv:	   show how rate settings are calculated
 *
 * If no options (except the device and -v's) are given, the current settings
 * are printed.
 *
 */

#define	VERSION	"0.1"

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <asm/atari_SCCserial.h>


#define	streq(a,b)	(!strcmp( (a), (b) ))
#define	strneq(a,b)	(!strncmp( (a), (b), strlen(b) ))


/***************************************************************************/
/*                                                                         */
/*							   Global Variables							   */
/*                                                                         */
/***************************************************************************/


/* verbosity level */
int Verbose = 0;

/* device name */
char Device[PATH_MAX] = "";

/* set to defaults? */
int SetDefaults = 0;

/* various clocks given on the command line? */
int RTxC_Given = 0;
int TRxC_Given = 0;
int PCLK_Given = 0;

/* number of substitutions on the command line */
int NSubst = 0;

/* clock values from the command line */
unsigned long RTxC, TRxC, PCLK;

/* fd for SCC serial device */
int SerialFd;

typedef struct {
	unsigned	bps;
	unsigned	clksrc;
	unsigned	divisor;
} BAUD;

#define	NBAUD	17

BAUD baud_table[NBAUD] = {
	{     50, 0, 0 },
	{     75, 0, 0 },
	{    110, 0, 0 },
	{    134, 0, 0 },
	{    150, 0, 0 },
	{    200, 0, 0 },
	{    300, 0, 0 },
	{    600, 0, 0 },
	{   1200, 0, 0 },
	{   1800, 0, 0 },
	{   2400, 0, 0 },
	{   4800, 0, 0 },
	{   9600, 0, 0 },
	{  19200, 0, 0 },
	{  38400, 0, 0 },
	{  57600, 0, 0 },
	{ 115200, 0, 0 }
};


/***************************** Prototypes *****************************/

int main( int argc, char *argv[] );
void parse_args( int argc, char *argv[] );
unsigned long parse_num( const char **p );
void usage( const char *format, ... );
void fatal( const char *format, ... );
void Vprintf( int minlev, const char *format, ... );
void print_settings( void );
void do_settings( void );
unsigned best_divisor( unsigned clksrc, unsigned base, unsigned bps );
const char *clksrc2str( unsigned clksrc );

/************************* End of Prototypes **************************/



int main( int argc, char *argv[] )

{
	parse_args( argc, argv );

	/* check arguments */
	if (!*Device)
		usage( "Device name missing!\n" );

	if (SetDefaults &&
		(RTxC_Given || TRxC_Given || PCLK_Given || NSubst > 0))
		usage( "\"default\" cannot be used together with other settings.\n" );

	if ((SerialFd = open( Device, O_RDONLY )) < 0)
		fatal( "Cannot open %s: %s\n", Device, strerror(errno) );
	
	/* do actions */
	if (SetDefaults) {

		/* reset to kernel boot time defaults */
		if (ioctl( SerialFd, TIOCDATSCC, NULL ) < 0)
			fprintf( stderr, "ioctl(TIOCDATSCC): %s\n", strerror(errno) );

	}
	else if (RTxC_Given || TRxC_Given || PCLK_Given || NSubst > 0) {

		/* do settings */
		do_settings();
		
	}
	else {

		/* just show current settings */
		print_settings();
		
	}

	close( SerialFd );
	return( 0 );
}


void parse_args( int argc, char *argv[] )

{	int			arg;
	const char	*p;

	for( arg = 1; arg < argc; ++arg ) {
		p = argv[arg];
		
		if (*p == '-') {
			/* options */
			for( ++p; *p; ++p ) {
				switch( *p ) {
				  case 'v':
					++Verbose;
					break;
				  default:
					usage( "Unknown option %c.", *p );
				}
			}
		}
		else if (strneq( p, "rtxc=" )) {
			if (RTxC_Given++ > 0)
				usage( "Multiple rtxc= settings\n" );
			p += 5;
			RTxC = (parse_num( &p ) + 8) / 16;
		}
		else if (strneq( p, "trxc=" )) {
			if (TRxC_Given++ > 0)
				usage( "Multiple trxc= settings\n" );
			p += 5;
			TRxC = (parse_num( &p ) + 8) / 16;
		}
		else if (strneq( p, "pclk=" )) {
			if (PCLK_Given++ > 0)
				usage( "Multiple pclk= settings\n" );
			p += 5;
			PCLK = (parse_num( &p ) + 8) / 16;
		}
		else if (streq( p, "subst" )) {
			unsigned long baud1, baud2;
			int i;
			
			/* need one more arg! */
			if (++arg >= argc)
				usage( "\"subst\" missing its argument!\n" );
			p = argv[arg];

			baud1 = parse_num( &p );
			if (*p != '=') usage( "'=' missing in substitution!\n" );
			++p;
			baud2 = parse_num( &p );

			for( i = 0; i < NBAUD; ++i ) {
				if (baud_table[i].bps == baud1) break;
			}
			if (i == NBAUD)
				usage( "%lu is no standard baud rate for substitution!\n" );
			baud_table[i].bps = baud2;
			++NSubst;
		}
		else if (streq( p, "default" )) {
			SetDefaults = 1;
		}
		else {
			if (*Device)
				usage( "Invalid argument \"%s\"", p );
			strcpy( Device, p );
		}
	}

}


unsigned long parse_num( const char **p )

{	const char		*end;
	unsigned long	val, fraction = 0, factor = 1, div, fr;
	
	if (!isdigit(**p))
		usage( "\"%s\" is no valid number!", *p );
		
	val = strtoul( *p, &end, 10 );
	*p = end;
	
	if (**p == '.') {
		++*p;
		if (isdigit(**p)) {
			fraction = strtoul( *p, &end, 10 );
			*p = end;
		}
	}

	if (tolower(**p) == 'k') {
		++*p;
		factor = 1000;
	}
	else if (tolower(**p) == 'm') {
		++*p;
		factor = 1000000;
	}

	for( div = 1, fr = fraction; fr > 0; ) {
		div *= 10;
		fr /= 10;
	}

	return( val * factor + (fraction * factor) / div );
}
	

void usage( const char *format, ... )

{	va_list	vap;
	
	va_start( vap, format );
	vfprintf( stderr, format, vap );
	va_end( vap );

	printf( "setSCCserial version " VERSION "\n" );
	printf( "Usage: setSCCserial [-v[v]] device [rtxc=freq] [trxc=freq]\n"
			"         [pclk=freq] {subst baud1=baud2} [default]\n" );

	exit( 1 );
}


void fatal( const char *format, ... )

{	va_list	vap;
	
	va_start( vap, format );
	vfprintf( stderr, format, vap );
	va_end( vap );
	exit( -1 );
}


void Vprintf( int minlev, const char *format, ... )

{	va_list	vap;

	if (Verbose < minlev) return;
	
	va_start( vap, format );
	vprintf( format, vap );
	va_end( vap );
}


void print_settings( void )

{	struct atari_SCCserial	SCCstruct;
	unsigned	clocks[3], clksrc, div;
	int 		i;

	if (ioctl( SerialFd, TIOCGATSCC, &SCCstruct ) < 0) {
		fprintf( stderr, "ioctl(TIOCGATSCC): %s\n", strerror(errno) );
		return;
	}

	printf( "RTxC: f=%u.%03u kHz, baud_base = %u\n",
			(SCCstruct.RTxC_base * 16) / 1000,
			(SCCstruct.RTxC_base * 16) % 1000,
			SCCstruct.RTxC_base );
	printf( "TRxC: f=%u.%03u kHz, baud_base = %u\n",
			(SCCstruct.TRxC_base * 16) / 1000,
			(SCCstruct.TRxC_base * 16) % 1000,
			SCCstruct.TRxC_base );
	printf( "PCLK: f=%u.%03u kHz, baud_base = %u\n",
			(SCCstruct.PCLK_base * 16) / 1000,
			(SCCstruct.PCLK_base * 16) % 1000,
			SCCstruct.PCLK_base );

	clocks[CLK_RTxC] = SCCstruct.RTxC_base;
	clocks[CLK_TRxC] = SCCstruct.TRxC_base;
	clocks[CLK_PCLK] = SCCstruct.PCLK_base;
		
	for( i = 0; i < NBAUD; ++i ) {
		clksrc = SCCstruct.baud_table[i].clksrc;
		div    = SCCstruct.baud_table[i].divisor;
			
		printf( "%7u: %s / %5u -> %7u.%03u bps ",
				baud_table[i].bps,
				clksrc2str( clksrc ), div,
				clocks[clksrc] / div,
				(clocks[clksrc] * 1000 / div) % 1000 );
		if (clksrc != CLK_PCLK && div <= 4)
			printf( "(direct mode 1:%u)\n", div*16 );
		else
			printf( "(via BRG value %u)\n", div/2-2 );
	}
}


void do_settings( void )

{	struct atari_SCCserial	SCCstruct;
	unsigned	clocks[3], divisors[3], errors[3];
	unsigned	clksrc, div, real_bps_t1000, delta, minerr;
	int 		i, minerr_src;

	if (ioctl( SerialFd, TIOCGATSCC, &SCCstruct ) < 0) {
		fprintf( stderr, "ioctl(TIOCGATSCC): %s\n", strerror(errno) );
		return;
	}

	clocks[CLK_RTxC] = RTxC_Given ? RTxC:  SCCstruct.RTxC_base;
	clocks[CLK_TRxC] = TRxC_Given ? TRxC : SCCstruct.TRxC_base;
	clocks[CLK_PCLK] = PCLK_Given ? PCLK : SCCstruct.PCLK_base;

	for( i = 0; i < NBAUD; ++i ) {

		Vprintf( 2, "Computation for %d bps:\n", baud_table[i].bps );
		
		for( clksrc = 0; clksrc < 3; ++clksrc ) {

			/* Compute divisor that best approximates the desired bps rate */
			div = best_divisor( clksrc, clocks[clksrc], baud_table[i].bps );
			divisors[clksrc] = div;

			/* Compute the error with this clock source and divisor */
			real_bps_t1000 = (((unsigned long long int)clocks[clksrc] * 10000 / div) + 5) / 10;
			delta = abs( (signed long)real_bps_t1000 - (signed long)baud_table[i].bps*1000 );
			errors[clksrc] = (unsigned long long int)delta * 100 / baud_table[i].bps;
			/* This is the real relative error * 100000 */

			Vprintf( 2, "  %s base_baud %u / %u = %u.%03u, error %u.%03u %%\n",
					 clksrc2str(clksrc),
					 clocks[clksrc],
					 div,
					 real_bps_t1000 / 1000, real_bps_t1000 % 1000,
					 errors[clksrc] / 1000, errors[clksrc] % 1000 );
		}

		/* look for the clock source with the least error */
		minerr = UINT_MAX;
		minerr_src = -1;
		for( clksrc = 0; clksrc < 3; ++clksrc ) {
			if (errors[clksrc] < minerr) {
				minerr = errors[clksrc];
				minerr_src = clksrc;
			}
		}

		baud_table[i].clksrc  = minerr_src;
		baud_table[i].divisor = divisors[minerr_src];

		Vprintf( 1, "%7u bps: %s / %u, error %u.%03u %%\n",
				 baud_table[i].bps,
				 clksrc2str( minerr_src ),
				 divisors[minerr_src],
				 errors[minerr_src] / 1000, errors[minerr_src] % 1000 );

		if (errors[minerr_src] >= 1000)
			Vprintf( 2, "  Warning: Error here > 1%% !\n" );
		Vprintf( 2, "\n" );	
	}

	SCCstruct.RTxC_base = clocks[CLK_RTxC];
	SCCstruct.TRxC_base = clocks[CLK_TRxC];
	SCCstruct.PCLK_base = clocks[CLK_PCLK];
	for( i = 0; i < NBAUD; ++i ) {
		SCCstruct.baud_table[i].clksrc  = baud_table[i].clksrc;
		SCCstruct.baud_table[i].divisor = baud_table[i].divisor;
	}
			
	if (ioctl( SerialFd, TIOCSATSCC, &SCCstruct ) < 0) {
		fprintf( stderr, "ioctl(TIOCSATSCC): %s\n", strerror(errno) );
		return;
	}
}


/* the BRG has a 16 bit register */
#define	MAX_DIVISOR		((65535+2)*2)

unsigned best_divisor( unsigned clksrc, unsigned base, unsigned bps )

{	unsigned div_t10, div, even_div;

	if (bps >= base)
		/* bps is at or over the maximum rate possible with this base_baud
		 * -> best divisor is always 1. This avoids returning 0 divisors.
		 */
		return( 1 );

	div_t10 = base * 10 / bps;
	/* Best integer divisor without considering restrictions: */
	div = (div_t10 + 5) / 10;
	/* Best even integer divisor: */
	even_div = ((div_t10 + 10) / 20) * 2;
	
	switch( clksrc ) {

	  case CLK_PCLK: /* 4, 6, 8, 10, ... */
		div = even_div;
		if (div < 4)
			div = 4;
		else if (div > MAX_DIVISOR)
			div = MAX_DIVISOR;
		break;

	  case CLK_RTxC: /* 1, 2, 4, 6, 8, ... */
		if (div != 1)
			div = even_div;
		if (div > MAX_DIVISOR)
			div = MAX_DIVISOR;
		break;

	  case CLK_TRxC: /* 1, 2, 4 only */
		if (div == 3)
			div = even_div;
		if (div > 4)
			div = 4;
		break;
	}

	return( div );
}


const char *clksrc2str( unsigned clksrc )

{
	switch( clksrc ) {
	  case CLK_RTxC:
		return( "RTxC" );
	  case CLK_TRxC:
		return( "TRxC" );
	  case CLK_PCLK:
		return( "PCLK" );
	}
}
