//-----------------------------------------------------------------------------
//
// KDE HTML Bookmarks
//
// (c) Martin R. Jones 1996
//

#include <qfile.h>
#include "bookmark.h"

#include "bookmark.moc"

#include <klocale.h>
#include <kapp.h>

//-----------------------------------------------------------------------------

KBookmark::KBookmark()
{
	children.setAutoDelete( TRUE );
	type = URL;
}

KBookmark::KBookmark( const char *_text, const char *_url )
{
	children.setAutoDelete( TRUE );
	text = _text;
	url = _url;
	type = URL;
}

void KBookmark::clear()
{
	KBookmark *bm;

	for ( bm = getChildren().first(); bm != NULL; bm = getChildren().next() )
	{
		bm->clear();
	}

	children.clear();
}

//-----------------------------------------------------------------------------

KBookmarkManager::KBookmarkManager()
{
}

void KBookmarkManager::read( const char *filename )
{
	QFile file( filename );

	if ( !file.open( IO_ReadOnly ) )
		return;

	root.clear();

	QString text;
	char buffer[256];

	do
	{
		file.readLine( buffer, 256 );
		text += buffer;
	}
	while ( !file.atEnd() );

	HTMLTokenizer *ht = new HTMLTokenizer;

	ht->begin();
	ht->write( text );
	ht->end();

	const char *str;

	while ( ht->hasMoreTokens() )
	{
		str = ht->nextToken();

		// Every tag starts with an escape character
		if ( str[0] == TAG_ESCAPE )
		{
			str++;
	
			if ( strncasecmp( str, "<title>", 7 ) == 0 )
			{
				QString t = "";
				bool bend = FALSE;

				do
				{
					if ( !ht->hasMoreTokens() )
						bend = TRUE;
					else
					{
						str = ht->nextToken();
						if ( str[0] == TAG_ESCAPE &&
								strncasecmp( str + 1, "</title>", 8 ) == 0 )
							bend = TRUE;
						else
							t += str;
					}
				}
				while ( !bend );

				title = t;
			}
			else if ( strncasecmp( str, "<DL>", 4 ) == 0 )
			{
				parse( ht, &root, "" );
			}
		}
    }

	delete ht;

	emit changed();
}

// parser based on HTML widget parser
//
const char *KBookmarkManager::parse( HTMLTokenizer *ht, KBookmark *parent,
	const char *_end )
{
	KBookmark *bm = parent;
	QString text;
	const char *str;

	parent->setType( KBookmark::Folder );

	while ( ht->hasMoreTokens() )
	{
		str = ht->nextToken();

		if (str[0] == TAG_ESCAPE )
		{
			str++;

			if ( _end[0] != 0 && strcasecmp( str, _end ) == 0 )
			{
				return str;
			}
			else if ( strncasecmp( str, "<dl>", 4 ) == 0 )
			{
				parse( ht, bm, "</dl>" );
			}
			else if ( strncasecmp( str, "<dt>", 4 ) == 0 )
			{
				bm = new KBookmark;
				parent->getChildren().append( bm );
			}
			else if ( strncasecmp( str, "<a ", 3 ) == 0 )
			{
				const char *p = str + 3;

				while ( *p != '>' )
				{
					if ( strncasecmp( p, "href=", 5 ) == 0 )
					{
						p += 5;

						text = "";
						bool quoted = FALSE;
						while ( ( *p != ' ' && *p != '>' ) || quoted )
						{
							if ( *p == '\"' )
								quoted = !quoted;
							else
								text += *p;
							p++;
						}

						bm->setURL( text );
			
						if ( *p == ' ' )
							p++;
					}
					else
					{
						char *p2 = strchr( p, ' ' );
						if ( p2 == 0L )
							p2 = strchr( p, '>');
						else
							p2++;
						p = p2;
					}
				}

				text = "";
			}
			else if ( strncasecmp( str, "<H3", 3 ) == 0 )
			{
				text = "";
			}
			else if ( strncasecmp( str, "</H3>", 5 ) == 0 ||
						strncasecmp( str, "</a>", 4 ) == 0 )
			{
				bm->setText( text );
			}
		}
		else if ( str[0] )
		{
			text += str;
		}
	}

	return NULL;
}

// write bookmarks file
//
void KBookmarkManager::write( const char *filename )
{
	QFile file( filename );

	if ( !file.open( IO_WriteOnly ) )
		return;

	QTextStream stream( &file );

	stream << "<!DOCTYPE KDEHELP-Bookmark-file>" << endl;
	stream << klocale->translate("<!-- Do not edit this file -->") << endl;
	stream << "<TITLE>" << title << "</TITLE>" << endl;

	stream << "<H1>" << title << "</H1>" << endl;

	stream << "<DL><p>" << endl;

	writeFolder( stream, &root );

	stream << "</DL><P>" << endl;
}

// write the contents of a folder (recursive)
//
void KBookmarkManager::writeFolder( QTextStream &stream, KBookmark *parent )
{
	KBookmark *bm;

	for ( bm = parent->getChildren().first(); bm != NULL;
			bm = parent->getChildren().next() )
	{
		if ( bm->getType() == KBookmark::URL )
		{
			stream << "<DT><A HREF=\"" << bm->getURL() << "\">"
				<< bm->getText() << "</A>" << endl;
		}
		else
		{
			stream << "<DT><H3>" << bm->getText() << "</H3>" << endl;
			stream << "<DL><P>" << endl;
			writeFolder( stream, bm );
			stream << "</DL><P>" << endl;
		}
	}
}

KBookmark *KBookmarkManager::getBookmark( int id )
{
	int currId = 0;

	return findBookmark( &root, id, currId );
}

KBookmark *KBookmarkManager::findBookmark( KBookmark *parent, int id,
	int &currId )
{
	KBookmark *bm;

	for ( bm = parent->getChildren().first(); bm != NULL;
			bm = parent->getChildren().next() )
	{
		if ( bm->getType() == KBookmark::URL )
		{
			if ( currId == id )
				return bm;
			currId++;
		}
		else
		{
			KBookmark *retbm;
			if ( ( retbm = findBookmark( bm, id, currId ) ) != NULL )
				return retbm;
		}
	}

	return NULL;
}

void KBookmarkManager::add( const char *_text, const char *_url )
{
	root.getChildren().append( new KBookmark( _text, _url ) );

	emit changed();
}

