/* 
   PXKDPSContext.m

   NSDPSContext for GNUstep GUI X/DPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Scott Christley <scottc@net-community.com>
   Date: March 1996
   Author:  Ovidiu Predescu <ovidiu@net-community.com>
   Date: May 1997
   
   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <stdlib.h>
#include <math.h>

// This is to prevent the annoying redefinition warnings with BOOL
// in the X Window System headers
#define BOOL XWINDOWSBOOL
#include <DPS/dpsXclient.h>
#undef BOOL

#include <Foundation/NSData.h>
#include <Foundation/NSArray.h>

#include <AppKit/PSMatrix.h>

#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKWindow.h>
#include <gnustep/xdps/PXKApplication.h>
#include <gnustep/xdps/PXKScreen.h>

#include "general.h"

// Backend instance variables structure
typedef struct _PXKDPSContext_struct
{
  Display *display;
  DPSContext dps_context;
  GC gc;
  PXKWindow* window;
  Window initialXWindow;
  float ctm[6], invctm[6];
  PSMatrix* ctmMatrix;
} PXKDPSContext_struct;

#define PXKDISPLAY (((PXKDPSContext_struct *)be_context_reserved)->display)
#define PXKDPS (((PXKDPSContext_struct *)be_context_reserved)->dps_context)
#define XGC (((PXKDPSContext_struct *)be_context_reserved)->gc)
#define WINDOW (((PXKDPSContext_struct *)be_context_reserved)->window)
#define INITIALXWINDOW \
    (((PXKDPSContext_struct *)be_context_reserved)->initialXWindow)
#define CTM (((PXKDPSContext_struct *)be_context_reserved)->ctm)
#define INVCTM (((PXKDPSContext_struct *)be_context_reserved)->invctm)
#define CTMMATRIX (((PXKDPSContext_struct *)be_context_reserved)->ctmMatrix)

enum {
  A_COEFF = 0,
  B_COEFF,
  C_COEFF,
  D_COEFF,
  TX_CONS,
  TY_CONS
};

#ifdef SIGN
#undef SIGN
#endif
#define SIGN(x) \
  ({typedef _tx = (x); \
      _tx _x = (x); (_x < 0 ? -1 : (_x == 0 ? 0 : 1)); })

#ifdef ABS
#undef ABS
#endif
#define ABS(x) \
  ({typedef _tx = (x); \
      _tx _x = (x); _x < 0 ? -_x : x; })

//
// The backend implementation of NSDPSContext
//
@implementation PXKDPSContext

- init
{
  NSMutableData *data = [NSMutableData data];

  return [self initWithMutableData: data
	       forDebugging: NO
	       languageEncoding: dps_ascii
	       nameEncoding: dps_strings
	       textProc: DPSDefaultTextBackstop
	       errorProc: DPSDefaultErrorProc];
}

// Default initializer
- initWithMutableData:(NSMutableData *)data
	 forDebugging:(BOOL)debug
     languageEncoding:(DPSProgramEncoding)langEnc
	 nameEncoding:(DPSNameEncoding)nameEnc
	     textProc:(DPSTextProc)tProc
	    errorProc:(DPSErrorProc)errorProc
{
  [super initWithMutableData: data
	 forDebugging: debug
	 languageEncoding: langEnc
	 nameEncoding: nameEnc
	 textProc: tProc
	 errorProc: errorProc];

  // Allocate backend structure
  be_context_reserved = calloc(1, sizeof(PXKDPSContext_struct));

  PXKDISPLAY = NULL;
  PXKDPS = NULL;

  return self;
}

- (void)dealloc
{
  DPSDestroySpace(DPSSpaceFromContext(PXKDPS));
  XFreeGC (PXKDISPLAY, XGC);
  XDestroyWindow (PXKDISPLAY, INITIALXWINDOW);
  [WINDOW release];
  [CTMMATRIX release];

  // Release backend structure
  free(be_context_reserved);

  [super dealloc];
}

@end

static void
myDPSErrorProc (DPSContext ctxt, DPSErrorCode errCode, unsigned arg1, unsigned args)
{
  DPSDefaultErrorProc (ctxt, errCode, arg1, arg1);
}

//
// Methods for XWindows implementation
//
@implementation PXKDPSContext (GNUstepXDPS)

- (Display*)xDisplay
{
  return PXKDISPLAY;
}

- (void)setXDisplay:(Display *)xdisplay
{
  PXKDISPLAY = xdisplay;
}

- (DPSContext)xDPSContext
{
  return PXKDPS;
}

- (void)createDPSContext
{
  int xScreen = [(PXKScreen *)[NSScreen mainScreen] xScreen];
  unsigned long valuemask = (GCForeground|GCBackground);
  XGCValues values;

  /* Create a X window to display the initial drawings. This window is never
     mapped. */
  INITIALXWINDOW = XCreateSimpleWindow (PXKDISPLAY,
					RootWindow(PXKDISPLAY, xScreen),
					0, 0, 100, 100, 1, 1,
					BlackPixel(PXKDISPLAY, xScreen));

  /* Create a GC for the initial window */
  values.foreground = BlackPixel(PXKDISPLAY, xScreen);
  values.background = WhitePixel(PXKDISPLAY, xScreen);
  XGC = XCreateGC(PXKDISPLAY, INITIALXWINDOW, valuemask, &values);

  /* Create the context if need be */
  if (!PXKDPS)
    {
      /* Pass None as the drawable argument; the program will execute correctly
         but will not render any text or graphics. */
      PXKDPS = XDPSCreateSimpleContext(PXKDISPLAY, None, XGC, 0, 0,
				       DPSDefaultTextBackstop,
				       (DPSErrorProc)myDPSErrorProc, NULL);
      if (PXKDPS == NULL)
	{
	  NSLog(@"Could not connect to DPS\n");
	  NSLog(@"Trying again...\n");
       	  PXKDPS = XDPSCreateSimpleContext(PXKDISPLAY, None, XGC, 0, 0,
					   DPSDefaultTextBackstop,
					   (DPSErrorProc)myDPSErrorProc, NULL);

	  if (PXKDPS == NULL)
	    {
	      NSLog(@"DPS is not available\n");
	      exit(1);
	    }
	}

      // Make it the active context
      DPSSetContext(PXKDPS);
    }

  PSWinitcontext (XGContextFromGC (XGC), INITIALXWINDOW, 0, 0);
  PSWGetTransform (PXKDPS, CTM, INVCTM);
  CTM[A_COEFF] = ABS(CTM[A_COEFF]);
  CTM[D_COEFF] = ABS(CTM[D_COEFF]);
  INVCTM[A_COEFF] = ABS(INVCTM[A_COEFF]);
  INVCTM[D_COEFF] = ABS(INVCTM[D_COEFF]);
  CTMMATRIX = [[PSMatrix matrixFrom:CTM] retain];
  PSWinitcontext (XGContextFromGC (XGC), None, 0, 0);
}

- (void)wait
{
  DPSWaitContext (PXKDPS);
}

- (void)setXDrawable:(PXKWindow*)w
{
  Drawable drawable;
  NSSize xSize = NSZeroSize;
  GContext gc;

  // Is it a valid window?
  if (!w) {
    drawable = None;
    gc = XGContextFromGC (XGC);
  }
  else {
    NSRect frame = [w frame];

    drawable = [w drawable];
    gc = XGContextFromGC ([w xGC]);
    xSize = [self XSizeFromUserSize:frame.size];
  }

//  /* If its already the drawable then don't set again */
//  if (drawable == [WINDOW drawable])
//    return;

  PSWinitcontext (gc, drawable, (int)0, (int)xSize.height);

//  [w retain];
//  [WINDOW release];
  WINDOW = w;
}

- (PXKWindow*)drawableWindow
{
  return WINDOW;
}

- (id)getCTM
{
  return CTMMATRIX;
}

/* From reasons I can't understand the conversion methods explained in
   "Programming the Display Postscript System with X" from Adobe at pages
   CLX-43/44 do not work correctly. I use here the linear transformations
   that map one set of coordinates to another. The mappings do not take
   into consideration the opposed directions of the two Y axes; thus we don't
   have to bother with the current drawing context. This mapping should be
   done in the places where these results are used.
   -- ovidiu 27/05/97
 */

- (NSPoint)userPointFromXPoint:(NSPoint)xPoint
{
  NSPoint userPoint;

  userPoint.x = INVCTM[A_COEFF] * xPoint.x;
  userPoint.y = INVCTM[D_COEFF] * xPoint.y;
  return userPoint;
}

- (NSPoint)XPointFromUserPoint:(NSPoint)userPoint
{
  NSPoint xPoint;

  xPoint.x = floor (CTM[A_COEFF] * userPoint.x);
  xPoint.y = floor (CTM[D_COEFF] * userPoint.y);
  return xPoint;
}

- (NSSize)userSizeFromXSize:(NSSize)xSize
{
  NSSize userSize;

  userSize.width = INVCTM[A_COEFF] * xSize.width;
  userSize.height = INVCTM[D_COEFF] * xSize.height;
  return userSize;
}

- (NSSize)XSizeFromUserSize:(NSSize)userSize
{
  NSSize xSize;

  xSize.width = floor (CTM[A_COEFF] * userSize.width);
  xSize.height = floor (CTM[D_COEFF] * userSize.height);
  return xSize;
}

@end
