open Lexing
open Unix
open Document
open Url
open Www
open Feed

(* Save to file fname, and apply continuation cont to this file *)
(* unprotected against Sys_error *)
let f cont dh fname endmsg =
  let buffer = String.create 1024
  and oc = open_out_bin fname in
  let red = ref 0 
  and size =   
    try Http_headers.contentlength dh.document_headers
    with Not_found -> 40000 (* duh *)
  in
    dh.document_feed.feed_schedule
      (fun () ->
	try
	 let n = dh.document_feed.feed_read buffer 0 1024 in
	 if n = 0 then begin
	     dclose true dh;
	     close_out oc;
	     Document.end_log dh endmsg;
	     cont fname (* cont is responsible for deleting fname *)
	     end
	 else begin
           output oc buffer 0 n;
      	   red := !red + n;
	   Document.progress_log dh (!red * 100 / size)
	   end
	with
	 Unix_error(_,_,_) | Sys_error _ ->
	   dclose true dh;
	   close_out oc;
	   Document.destroy_log dh false;
	   Msys.rm fname;
	   Error.default#f (I18n.sprintf
	       "Error during retrieval of %s" 
		  (Url.string_of dh.document_id.document_url))
	   )

let tofile cont dh fname endmsg =
  try
    f cont dh fname endmsg
  with Sys_error msg -> 
    dclose true dh;
    Document.destroy_log dh false;
    Error.default#f (I18n.sprintf "Cannot save to %s\n(%s)" fname msg)


let rec interactive cont dh =
  (* The initial content of the requester *)
  let url = Url.string_of dh.document_id.document_url in
  let path = 
    match dh.document_id.document_url.path with Some p -> p | None -> "" in

  Fileselect.f (I18n.sprintf "Save document")
    (function [] -> dclose true dh
            | [fname] -> begin
	       try 
		 let endmsg = (I18n.sprintf "URL %s\nsaved as %s" url fname) in
		  f cont dh fname endmsg;
		  Document.add_log dh 
			(I18n.sprintf "Saving %s\nto %s" url fname)
			(* channel is not closed ! *)
			(fun () -> Msys.rm fname)
	       with Sys_error msg -> 
		  Error.default#f (I18n.sprintf "Cannot save to %s\n(%s)" fname msg);
		  interactive cont dh
	       end
            | l -> raise (Failure "multiple selection"))
    "*"
    (Filename.basename path)
    false false    


let transfer wr dh =
  wr.www_logging (I18n.sprintf "Saving...");
  interactive 
    (fun s -> wr.www_logging "")
    dh


let save_from_string s f =
  try
   let oc = open_out_bin f in
     begin try
      output_string oc s; Error.default#ok (I18n.sprintf "Document saved")
     with
       Sys_error e ->
      	 Error.default#f (I18n.sprintf "Cannot save to %s\n(%s)" f e)
     end;
     close_out oc
  with
    Sys_error e ->
      	 Error.default#f (I18n.sprintf "Cannot save to %s\n(%s)" f e)

let copy_file src dst =
  try
    let ic = open_in_bin src
    and oc = open_out_bin dst 
    and buf = String.create 2048 in
    let rec copy () =
      let n = input ic buf 0 2048 in
      if n <> 0 then begin output oc buf 0 n; copy() end
    in
    begin try 
     copy(); Error.default#ok (I18n.sprintf "Document saved")
    with 
     Sys_error e ->
      Error.default#f (I18n.sprintf "Cannot save to %s\n(%s)" dst e)
    end;
    close_in ic; 
    close_out oc
  with
    Sys_error e ->
      Error.default#f (I18n.sprintf "Cannot save to %s\n(%s)" dst e)

let document did =
  let open_selection_box act =
    Fileselect.f (I18n.sprintf "Save to file")
      (function [] -> ()
      	      | [s] -> act s
	      | l -> raise (Failure "multiple selection"))
      "*" (* should be better *)
      (Filename.basename (Url.string_of did.document_url))
      false
      true
  in      	      
    try
      match Cache.find did with
       {document_data = MemoryData buf} ->
        open_selection_box (save_from_string (Ebuffer.get buf))
    |  {document_data = FileData (f, _)} ->
        open_selection_box (copy_file f)
    with
      Not_found ->
       Error.default#f ("Document has been flushed from cache.")

