#include "cthugha.h"
#include "display.h"
#include "options.h"
#include "keys.h"
#include "action.h"
#include "information.h"
#include "sound.h"
#include "translate.h"

#include <string.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#undef COLORS					/* comes again in ncurses */
#include <ncurses.h>

#ifdef CTH_X11
    #include <sys/ipc.h>
    #include <sys/shm.h>
    #include <X11/extensions/XShm.h>

    Status mit_shm;
    XShmSegmentInfo shminfo;
    XImage * image;
#endif

/* possible display-function */
int screen_up();	int screen_down();	int screen_2hor();
int screen_r2hor();	int screen_4hor();	int screen_2verd();
int screen_r2verd();	int screen_4kal();	int screen_mirrorH();
int screen_mirrorHV();	int screen_mirrorV();

opt_data screens[] = {
    { screen_up,	1,	"Up",		"Up Display" },
    { screen_down,	1,	"Down",		"Upside Down"},
    { screen_2hor,	1,	"2hor",		"Hor. Split out"},
    { screen_r2hor,	1,	"r2hor",	"Hor. Split in"},
    { screen_4hor,	1,	"4hor",		"Kaleidoscope"},
    { screen_2verd,	1,	"2verd",	"90deg rot. mirror"},
    { screen_r2verd,	1,	"r2verd",	"90deg rot. mirror II"},
    { screen_4kal,	1,	"4kal",		"90deg Kaleidoscope"},
    { screen_mirrorH,	1,	"mirrorH",	"Mirror horizontal"},
    { screen_mirrorHV,	1,	"mirrorHV",	"Mirror horiz&vert"},
    { screen_mirrorV,	1,	"mirrorV",	"Mirror vertical"}
};
int nr_screens = sizeof(screens) / sizeof(opt_data);
int screen_sizes[][2] = {
    {1,1},	/* up */    {1,1},	/* down */
    {1,1},	/* 2hor */	{1,1},	/* r2hor */
    {1,1},	/* 4hor */	{1,1},	/* 2verd */
    {1,1},	/* r2verd */	{1,1},	/* 4 kal */
    {2,1},	/* mirrorH */	 {2,2},	/* mirrorHV */
    {1,2}	/* mirrorV */
};

int (*update_screen)(void);			/* pointer to display-funct. */

int screen_width = 0;
int bytes_per_line = 0;
int screen_height = 0;

#define	SCREEN_SIZE	( bytes_per_line * screen_height )

char * buff1;					/* Screen-Memory 1 */
char * buff2;					/* Screen-Memory 2 */

unsigned char * active_buffer;			/* Next on screen */
unsigned char * passive_buffer;			/* right now on screen */

char screen_first[256] = "";			/* Start with this scrn-fkt */

int display_frames = 0;				/* counter for speed-tests */
time_t display_start;

int display_clear = 1;				/* clear screen */

int display_tile_x = 1;				/* tile smal buffers on scrn */
int display_tile_y = 1;
static int tiles_x = 1, tiles_y = 1;		/* how many tiles */
int display_mode = 0;				/* graphic mode */

#ifdef CTH_console
    int screen_modes[] = {			/* allowed modes */
	G320x200x256,		/* 0 */
	G320x240x256,		/* 1 */
	G320x400x256,		/* 2 */
	G360x480x256,		/* 3 */
	G640x480x256,		/* 4 */
	G800x600x256,		/* 5  here it starts getting tiny */
	G1024x768x256,		/* 6 */
	G1280x1024x256,		/* 7 */
	G1600x1200x256,		/* 8  only for completeness */
    };
    int nr_screen_modes = sizeof(screen_modes) / sizeof(int);

    GraphicsContext display_screen_context;
    GraphicsContext display_virtual_context;

    int display_copybox = 0;			/* use copybox. Not available
						   at some screen-modes */
    int display_direct = 1;			/* draw directly to screen */
#endif

int bpp = 8;					/* bits per pixel */
int bypp = 1;					/* bytes per pixel */

unsigned char * display_mem = NULL;		/* memory for screen-fkt
						   to draw to */
#ifdef CTH_X11
    unsigned char * display_bitmap = NULL;	/* memory for X11-bitmap */
    int display_mit_shm = 1;			/* use MIT-SHM if possible */
    int display_on_root = 0;			/* display on root window */
#endif

int display_text_line=-1;			/* display at line */
char display_text[MAX_TEXT_DISPLAY+1];		/* text to display */
char display_font[ 256 * 8 * 8 ];
int display_text_on_screen = 0;			/* text is currently on screen;
						   also used to count how long
						   the text is on the screen 
						   (counts down, on 0 the text
						   is removed	*/
int display_syncwait = 
#ifndef CTH_X11
	0;			/* Wait for sync on display */
#else
	1;			/* call XSync */
#endif

int window_width;
int window_height;
int draw_mode = DM_direct;			/* how drawing is done */


/* 
 * Initialization of ncurses
 */
#ifndef CTH_saverX11
int init_ncurses() {
    if( !initscr() )
	return 1;

    timeout(0);					/* no timeout on getch(); */
    keypad(stdscr, TRUE);			/* allow function keys */
    meta(stdscr, TRUE);				/* 8-bit clean */
    noecho();					/* don't print out keys */
    cbreak();					/* don't wait for cr */
    start_color();

    clear();
    refresh();
    
    return 0;
}
#endif

/*
 * Cleanup of ncurses (makes cursor visible again)
 */
#ifndef CTH_saverX11
int exit_ncurses() {
    echo();
    endwin();
    return 0;
}
#endif


/*
 * Initialization of display 
 *  - load palettes and PCX 
 *  - allocate buffer
 *  - does not switch to graph mode
 */
int init_display() {
  
#ifdef CTH_console
    vga_init();					/* initialize svgalib */
#endif
    
#ifdef CTH_X11
    static int X[] = {320,320,320,260,640,800,1024,1280,1600};
    static int Y[] = {200,240,400,480,480,600, 768,1024,1200};
    static int nr_screen_modes = sizeof(X) / sizeof(int);

    if ( display_mode == -1) {
	/* a special display resolution is given */
	tiles_x = (display_tile_x ? (screen_width/BUFF_WIDTH) : 1);
	tiles_y = (display_tile_y ? (screen_height/BUFF_HEIGHT):1);
    } else {
	/* use one of the default resolutions */
	if ( (display_mode >= nr_screen_modes) || (display_mode < 0))
	    display_mode = 0;
	
	tiles_x = (display_tile_x ?(X[display_mode]/BUFF_WIDTH) : 1);
	tiles_y = (display_tile_y ?(Y[display_mode]/BUFF_HEIGHT): 1);
    }
    /* make buffer big enough */
    tiles_x = max(tiles_x,1);
    tiles_y = max(tiles_y,1);

    /* calculate real size of display */
    screen_width = max(screen_width, BUFF_WIDTH * tiles_x);
    screen_height = max(screen_height, BUFF_HEIGHT * tiles_y);

#endif

    /* allocate memory for buffers */
    if( (buff1 = malloc(BUFF_SIZE+6*BUFF_WIDTH)) == NULL) {
	printfe("Can not allocate memory for buffer 1.\n");
	return 1;
    }
    if( (buff2 = malloc(BUFF_SIZE+6*BUFF_WIDTH)) == NULL) {
	printfe("Can not allocate memory for buffer 2.\n");
	return 1;
    }

    /* The buffer has a border of 3 lines on top on bottom. These
       lines are set black before the 'flame' function is called.
       The border is not displayed to the screen
       */
    active_buffer  = buff1 + 3*BUFF_WIDTH;
    passive_buffer = buff2 + 3*BUFF_WIDTH;
	
    /* clear active buffer */
    memset( active_buffer, BUFF_SIZE, 0);

    /* initiaize (load) the palettes (internal and external) */
    if( load_palettes() )
	return 1;
		
    /* load the pcx-files */
    if ( init_pcx() )
	return 1;

    /* check if any palettes in use */
    if ( nr_palettes == 0) {
	printfe("No palettes specified.\n");
	return 1;
    }

    return 0;
}

/*
 * Clean upt display
 * Turns back to text mode in console version
 * Deallocates XImage in X11 versions
 */
int exit_display() {
    exit_graph_mode();
    return 0;
}

/*
 * Allocate the XImage
 */
#ifdef CTH_X11
int alloc_image() {

    /* Check if MIT-SHM is available */
    if ( display_mit_shm) {
	int d1,d2;    Bool d3;		/* dummy variables */
	mit_shm = XShmQueryVersion(cth_display,&d1,&d2,&d3);
    }

    if ( mit_shm) {
	/* create XImage */
	if( (image = XShmCreateImage(cth_display,
				     cth_visual,
				     cth_planes,
				     ZPixmap,	/* format */
				     NULL,		/* data */
				     &shminfo, 
				     screen_width, screen_height)) == NULL) {
	    printfe("Can not create XImage.\n");
	    return 1;
	}

	/* maybe the size changed */
	bytes_per_line = image->bytes_per_line;
	screen_height = image->height;

	/* create Shared Memory */
	if ( (shminfo.shmid = shmget(IPC_PRIVATE, SCREEN_SIZE,
				     IPC_CREAT|0777)) == -1 ) {
	    printfee("Can not create shared memory segment");
	    return 1;
	}
	/* attach Shard Memory */
	if( (shminfo.shmaddr = image->data = shmat(shminfo.shmid, 0,0)) 
	   == (void*)-1) {
	    printfee("Can not attach shared memory segment");
	    return 1;
	}
	    
	/* allow X11 to write */
	shminfo.readOnly = False;

	/* Attach X11 with Shared Memory */
	if( XShmAttach(cth_display, &shminfo) == 0) {
	    printfe("Can not X-attach shared memory segment.\n");
	    return 1;
	}

	/* keep pointer to data */
	display_bitmap = image->data;

    } else {	/* don't use MIT-SHM Extensions */

	/* allocate memory for bitmap without Shard Memory*/
	if( (display_bitmap = (unsigned char *)malloc( SCREEN_SIZE)) == NULL) {
	    printfe("Can not allocate memory for bitmap.\n");
	    return 1;
	}
	/* create Image */
	if( (image = XCreateImage(cth_display, cth_visual, cth_planes, 
				  ZPixmap, 0, display_bitmap, 
				  screen_width, screen_height,
				  XBitmapPad(cth_display),  
				  screen_width*bypp)) == NULL) {
	    printfe("Can not create XImage.\n");
	    return 1;
	}
	
	/* maybe the size changed */
	bytes_per_line = image->bytes_per_line;
	screen_height = image->height;
    }
    return 0;
}
#endif

/*
 * free the XImage
 */
#ifdef CTH_X11
int free_image() {
    if ( mit_shm) {
	/* remove Shared Memory if needed */
	XShmDetach(cth_display, &shminfo);
	XDestroyImage(image);
	shmdt(shminfo.shmaddr);
	shmctl(shminfo.shmid,IPC_RMID,0);
    } else {
	XDestroyImage(image);
    }
    return 0;
}
#endif

/*
 * Resize window
 * Reallocates the XImage
 */
#ifdef CTH_X11
int resize_display(int new_width, int new_height) {

    /* calulcate new real size of display */
    tiles_x = (display_tile_x ? (new_width/BUFF_WIDTH) : 1);
    tiles_y = (display_tile_y ? (new_height/BUFF_HEIGHT):1);

    /* make buffer big enough */
    tiles_x = max(tiles_x,1);
    tiles_y = max(tiles_y,1);

    /* calculate real size of display */
    screen_width = max(screen_width, BUFF_WIDTH * tiles_x);
    screen_height = max(screen_height, BUFF_HEIGHT * tiles_y);

    /* release the old image */
    if( free_image() )
	return 1;

    /* allocate a new image */
    if( alloc_image() )
	return 1;

    return 0;
}
#endif


/*
 * Turn on graph-mode
 *   - turnon graph mode
 *   - set X11 palette
 *   - allocate XImage
 */
int init_graph_mode() {

#ifdef CTH_console
    vga_modeinfo * mi;

    if ( (display_mode >= nr_screen_modes) || (display_mode < 0))
	display_mode = 0;

    /* create screen-context */
    vga_setmode(screen_modes[ display_mode ]);		
    gl_setcontextvga(screen_modes[ display_mode ]);

    gl_getcontext(&display_screen_context);
/*    gl_enablepageflipping( &display_screen_context);  */
    /* direct display only in 320x200 */
    if ( (display_mode != 0) || (bpp != 8))
	display_direct = 0;

    
    mi = vga_getmodeinfo(screen_modes[ display_mode ]);

    /* can't use copybox if screen-mode is planar 256 colors */
    display_copybox = (mi->flags & 4) ? 0 : 1;
     
    /* prepare font */
    gl_expandfont(8,8,255, gl_font8x8, display_font);
    gl_setfont( 8,8, display_font);
    gl_setwritemode( WRITEMODE_MASKED);
    gl_colorfont(8,8, 255, display_font);				

    /* create context for drawing in memory */
    gl_setcontextvgavirtual(screen_modes[ display_mode ]);
    gl_getcontext(&display_virtual_context);
/*    gl_enablepageflipping( &display_virtual_context);  */

    screen_width = WIDTH/BYTESPERPIXEL;		/* from SVGALIB */
    screen_height = HEIGHT;			/* from SVGALIB */
    
    if ( screen_width < BUFF_WIDTH) {
	exit_graph_mode();
	printfe("Screen must not be smaller than buffer.");
	return 1;
    }

    tiles_x = (display_tile_x ? (screen_width/BUFF_WIDTH) : 1);
    tiles_y = (display_tile_y ? (screen_height/BUFF_HEIGHT) : 1);

    tiles_x = max(tiles_x, 1);
    tiles_y = max(tiles_y, 1);
#endif

#ifdef CTH_X11
    alloc_image();
#endif

#ifdef CTH_normalX11
    clear();
    refresh();
#endif

    init_palettes();				/* really set first palette */

    display_start = time(0);
	
    return 0;
}

/*
 * Switch back to text mode
 */
int exit_graph_mode() {
#ifdef CTH_console
    vga_setmode(TEXT);
#endif 
#ifdef CTH_X11
    free_image();
#endif
    exit_palettes();
    return 0;
}


/*
 * Change/Select how buffer is displayed on screen
 * passes only the parameters to 'opt_change' to display the menu
 */
int change_update_screen(int to) {
    update_screen = opt_change(to, screens, nr_screens, update_screen)->data;
#ifndef CTH_saverX11
    update_status();
#endif
    display_clear = 1;
    return 0;
}
int select_screen() {
    int i;
    if( (i=display_selection(screens, nr_screens, 
			     update_screen, "Select Screens\n")) >= 0) 
	change_update_screen(i);
    return i;
}

/* 
 * Display a text on the screen 
 */
/* modified to allow status to change without reseting the counter...  clp */
int display_print(char * text, int line, int reset) {

#ifndef CTH_saverX11
    clear_status();				/* status is no longer disp. */
#ifndef CTH_X11
    if ( text == NULL) {			/* deactivate display */
	if ( display_text_on_screen) {		/* something on display */
	    display_text_on_screen = 0;
	    update_palette();
	    display_clear = 1;
	}
    } else {					/* disp. something new */
	strncpy( display_text, text, MAX_TEXT_DISPLAY);	/* copy text */
	strcat( display_text, "\n");		/* makes display simplier */
	if ( line < 0)				/* determine line on display */
	    display_text_line = rand() % (screen_height - 8);
	else
	    display_text_line = line;

	if ( reset )
	    display_text_on_screen = 255;
	update_palette();
    }
#else
    if ( text == NULL) 
	display_text[0] = '\0';
    else
	strncpy(display_text, text, MAX_TEXT_DISPLAY);
    erase();
    mvprintw(0,0, display_text);
    refresh();
#endif

#endif /* CTH_saverX11 */

    return 0;
}

#ifndef CTH_saverX11
/*
 * Selection-stuff
 */
int sel_line = 0;				/* nr. of line cur. selected */
opt_data * sel_what = NULL;
int sel_nr = 0;
char * sel_title = NULL;

char * name_dest_str(char * name, char * desc) {
    static char str[MAX_NAME_LEN + MAX_DESC_LEN + 1];

    sprintf(str, "%-8s %25s", name, desc);
    str[34] = '\0';
    return str;
}

int display_selection(opt_data * what, int nr, void * old, char * title) {
    int s;

    sel_what = what;
    sel_nr = nr;
    sel_title = title;
    sel_line = opt_number(old,what,nr);

#ifndef CTH_X11
    while( (s=display_do_selection()) == -2);
#else
    s=display_do_selection();
#endif

    return s;
}

#define SEL_SIZE	5
int display_do_selection() {
    char s[256];				/* temporary string */
    char text[20000];				/* text disp. on display */

    int i, key;

    if ( sel_nr <= 0)
	return -2;

    /* create text for selection */
    strcpy(text, sel_title);
    for(i= sel_line - SEL_SIZE; i < sel_line + SEL_SIZE; i++) {
	if ( i < 0)
	    sprintf(s,"\n");
	else if ( i == sel_line)
	    sprintf(s, ">%34s%4s<\n", 
		    name_dest_str(sel_what[i].name, sel_what[i].desc),
		    sel_what[i].use ? "ON" : "OFF");
	else if ( i < sel_nr)
	    sprintf(s, " %34s%4s \n", 
		    name_dest_str(sel_what[i].name, sel_what[i].desc),
		    sel_what[i].use ? "ON" : "OFF");
	else
	    s[0] = '\0';
					
	strcat( text, s);
    }
    
    display_print(text, 0,1);
    
#ifndef CTH_X11	
    /* business as usual */
    display_sound();
#endif
    
    while( (key = getkey() ) != Z_NOKEY  ) {
	switch( key ) {
	case Z_UP:
	    if (sel_line > 0)
		sel_line --;
	    break;
	case Z_DOWN:
	    if (sel_line < (sel_nr-1))
		sel_line ++;
	    break;
	case Z_PGUP:
	    sel_line -= 2*SEL_SIZE-1;
	    if ( sel_line < 0)
		sel_line = 0;
	    break;
	case Z_PGDN:
	    sel_line += 2*SEL_SIZE+1;
	    if ( sel_line >= (sel_nr-1))
		sel_line = sel_nr -1;
	    break;
	case Z_HOME:
	    sel_line = 0;
	    break;
	case Z_END:
	    sel_line = sel_nr -1;
	    break;
	case Z_SPACE:
	    sel_what[sel_line].use = 1 - sel_what[sel_line].use;
	    break;
	case Z_ENTER:
	    sel_what[sel_line].use = 1;
	    display_print(NULL, 0,1);
	    sel_nr = 0;
	    return sel_line;
/*	default: */
	case Z_ESC:
	    display_print(NULL, 0,1);
	    sel_nr = 0;
	    return -1;
	}
    }
    return -2;
}
#else
int display_selection(opt_data * what, int nr, void * old, char * title) {
    return opt_number(old,what,nr);;
}
#endif /* CTH_saverX11 */    	

/*****************************************************************************/

/*
 * exchange active and passive buffer
 */
void swap_buffers() {
    char * tmp;
    tmp = active_buffer; active_buffer = passive_buffer; passive_buffer = tmp;
}

/*
 * bring the current active text to the screen
 */
void show_text() {

#ifndef CTH_X11
    if ( ! display_text_on_screen ) 
	return;
		
    display_text_on_screen --;			/* decrease counter */

    /* keep on screen if status */
    if ( status_active() && (display_text_on_screen <= 0)) {
	display_text_on_screen = 255;
    }
    /* check, if text should be removed */
    if ( display_text_on_screen <= 0) {	
	display_text_on_screen = 0;
	update_palette();				/* reset palette */
	display_clear = 1;
    } else {
	/* bring text to screen (line by line) */
	char * text = display_text;
	char * ntext = strchr(text,'\n');
	int line = display_text_line;
	while ( ntext != NULL) {
	    gl_writen( 0,line, min(ntext-text, screen_width/8), text);
	    line += 8;
	    if ( line >= screen_height)
		line = 0;
	    text = ntext+1;
	    ntext = strchr(text, '\n');
	}
    }
#endif
}

/*****************************************************************************
 some useful definitions
*/

#define SB_HEIGHT	(min(BUFF_HEIGHT, screen_height))

#define SCREEN_OFFSET_X	(max(bypp*((screen_width-BUFF_WIDTH*tile_x*size_x)/2),0))
#define SCREEN_OFFSET_Y	(max(((screen_height-SB_HEIGHT *tile_y*size_y) / 2),0))
#define SCREEN_OFFSET	(bytes_per_line * SCREEN_OFFSET_Y + SCREEN_OFFSET_X)

#define SB_HEIGHT2	(SB_HEIGHT/2)
#define SB_SIZE2	(SB_HEIGHT2 * BUFF_WIDTH)

#define BUFF_WIDTH2	(BUFF_WIDTH/2)
#define screen_width2	(screen_width/2)

/*
 * clear border around display if necessary 
 */
int clear_border(int tile_x,int tile_y, int size_x,int size_y) {
#ifdef CTH_console
    if ( display_text_on_screen || display_clear ) {
	gl_fillbox(0,0, 
		   bytes_per_line, SCREEN_OFFSET_Y,
		   0);
	gl_fillbox(0,SCREEN_OFFSET_Y,	
		   SCREEN_OFFSET_X, SB_HEIGHT*tile_y*size_y,	
		   0);
	gl_fillbox(bytes_per_line-SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
		   SCREEN_OFFSET_X, SB_HEIGHT*tile_y*size_y,
		   0);
	gl_fillbox(0,screen_height-SCREEN_OFFSET_Y,
		   bytes_per_line, SCREEN_OFFSET_Y,			
		   0);

	display_clear = 0;

	return 1;
    }
#endif
#ifdef CTH_X11
    int i;
    if ( display_clear ) {
	/* upper border */
	memset(display_bitmap, 0, bytes_per_line*SCREEN_OFFSET_Y);
	/* left & right border */
	for(i=SCREEN_OFFSET_Y; i < screen_height-SCREEN_OFFSET_Y; i++) {
	    memset(display_bitmap + i*bytes_per_line, 0, SCREEN_OFFSET_X);
	    memset(display_bitmap + i*bytes_per_line 
		   + bytes_per_line - SCREEN_OFFSET_X, 0, SCREEN_OFFSET_X);
	}
	/* lower border */
	memset(display_bitmap+(screen_height-SCREEN_OFFSET_Y)*bytes_per_line, 
	       0,bytes_per_line * SCREEN_OFFSET_Y );

	display_clear = 0;

	return 1;
    }
#endif
    return 0;
}

/*
 * tile buffer to screen
 */
void tile_buffer(int tile_x,int tile_y, int size_x,int size_y) {
#ifdef CTH_console
    int i;

    for(i=1; i < tile_x; i++)
    	gl_copybox( SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
    		    BUFF_WIDTH*size_x, SB_HEIGHT*size_y,
    		    SCREEN_OFFSET_X + i*BUFF_WIDTH*size_x, SCREEN_OFFSET_Y);
    		    
    for(i=1; i < tile_y; i++)
    	gl_copybox( SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
    		    BUFF_WIDTH*tile_x*size_x, SB_HEIGHT*size_y,
    		    SCREEN_OFFSET_X, SCREEN_OFFSET_Y + i*SB_HEIGHT*size_y);
    		    
#endif 
#ifdef CTH_X11
    int i,j;
    
    for(i=1; i < tile_x; i++)
	for(j=0; j < BUFF_HEIGHT*size_y; j++)
	    memcpy(display_mem + i*bypp*BUFF_WIDTH*size_x + j*bytes_per_line,
		   display_mem + j*bytes_per_line,
		   bypp*BUFF_WIDTH*size_x);
    
    for(i=1; i < tile_y; i++)
	memcpy(display_mem + i*bytes_per_line*SB_HEIGHT*size_y,
	       display_mem,
	       size_y*SB_HEIGHT*bytes_per_line);
    
#endif
}

/*
 * calculate information for tile and offset 
 */
void get_tile_size(int * tile_x, int * tile_y, int * size_x, int * size_y) {

    *size_x = screen_sizes[ opt_number(update_screen, screens, nr_screens)][0];
    *tile_x = max(tiles_x / *size_x, 1);
	
    *size_y = screen_sizes[ opt_number(update_screen, screens, nr_screens)][1];
    *tile_y = max(tiles_y / *size_y, 1);

}

#ifdef CTH_console
    /* draw into buffer used by 'libvgagl' */
    #define display_base VBUF
#endif
#ifdef CTH_X11
    /* draw into buffer allocated in 'alloc_image' */
    #define display_base display_bitmap
#endif

/*
 * Bring the active Buffer to screen.
 */
int display() {
    int tile_x, tile_y, size_x, size_y;
    int clear;
    
    bytes_per_line = bypp * screen_width;
    
#ifdef CTH_console
    if( display_direct )
	gl_setcontext( &display_screen_context);
    else
	gl_setcontext( &display_virtual_context);
#endif

    do {
	get_tile_size(&tile_x, &tile_y, &size_x, &size_y);
	display_mem = display_base + SCREEN_OFFSET;
    } while( update_screen() );			/* draw the buffer */

    clear = clear_border(tile_x,tile_y, size_x,size_y);	 
    tile_buffer(tile_x,tile_y, size_x,size_y);	
    show_text();				/* bring active text to scrn */

#ifdef CTH_console
    if( display_syncwait )
	vga_waitretrace();

    /* copy virtual screen to physical screen */
    if( !display_direct) {
	if( clear || !display_copybox)		/* draw full screen */
	    gl_copyscreen( &display_screen_context);
	else {					/* or only a part */
	    gl_setcontext( &display_screen_context);
	    gl_copyboxfromcontext( &display_virtual_context,
				  SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
				  BUFF_WIDTH * tile_x * size_x, 
				  SB_HEIGHT * tile_y * size_y,
				  SCREEN_OFFSET_X, SCREEN_OFFSET_Y);
	}
    }
#endif
#ifdef CTH_X11
    {
	int ss_offset_x = max((window_width - screen_width) >> 1, 0);
	int ss_offset_y = max((window_height - screen_height) >> 1, 0);

	if(clear)		/* draw full screen */
	    if ( mit_shm) {
		XShmPutImage(cth_display, cth_window, cth_gc, image,0,0,
			     ss_offset_x,  ss_offset_y, 
			     screen_width, screen_height, 
			     0);
	    } else {
		XPutImage(cth_display, cth_window, cth_gc, image, 0,0,
			  ss_offset_x, ss_offset_y,
			  screen_width, screen_height);
	    }
	else			/* or only a part */
	    if ( mit_shm) {
		XShmPutImage(cth_display, cth_window, cth_gc, image,
			     SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
			     ss_offset_x + SCREEN_OFFSET_X,
			     ss_offset_y + SCREEN_OFFSET_Y,
			     BUFF_WIDTH * tile_x * size_x,
			     SB_HEIGHT * tile_y * size_y, 
			     0);
	    } else {
		XPutImage(cth_display, cth_window, cth_gc, image, 
			     SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
			     ss_offset_x + SCREEN_OFFSET_X,
			     ss_offset_y + SCREEN_OFFSET_Y,
			     BUFF_WIDTH * tile_x * size_x,
			     SB_HEIGHT * tile_y * size_y);
	    }
    

	if ( display_syncwait )
	    XSync(cth_display, 0);
    }
#endif

    swap_buffers();
    display_frames ++;
    
    return 0;
}

/*****************************************************************************
 * Helping macros and fuctions for the Screen-functions
 ****************************************************************************/

#define memeq(dest,src)	    *(dest)=*(src)
#define tbleq1(dest,src)    *((unsigned char*)(dest))  = bitmap_colors[*(src)] 
#define tbleq2(dest,src)    *((unsigned short*)(dest)) = bitmap_colors[*(src)] 
#define tbleq4(dest,src)    *((unsigned int*)(dest))   = bitmap_colors[*(src)] 
								
inline void tblcpy1(unsigned char * dst, unsigned char * src, int n) {
    for(;n>0;n--) {
	tbleq1(dst,src);
	src ++; dst ++;
    }
}
inline void tblcpy2(unsigned char * dst, unsigned char * src, int n) {
    for(;n>0;n--) {
	tbleq2(dst,src);
	src ++; dst +=2;
    }
}
inline void tblcpy4(unsigned char * dst, unsigned char * src, int n) {
    for(;n>0;n--) {
	tbleq4(dst,src);
	src ++; dst +=4;
    }
}			    

#ifdef CTH_console
    #define SCRN(scrn)	{ scrn_##scrn(memcpy, memeq, 1) }
#endif
#ifdef CTH_X11
    #define SCRN(scrn)							\
        switch(draw_mode) {						\
        case DM_direct:		scrn_##scrn(memcpy,  memeq,  1 ) break;	\
	case DM_mapped1:	scrn_##scrn(tblcpy1, tbleq1, 1 ) break;	\
	case DM_mapped2:	scrn_##scrn(tblcpy2, tbleq2, 2 ) break;	\
	case DM_mapped4:	scrn_##scrn(tblcpy4, tbleq4, 4 ) break;	\
    }
#endif

/*****************************************************************************
 * Screen-functions
 ****************************************************************************/

#define scrn_up(cpy,eq,bytes_per_pixel)			\
    for(i=SB_HEIGHT; i != 0; i--) {			\
	##cpy( scrn, tmp, BUFF_WIDTH);			\
	scrn += bytes_per_line;				\
	tmp += BUFF_WIDTH;				\
    }
int screen_up() {
    int i;
    unsigned char * scrn = display_mem;
    unsigned char * tmp = active_buffer;
		
    SCRN(up)
    return 0;
}


#define scrn_down(cpy,eq,bytes_per_pixel)		\
    for(i=SB_HEIGHT; i != 0; i--) {			\
	##cpy( scrn, tmp, BUFF_WIDTH);			\
	tmp += BUFF_WIDTH;				\
	scrn -= bytes_per_line;				\
    }
int screen_down() {
    int i;
    unsigned char * scrn = display_mem + bytes_per_line * (SB_HEIGHT-1);
    unsigned char * tmp = active_buffer;
    SCRN(down)
    return 0;
}


#define scrn_2hor_a(cpy,eq,bytes_per_pixel)		\
    for(i=SB_HEIGHT2; i != 0; i--) {			\
	##cpy( scrn, tmp, BUFF_WIDTH);			\
	scrn += bytes_per_line;				\
	tmp += BUFF_WIDTH;				\
    }
#define scrn_2hor_b(cpy,eq,bytes_per_pixel)		\
    for(i=SB_HEIGHT2; i != 0; i--) {			\
	##cpy( scrn, tmp, BUFF_WIDTH);			\
	scrn -= bytes_per_line;				\
	tmp += BUFF_WIDTH;				\
    }
int screen_2hor() {
    unsigned char * tmp, * scrn;
    int i;
	
    /* lower half of buffer maps to upper half of screen */
    tmp = active_buffer + SB_SIZE2;
    scrn = display_mem;
    SCRN(2hor_a)
				
    /* lower half of buffer get turned around to lower half of screen*/
    tmp = active_buffer + SB_SIZE2;
    scrn = display_mem + bytes_per_line*(SB_HEIGHT-1);
    SCRN(2hor_b)

    return 0;
}

		
int screen_r2hor() {
    unsigned char * tmp, * scrn;
    int i;
	
    /* lower half of buffer maps to lower half of screen */
    tmp = active_buffer + SB_SIZE2;
    scrn = display_mem + SB_HEIGHT2*bytes_per_line;
    SCRN(2hor_a)
		
    /* lower half of buffer get turned around to upper half of screen*/
    tmp = active_buffer + SB_SIZE2;
    scrn = display_mem + bytes_per_line*(SB_HEIGHT2-1);
    SCRN(2hor_b)
    
    return 0;
}


#define scrn_4hor_a(cpy,eq,bytes_per_pixel)			\
    for(y=SB_HEIGHT2; y != 0; y--) {				\
								\
	/* left half */						\
	##cpy( scrn, tmp, BUFF_WIDTH2);				\
	scrn += BUFF_WIDTH2*bytes_per_pixel;			\
	tmp += BUFF_WIDTH2;					\
								\
	/* right half */					\
	for(x=BUFF_WIDTH2; x != 0; x--)	{			\
	    ##eq(scrn, tmp);					\
	    scrn += bytes_per_pixel;				\
	    tmp --;						\
	}							\
	tmp += BUFF_WIDTH;					\
	scrn += bytes_per_line - BUFF_WIDTH*bytes_per_pixel;	\
    }
#define scrn_4hor_b(cpy,eq,bytes_per_pixel)			\
    for(y= SB_HEIGHT2; y != 0; y--) {				\
								\
	/* left half */						\
	##cpy( scrn, tmp, BUFF_WIDTH2);				\
	scrn += BUFF_WIDTH2*bytes_per_pixel;			\
	tmp += BUFF_WIDTH2;					\
								\
	/* right half */					\
	for(x=BUFF_WIDTH2; x != 0; x--)	{			\
	    ##eq(scrn, tmp);					\
	    scrn += bytes_per_pixel;				\
	    tmp --;						\
	}							\
	tmp -= BUFF_WIDTH;					\
	scrn += bytes_per_line - BUFF_WIDTH*bytes_per_pixel;	\
    }
int screen_4hor() {
    unsigned char * tmp, * scrn;
    int x,y;
	
    /* upper half of screen */	
    tmp = active_buffer + SB_SIZE2;
    scrn = display_mem;
    SCRN(4hor_a)

    /* lower half of screen */
    tmp = active_buffer + BUFF_WIDTH*SB_HEIGHT - BUFF_WIDTH;
    scrn = display_mem + bytes_per_line*SB_HEIGHT2;
    SCRN(4hor_b)

    return 0;
}


#define scrn_2verd(cpy,eq,bytes_per_pixel)		\
    for(y=SB_HEIGHT; y != 0; y--) {			\
	for(x=BUFF_WIDTH2; x != 0; x--) {		\
	    ##eq(scrn, tmp);				\
            scrn += bytes_per_pixel;			\
	    tmp += BUFF_WIDTH;				\
	}						\
	for(x=BUFF_WIDTH2; x != 0; x--) {		\
	    ##eq(scrn, tmp);				\
            scrn += bytes_per_pixel;			\
	    tmp -= BUFF_WIDTH;				\
	}						\
	tmp ++;						\
	scrn += bytes_per_line - bytes_per_pixel*BUFF_WIDTH;		\
    }

int screen_2verd() {
    if( BUFF_WIDTH2 <= BUFF_HEIGHT) {
	int x,y;
	unsigned char * tmp = active_buffer;
	unsigned char * scrn = display_mem;
	SCRN(2verd)
    } else {
	change_update_screen(CHANGE_NEXT);
	return 1;
    }
    return 0;
}


#define scrn_r2verd(cpy,eq,bytes_per_pixel)			\
    for(y=SB_HEIGHT; y != 0; y--) {				\
	for(x=BUFF_WIDTH2; x != 0; x--) {			\
	    ##eq(scrn, tmp);					\
	    scrn -= bytes_per_pixel;				\
	    tmp += BUFF_WIDTH;					\
	}							\
	for(x=BUFF_WIDTH2; x != 0; x--) {			\
	    ##eq(scrn, tmp);					\
	    scrn -= bytes_per_pixel;				\
	    tmp -= BUFF_WIDTH;					\
	}							\
	tmp ++;							\
	scrn -= bytes_per_line - bytes_per_pixel*BUFF_WIDTH;	\
    }
int screen_r2verd() {
    if( BUFF_WIDTH2 <= BUFF_HEIGHT) {
	int x,y;
	unsigned char * tmp = active_buffer;
	unsigned char * scrn = display_mem + 
	    bytes_per_line*(SB_HEIGHT-1) + bypp*(BUFF_WIDTH-1);
	SCRN(r2verd)
    } else {
	change_update_screen(CHANGE_NEXT);
	return 1;
    }
    return 0;
}


#define scrn_4kal_a(cpy,eq,bytes_per_pixel)			\
    for(y=SB_HEIGHT2; y != 0; y--) {				\
	for(x=BUFF_WIDTH2; x != 0; x--) {			\
	    ##eq(scrn, tmp);					\
	    scrn += bytes_per_pixel;				\
	    tmp += BUFF_WIDTH;					\
	}							\
	for(x=BUFF_WIDTH2; x != 0; x--) {			\
	    ##eq(scrn, tmp);					\
	    scrn += bytes_per_pixel;				\
	    tmp -= BUFF_WIDTH;					\
	}							\
	tmp ++;							\
	scrn += bytes_per_line - bytes_per_pixel*BUFF_WIDTH;	\
    }
#define scrn_4kal_b(cpy,eq,bytes_per_pixel)			\
    for(y=SB_HEIGHT2; y != 0; y--) {				\
	for(x=BUFF_WIDTH2; x != 0; x--) {			\
	    ##eq(scrn, tmp);					\
	    scrn -= bytes_per_pixel;				\
	    tmp += BUFF_WIDTH;					\
	}							\
	for(x=BUFF_WIDTH2; x != 0; x--) {			\
	    ##eq(scrn, tmp);					\
	    scrn -= bytes_per_pixel;				\
	    tmp -= BUFF_WIDTH;					\
	}							\
	tmp ++;							\
	scrn -= bytes_per_line - bytes_per_pixel*BUFF_WIDTH;	\
    }
int screen_4kal() {
    if( BUFF_WIDTH2 <= BUFF_HEIGHT) {
	unsigned char * tmp, * scrn;
	int x,y;
	
	tmp = active_buffer;
	scrn = display_mem;
	/* upper half */
	SCRN(4kal_a)

	tmp = active_buffer;
	scrn = display_mem + bytes_per_line * (SB_HEIGHT-1) + 
	    bypp*(BUFF_WIDTH-1);
	/* lower half */
	SCRN(4kal_b)
    } else {
	change_update_screen(CHANGE_NEXT);
	return 1;
    }
    return 0;
}


/*
 * big screen functions
 */
#define scrn_mirrorH(cpy,eq,bytes_per_pixel)			\
    for(i=SB_HEIGHT; i != 0; i--) {				\
	/* left part maps directly */				\
	##cpy(dst1, src, BUFF_WIDTH);				\
	    							\
	/* right part gets turned around */			\
	for(j=BUFF_WIDTH; j != 0; j--) {			\
	    ##eq(dst2, src);					\
	    src ++;						\
	    dst2 -= bytes_per_pixel;				\
	}							\
	/* move pointer to next line */				\
	dst1 += bytes_per_line;					\
	dst2 += bytes_per_line + bytes_per_pixel*BUFF_WIDTH;	\
    }
int screen_mirrorH() {
    if( 2*BUFF_WIDTH <= screen_width) {
	int i,j;
	unsigned char * src = active_buffer;
	unsigned char * dst1 = display_mem;
	unsigned char * dst2 = display_mem + 2*bypp*BUFF_WIDTH - bypp;
	SCRN(mirrorH)
    } else {
	change_update_screen(CHANGE_NEXT);
	return 1;
    }
    return 0;
}


#define scrn_mirrorHV(cpy,eq,bytes_per_pixel)			\
    for(i=SB_HEIGHT; i != 0; i--) {				\
	/* upper left maps direct */				\
	##cpy(dst1, src, BUFF_WIDTH);				\
	/* lower left maps direct */				\
	##cpy(dst3, src, BUFF_WIDTH);				\
	/* upper & lower right gets turned around */		\
	for(j=BUFF_WIDTH; j != 0; j--) {			\
	    ##eq(dst2, src);					\
	    ##eq(dst4, src);					\
	    src ++;						\
	    dst2 -= bytes_per_pixel;				\
	    dst4 -= bytes_per_pixel;				\
	}							\
	/* move pointer to next line */				\
	dst1 +=  bytes_per_line;				\
	dst2 +=  bytes_per_line + bytes_per_pixel*BUFF_WIDTH;	\
	dst3 += -bytes_per_line;				\
	dst4 += -bytes_per_line + bytes_per_pixel*BUFF_WIDTH;	\
    }
int screen_mirrorHV() {
    if( (2*BUFF_WIDTH <= screen_width) && (2*SB_HEIGHT <= screen_height) ) {
	int i,j;
	unsigned char * src = active_buffer;
	unsigned char * dst1 = display_mem;
	unsigned char * dst2 = display_mem + 2*bypp*BUFF_WIDTH - bypp;
	unsigned char * dst3 = display_mem + 2*SB_HEIGHT*bytes_per_line 
	    - bytes_per_line;
	unsigned char * dst4 = display_mem + 2*SB_HEIGHT*bytes_per_line 
	    + 2*bypp*BUFF_WIDTH - bypp - bytes_per_line;
	SCRN(mirrorHV)
    } else {
	change_update_screen(CHANGE_NEXT);
	return 1;
    }
    return 0;
}



#define scrn_mirrorV(cpy,eq,bytes_per_pixel)			\
    for(i=SB_HEIGHT; i != 0; i--) {				\
	/* do 1 line */						\
	##cpy(dst1, src, BUFF_WIDTH);				\
	##cpy(dst2, src, BUFF_WIDTH);				\
	/* move pointer to next line */				\
	src += BUFF_WIDTH;					\
	dst1 += bytes_per_line;					\
	dst2 -= bytes_per_line;					\
    }
int screen_mirrorV() {
    if( 2*SB_HEIGHT <= screen_height ) {
	int i;
	unsigned char * src = active_buffer;
	unsigned char * dst1 = display_mem;
	unsigned char * dst2 = display_mem + 2*SB_HEIGHT*bytes_per_line 
	    - bytes_per_line;
	SCRN(mirrorV)
    } else {
	change_update_screen(CHANGE_NEXT);
	return 1;
    }
    return 0;
}








