/*
 * main.c
 * Copyright (c) 1994-1998 by Christopher Heng. All rights reserved.
 * You may not remove or alter the copyright notice and/or the conditions for
 * use and distribution. See COPYING for additional conditions for use and
 * distribution. (The file COPYING contains the GNU General Public License
 * version 2.)
 *
 * Program to undigest a file comprising one or more digest files into
 * separate digests.
 *
 * Usage: splitdigest [options] [file...]
 * Options include
 *	-C <config>	Use configuration file specified.
 *	-d <outdir> Output directory to place files (will be created
 *			if it does not exist).
 *	-h		Display usage.
 *	-l		Suppress the removal of the Content-Length: header.
 *	-o <outfile> Write all output to outfile. -o- write to stdout.
 *	-v		Verbose.
 *
 * Obsolete options (will be removed in future releases, if any).
 *	-c		Compress output file.
 *	-t		Leave output file as uncompressed text file (default)
 *
 *
 * Set your editor to have tab stops every four spaces to preserve
 * your sanity while reading this file.
 *
 * $Id: main.c,v 2.7 1998/03/30 13:18:26 chris Released $
 */

/*
	Note that this program is only designed to be compiled and linked
	on Unix systems. It will not compile/link/work correctly on
	MSDOS or Win32 systems (or rather, I have not bothered to try to
	get it to work there). The presence of macros like
	MSDOS or WIN32 are due to my updating this program from a Windows 95
	editor, so I find it convenient to have my Windows compiler do a
	syntax check for obvious errors. I did not even attempt to link it
	or try to make it compatible.
*/

#include "sysdep.h"

#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>

#if defined(UNIX)
#include <unistd.h>
#endif

#if defined(MSDOS) || defined(WIN32)
#include <io.h>
#endif

#if defined(HAVE_GETOPT_H)
#include <getopt.h>
#elif defined(NEED_GETOPT_DECLARATION)
	/* note that you'll still need a version of getopt() to link... */
	/* hence this declaration is no use to you if getopt() is not */
	/* in your library. */
extern char * optarg;	/* argument if option has one */
extern int opterr;		/* 0 = don't print err msg, 1 = print */
extern int optind;		/* next argument to process */
extern int optopt ;		/* current option */
extern int getopt (int argc, char * const * argv, const char * optlist );
#endif

#if defined(__WATCOMC__) || defined(_MSC_VER)
#include <direct.h>		/* getcwd(), mkdir(), chdir() */
#elif defined(__BORLANDC__)
#include <dir.h>		/* getcwd(), mkdir(), chdir() */
#endif

#include "cftables.h"
#include "splitdigest.h"
#include "version.h"

/* macros */
#define	OUTPUT_TO_STDOUT_STRING	"-"

/* the following are for easy identification for bug reports and */
/* debugging purposes */
static char verstr[]		= VERSTR;
static char shusagestr[]	= SHUSAGESTR ;

/* global variables */
int compression = DEFTOCOMPRESS ;	/* whether to compress or not */
									/* (obsolete) */
int verbose ;						/* verbosity flag */
int kill_contentlen = 1 ;			/* kill Content-Length lines */
char * pgmname ;					/* program name */
char * configfile = DEFCONFIG;		/* configuration filename */
char * infilename = "stdin" ;
char * newdir = NULL ;				/* directory to extract the files */
char * output_filename ;			/* output filename */
int	pipe_output ;					/* whether we're to output to stdout*/

/* local functions */
static void cleanuponsig ( int sig );
static void cleanuponexit( void );
static int changedir( char * dir );

int main ( int argc, char ** argv )
{

	static char optstring[] = "C:cd:hlo:tvV" ; /* options for getopt() */
	char * currdir ;
	FILE * infile ;
	int i, c ;

	/* form pgmname without path prefix */
	if ((pgmname = strrchr( argv[0], '/' )) == NULL)
		pgmname = argv[0] ;
	else pgmname++;

	/* trap SIGINT so that we can clean up temporary files */
	if (signal( SIGINT, SIG_IGN ) != SIG_IGN) {
		signal( SIGINT, cleanuponsig );
	}
	/* register an atexit function */
	atexit( cleanuponexit );

	while ( (c = getopt(argc, argv, optstring)) != EOF ) {
		switch( c ) {
			case 'C':	/* specify another config file */
				configfile = optarg ;
				break;
			case 'c':	/* compress output file */
				compression = 1; /* obsolete */
				break ;
			case 'd':	/* output directory specified */
				newdir = optarg;
				break ;
			case 'h':	/* usage help requested */
				fprintf( stdout, "%s\n%s", verstr, USAGESTR );
				return EXIT_SUCCESS;	/* quit immediately */
			case 'l':	/* preserve Content-Length header */
				kill_contentlen = 0;
				break ;
			case 'o':	/* specify output file */
				if (!strcmp( optarg, OUTPUT_TO_STDOUT_STRING ))
					pipe_output = 1 ;
				else
					output_filename = optarg ;
				break ;
			case 't':	/* switch off compression (obsolete) */
				compression = 0;
				break ;
			case 'v':	/* verbose mode. At present, we don't do much. */
				fprintf( stderr, "%s\n", verstr );
				verbose++ ;
				break ;
			case 'V':	/* show version number and quit */
				fprintf( stdout, "%s\n", verstr );
				return EXIT_SUCCESS;
			default:	/* bad option */
				fprintf( stderr, "\n%s\n", shusagestr);
				return EXIT_FAILURE;
		}
	}
	/* get current directory and save it */
	for (currdir = NULL, i = 128; currdir == NULL ; i+=128) {
		if ((currdir = getcwd( NULL, i )) == NULL) {
			if (errno == ERANGE)
				continue ;
			else
				goto nomemleft ;
		}
	}

	/* make directory and change to it if applicable */
	if (newdir != NULL) {
		if (verbose) {
			fprintf( stderr, "Changing directory to %s.\n", newdir );
		}
		if (changedir(newdir))
			return EXIT_FAILURE ;
	}
	/* load the configuration tables */
	if (inittables())
		return EXIT_FAILURE ;
	/* if no args assume stdin is the input */
	if (optind >= argc) {
		if (isatty(fileno(stdin))) {
			fprintf( stderr, "%s\n", shusagestr );
			return EXIT_FAILURE ;
		}
		uncatfile( stdin );
	}
	else {	/* invoked uncatfile for each arg */

		/* remove any file with the same name as our output file */
		if (output_filename != NULL)
			unlink( output_filename );

		for (i = optind; i < argc; i++) {
			if (argv[i][0] == '/') {
				if ((infilename = malloc(strlen(argv[i])+1))
					== NULL)
					goto nomemleft ;
				strcpy(infilename, argv[i]);
			}
			else {
			/* we need to make infilename a full path because */
			/* we may have changed directory if the user */
			/* has specified -o xxx */
			    if ((infilename = malloc(
				strlen(currdir)+strlen(argv[i])+2 ))==NULL) {
nomemleft:				
				fprintf( stderr, "%s: out of memory.\n",
					pgmname );
				return EXIT_FAILURE ;
			    }
			    strcpy(infilename, currdir);
			    strcat(infilename, "/");
			    strcat(infilename, argv[i]);
			}
			if (verbose) {
				fprintf( stderr, "Undigesting %s.\n", infilename );
			}
			if ((infile = fopen ( infilename, READMODE ))==NULL) {
				fprintf( stderr, "%s: cannot open %s. "
					"Skipped.\n", pgmname, argv[i] );
				continue;
			}
			uncatfile( infile );
			fclose(infile);
			free(infilename);
		}
	}
	return EXIT_SUCCESS;

}

/* clean up on SIGINT */
#if defined(__BORLANDC__)
#pragma argsused
#endif
#if defined(_MSC_VER)
#pragma warning( disable: 4100 )
#endif
#if defined(__WATCOMC__) && __WATCOMC__ <= 1000
#pragma off(unreferenced)
#endif
static void cleanuponsig ( int sig )
#if defined(__WATCOMC__) && __WATCOMC__ <= 1000
#pragma on( unreferenced )
#endif
{
	exit(EXIT_SIGINT) ;	/* the atexit() function will do the cleaning*/
}
#if defined(_MSC_VER)
#pragma warning( default: 4100 )
#endif

/* atexit cleanup function */
static void cleanuponexit( void )
{
	realcleanup();
}

/* change directory to specified directory. Create if it does not exist */
static int changedir ( char * dir )
{
	struct stat buf ;
#if defined(UNIX)
	int mask, mode ;
#endif

	/* we stat to see if the "dir" string is a directory. If */
	/* it doesn't exist we'll create it. If it is a file */
	/* we'll abort. We'll also abort if the directory cannot */
	/* be created. The directory is created with the default */
	/* umask */

	if (!access(dir, F_OK)) { /* directory/file exists */
		/* now make sure it is a directory */
		if (stat(dir, &buf)) {
			fprintf( stderr, "%s: Error - can't stat %s.\n",
					pgmname, dir );
			return -1 ;
		}
		if (! S_ISDIR(buf.st_mode)) {
			fprintf( stderr, "%s: Error - %s is not a directory.\n",
					pgmname, dir );
			return -1 ;
		}
	}
	else {
#if defined(UNIX)
		/* get umask */
		mask = umask( 0 );
		umask ( mask ); /* restore to original */
		mode = S_IRUSR|S_IWUSR|S_IXUSR|
			((mask&S_IRGRP)?0:S_IRGRP)|
			((mask&S_IWGRP)?0:S_IWGRP)|
			((mask&S_IXGRP)?0:S_IXGRP)|
			((mask&S_IROTH)?0:S_IROTH)|
			((mask&S_IWOTH)?0:S_IWOTH)|
			((mask&S_IXOTH)?0:S_IXOTH);
		if (mkdir(dir, mode)) { /* make dir if it doesn't exist */
#else	/* assume MSDOS or WIN32 */
		if (mkdir( dir )) {
#endif
			/* couldn't make directory. exit with error message */
			fprintf ( stderr, "%s: Error - Unable to make %s.\n",
				pgmname, dir );
			return -1 ;
		}
	}
	/* now change to the directory */
	if (chdir(dir)) {
		fprintf ( stderr, "%s: Error - Unable to change to directory %s\n",
			pgmname, dir );
		return -1 ;
	}
	return 0;
}
