/*
 *  iptables module to match inet_addr_type() of an ip.
 */

#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <net/route.h>

#include <linux/netfilter_ipv4/ipt_addrtype.h>
#include <linux/netfilter_ipv4/ip_tables.h>

MODULE_LICENSE("GPL");

static inline int match_type(u_int32_t addr, u_int16_t mask)
{
	return !!(mask & (1 << inet_addr_type(addr)));
}

static int match(const struct sk_buff *skb, const struct net_device *in,
		 const struct net_device *out, const void *matchinfo,
		 int offset, int *hotdrop)
{
	const struct ipt_addrtype_info *info = matchinfo;
	const struct iphdr *iph = skb->nh.iph;
	int ret = 1;

	if (info->source)
		ret &= match_type(iph->saddr, info->source)^info->invert_source;
	if (info->dest)
		ret &= match_type(iph->daddr, info->dest)^info->invert_dest;
	
	return ret;
}

static int checkentry(const char *tablename, const struct ipt_ip *ip,
		      void *matchinfo, unsigned int matchsize,
		      unsigned int hook_mask)
{
	if (matchsize != IPT_ALIGN(sizeof(struct ipt_addrtype_info))) {
		printk(KERN_ERR "ipt_addrtype: invalid size (%u != %u)\n.",
		       matchsize, IPT_ALIGN(sizeof(struct ipt_addrtype_info)));
		return 0;
	}

	return 1;
}

static struct ipt_match addrtype_match = { 
	.name = "addrtype",
	.match = match,
	.checkentry = checkentry,
	.me = THIS_MODULE
};

static int __init init(void)
{
	return ipt_register_match(&addrtype_match);
}

static void __exit fini(void)
{
	ipt_unregister_match(&addrtype_match);

}

module_init(init);
module_exit(fini);
