#ifndef _NFNETLINK_H
#define _NFNETLINK_H
#include <linux/types.h>

/* Generic structure for encapsulation optional netfilter information.
 * It is reminiscent of sockaddr, but with sa_family replaced
 * with attribute type. 
 * ! This should someday be put somewhere generic as now rtnetlink and
 * ! nfnetlink use the same attributes methods. - J. Schulist.
 */

struct nfattr
{
	unsigned short  nfa_len;
	unsigned short  nfa_type;
};

#define NFA_ALIGNTO     4
#define NFA_ALIGN(len)	(((len) + NFA_ALIGNTO - 1) & ~(NFA_ALIGNTO - 1))
#define NFA_OK(nfa,len)	((len) > 0 && (nfa)->nfa_len >= sizeof(struct nfattr) \
	&& (nfa)->nfa_len <= (len))
#define NFA_NEXT(nfa,attrlen)	((attrlen) -= NFA_ALIGN((nfa)->nfa_len), \
	(struct nfattr *)(((char *)(nfa)) + NFA_ALIGN((nfa)->nfa_len)))
#define NFA_LENGTH(len)	(NFA_ALIGN(sizeof(struct nfattr)) + (len))
#define NFA_SPACE(len)	NFA_ALIGN(NFA_LENGTH(len))
#define NFA_DATA(nfa)   ((void *)(((char *)(nfa)) + NFA_LENGTH(0)))
#define NFA_PAYLOAD(nfa) ((int)((nfa)->nfa_len) - NFA_LENGTH(0))

/* General form of address family dependent message.
 */
struct nfgenmsg {
	unsigned char nfgen_family;
};

#define NFM_NFA(n)      ((struct nfattr *)(((char *)(n)) \
        + NLMSG_ALIGN(sizeof(struct nfgenmsg))))
#define NFM_PAYLOAD(n)  NLMSG_PAYLOAD(n, sizeof(struct nfgenmsg))


#ifndef NETLINK_NETFILTER
#define NETLINK_NETFILTER	10
#endif

/* netfilter netlink message types are split in two pieces:
 * 8 bit subsystem, 8bit operation.
 */

#define NFNL_SUBSYS_ID(x)	((x & 0xff00) >> 8)
#define NFNL_MSG_TYPE(x)	(x & 0x00ff)

enum nfnl_subsys_id {
	NFNL_SUBSYS_NONE = 0,
	NFNL_SUBSYS_CTNETLINK,
	NFNL_SUBSYS_CTNETLINK_EXP,
	NFNL_SUBSYS_IPTNETLINK,
	NFNL_SUBSYS_QUEUE,
	NFNL_SUBSYS_ULOG,
	NFNL_SUBSYS_COUNT,
};

#ifdef __KERNEL__

#include <linux/capability.h>

struct nfnl_callback
{
	kernel_cap_t cap_required; /* capabilities required for this msg */
	int (*call)(struct sock *nl, struct sk_buff *skb, 
		struct nlmsghdr *nlh, int *errp);
};

struct nfnetlink_subsystem
{
	/* Internal use. */
	struct list_head list;
	
	const char *name;
	__u8 subsys_id;		/* nfnetlink subsystem ID */
	__u8 cb_count;		/* number of callbacks */
	u_int32_t attr_count;	/* number of nfattr's */
	struct nfnl_callback cb[0]; /* callback for individual types */
};

extern void __nfa_fill(struct sk_buff *skb, int attrtype,
        int attrlen, const void *data);
#define NFA_PUT(skb, attrtype, attrlen, data) \
({ if (skb_tailroom(skb) < (int)NFA_SPACE(attrlen)) goto nfattr_failure; \
   __nfa_fill(skb, attrtype, attrlen, data); })

extern struct semaphore nfnl_sem;
#define nfnl_exlock()		do { } while(0)
#define nfnl_exunlock()		do { } while(0)
#define nfnl_exlock_nowait()	(0)

#define nfnl_shlock()		down(&nfnl_sem)
#define nfnl_shlock_nowait()	down_trylock(&nfnl_sem)

#ifndef CONFIG_NF_NETLINK
#define nfnl_shunlock()		up(&nfnl_sem)
#else
#define nfnl_shunlock()		do { up(&nfnl_sem); \
                             		if(nfnl && nfnl->receive_queue.qlen) \
                                     		nfnl->data_ready(nfnl, 0); \
                        	} while(0)
#endif

extern void nfnl_lock(void);
extern void nfnl_unlock(void);

extern struct nfnetlink_subsystem *nfnetlink_subsys_alloc(int cb_count);
extern int nfnetlink_subsys_register(struct nfnetlink_subsystem *n);
extern int nfnetlink_subsys_unregister(struct nfnetlink_subsystem *n);

extern int nfnetlink_check_attributes(struct nfnetlink_subsystem *subsys,
				      struct nlmsghdr *nlh, 
				      struct nfattr *cda[]);
extern int nfattr_parse(struct nfattr *tb[], int maxattr, 
			struct nfattr *nfa, int len);
extern int nfnetlink_send(struct sk_buff *skb, u32 pid, unsigned group, 
			  int echo);

#endif	/* __KERNEL__ */
#endif	/* _NFNETLINK_H */
