/* Netfilter messages via netlink socket. Allows for user space
 * protocol helpers and general trouble making from userspace.
 *
 * (C) 2001 by Jay Schulist <jschlst@samba.org>,
 * (C) 2002 by Harald Welte <laforge@gnumonks.org>
 *
 * Initial netfilter messages via netlink development funded and
 * generally made possible by Network Robots, Inc. (www.networkrobots.com)
 *
 * Further development of this code funded by Astaro AG (http://www.astaro.com)
 *
 * This software may be used and distributed according to the terms
 * of the GNU General Public License, incorporated herein by reference.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include <linux/fcntl.h>
#include <linux/skbuff.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <net/sock.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/list.h>

#include <linux/netfilter.h>
#include <linux/netlink.h>
#include <linux/nfnetlink.h>

MODULE_LICENSE("GPL");

static char __initdata nfversion[] = "0.12";

#if 1
static int nf_debug_level = 1;
#define nf_debug(level, format, arg...)					\
do {									\
	if (nf_debug_level > level)					\
		printk(KERN_DEBUG "%s: " format, __FUNCTION__, ## arg);	\
} while(0)
#else
#define nf_debug(level, format, arg...)
#endif

static struct sock *nfnl = NULL;
static LIST_HEAD(subsys_list);
static struct nfnetlink_subsystem *subsys_table[NFNL_SUBSYS_COUNT];
DECLARE_MUTEX(nfnl_sem);

void nfnl_lock(void)
{
	nfnl_shlock();
	nfnl_exlock();
}

void nfnl_unlock(void)
{
	nfnl_exunlock();
	nfnl_shunlock();
}

struct nfnetlink_subsystem *nfnetlink_subsys_alloc(int cb_count)
{
	int size;
	struct nfnetlink_subsystem *ss;

	size = sizeof(struct nfnetlink_subsystem)
		+ (cb_count * sizeof(struct nfnl_callback));

	ss = kmalloc(size, GFP_KERNEL);
	if (!ss)
		return NULL;
	memset(ss, 0, size);

	return ss;
}

int nfnetlink_subsys_register(struct nfnetlink_subsystem *n)
{
	MOD_INC_USE_COUNT;

	nf_debug(0, "registering subsystem ID %u\n", n->subsys_id);

	nfnl_lock();
	list_add(&n->list, &subsys_list);
	subsys_table[n->subsys_id] = n;
	nfnl_unlock();

	return 0;
}

int nfnetlink_subsys_unregister(struct nfnetlink_subsystem *n)
{
	nf_debug(0, "unregistering subsystem ID %u\n", n->subsys_id);

	nfnl_lock();
	subsys_table[n->subsys_id] = NULL;
	list_del(&n->list);
	nfnl_unlock();

	MOD_DEC_USE_COUNT;

	return 0;
}

struct nfnl_callback *nfnetlink_find_client(u_int16_t nlmsg_type)
{
	struct nfnetlink_subsystem *ss;
	u_int8_t subsys_id = NFNL_SUBSYS_ID(nlmsg_type);
	u_int8_t type = NFNL_MSG_TYPE(nlmsg_type);

	if (subsys_id >= NFNL_SUBSYS_COUNT
	    || subsys_table[subsys_id] == NULL)
		return NULL;

	ss = subsys_table[subsys_id];

	if (type >= ss->cb_count) {
		nf_debug(0, "msgtype %u >= %u, returning\n", type, 
			 ss->cb_count);
		return NULL;
	}

	return &ss->cb[type];
}

void __nfa_fill(struct sk_buff *skb, int attrtype, int attrlen,
		const void *data)
{
	struct nfattr *nfa;
	int size = NFA_LENGTH(attrlen);

	nfa = (struct nfattr *)skb_put(skb, NFA_ALIGN(size));
	nfa->nfa_type = attrtype;
	nfa->nfa_len  = size;
	memcpy(NFA_DATA(nfa), data, attrlen);
}

int nfattr_parse(struct nfattr *tb[], int maxattr, struct nfattr *nfa, int len)
{
	memset(tb, 0, sizeof(struct nfattr *) * maxattr);

	while (NFA_OK(nfa, len)) {
		unsigned flavor = nfa->nfa_type;
		if (flavor && flavor <= maxattr)
			tb[flavor-1] = nfa;
		nfa = NFA_NEXT(nfa, len);
	}

	return 0;
}

/**
 * nfnetlink_check_attributes - check and parse nfnetlink attributes
 *
 * subsys: nfnl subsystem for which this message is to be parsed
 * nlmsghdr: netlink message to be checked/parsed
 * cda: array of pointers, needs to be at least subsys->attr_count big
 *
 */
int
nfnetlink_check_attributes(struct nfnetlink_subsystem *subsys,
			   struct nlmsghdr *nlh, struct nfattr *cda[])
{
	int min_len;

	memset(cda, 0, sizeof(struct nfattr *) * subsys->attr_count);

	/* check attribute lengths. */
	min_len = sizeof(struct nfgenmsg);
	if (nlh->nlmsg_len < min_len)
		return -EINVAL;

	if (nlh->nlmsg_len > min_len) {
		struct nfattr *attr = NFM_NFA(NLMSG_DATA(nlh));
		int attrlen = nlh->nlmsg_len - NLMSG_ALIGN(min_len);

		while (NFA_OK(attr, attrlen)) {
			unsigned flavor = attr->nfa_type;
			if (flavor) {
				if (flavor > subsys->attr_count)
					return -EINVAL;
				cda[flavor - 1] = attr;
			}
			attr = NFA_NEXT(attr, attrlen);
		}
	} else
		return -EINVAL;

        return 0;
}

int nfnetlink_send(struct sk_buff *skb, u32 pid, unsigned group, int echo)
{
	int allocation = in_interrupt() ? GFP_ATOMIC : GFP_KERNEL;
	int err = 0;

	NETLINK_CB(skb).dst_groups = group;
	if (echo)
		atomic_inc(&skb->users);
	netlink_broadcast(nfnl, skb, pid, group, allocation);
	if (echo)
		err = netlink_unicast(nfnl, skb, pid, MSG_DONTWAIT);

	return err;
}

/* Process one complete nfnetlink message. */
static inline int nfnetlink_rcv_msg(struct sk_buff *skb,
				    struct nlmsghdr *nlh, int *errp)
{
	struct nfnl_callback *nc;
	int type, err = 0;

	nf_debug(0, "entered; subsys=%u, msgtype=%u\n",
		 NFNL_SUBSYS_ID(nlh->nlmsg_type),
		 NFNL_MSG_TYPE(nlh->nlmsg_type));

	/* Only requests are handled by kernel now. */
	if (!(nlh->nlmsg_flags & NLM_F_REQUEST)) {
		nf_debug(0, "received non-request message\n");
		return 0;
	}

	/* Unknown message: reply with EINVAL */
	type = nlh->nlmsg_type;
	if (NFNL_SUBSYS_ID(type) > NFNL_SUBSYS_COUNT) {
		nf_debug(0, "subsys_id > subsys_count\n");
		goto err_inval;
	}

	/* All the messages must have at least 1 byte length */
	if (nlh->nlmsg_len < NLMSG_LENGTH(sizeof(struct nfgenmsg))) {
		nf_debug(0, "received message was too short\n");
		return 0;
	}

	nc = nfnetlink_find_client(type);
	if (!nc) {
		nf_debug(0, "unable to find client for type %d\n", type);
		goto err_inval;
	}

	if (nc->cap_required && 
	    !cap_raised(NETLINK_CB(skb).eff_cap, nc->cap_required)) {
		nf_debug(0, "permission denied for type %d\n", type);
		*errp = -EPERM;
		return -1;
	}

	err = nc->call(nfnl, skb, nlh, errp);
	*errp = err;
	return err;

err_inval:
	*errp = -EINVAL;
	return -1;
}

/* Process one packet of messages. */
static inline int nfnetlink_rcv_skb(struct sk_buff *skb)
{
	int err;
	struct nlmsghdr *nlh;

	while (skb->len >= NLMSG_SPACE(0)) {
		u32 rlen;

		nlh = (struct nlmsghdr *)skb->data;
		if (nlh->nlmsg_len < sizeof(struct nlmsghdr)
		    || skb->len < nlh->nlmsg_len)
			return 0;
		rlen = NLMSG_ALIGN(nlh->nlmsg_len);
		if (rlen > skb->len)
			rlen = skb->len;
		if (nfnetlink_rcv_msg(skb, nlh, &err)) {
			if (!err)
				return -1;
			netlink_ack(skb, nlh, err);
		} else
			if (nlh->nlmsg_flags & NLM_F_ACK)
				netlink_ack(skb, nlh, 0);
		skb_pull(skb, rlen);
	}

	return 0;
}

static void nfnetlink_rcv(struct sock *sk, int len)
{
	do {
		struct sk_buff *skb;

		if (nfnl_shlock_nowait())
			return;

		while ((skb = skb_dequeue(&sk->receive_queue)) != NULL) {
			if (nfnetlink_rcv_skb(skb)) {
				if (skb->len)
					skb_queue_head(&sk->receive_queue, skb);
				else
					kfree_skb(skb);
				break;
			}
			kfree_skb(skb);
		}

		up(&nfnl_sem);
	} while(nfnl && nfnl->receive_queue.qlen);
}

void __exit nfnetlink_exit(void)
{
	printk("Netfilter removing netlink socket.\n");
	sock_release(nfnl->socket);
	return;
}

int __init nfnetlink_init(void)
{
	int i;
	printk("Netfilter messages via NETLINK v%s.\n", nfversion);

	for (i = 0; i < NFNL_SUBSYS_COUNT; i++)
		subsys_table[i] = NULL;

	nfnl = netlink_kernel_create(NETLINK_NETFILTER, nfnetlink_rcv);
	if (!nfnl) {
		printk(KERN_ERR "cannot initialize nfnetlink!\n");
		return -1;
	}

	return 0;
}

module_init(nfnetlink_init);
module_exit(nfnetlink_exit);

EXPORT_SYMBOL_GPL(nfnetlink_subsys_alloc);
EXPORT_SYMBOL_GPL(nfnetlink_subsys_register);
EXPORT_SYMBOL_GPL(nfnetlink_subsys_unregister);
EXPORT_SYMBOL_GPL(nfnetlink_check_attributes);
EXPORT_SYMBOL_GPL(nfnetlink_send);
EXPORT_SYMBOL_GPL(__nfa_fill);
