/*
 * Connection state tracking for netfilter.  This is separated from,
 * but required by, the (future) NAT layer; it can also be used by an iptables
 * extension.
 *
 * 16 Dec 2003: Yasuyuki Kozakai @USAGI <yasuyuki.kozakai@toshiba.co.jp>
 *	- generalize L3 protocol dependent part.
 *
 * Derived from include/linux/netfiter_ipv4/ip_conntrack.h
 */

#ifndef _NF_CONNTRACK_H
#define _NF_CONNTRACK_H

enum nf_conntrack_info
{
	/* Part of an established connection (either direction). */
	NF_CT_ESTABLISHED,

	/* Like NEW, but related to an existing connection, or ICMP error
	   (in either direction). */
	NF_CT_RELATED,

	/* Started a new connection to track (only
           NF_CT_DIR_ORIGINAL); may be a retransmission. */
	NF_CT_NEW,

	/* >= this indicates reply direction */
	NF_CT_IS_REPLY,

	/* Number of distinct NF_CT types (no NEW in reply dirn). */
	NF_CT_NUMBER = NF_CT_IS_REPLY * 2 - 1
};

/* Bitset representing status of connection. */
enum nf_conntrack_status {
	/* It's an expected connection: bit 0 set.  This bit never changed */
	NF_S_EXPECTED_BIT = 0,
	NF_S_EXPECTED = (1 << NF_S_EXPECTED_BIT),

	/* We've seen packets both ways: bit 1 set.  Can be set, not unset. */
	NF_S_SEEN_REPLY_BIT = 1,
	NF_S_SEEN_REPLY = (1 << NF_S_SEEN_REPLY_BIT),

	/* Conntrack should never be early-expired. */
	NF_S_ASSURED_BIT = 2,
	NF_S_ASSURED = (1 << NF_S_ASSURED_BIT),

	/* Connection is confirmed: originating packet has left box */
	NF_S_CONFIRMED_BIT = 3,
	NF_S_CONFIRMED = (1 << NF_S_CONFIRMED_BIT),
};

#ifdef __KERNEL__
#include <linux/config.h>
#include <linux/netfilter/nf_conntrack_tuple.h>
#include <linux/bitops.h>
#include <linux/compiler.h>
#include <asm/atomic.h>

#include <linux/netfilter/nf_conntrack_tcp.h>
#include <linux/netfilter/ipv4/nf_conntrack_icmp.h>
#include <linux/netfilter/ipv6/nf_conntrack_icmpv6.h>
#include <linux/netfilter/nf_conntrack_sctp.h>

/* per conntrack: protocol private data */
union nf_conntrack_proto {
	/* insert conntrack proto private data here */
	struct nf_ct_sctp sctp;
	struct nf_ct_tcp tcp;
	struct nf_ct_icmp icmp;
	struct nf_ct_icmpv6 icmpv6;
};

union nf_conntrack_expect_proto {
	/* insert expect proto private data here */
};

/* Add protocol helper include file here */
#include <linux/netfilter/nf_conntrack_ftp.h>

/* per conntrack: application helper private data */
union nf_conntrack_help {
	/* insert conntrack helper private data (master) here */
	struct nf_ct_ftp_master ct_ftp_info;
};

#include <linux/types.h>
#include <linux/skbuff.h>

#ifdef CONFIG_NETFILTER_DEBUG
#define NF_CT_ASSERT(x)							\
do {									\
	if (!(x))							\
		/* Wooah!  I'm tripping my conntrack in a frenzy of	\
		   netplay... */					\
		printk("NF_CT_ASSERT: %s:%i(%s)\n",			\
		       __FILE__, __LINE__, __FUNCTION__);		\
} while(0)
#else
#define NF_CT_ASSERT(x)
#endif

struct nf_conntrack_counter
{
	u_int64_t packets;
	u_int64_t bytes;
};

struct nf_conntrack_helper;

#include <linux/netfilter/ipv4/nf_conntrack_ipv4.h>
struct nf_conn
{
	/* Usage count in here is 1 for hash table/destruct timer, 1 per skb,
           plus 1 for any connection(s) we are `master' for */
	struct nf_conntrack ct_general;

	/* XXX should I move this to the tail ? - Y.K */
	/* These are my tuples; original and reply */
	struct nf_conntrack_tuple_hash tuplehash[NF_CT_DIR_MAX];

	/* Have we seen traffic both ways yet? (bitset) */
	unsigned long status;

	/* Timer function; drops refcnt when it goes off. */
	struct timer_list timeout;

#ifdef CONFIG_NF_CT_ACCT
	/* Accounting Information (same cache line as other written members) */
	struct nf_conntrack_counter counters[NF_CT_DIR_MAX];
#endif
	/* If we were expected by an expectation, this will be it */
	struct nf_conn *master;
	
	/* Current number of expected connections */
	unsigned int expecting;

	/* Helper. if any */
	struct nf_conntrack_helper *helper;

	/* features - nat, helper, ... used by allocating system */
	u_int32_t features;

	/* Storage reserved for other modules: */

	union nf_conntrack_proto proto;

#if defined(CONFIG_NF_CONNTRACK_MARK)
	unsigned long mark;
#endif

	/* These members are dynamically allocated. */

	union nf_conntrack_help *help;

	/* Layer 3 dependent members. (ex: NAT) */
	union {
		struct nf_conntrack_ipv4 *ipv4;
	} l3proto;
	void *data[0];
};

struct nf_conntrack_expect
{
	/* Internal linked list (global expectation list) */
	struct list_head list;

	/* We expect this tuple, with the following mask */
	struct nf_conntrack_tuple tuple, mask;
 
	/* Function to call after setup and insertion */
	void (*expectfn)(struct nf_conn *new,
			 struct nf_conntrack_expect *this);

	/* The conntrack of the master connection */
	struct nf_conn *master;

	/* Timer function; deletes the expectation. */
	struct timer_list timeout;

#ifdef CONFIG_NF_NAT_NEEDED
	/* This is the original per-proto part, used to map the
	 * expected connection the way the recipient expects. */
	union nf_conntrack_manip_proto saved_proto;
	/* Direction relative to the master connection. */
	enum nf_conntrack_dir dir;
#endif
};

static inline struct nf_conn *
tuplehash_to_ctrack(const struct nf_conntrack_tuple_hash *hash)
{
	return container_of(hash, struct nf_conn,
			    tuplehash[hash->tuple.dst.dir]);
}

/* get master conntrack via master expectation */
#define master_ct(conntr) (conntr->master)

/* Alter reply tuple (maybe alter helper). */
extern void
nf_conntrack_alter_reply(struct nf_conn *conntrack,
			 const struct nf_conntrack_tuple *newreply);

/* Is this tuple taken? (ignoring any belonging to the given
   conntrack). */
extern int
nf_conntrack_tuple_taken(const struct nf_conntrack_tuple *tuple,
			 const struct nf_conn *ignored_conntrack);

/* Return conntrack_info and tuple hash for given skb. */
static inline struct nf_conn *
nf_ct_get(struct sk_buff *skb, enum nf_conntrack_info *ctinfo)
{
	*ctinfo = skb->nfctinfo;
	return (struct nf_conn *)skb->nfct;
}

/* decrement reference count on a conntrack */
extern inline void nf_ct_put(struct nf_conn *ct);

/* call to create an explicit dependency on nf_conntrack. */
extern void need_nf_conntrack(void);

extern int nf_ct_invert_tuplepr(struct nf_conntrack_tuple *inverse,
				const struct nf_conntrack_tuple *orig);

/* Refresh conntrack for this many jiffies */
extern void nf_ct_refresh_acct(struct nf_conn *ct,
			       enum nf_conntrack_info ctinfo,
			       const struct sk_buff *skb,
			       unsigned long extra_jiffies);

/* These are for NAT.  Icky. */
/* Call me when a conntrack is destroyed. */
extern void (*nf_conntrack_destroyed)(struct nf_conn *conntrack);

/* Fake conntrack entry for untracked connections */
extern struct nf_conn nf_conntrack_untracked;

extern int nf_ct_no_defrag;

/* Iterate over all conntracks: if iter returns true, it's deleted. */
extern void
nf_ct_iterate_cleanup(int (*iter)(struct nf_conn *i, void *data), void *data);

/* It's confirmed if it is, or has been in the hash table. */
static inline int is_confirmed(struct nf_conn *ct)
{
	return test_bit(NF_S_CONFIRMED_BIT, &ct->status);
}

extern unsigned int nf_conntrack_htable_size;

struct nf_conntrack_stat
{
	unsigned int searched;
	unsigned int found;
	unsigned int new;
	unsigned int invalid;
	unsigned int ignore;
	unsigned int delete;
	unsigned int delete_list;
	unsigned int insert;
	unsigned int insert_failed;
	unsigned int drop;
	unsigned int early_drop;
	unsigned int error;
	unsigned int expect_new;
	unsigned int expect_create;
	unsigned int expect_delete;
};

#define NF_CT_STAT_INC(count) (__get_cpu_var(nf_conntrack_stat).count++)

/* eg. PROVIDES_CONNTRACK(ftp); */
#define PROVIDES_CONNTRACK(name)                        \
        int needs_nf_conntrack_##name;                  \
        EXPORT_SYMBOL(needs_nf_conntrack_##name)

/*. eg. NEEDS_CONNTRACK(ftp); */
#define NEEDS_CONNTRACK(name)                                           \
        extern int needs_nf_conntrack_##name;                           \
        static int *need_nf_conntrack_##name __attribute_used__ = &needs_nf_conntrack_##name

/* no helper, no nat */
#define	NF_CT_F_BASIC	0
/* for helper */
#define	NF_CT_F_HELP	1
/* for nat. */
#define	NF_CT_F_NAT	2
#define NF_CT_F_NUM	4

extern int
nf_conntrack_register_cache(u_int32_t features, const char *name, size_t size,
			    int (*init_conntrack)(struct nf_conn *, u_int32_t));
extern void
nf_conntrack_unregister_cache(u_int32_t features);

#endif /* __KERNEL__ */
#endif /* _NF_CONNTRACK_H */
