/*
 *	openssl.c
 *	Release $Name: MATRIXSSL_1_0_BETA $
 *
 *	MatrixSSL OpenSSL compatibility layer. http://www.openssl.org
 *	This layer allows optimized OpenSSL hardware cryptography libraries
 *	to be linked with MatrixSSL.
 *
 *	Copyright (c) PeerSec Networks, 2002-2004. All Rights Reserved.
 *	The latest version of this code is available at http://www.matrixssl.org
 *
 *	This software is open source; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This General Public License does NOT permit incorporating this software 
 *	into proprietary programs.  If you are unable to comply with the GPL, a 
 *	commercial license for this software may be purchased from PeerSec Networks
 *	at http://www.peersec.com
 *	
 *	This program is distributed in WITHOUT ANY WARRANTY; without even the 
 *	implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
 *	See the GNU General Public License for more details.
 *	
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *	http://www.gnu.org/copyleft/gpl.html
 */
/******************************************************************************/

#include "openssl.h"
#include "../cryptoLayer.h"
#include <sys/stat.h>

/******************************************************************************/
/*
	MD5 wrapper:

	void matrixMd5Init(sslMd5Context_t *ctx);
	void matrixMd5Update(sslMd5Context_t *ctx, const unsigned char *buf, 
		unsigned long len);
	void matrixMd5Final(sslMd5Context_t *ctx, unsigned char *hash);
*/

void matrixMd5Init(sslMd5Context_t *ctx)
{
	MD5_Init(ctx);
}

void matrixMd5Update(sslMd5Context_t *ctx, const unsigned char *buf, 
						unsigned long len)
{
	MD5_Update(ctx, buf, len);
}

void matrixMd5Final(sslMd5Context_t *ctx, unsigned char *hash)
{
	MD5_Final(hash, ctx);
}

/******************************************************************************/
/*
	Sha1 wrapper:

	int SHA_Init(SHA_CTX *c);
	int SHA_Update(SHA_CTX *c, const void *data, unsigned long len);
	int SHA_Final(unsigned char *md, SHA_CTX *c);

*/
void matrixSha1Init(sslSha1Context_t *ctx)
{
	SHA1_Init(ctx);
}

void matrixSha1Update(sslSha1Context_t *ctx, const unsigned char *buf, 
						unsigned long len)
{
	SHA1_Update(ctx, buf, len);
}
void matrixSha1Final(sslSha1Context_t *ctx, unsigned char *hash)
{
	SHA1_Final(hash, ctx);
}

/******************************************************************************/
/*
	ARC4 wrappers:

	void matrixArc4Init(sslCipherContext_t *ctx, unsigned char *key, 
		int keylen);
	int matrixArc4(sslCipherContext_t *ctx, char *in, char *out, int len);

*/
#ifdef USE_ARC4
void matrixArc4Init(sslCipherContext_t *ctx, unsigned char *key, int keylen)
{
	RC4_set_key(&(ctx->rc4), keylen, key);
}

int matrixArc4(sslCipherContext_t *ctx, char *in, char *out, int len)
{
	RC4(&(ctx->rc4), len, in, out);
	return len;
}
#endif /* USE_ARC4 */
/******************************************************************************/
/*
	3DES wrappers:

int matrix3desInit(sslCipherContext_t *ctx, char *IV, char *key, int keylen);
int matrix3desEncrypt(sslCipherContext_t *ctx, char *pt, char *ct, int len);
int matrix3desDecrypt(sslCipherContext_t *ctx, char *ct, char *pt, int len);

	FUTURE - implement wrappers					
*/

/******************************************************************************/
/*
	RSA wrappers: 

	int matrixRsaEncryptPub(sslRsaKey_t *key, char *in, int inlen, char *out, 
		int outlen);
	int matrixRsaDecryptPriv(sslRsaKey_t *key, char *in, int inlen, char *out, 
		int outlen);
	int matrixSslReadPrivKey(char *fileName, char *password, sslRsaKey_t **key);
	int matrixSslReadCert(char *fileName, unsigned char **out, int *outLen);

*/
int matrixRsaEncryptPub(sslRsaKey_t *key, char *in, int inlen, char *out, 
						int outlen)
{	
	return RSA_public_encrypt(inlen, in, out, key, RSA_SSLV23_PADDING);
}

int matrixRsaDecryptPriv(sslRsaKey_t *key, char *in, int inlen, char *out, 
						 int outlen)
{
	return RSA_private_decrypt(inlen, in, out, key, RSA_SSLV23_PADDING);
}

int matrixSslReadPrivKey(char *fileName, char *password, sslRsaKey_t **key)
{
	FILE			*fp;

	if ((fp = fopen(fileName, "r")) == NULL) {
		return -1;
	}
	
	*key = NULL;
	PEM_read_RSAPrivateKey(fp, key, NULL, (void*)password);
	fclose(fp);
	
	return 0;
}

int matrixSslReadCert(char *fileName, unsigned char **out, int *outLen)
{
	FILE	*fp;
	X509	*cert = NULL;
	BIO		*outBio;
	
	if ((fp = fopen(fileName, "r")) == NULL) {
		return -1;
	}
	
	cert = PEM_read_X509(fp, NULL, NULL, NULL);
	fclose(fp);

	if (cert == NULL) {
		return -1;
	}

	outBio = BIO_new(BIO_s_mem());
	i2d_X509_bio(outBio, cert);

	*out = sslMalloc(outBio->num_write);
	BIO_read(outBio, *out, outBio->num_write);

	*outLen = outBio->num_read;
	
	BIO_free(outBio);
	X509_free(cert);
	return 0;
}

void matrixSslFreeKey(sslRsaKey_t *key)
{
	RSA_free(key);
}

/******************************************************************************/
