#include <linux/module.h>
#include <linux/netfilter_ipv4/ip_tables.h>
#include <linux/version.h>

#include <linux/netfilter_ipv4/ipt_ipp2p.h>

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)

#if defined(MODVERSIONS)
    #include <linux/modversions.h>
#endif
#endif

#define get_u16(X,O)  (*(__u16 *)(X + O))
#define IPP2P_VERSION "0.6"

MODULE_AUTHOR("Eicke Friedrich <tady at gmx dot net>");
MODULE_DESCRIPTION("An extension to iptables to identify P2P traffic.");
MODULE_LICENSE("GPL");


/*Search for SoulSeek commands*/
int
search_soul (unsigned char *haystack, int packet_len, int head_len)
{
    unsigned char *t = haystack;
    int cmd;    
    t += head_len;

    cmd = get_u16(t, 0);	
    if (cmd == (packet_len - head_len - 4))
    {
	if ((t[2] == 0x00) && (t[3] == 0x00) && (t[4] == 0x01)) return 1; //CONNECT: 		xx xx 00 00 01
	if ((t[2] == 0x00) && (t[3] == 0x00) && (t[4] == 0x28)) return 1; //TRANSFER REQUEST: 	xx xx 00 00 28	
    }
    return 0;

}


/*Search for appleJuice commands*/
int
search_apple (unsigned char *haystack, int packet_len, int head_len)
{
    unsigned char *t = haystack;
    t += head_len;
    
    if ((memcmp(t, "ajprot", 6) == 0) && (t[6] == 0x0d) && (t[7] == 0x0a))  return 1;
    
    return 0;
}


/*Search for BitTorrent commands*/
int
search_bittorrent (unsigned char *haystack, int packet_len, int head_len)
{

    unsigned char *t = haystack;
    if (*(haystack+head_len) != 0x13) return 0; //Bail out of first byte != 0x13
    
    t += head_len + 1;
    
    if (memcmp(t, "BitTorrent protocol", 19) == 0) return 1;        
    return 0;
}



/*check for Kazaa get command*/
int
search_kazaa (unsigned char *haystack, int packet_len, int head_len)
{
    unsigned char *t = haystack;

    if (!((*(haystack + packet_len - 2) == 0x0d) && (*(haystack + packet_len - 1) == 0x0a))) return 0;    

    t += head_len;
    if (memcmp(t, "GET /.hash=", 11) == 0)
	return 1;
    else
	return 0;
}


/*check for gnutella get command*/
int
search_gnu (unsigned char *haystack, int packet_len, int head_len)
{
    unsigned char *t = haystack;

    if (!((*(haystack + packet_len - 2) == 0x0d) && (*(haystack + packet_len - 1) == 0x0a))) return 0;    

    t += head_len;
    if (memcmp(t, "GET /get/", 9) == 0)	return 1;
    if (memcmp(t, "GET /uri-res/", 13) == 0) return 1; 
    
    return 0;
}


/*check for gnutella get commands and other typical data*/
int
search_all_gnu (unsigned char *haystack, int packet_len, int head_len)
{
    unsigned char *t = haystack;
    int c;    

    if (!((*(haystack + packet_len - 2) == 0x0d) && (*(haystack + packet_len - 1) == 0x0a))) return 0;

    t += head_len;

    if (memcmp(t, "GNUTELLA CONNECT/", 17) == 0) return 1;        
    if (memcmp(t, "GNUTELLA/", 9) == 0) return 1;    

    if ((memcmp(t, "GET /get/", 9) == 0) || (memcmp(t, "GET /uri-res/", 13) == 0))
    {        
        c = head_len + 8;
	t += 8;
	while (c < packet_len - 22)
	{
	    if (t[0] == 0x0d)
	    {
		t++;
		c++;
		if (t[0] == 0x0a)
		{
		    t++;
		    c++;
		    if ((memcmp(t, "X-Gnutella-", 11) == 0) || (memcmp(t, "X-Queue:", 8) == 0)) return 1;
		}
	    }
	    else
	    {
		t++;
		c++;
	    }    
	}
    }
    return 0;
}


/*check for KaZaA download commands and other typical data*/
int
search_all_kazaa (unsigned char *haystack, int packet_len, int head_len)
{
    unsigned char *t = haystack;
    int c;    

    if (!((*(haystack + packet_len - 2) == 0x0d) && (*(haystack + packet_len - 1) == 0x0a))) return 0;

    t += head_len;
    if (memcmp(t, "GIVE ", 5) == 0) return 1;    
    
    if (memcmp(t, "GET /", 5) == 0)
    {
        c = head_len + 8;
	t += 8;
	while (c < packet_len - 22)
	{
	    if (t[0] == 0x0d)
	    {
		t++;
		c++;
		if (t[0] == 0x0a)
		{
		    t++;
		    c++;
    		    if ( memcmp(t, "X-Kazaa-Username: ", 18) == 0 ) return 1;
		}
	    }
	    else
	    {
		t++;
		c++;
	    }    
	}
    }
    
    return 0;
}

/*fast check for edonkey file segment transfer command*/
int
search_edk (unsigned char *haystack, int packet_len, int head_len)
{
    if (*(haystack+head_len) != 0xe3) 
	return 0;
    else
    {
	if (*(haystack+head_len+5) == 0x47) 
	    return 1;
	else 	
	    return 0;
    }
}



/*intensive but slower search for some edonkey packets including size-check*/
int
search_all_edk (unsigned char *haystack, int packet_len, int head_len)
{
    unsigned char *t = haystack;
    int cmd;
    
    if (*(haystack+head_len) == 0xd4)
    {
	t += head_len;	
	cmd = get_u16(t, 1);	
	if (cmd == (packet_len - head_len - 5))
	{
	    if (t[5] == 0x82) return 1;
	    if (t[5] == 0x15) return 1;	    
	}
	return 0;    
    }
    
    
    if (*(haystack+head_len) == 0xc5) //search for additional eMule packets
    {
	t += head_len;	
	cmd = get_u16(t, 1);	

	if (cmd == (packet_len - head_len - 5))
	{
	    if (t[5] == 0x01) return 1;	
	    if (t[5] == 0x02) return 1;		    	    
	    if (t[5] == 0x60) return 1;
	    if (t[5] == 0x81) return 1;
	    if (t[5] == 0x82) return 1;	    
	    if (t[5] == 0x85) return 1;	    
	    if (t[5] == 0x86) return 1;
	    if (t[5] == 0x87) return 1;
	    if (t[5] == 0x40) return 1;
	    if (t[5] == 0x92) return 1;
	    if (t[5] == 0x93) return 1;	
	    if (t[5] == 0x12) return 1;		    
	    return 0;
	}
	
	return 0;
    }


    if (*(haystack+head_len) != 0xe3) 
	return 0;
    else
    {
	t += head_len;	
	cmd = get_u16(t, 1);	
	if (cmd == (packet_len - head_len - 5)) 
	{
	    if (t[5] == 0x01) return 1;	//Client: hello or Server:hello
	    if (t[5] == 0x50) return 1;	//Client: file status
	    if (t[5] == 0x16) return 1;	//Client: search
	    if (t[5] == 0x58) return 1;	//Client: file request
	    if (t[5] == 0x48) return 1;	//???
	    if (t[5] == 0x54) return 1;	//???	    	    
	    if (t[5] == 0x47) return 1;	//Client: file segment request
	    if (t[5] == 0x46) return 1; //Client: download segment	    
	    if (t[5] == 0x4c) return 1;	//Client: Hello-Answer
	    if (t[5] == 0x4f) return 1;	//Client: file status request
	    if (t[5] == 0x59) return 1;	//Client: file request answer
	    if (t[5] == 0x65) return 1;	//Client: ???
	    if (t[5] == 0x66) return 1;	//Client: ??? 	    
	    if (t[5] == 0x51) return 1;	//Client: ??? 	    	    
	    if (t[5] == 0x52) return 1;	//Client: ??? 	    	        
	    if (t[5] == 0x4d) return 1;	//Client: ??? 	    
	    if (t[5] == 0x5c) return 1;	//Client: ??? 	    
	    if (t[5] == 0x38) return 1;	//Client: ??? 	    
	    if (t[5] == 0x69) return 1;	//Client: ??? 	    	    
	    if (t[5] == 0x19) return 1;	//Client: ??? 	    	    
	    if (t[5] == 0x42) return 1;	//Client: ??? 	    
	    if (t[5] == 0x34) return 1;	//Client: ???
	    if (t[5] == 0x94) return 1;	//Client: ???	    
	    if (t[5] == 0x1c) return 1;	//Client: ???	    	    
	    if (t[5] == 0x6a) return 1;	//Client: ???	    
	    return 0;	
	}
	else
	{
	    if (cmd > packet_len - head_len - 5) 
	    {
		if ((t[3] == 0x00) && (t[4] == 0x00))
		{
		    if (t[5] == 0x01) return 1;
		    if (t[5] == 0x4c) return 1;
		} 
		return 0;
		
	    }	//non edk packet
	    if (t[cmd+5] == 0xe3) return 1;		//found another edk-command
	    if (t[cmd+5] == 0xc5) return 1;		//found an emule-command	    
	    return 0;
	}
    }
}


/*fast check for Direct Connect send command*/
int
search_dc (unsigned char *haystack, int packet_len, int head_len)
{
    unsigned char *t = haystack;

    if (*(haystack+head_len) != 0x24 ) 
	return 0;
    else
    {
	t += head_len + 1;
        if (memcmp(t, "Send|", 5) == 0)
	    return 1;
	else
	    return 0;
    }	

}


/*intensive but slower check for all direct connect packets*/
int
search_all_dc (unsigned char *haystack, int packet_len, int head_len)
{
    unsigned char *t = haystack;

    if ((*(haystack + head_len) == 0x24) && (*(haystack + packet_len - 1) == 0x7c)) 
    {
    	t += head_len + 1;
	if (memcmp(t, "Lock ", 5) == 0)	 return 1;	//hub: hello
	if (memcmp(t, "Key ", 4) == 0)	 return 1;	//client: hello
	if (memcmp(t, "Hello ", 6) == 0) return 1;	//hub:connected
	if (memcmp(t, "MyNick ", 7) == 0) return 1;	//client-client: hello
	if (memcmp(t, "Search ", 7) == 0) return 1;	//client: search
	if (memcmp(t, "Send", 4) == 0)	 return 1;	//client: start download
	return 0;
    }
    else
	return 0;
	

}


//typedef int (*p2p_function) (unsigned char *, int, int);

static struct {
    int command;
    int short_hand;			/*1=ipp2p; 4=ipp2p-data; 5=none*/
    int packet_len;
    int (*function_name) (unsigned char *, int, int);
} matchlist[] = {
    {2,1,40, &search_all_edk}, 		/*edk || ipp2p */
    {8,4,200, &search_kazaa}, 		/*kazaa-data || ipp2p-data */    
    {16,4,60, &search_edk},		/*edk-data || ipp2p-data*/
    {32,4,26, &search_dc},		/*dc-data || ipp2p-data*/    
    {64,1,25, search_all_dc}, 		/*dc || ipp2p */
    {128,4,40, &search_gnu},		/*gnu-data || ipp2p-data*/    
    {256,1,35, &search_all_gnu}, 	/*gnu || ipp2p */
    {512,1,35, &search_all_kazaa}, 	/*kazaa || ipp2p */
    {1024,5,40, &search_bittorrent},	/*bit */    
    {2048,5,20, &search_apple},		/*apple */    
    {4096,5,20, &search_soul},		/*soul */    
    {0,0,0,NULL}
};


static int
match(const struct sk_buff *skb,
      const struct net_device *in,
      const struct net_device *out,
      const void *matchinfo,
      int offset,

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
      const void *hdr,
      u_int16_t datalen,
#endif

      int *hotdrop)
{
    const struct ipt_p2p_info *info = matchinfo;
    unsigned char  *haystack;
    struct iphdr *ip = skb->nh.iph;
    int p2p_result = 0, i = 0;
    int head_len;


    int hlen = ntohs(ip->tot_len)-(ip->ihl*4);	//hlen = packet-data length    
    haystack=(char *)ip+(ip->ihl*4);		//haystack = packet data    


    if (((*(haystack+13)) & 1) == 1) return 0;  //if FIN bit is set bail out
    if (((*(haystack+13)) & 2) == 2) return 0;  //if SYN bit is set bail out
    if (((*(haystack+13)) & 4) == 4) return 0;  //if RST bit is set bail out
	    

    head_len = (*(haystack+12))/4; //get TCP-Header-Size

    while (matchlist[i].command) {
	if ((((info->cmd & matchlist[i].command) == matchlist[i].command) || ((info->cmd & matchlist[i].short_hand) == matchlist[i].short_hand)) && (hlen > matchlist[i].packet_len)) {
	    p2p_result = matchlist[i].function_name(haystack, hlen, head_len);
	    if (p2p_result) return p2p_result;
	}
	i++;
    }
    return p2p_result;
}



static int
checkentry(const char *tablename,
            const struct ipt_ip *ip,
	    void *matchinfo,
	    unsigned int matchsize,
	    unsigned int hook_mask)
{
        /* Must specify -p tcp */
    if (ip->proto != IPPROTO_TCP || (ip->invflags & IPT_INV_PROTO)) {
	printk("ipp2p: Only works on TCP packets, use -p tcp\n");
	return 0;
    }
							

    return 1;
}
									    



static struct ipt_match ipp2p_match = { 
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	{ NULL, NULL }, 
	"ipp2p", 
	&match, 
	&checkentry, 
	NULL, 
	THIS_MODULE
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
	.name		= "ipp2p",
	.match		= &match,
	.checkentry	= &checkentry,
	.me		= THIS_MODULE,
#endif
};


static int __init init(void)
{
    printk(KERN_INFO "IPP2P v%s loading\n", IPP2P_VERSION);
    return ipt_register_match(&ipp2p_match);
}
	
static void __exit fini(void)
{
    ipt_unregister_match(&ipp2p_match);
    printk(KERN_INFO "IPP2P v%s unloaded\n", IPP2P_VERSION);
}
	
module_init(init);
module_exit(fini);


