#-*- coding: utf-8 -*-

# Copyright 2010-2012 Calculate Ltd. http://www.calculate-linux.org
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0 #
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.

#   mode -     read only or writeable variable
#   value    - default variable value
#   select   - list of posible values for variable
#   hide - flag, if it is True, then the variable is not printable
#   printval - print value of variable

from calculate.lib.datavars import Variable,ReadonlyVariable,VariableError
import os, glob, sys
try:
    import OpenSSL
except:
    OpenSSL = None

from calculate.lib.cl_lang import setLocalTranslate
from calculate.lib.utils.files import readLinesFile
#from calculate.api.cert_cmd import find_id_cert
setLocalTranslate('cl_core3',sys.modules[__name__])

class VariableClAllReqId(Variable):
    def get(self):
        data_path = self.Get('cl_core_data')
        result = []
        cert_dir =  data_path + '/client_certs/'
        for filename in glob.glob(cert_dir+"*"):
            if filename.endswith('.csr'):
                temp = filename.split('.')[0].split('/')
                id = temp[len(temp)-1]
                try:
                    result.append(id)
                except:
                    pass
        return result

class VariableClReqId(Variable):
    """
    Certificate Identification
    """
    type = "choice"
    opt = ["-r"]
    metavalue = "REQ_ID"
    untrust = True

    def init(self):
        self.help = _("request identifier")
        self.label = _("Request identifier")

    def choice(self):
        return self.Get('cl_all_req_id')

    def check(self, req_id):
        try:
            int(req_id)
        except (ValueError, TypeError):
            raise VariableError(_("The request ID must be int"))

    def raiseNothingValue(self):
        raise VariableError(_("Not found any requests"))

    def get(self):
        #if not self.choice():
        #    self.raiseNothingValue()
        return ""

class VariableClPageLimit(Variable):
    """
    Certificate Identification
    """
    type = "int"
    opt = ["--page-limit"]
    metavalue = "PAGE_LIMIT"
    element = 'input'

    def init(self):
        self.help = _("set the page limit value")
        self.label = _("Page limit")

    def check(self, limit):
        try:
            int(limit)
        except ValueError:
            raise VariableError(_("The limit number must be int"))

class VariableClPageCount(Variable):
    """
    Certificate Identification
    """
    type = "int"
    opt = ["--page-count"]
    metavalue = "PAGE_COUNT"
    element = 'input'

    def init(self):
        self.help = _("set the page count value")
        self.label = _("Page count")

    def check(self, count):
        try:
            int(count)
        except ValueError:
            raise VariableError(_("The count number must be int"))

class VariableClPageOffset(Variable):
    """
    Certificate Identification
    """
    type = "int"
    opt = ["--page-offset"]
    metavalue = "PAGE_OFFSET"
    element = 'input'

    def init(self):
        self.help = _("set the page offset value")
        self.label = _("Page offset")

    def check(self, offset):
        try:
            int(offset)
        except ValueError:
            raise VariableError(_("The offset number must be int"))

class VariableClReqBaseData(Variable):
    """
    """
    def get(self):
        req_id = self.Get('cl_req_id')
        serv_certbase = self.Get('cl_core_serv_database')
        certbase = self.Get('cl_core_database')

        for line in readLinesFile(certbase):
            if line.split()[0] == str(req_id):
                return line.strip().split()
        return ['']*7

class VariableClReqData(Variable):
    """
    """
    def get(self):
        req_id = self.Get('cl_req_id')
        data_path = self.Get('cl_core_data')
        
        req_file =  data_path + '/client_certs/%s.csr' %req_id
        if OpenSSL and  os.path.exists(req_file):
            fp = open(req_file, 'r')
            request = fp.read()
            fp.close()
            reqobj = OpenSSL.crypto.load_certificate_request \
                     (OpenSSL.SSL.FILETYPE_PEM, request)
            Subject = reqobj.get_subject().get_components()
            return Subject
        return [['','']]*6

class VariableClReqIp(ReadonlyVariable):
    """
    Ip Request
    """

    def init(self):
        self.help = _("request IP address")
        self.label = _("Request IP address")

    def uncompatible(self):
        return 'Ip adress'

    def get(self):
        return self.Get('cl_req_base_data')[4]

class VariableClReqMac(ReadonlyVariable):
    """
    Mac Adress Request
    """

    def init(self):
        self.help = _("request MAC adress")
        self.label = _("Request MAC address")

    def uncompatible(self):
        return 'Mac adress'

    def get(self):
        return self.Get('cl_req_base_data')[5]

class VariableClReqDate(ReadonlyVariable):
    """
    Date send Request
    """

    def init(self):
        self.help = _("request date")
        self.label = _("Request date")

    def uncompatible(self):
        return 'Request Date'

    def get(self):
        words = self.Get('cl_req_base_data')
        if words:
            return '%s %s' %(words[2], words[3])
        else:
            return ""

class VariableClReqUserName(ReadonlyVariable):
    """
    UserName Owner Request
    """

    def init(self):
        self.help = _("request owner username")
        self.label = _("Request owner username")

    def uncompatible(self):
        return 'User name request owner'

    def get(self):
        Subject = self.Get('cl_req_data')
        for item in Subject:
            if item[0] == 'OU':
                return item[1]
        return ''

class VariableClReqLocation(ReadonlyVariable):
    """
    Location Owner Request
    """

    def init(self):
        self.help = _("request location")
        self.label = _("Request location")

    def uncompatible(self):
        return 'Location'

    def get(self):
        Subject = self.Get('cl_req_data')
        for item in Subject:
            if item[0] == 'L':
                return item[1]
        return ''

class VariableClReqGroup(Variable):
    """
    Certificate Group
    """
    type = "choice"
    opt = ["-g"]
    metavalue = "REQ_GROUP"

    def init(self):
        self.help = _("set the certificate group")
        self.label = _("Certificate group")

    def choice(self):
        group_rights = self.Get('cl_core_group_rights')

        t = open(group_rights, 'r').read()
        result = []
        for line in t.splitlines():
            words = line.split()

            if not words[0].startswith('#'):
                result.append(words[0])
        if not 'all' in result:
            result.append('all')
        return result

    def get(self):
        req_id = self.Get('cl_req_id')
        data_path = self.Get('cl_core_data')
        
        cert_file =  data_path + '/client_certs/%s.crt' %req_id
        if OpenSSL is None:
            return ""
        try:
            fp = open(cert_file, 'r')
            cert = fp.read()
            fp.close()
            certobj = OpenSSL.crypto.load_certificate \
                      (OpenSSL.SSL.FILETYPE_PEM, cert)
            com = certobj.get_extension(certobj.get_extension_count()-1).get_data()
            return com.split(':')[1]
        except (OpenSSL.crypto.Error,IOError) as e:
            return ""

    def check(self, group):
        group_rights = self.Get('cl_core_group_rights')

        if group == 'all':
            return
        t = open(group_rights, 'r').read()
        for line in t.splitlines():
            words = line.split()
            if words[0].startswith('#'):
                continue
            if group == words[0]:
                return
        raise VariableError(_("Group %s does not exist") %group)
