﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/personalize-events/PersonalizeEvents_EXPORTS.h>
#include <aws/personalize-events/PersonalizeEventsRequest.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/personalize-events/model/Event.h>
#include <utility>

namespace Aws
{
namespace PersonalizeEvents
{
namespace Model
{

  /**
   */
  class PutEventsRequest : public PersonalizeEventsRequest
  {
  public:
    AWS_PERSONALIZEEVENTS_API PutEventsRequest() = default;

    // Service request name is the Operation name which will send this request out,
    // each operation should has unique request name, so that we can get operation's name from this request.
    // Note: this is not true for response, multiple operations may have the same response name,
    // so we can not get operation's name from response.
    inline virtual const char* GetServiceRequestName() const override { return "PutEvents"; }

    AWS_PERSONALIZEEVENTS_API Aws::String SerializePayload() const override;


    ///@{
    /**
     * <p>The tracking ID for the event. The ID is generated by a call to the <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/API_CreateEventTracker.html">CreateEventTracker</a>
     * API.</p>
     */
    inline const Aws::String& GetTrackingId() const { return m_trackingId; }
    inline bool TrackingIdHasBeenSet() const { return m_trackingIdHasBeenSet; }
    template<typename TrackingIdT = Aws::String>
    void SetTrackingId(TrackingIdT&& value) { m_trackingIdHasBeenSet = true; m_trackingId = std::forward<TrackingIdT>(value); }
    template<typename TrackingIdT = Aws::String>
    PutEventsRequest& WithTrackingId(TrackingIdT&& value) { SetTrackingId(std::forward<TrackingIdT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The user associated with the event.</p>
     */
    inline const Aws::String& GetUserId() const { return m_userId; }
    inline bool UserIdHasBeenSet() const { return m_userIdHasBeenSet; }
    template<typename UserIdT = Aws::String>
    void SetUserId(UserIdT&& value) { m_userIdHasBeenSet = true; m_userId = std::forward<UserIdT>(value); }
    template<typename UserIdT = Aws::String>
    PutEventsRequest& WithUserId(UserIdT&& value) { SetUserId(std::forward<UserIdT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>The session ID associated with the user's visit. Your application generates
     * the sessionId when a user first visits your website or uses your application.
     * Amazon Personalize uses the sessionId to associate events with the user before
     * they log in. For more information, see <a
     * href="https://docs.aws.amazon.com/personalize/latest/dg/recording-item-interaction-events.html">Recording
     * item interaction events</a>.</p>
     */
    inline const Aws::String& GetSessionId() const { return m_sessionId; }
    inline bool SessionIdHasBeenSet() const { return m_sessionIdHasBeenSet; }
    template<typename SessionIdT = Aws::String>
    void SetSessionId(SessionIdT&& value) { m_sessionIdHasBeenSet = true; m_sessionId = std::forward<SessionIdT>(value); }
    template<typename SessionIdT = Aws::String>
    PutEventsRequest& WithSessionId(SessionIdT&& value) { SetSessionId(std::forward<SessionIdT>(value)); return *this;}
    ///@}

    ///@{
    /**
     * <p>A list of event data from the session.</p>
     */
    inline const Aws::Vector<Event>& GetEventList() const { return m_eventList; }
    inline bool EventListHasBeenSet() const { return m_eventListHasBeenSet; }
    template<typename EventListT = Aws::Vector<Event>>
    void SetEventList(EventListT&& value) { m_eventListHasBeenSet = true; m_eventList = std::forward<EventListT>(value); }
    template<typename EventListT = Aws::Vector<Event>>
    PutEventsRequest& WithEventList(EventListT&& value) { SetEventList(std::forward<EventListT>(value)); return *this;}
    template<typename EventListT = Event>
    PutEventsRequest& AddEventList(EventListT&& value) { m_eventListHasBeenSet = true; m_eventList.emplace_back(std::forward<EventListT>(value)); return *this; }
    ///@}
  private:

    Aws::String m_trackingId;
    bool m_trackingIdHasBeenSet = false;

    Aws::String m_userId;
    bool m_userIdHasBeenSet = false;

    Aws::String m_sessionId;
    bool m_sessionIdHasBeenSet = false;

    Aws::Vector<Event> m_eventList;
    bool m_eventListHasBeenSet = false;
  };

} // namespace Model
} // namespace PersonalizeEvents
} // namespace Aws
