package Lire::Logger;

use strict;

use base qw/ Exporter /;
use vars qw/ @EXPORT $lr_tag /;

use File::Basename;

=pod

=head1 NAME

Lire::Logger - Lire's logging interface.

=head1 SYNOPSIS

Any program:

    use Lire::Logger qw/ lr_info lr_debug lr_warn /;

    lr_info( "doing stuff " );
    lr_debug( "debugging information" );
    lr_warn( "encountered unexpected value: ", $value );
    open( CFGFILE, "my.conf" ) or lr_err( "open error: ", $! );

=head1 DESCRIPTION

This module contains functions that should be used by everything in
the Lire framework which should need logging.

=cut

BEGIN {
    my $prog    = basename $0;
    my $super   = $ENV{'LR_SUPERSERVICE'} || 'all';
    my $service = $ENV{'LR_SERVICE'} || 'all';
    my $lr_id   = $ENV{'LR_ID'}  || "UNSET";
    $lr_tag     = "$super $service $lr_id $prog";

    @EXPORT = qw/ lr_emerg lr_crit lr_err lr_warn lr_notice lr_info
                  lr_debug /;
}

=pod

All logging functions take any number of parameters that will be
joined together to form the message (like print(), die(), warn() and
friends).

=cut

sub label_msg {
    my ( $label, @msg ) = @_;

    # Maybe remove trailing newline of last part of the message.
    chomp $msg[$#msg];

    # Add $lr_tag err in front of all lines
    my $msg = join "", @msg, "\n";
    $msg =~ s/^/$lr_tag $label /mg;

    $msg;
}

=pod

=head2 lr_emerg()

This logs a message at the C<emerg> level and aborts your program. This
probably shouldn't be used.

=cut

sub lr_emerg {
    my $msg = label_msg( "emerg", @_ );
    die $msg;
}

=pod

=head2 lr_crit()

This logs a message at the C<crit> level and aborts your program. This
should only be used when something is really broken in the Lire
program or the environment. It is used a few places in Lire when
assertion that should really never fail (like a DLF file with the
wrong number of fields).

=cut

sub lr_crit {
    my $msg = label_msg( "crit", @_ );
    die $msg;
}

=pod

=head2 lr_err()

This logs a message at the C<err> level and usually aborts your
program. (It is the equivalent of a die().) The program won't abort if
it's used in an eval block. Uses this for error condition.

=cut

sub lr_err {
    my $msg = label_msg( "err", @_ );
    die $msg;
}

=pod

=head2 lr_warn()

This logs a message at the C<warning> level. Perl's builtin warn is
mapped to this function. Use this for non-fatal errors.

=cut

sub lr_warn {
    my $msg = label_msg( "warning", @_ );
    print STDERR $msg;
}

=pod

=head2 lr_notice()

This logs a message at the C<notice> level. This should be used for
significant informational messages that the user should see. (By
default, the user will only see messages at level C<notice> or
higher.)

=cut

sub lr_notice {
    my $msg = label_msg( "notice", @_ );
    print STDERR $msg;
}

=pod

=head2 lr_info()

This logs a message at the C<info> level. Use this for general
informational messages.

=cut

sub lr_info {
    my $msg = label_msg( "info", @_ );
    print STDERR $msg;
}

=pod

=head2 lr_debug()

This logs a message at the C<debug> level. Use this for debugging messages.

=cut

sub lr_debug {
    my $msg = label_msg( "debug", @_ );
    print STDERR $msg;
}

1;

__END__

=pod

=head1 SEE ALSO

Lire::Program(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: Logger.pm,v 1.10 2006/07/23 13:16:29 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2003 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html. 

=cut

