/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.infinispan.module.certificates;

import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.KeyPair;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import org.infinispan.commons.api.Lifecycle;
import org.infinispan.factories.annotations.ComponentName;
import org.infinispan.factories.annotations.Inject;
import org.infinispan.factories.annotations.Start;
import org.infinispan.factories.annotations.Stop;
import org.infinispan.factories.scopes.Scope;
import org.infinispan.factories.scopes.Scopes;
import org.infinispan.manager.EmbeddedCacheManager;
import org.infinispan.notifications.Listener;
import org.infinispan.notifications.cachemanagerlistener.CacheManagerNotifier;
import org.infinispan.notifications.cachemanagerlistener.annotation.Merged;
import org.infinispan.notifications.cachemanagerlistener.annotation.ViewChanged;
import org.infinispan.notifications.cachemanagerlistener.event.ViewChangedEvent;
import org.infinispan.remoting.transport.Address;
import org.infinispan.util.concurrent.BlockingManager;
import org.jboss.logging.Logger;
import org.keycloak.common.util.CertificateUtils;
import org.keycloak.common.util.KeyUtils;
import org.keycloak.common.util.Time;
import org.keycloak.infinispan.module.certificates.JGroupsCertificate;
import org.keycloak.infinispan.module.certificates.JGroupsCertificateHolder;
import org.keycloak.infinispan.module.certificates.ReloadCertificateFunction;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.storage.configuration.ServerConfigStorageProvider;

@Listener
@Scope(value=Scopes.GLOBAL)
public class CertificateReloadManager
implements Lifecycle {
    private static final Logger logger = Logger.getLogger(MethodHandles.lookup().lookupClass());
    public static final String CERTIFICATE_ID = "crt_jgroups";
    private static final String JGROUPS_SUBJECT = "jgroups";
    private static final Duration RETRY_WAIT_TIME = Duration.ofMinutes(1L);
    private static final Duration BOOT_PERIOD = Duration.ofMillis(500L);
    private final KeycloakSessionFactory sessionFactory;
    private final JGroupsCertificateHolder certificateHolder;
    private volatile long rotationSeconds;
    private final AutoCloseableLock lock;
    private ScheduledFuture<?> scheduledFuture;
    private ScheduledFuture<?> bootFuture;
    @Inject
    EmbeddedCacheManager cacheManager;
    @Inject
    CacheManagerNotifier notifier;
    @ComponentName(value="org.infinispan.executors.expiration")
    @Inject
    ScheduledExecutorService scheduledExecutorService;
    @Inject
    BlockingManager blockingManager;

    public CertificateReloadManager(KeycloakSessionFactory sessionFactory, JGroupsCertificateHolder certificateHolder, int rotationDays) {
        this.sessionFactory = sessionFactory;
        this.certificateHolder = certificateHolder;
        this.rotationSeconds = TimeUnit.DAYS.toSeconds(rotationDays);
        this.lock = new AutoCloseableLock(new ReentrantLock());
    }

    @Start
    public void start() {
        logger.info((Object)"Starting JGroups certificate reload manager");
        this.notifier.addListener((Object)this);
        this.scheduleNextRotation();
        this.certificateHolder.setExceptionHandler(this::onInvalidCertificate);
        this.lock.lock();
        try (AutoCloseableLock autoCloseableLock = this.lock;){
            this.bootFuture = this.scheduledExecutorService.scheduleAtFixedRate(() -> this.blockingManager.runBlocking(this::bootReload, (Object)"boot-reload"), BOOT_PERIOD.toMillis(), BOOT_PERIOD.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    @Stop
    public void stop() {
        logger.info((Object)"Stopping JGroups certificate reload manager");
        this.notifier.removeListener((Object)this);
        this.lock.lock();
        try (AutoCloseableLock autoCloseableLock = this.lock;){
            if (this.scheduledFuture == null) {
                return;
            }
            this.scheduledFuture.cancel(true);
        }
    }

    public void rotateCertificate() {
        logger.info((Object)"Rotating JGroups certificate");
        this.lock.lock();
        try (AutoCloseableLock autoCloseableLock = this.lock;){
            KeycloakModelUtils.runJobInTransaction((KeycloakSessionFactory)this.sessionFactory, this::replaceCertificateInTransaction);
            this.sendReloadNotification();
        }
        catch (RuntimeException e) {
            logger.warn((Object)"Failed to rotate JGroups certificate", (Throwable)e);
            this.retry(this::rotateCertificate, "retry-rotate");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void reloadCertificate() {
        logger.info((Object)"Reloading JGroups Certificate");
        this.lock.lock();
        try (AutoCloseableLock autoCloseableLock = this.lock;){
            Optional maybeCrt;
            if (this.bootFuture != null) {
                this.bootFuture.cancel(true);
                this.bootFuture = null;
            }
            if ((maybeCrt = (Optional)KeycloakModelUtils.runJobInTransactionWithResult((KeycloakSessionFactory)this.sessionFactory, CertificateReloadManager::loadCertificateInTransaction)).isEmpty()) {
                return;
            }
            JGroupsCertificate crt = JGroupsCertificate.fromJson((String)maybeCrt.get());
            this.certificateHolder.useCertificate(crt);
        }
        catch (IOException | GeneralSecurityException e) {
            logger.warn((Object)"Failed to reload JGroups certificate", (Throwable)e);
            this.retry(this::reloadCertificate, "retry-reload");
        }
        finally {
            this.scheduleNextRotation();
        }
    }

    @ViewChanged
    @Merged
    public void onViewChanged(ViewChangedEvent event) {
        logger.debug((Object)"On view changed");
        this.reloadCertificate();
    }

    public JGroupsCertificate currentCertificate() {
        return this.certificateHolder.getCertificateInUse();
    }

    public void setRotationSeconds(long seconds) {
        this.rotationSeconds = seconds;
    }

    public long getRotationSeconds() {
        return this.rotationSeconds;
    }

    public boolean isCoordinator() {
        return this.cacheManager.isCoordinator();
    }

    public boolean hasRotationTask() {
        this.lock.lock();
        try (AutoCloseableLock autoCloseableLock = this.lock;){
            boolean bl = this.scheduledFuture != null;
            return bl;
        }
    }

    private void bootReload() {
        logger.debug((Object)"[Boot] reloading certificate.");
        this.lock.lock();
        try (AutoCloseableLock autoCloseableLock = this.lock;){
            Optional maybeCrt = (Optional)KeycloakModelUtils.runJobInTransactionWithResult((KeycloakSessionFactory)this.sessionFactory, CertificateReloadManager::loadCertificateInTransaction);
            if (maybeCrt.isEmpty()) {
                return;
            }
            JGroupsCertificate crt = JGroupsCertificate.fromJson((String)maybeCrt.get());
            this.certificateHolder.useCertificate(crt);
        }
        catch (IOException | GeneralSecurityException e) {
            logger.warn((Object)"Exception on boot reload cycle. Ignoring it.", (Throwable)e);
        }
    }

    private void onInvalidCertificate() {
        logger.info((Object)"On certificate exception");
        this.blockingManager.runBlocking(this::reloadCertificate, (Object)"invalid-certificate");
    }

    private void onCertificateReloadResponse(Address address, Void unused, Throwable throwable) {
        if (throwable != null) {
            logger.warnf(throwable, "Node %s failed to handle JGroups certificate reload notification.", (Object)address);
            this.retry(() -> this.sendReloadNotification(address), "retry-notification");
        }
    }

    private void scheduleNextRotation() {
        this.lock.lock();
        try (AutoCloseableLock autoCloseableLock = this.lock;){
            if (this.scheduledFuture != null) {
                this.scheduledFuture.cancel(false);
            }
            if (!this.isCoordinator()) {
                return;
            }
            JGroupsCertificate crt = this.certificateHolder.getCertificateInUse();
            Duration delay = this.delayUntilNextRotation(Instant.ofEpochMilli(crt.getGeneratedMillis()), crt.getCertificate().getNotAfter().toInstant());
            logger.debugf("Next rotation in %s", (Object)delay);
            if (delay.isZero()) {
                this.blockingManager.runBlocking(this::rotateCertificate, (Object)"rotate");
                return;
            }
            this.scheduledFuture = this.scheduledExecutorService.schedule(() -> this.blockingManager.runBlocking(this::rotateCertificate, (Object)"rotate"), delay.toSeconds(), TimeUnit.SECONDS);
        }
    }

    private void replaceCertificateInTransaction(KeycloakSession session) {
        ServerConfigStorageProvider storage = (ServerConfigStorageProvider)session.getProvider(ServerConfigStorageProvider.class);
        JGroupsCertificate holder = this.certificateHolder.getCertificateInUse();
        storage.replace(CERTIFICATE_ID, holder::isSameAlias, () -> CertificateReloadManager.generateSelfSignedCertificate(this.rotationSeconds * 2L));
    }

    private static Optional<String> loadCertificateInTransaction(KeycloakSession session) {
        return ((ServerConfigStorageProvider)session.getProvider(ServerConfigStorageProvider.class)).find(CERTIFICATE_ID);
    }

    private Duration delayUntilNextRotation(Instant certificateStartInstant, Instant certificateEndInstant) {
        long secondsLeft;
        Instant rotationInstant = certificateStartInstant.plus(Duration.ofSeconds(this.rotationSeconds));
        Instant rotationInstantOldCertificate = certificateStartInstant.plus(Duration.between(certificateStartInstant, certificateEndInstant).dividedBy(2L));
        if (rotationInstantOldCertificate.isBefore(rotationInstant)) {
            rotationInstant = rotationInstantOldCertificate;
        }
        return (secondsLeft = Instant.ofEpochSecond(Time.currentTime()).until(rotationInstant, ChronoUnit.SECONDS)) > 0L ? Duration.ofSeconds(secondsLeft) : Duration.ZERO;
    }

    private void sendReloadNotification() {
        this.cacheManager.executor().allNodeSubmission().submitConsumer((Function)ReloadCertificateFunction.getInstance(), this::onCertificateReloadResponse);
    }

    private void sendReloadNotification(Address destination) {
        this.cacheManager.executor().filterTargets(arg_0 -> ((Address)destination).equals(arg_0)).submitConsumer((Function)ReloadCertificateFunction.getInstance(), this::onCertificateReloadResponse);
    }

    private void retry(Runnable runnable, String traceId) {
        this.scheduledExecutorService.schedule(() -> this.blockingManager.runBlocking(runnable, (Object)traceId), RETRY_WAIT_TIME.toSeconds(), TimeUnit.SECONDS);
    }

    public static String generateSelfSignedCertificate(long validForSeconds) {
        Date endDate = Date.from(Instant.now().plus(validForSeconds, ChronoUnit.SECONDS));
        KeyPair keyPair = KeyUtils.generateRsaKeyPair((int)2048);
        X509Certificate certificate = CertificateUtils.generateV1SelfSignedCertificate((KeyPair)keyPair, (String)JGROUPS_SUBJECT, (BigInteger)BigInteger.valueOf(System.currentTimeMillis()), (Date)endDate);
        logger.debugf("Created JGroups certificate. Valid until %s", (Object)certificate.getNotAfter());
        JGroupsCertificate entity = new JGroupsCertificate();
        entity.setCertificate(certificate);
        entity.setKeyPair(keyPair);
        entity.setAlias(UUID.randomUUID().toString());
        entity.setGeneratedMillis(System.currentTimeMillis());
        return JGroupsCertificate.toJson(entity);
    }

    private record AutoCloseableLock(ReentrantLock innerLock) implements AutoCloseable
    {
        public void lock() {
            this.innerLock.lock();
        }

        @Override
        public void close() {
            this.innerLock.unlock();
        }
    }
}

