"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateFieldContext = void 0;
var _lodash = require("lodash");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @template T = unknown - mapping Context
 * @template U = unknown - search Context
 * @param mappings - mapping from get mappings request
 * @param hits - search response that contains a sample document
 * @returns a string that describes fields, types, and sample values
 */
const generateFieldContext = (mappings, hits) => {
  var _hits$body$hits$hits$;
  const flattenedFields = flattenMappings(mappings);
  const source = (_hits$body$hits$hits$ = hits.body.hits.hits[0]) === null || _hits$body$hits$hits$ === void 0 ? void 0 : _hits$body$hits$hits$._source;
  return Object.entries(flattenedFields).filter(([, type]) => type !== 'alias') // PPL doesn't support 'alias' type
  .map(([field, type]) => {
    return `- ${field}: ${type} (${extractValue(source, field, type)})`;
  }).join('\n');
};
exports.generateFieldContext = generateFieldContext;
const extractValue = (source, field, type) => {
  const value = (0, _lodash.get)(source, field);
  if (value === undefined) return null;
  if (['text', 'keyword'].includes(type)) return `"${value}"`;
  return value;
};

/**
 * Flatten mappings response to an object of fields and types.
 *
 * @template T = unknown - Context
 * @param mappings - mapping from get mappings request
 * @returns an object of fields and types
 */
const flattenMappings = mappings => {
  const fields = {};
  Object.values(mappings.body).forEach(body => parseProperties(body.mappings.properties, undefined, fields));
  return fields;
};
const parseProperties = (properties, prefixes = [], fields) => {
  Object.entries(properties || {}).forEach(([key, value]) => {
    if (value.properties) {
      parseProperties(value.properties, [...prefixes, key], fields);
    } else {
      fields[[...prefixes, key].join('.')] = value.type;
    }
  });
  return fields;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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