!===============================================================================
! Copyright (C) 2025 Intel Corporation
!
! This software and the related documents are Intel copyrighted  materials,  and
! your use of  them is  governed by the  express license  under which  they were
! provided to you (License).  Unless the License provides otherwise, you may not
! use, modify, copy, publish, distribute,  disclose or transmit this software or
! the related documents without Intel's prior written permission.
!
! This software and the related documents  are provided as  is,  with no express
! or implied  warranties,  other  than those  that are  expressly stated  in the
! License.
!===============================================================================

!
!    Content : Intel(R) oneAPI Math Kernel Library (oneMKL) Inspector-Executor
!              Sparse BLAS Fortran example for the mkl_sparse_?_convert_dense
!              routine to convert from compressed sparse row (CSR) sparse matrix
!              format to a dense matrix representation.
!
!===============================================================================
!
!  Example program for using Intel(R) oneMKL Inspector-Executor Sparse BLAS
!  routines for conversion from sparse to dense matrix representation.
!
!  The following Inspector-Executor Sparse BLAS routines are used in the example:
!
!    Initialization/Destruction stage:
!           mkl_sparse_d_create_csr
!           mkl_sparse_destroy
!
!    Format conversion functions:
!           mkl_sparse_d_convert_dense
!
!
! Consider the matrix A below to be represented in compressed sparse row (CSR)
! format (see 'Sparse Matrix Storage Schemes' in the Intel(R) oneMKL C Reference
! Manual):
!        |  1  -2   0   0 |
!        |  3  -4   0   0 |
!    A = |  0   0   5  -6 |.
!        |  0   0   7  -8 |
!        |  9 -10   0   0 |
!        | 11 -12   0   0 |
!
!
! A CSR representation of the matrix with 1-based indexing and three arrays is
! given below:
!
!      csrNrows  = 6
!      csrNcols  = 4
!      csrNnz    = 12
!      csrIndex  = SPARSE_INDEX_BASE_ONE
!      csrRowPtr = (1       3       5       7       9      11      13)
!      csrColIdx = (1   2   1   2   3   4   3   4   1   2   1   2)
!      csrValues = (1  -2   3  -4   5  -6   7  -8   9 -10  11 -12)
!
! This example presents:
!    * mkl_sparse_d_create_csr() usage to create a handle for a sparse matrix in
!      CSR format
!    * mkl_sparse_d_convert_dense() usage to convert a sparse matrix from CSR to
!      dense format
!    * Validation of the conversion result by comparing the converted dense
!      matrix with the reference one
!
! Note: The mkl_sparse_?_convert_dense() API supports conversion from an input
!       matrix in any of the following sparse formats:
!       - Compressed Sparse Row (CSR)
!       - Compressed Sparse Column (CSC)
!       - Block Compressed Sparse Row (BSR)
!       - Coordinate (COO)
!       The arguments for the call to mkl_sparse_?_convert_dense will be the same
!       for all those formats. The only difference will be in the call to the
!       initialization routine for the sparse matrix, which needs to match the
!       corresponding sparse format.
!
!===============================================================================
!

SUBROUTINE print_int_value(name, val)
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER, INTENT(IN) :: val
    PRINT *, TRIM(ADJUSTL(name)), " = ", val
END SUBROUTINE print_int_value

SUBROUTINE print_int_array(name, array, len)
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER, INTENT(IN), DIMENSION(*) :: array
    INTEGER, INTENT(IN) :: len
    INTEGER :: i
    WRITE(*, '(A, A, A)', ADVANCE="NO") " ", TRIM(ADJUSTL(name)), " ="
    DO i = 1, len
        WRITE(*,  '(I3)', ADVANCE="NO") array(i)
        IF (i < len) WRITE(*, '(A)', ADVANCE="NO") ", "
    END DO
    PRINT *
END SUBROUTINE print_int_array

SUBROUTINE print_index(name, idxBase)
    USE MKL_SPBLAS
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    INTEGER(C_INT), INTENT(IN) :: idxBase
    CHARACTER(LEN=25) :: idxBaseStr
    IF (idxBase == SPARSE_INDEX_BASE_ZERO) THEN
        idxBaseStr = "SPARSE_INDEX_BASE_ZERO"
    ELSE
        idxBaseStr = "SPARSE_INDEX_BASE_ONE"
    END IF
    PRINT *, TRIM(ADJUSTL(name)), " = ", TRIM(idxBaseStr)
END SUBROUTINE print_index

SUBROUTINE print_flt_array(name, array, len)
    IMPLICIT NONE
    CHARACTER(LEN=*), INTENT(IN) :: name
    DOUBLE PRECISION, INTENT(IN), DIMENSION(*) :: array
    INTEGER, INTENT(IN) :: len
    INTEGER :: i
    WRITE(*, '(A, A, A)', ADVANCE="NO") " ", TRIM(ADJUSTL(name)), " ="
    DO i = 1, len
        WRITE(*,  '(F4.0)', ADVANCE="NO") array(i)
        IF (i < len) WRITE(*, '(A)', ADVANCE="NO") ", "
    END DO
    PRINT *
END SUBROUTINE print_flt_array

PROGRAM sparse_convert_dense_example
    USE MKL_SPBLAS
    USE ISO_C_BINDING
    IMPLICIT NONE

    INTEGER, PARAMETER :: M = 6, N = 4, NNZ = 12
    INTEGER :: exit_status
    TYPE(SPARSE_MATRIX_T) :: csrA
    TYPE(MATRIX_DESCR) :: descrA
    INTEGER(C_INT) :: indexing
    INTEGER(C_INT) :: info
    INTEGER(C_INT) :: i, j
    DOUBLE PRECISION, ALLOCATABLE :: denseMatrix(:)
    DOUBLE PRECISION, ALLOCATABLE :: denseMatrixRef(:)
    INTEGER         , ALLOCATABLE :: csrRowPtr(:), csrColIdx(:)
    DOUBLE PRECISION, ALLOCATABLE :: csrValues(:)
    INTEGER :: conversion_passed

    ! Initialize parameters
    indexing = SPARSE_INDEX_BASE_ONE
    descrA%type = SPARSE_MATRIX_TYPE_GENERAL

    ! Allocate data
    ALLOCATE(denseMatrix(M*N))
    ALLOCATE(denseMatrixRef(M*N))
    ALLOCATE(csrColIdx(NNZ))
    ALLOCATE(csrRowPtr(M+1))
    ALLOCATE(csrValues(NNZ))

    csrRowPtr = (/ 1, 3, 5, 7, 9, 11, 13 /)
    csrColIdx = (/ 1, 2, 1, 2, 3, 4, 3, 4, 1, 2, 1, 2 /)
    csrValues = (/ 1.0D0, -2.0D0, 3.0D0, -4.0D0, 5.0D0, -6.0D0, 7.0D0, -8.0D0, 9.0D0, -10.0D0, 11.0D0, -12.0D0 /)
    denseMatrixRef = (/  &
        1.0D0, 3.0D0, 0.0D0, 0.0D0,  9.0D0, 11.0D0, &
       -2.0D0,-4.0D0, 0.0D0, 0.0D0,-10.0D0,-12.0D0, &
        0.0D0, 0.0D0, 5.0D0, 7.0D0,  0.0D0,  0.0D0, &
        0.0D0, 0.0D0,-6.0D0,-8.0D0,  0.0D0,  0.0D0  &
    /)

    PRINT *, ""
    PRINT *, "Example program for conversion from a sparse matrix in CSR format to"
    PRINT *, " a dense matrix using IE Sparse BLAS APIs"
    PRINT *, "-------------------------------------------------------------------------"

    ! Create sparse matrix handle: csrA
    exit_status = 0
    info = mkl_sparse_d_create_csr(csrA, indexing, M, N, csrRowPtr(1), csrRowPtr(2), csrColIdx, csrValues)
    IF (info /= SPARSE_STATUS_SUCCESS) THEN
        PRINT *, " Error in mkl_sparse_d_create_csr: ", info
        exit_status = 1
        GOTO 999
    END IF

    PRINT *, "[Input] Matrix arrays in CSR format:"
    CALL print_int_value("nrows", M)
    CALL print_int_value("ncols", N)
    CALL print_int_value("nnz  ", NNZ)
    CALL print_index("index", indexing)
    CALL print_int_array("csrRowIdx", csrRowPtr, M+1)
    CALL print_int_array("csrColIdx", csrColIdx, NNZ)
    CALL print_flt_array("csrValues", csrValues, NNZ)
    PRINT *, ""

    ! Convert from CSR to DENSE format
    info = mkl_sparse_d_convert_dense(csrA, descrA, SPARSE_LAYOUT_COLUMN_MAJOR, M, denseMatrix)
    IF (info /= SPARSE_STATUS_SUCCESS) THEN
        PRINT *, " Error in mkl_sparse_d_convert_dense: ", info
        exit_status = 1
        GOTO 999
    END IF

    PRINT *, "[Output] Matrix array in dense format (converted from CSR):"
    CALL print_int_value("nrows", M)
    CALL print_int_value("ncols", N)
    PRINT *, "denseMatrix:"
    DO i = 1, M
        WRITE(*, '(A)', ADVANCE="NO") "    "
        DO j = 1, N
            WRITE(*, '(F7.0)', ADVANCE="NO") denseMatrix((i-1)*N + j)
        END DO
        PRINT *
    END DO
    PRINT *, ""

    ! Validation of the conversion result
    conversion_passed = 1
    DO i = 1, M*N
        IF (denseMatrix(i) /= denseMatrixRef(i)) THEN
            conversion_passed = 0
            exit_status = 1
            EXIT
        END IF
    END DO

    IF (conversion_passed == 1) THEN
        PRINT *, " The conversion from CSR format to dense passed"
    ELSE
        PRINT *, " The conversion from CSR format to dense failed"
        GOTO 999
    END IF
    PRINT *, ""

999 CONTINUE
    ! Release matrix handle
    info = mkl_sparse_destroy(csrA)
    IF (info /= SPARSE_STATUS_SUCCESS) THEN
        PRINT *, " Error in mkl_sparse_destroy: ", info
        exit_status = 1
    END IF

    call exit(exit_status)

END PROGRAM sparse_convert_dense_example
