/*
 * Copyright (C) 2013 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

define('data_utils', ['./utils'], function (Utils) {

"use strict"

var RowState = {
  collapsed: '+',
  expanded: '-',
  simple: ' ',
  waiting: 'w',
  none: ''
};

function buildValueDecoration(bar) {
  if (!Array.isArray(bar)) return '';

  var barsContent = '';
  bar.forEach(function (barSegment) {
    if (barSegment.width === '0%') return;

    var classes = Array.isArray(barSegment.classes) ? barSegment.classes : [];
    classes.push('idvcgrid_cell_item');

    var colorStr = '';
    if (barSegment.color) colorStr = 'color:' + barSegment.color + ';';

    barsContent += '<span class="' + classes.join(' ') + '" style="' +
    colorStr + 'width: ' + barSegment.width + ';"></span>';
  });

  if (barsContent) {
    var classes = Array.isArray(bar.classes) ? bar.classes : [];
    classes.push('idvcgrid_cell_items_container');
    return '<div class="' + classes.join(' ') + '">' + barsContent + '</div>';
  }

  return '';
}

function buildValueText(value, width) {
  if (value === undefined || value === null) return '';

  var result = value;

  if (width && value) {
    result = '<span class="idvcgrid_cell_text" style="width: ' + width + '">' + value + '</span>';
  }

  return result;
}

function decoratePercent(val, decimal) {
  if (val === '') {
    return val;
  }

  if (val > 100) val = 99.9999;

  var text = val;
  if (decimal !== undefined) {
    text = (+val).toFixed(decimal);
  }

  var result = '<span class="idvcgrid_cell_text">' + text + '%</span>';
  if (val > 0) {
    var color = 'rgb(' + Math.floor(val/100 * 255) + ', 0, ' +
      Math.floor((100 - val) / 100 * 255) + ')';

    result += buildValueDecoration([{color: color, width: text + '%'}]);
  }

  return result;
}

function decorateValue(val, maxValue, format) {
  if (val === '') {
    return val;
  }

  if (val > 0) {
    var text = val;
    if (typeof(format) === 'number') {
      text = (+val).toFixed(format);
    } else if (typeof(format) === 'function') {
      text = format(val);
    }

    if (maxValue) {
      var red = 255;
      if (val * 2 < maxValue) {
        red = Math.floor(2 * val / maxValue * 255);
      }

      var color = 'rgb(' + red + ',' + (255 - red) + ',0)';
      var width = Math.floor(val / maxValue * 100);
      if (width > 6) {
        return text + buildValueDecoration([{color: color, width: width + '%'}]);
      } else {
        return text;
      }
    } else {
      return text;
    }
  } else {
    return '0';
  }
}

function decorateValueEx(val, width, params) {
  var valueText = buildValueText(val, width);
  var bars = buildValueDecoration(params);
  if (valueText || bars) {
    return '<div class="idvcgrid_cell_content_container">' +
      valueText +
      bars +
      '</div>';
  }

  return ''
}

function decorateMemorySize(value, isFull) {
  function formatMemoryValue(value) {
    var classes = ['b', 'Kb ', 'Mb ', 'Gb'];
    var chunk = 1024;
    var result = '';
    var len = classes.length;
    for (var i = 0; i < len && value > 0; i++) {
      var newValue = Math.floor(value / chunk);
      result = (value - newValue * chunk).toFixed(0) + classes[i] + result;
      value = newValue;
    }

    if (!result.length) result = '0b';

    return result;
  }
  function formatMemoryValueShort(value) {
    var classes = ['b', 'Kb', 'Mb', 'Gb'];
    var chunk = 1024;
    var result = '0b';
    var len = classes.length;
    for (var i = 0; i < len && value > 1; i++) {
      if (value > 1) {
        result = value.toFixed((i > 0) ? 3 : 0) + classes[i];
      }
      value /= chunk;
    }

    return result;
  }

  if (isFull) {
    return formatMemoryValue(value);
  }

  return formatMemoryValueShort(value);
}

function decorateTime(val) {
  return (val.toFixed(0) / 1000) + 's';
}

function addSortingIndicator(caption, params) {
  var indicator = '';
  var sortingClass;

  var baseSortingClass = 'idvcgrid_column_sorting';
  if (caption.indexOf(baseSortingClass) < 0) {
    if (params.sortingWait !== undefined) {
      sortingClass = baseSortingClass + ' idvcgrid_sorting_waiting';
    } else if (params.sortingForward !== undefined) {
      sortingClass = baseSortingClass;
      if (!params.sortingForward) sortingClass += ' idvcgrid_sorting_backward';
    }

    if (sortingClass) indicator = '<span class="' + sortingClass + '"></span>';
  }

  return caption + indicator;
}

function decorateCaption(text, params) {
  var caption = '';
  var style = '';
  if (params && params.levelHeight &&
      params.height < params.levelHeight * 1.5) {
    style = ' style="white-space: nowrap;"';
  }

  caption = '<span class="idvcgrid_header_section_text"' +
              style + '>' + text +
              '</span>';

  caption = addSortingIndicator(caption, params);

  return caption;
}

function buildHeaderLegend(params) {
  if (!Array.isArray(params)) return '';

  var classes = ['idvcgrid_header_legend'].concat(params.classes).join(' ');

  var result = '<div class="' + classes + '">';

  params.forEach(function (param) {
    var colorClasses = ['idvcgrid_legend_color'].concat(param.colorClasses).join(' ');
    var textClasses = ['idvcgrid_legend_text'].concat(param.textClasses).join(' ');

    result += '<span class="' + colorClasses + '" style="color: ' + param.color + ';"></span>' +
      '<span class="' + textClasses + '">' + param.text + '</span>';
  });

  result += '</div>';

  return result;
}

function buildHeaderWithLegend(caption, legend) {
  var result =
    '<div class="idvcgrid_header_container">' +
      '<div class="idvcgrid_header_caption">' + caption +
      '</div>' +
      buildHeaderLegend(legend) +
    '</div>';

  return result;
}

var escapeHTML = (function() {
  var replacement = {
    '<': '&lt;',
    '>': '&gt;',
    '&': '&amp;',
    '"': '&quot;'
  };

  return function(val) {
    if (typeof val === 'string') {
      return val.replace(/[<>&"]/g, function(c) {
        return replacement[c];
      });
    }

    return val;
  };
})();

var unEscapeHTML = (function() {
  var replacement = {
    '&lt;'  : '<',
    '&gt;'  : '>',
    '&amp;' : '&',
    '&quot;': '"'
  };

  return function(val) {
    if (typeof val === 'string') {
      return val.replace(/\&[a-z]{2,4};/gi, function(c) {
        return replacement[c] || c;
      });
    }

    return val;
  };
})();

function getExpandWidgetStyle(text) {
  switch (text) {
    case RowState.collapsed:
      return ' idvcgrid_collapsed';
    case RowState.expanded:
      return ' idvcgrid_expanded';
    case RowState.simple:
      return ' idvcgrid_empty';
    case RowState.waiting:
      return ' idvcgrid_waiting';
    default:
      return '';
  }
}

function decorateSearchStr(content, searchStr, params) {
  function escapeSpecialChars(str) {
    return str.replace(/[.*+?^=!:${}()\-[\]|\/\\]/g, '\\$&');
  }

  var result = content;

  params = params || {};
  var className = params.className || 'idvcgrid_search_str';
  var currentIndex = (params.currentIndex !== undefined) ? params.currentIndex : -1;
  var currentClassName = params.currentClassName || 'idvcgrid_current_search_str';

  var flags = 'g';
  if (params.ignoreCase) {
    flags += 'i';
  }

  var index = 0;
  if (searchStr) {
    result = result.toString().replace(new RegExp(
        escapeSpecialChars(searchStr), flags), function(repl) {
      var name = className;
      if (index === currentIndex) {
        name = currentClassName;
      }
      index++;
      return '<span class="' + name + '">' + repl + '</span>'
    });
  }

  return result;
}

function decorateSearchPositions(content, searchPositions, params) {
  var result = content;

  if (searchPositions.length) {
    params = params || {};
    var className = params.className || 'idvcgrid_search_str';
    var currentIndex = (params.currentIndex !== undefined) ? params.currentIndex : -1;
    var currentClassName = params.currentClassName || 'idvcgrid_current_search_str';

    var currentPosition;
    if (currentIndex >= 0) {
      currentPosition = searchPositions[currentIndex]
    }

    for (var i = searchPositions.length - 1; i >= 0; i--) {
      var name = className;
      var position = searchPositions[i];
      if (currentPosition === position) {
        name = currentClassName;
      }
      var strBefore = result.slice(0, position.start);
      var searchStr = result.slice(position.start, position.end);
      var strAfter = result.slice(position.end);

      result = strBefore +
        '<span class="' + name + '">' + searchStr + '</span>' +
        strAfter;
    }
  }

  return result;
}

var levelIndent;
function getLevelIndent() {
  if (!levelIndent) {
    var expandDiv = document.createElement('div');
    expandDiv.className = 'idvcgrid_expand_collapse idvcgrid_collapsed';
    expandDiv.style.position = 'absolute';

    document.body.appendChild(expandDiv);

    var style = window.getComputedStyle(expandDiv, null);

    levelIndent = style.left;

    document.body.removeChild(expandDiv);
  }

  return levelIndent;
}

function getLevelSize(area) {
  var levelSize;

  if (area) {
    var expandDiv = document.createElement('div');
    expandDiv.className = 'idvcgrid_expand_collapse idvcgrid_collapsed';
    expandDiv.style.marginLeft = getLevelIndent();
    expandDiv.style.position = 'absolute';
    expandDiv.style.left = '0';

    area.appendChild(expandDiv);

    levelSize = {
      offset: expandDiv.offsetLeft,
      width: expandDiv.offsetWidth
    };

    area.removeChild(expandDiv);
  }

  return levelSize;
};

function getLevelsStart(area) {
  if (!area) return 0;

  var cellDiv = document.createElement('div');
  cellDiv.className = 'idvcgrid_cell';
  cellDiv.style.position = 'absolute';
  cellDiv.style.left = '0';
  cellDiv.style.top = '0';

  area.appendChild(cellDiv);

  var style = window.getComputedStyle(cellDiv, null);
  var levelsStart = parseInt(style.paddingLeft);

  area.removeChild(cellDiv);

  return levelsStart;
}

function getMarginLeft(level) {
  if (level) {
    return Utils.Consts.browserPrefix + 'calc(' + (level + '*' + getLevelIndent()) + ')';
  }

  return '0';
}

function decorateExpand(id, state, level) {
  if (state === RowState.none) return '';

  return '<span class="idvcgrid_widget idvcgrid_expand_collapse' +
          getExpandWidgetStyle(state) +
          '" id="' + id +
          '" style="margin-left: ' + getMarginLeft(level) + '" data-hierarchical-level=' + level + '></span>';
}

function decorateMore(moreText, id, level) {
  return '<span class="idvcgrid_widget idvcgrid_more_text' +
            '" onclick="event.stopPropagation();" ondblclick="event.preventDefault();" id="' + id +
            '" style="margin-left: ' + getMarginLeft(level) + '">' + moreText + '</span>';
}

function getContentLayout(content, controlArea, findSearch) {
  var result = {text: content, margin: 0};

  if (content.indexOf('idvcgrid_expand_collapse', 0) >= 0 &&
      controlArea) {
    var marginLeft = 'margin-left: ';
    var pos = content.indexOf(marginLeft) + marginLeft.length;
    var endMargin = content.indexOf(')', pos) + 1;
    var calcMargin = (endMargin > pos) ? content.slice(pos, endMargin) : '0';
    pos = content.indexOf('>', pos) + 1;
    pos = content.indexOf('>', pos) + 1;

    var textContent = content.slice(pos);

    var classAttr = 'class="';
    var classStart = content.indexOf(classAttr) + classAttr.length;
    var classEnd = content.indexOf('"', classStart);
    var className = content.slice(classStart, classEnd);
    className = className.replace('idvcgrid_expanded', 'idvcgrid_collapsed');

    var expandDiv = document.createElement('div');
    expandDiv.className = className;
    expandDiv.style.marginLeft = calcMargin;
    expandDiv.style.position = 'absolute';
    expandDiv.style.left = '0';

    controlArea.appendChild(expandDiv);

    var expandStyle = window.getComputedStyle(expandDiv, null);
    var expandDivRect = expandDiv.getBoundingClientRect();
    var contentAreaRect = controlArea.getBoundingClientRect();

    var margin = expandDivRect.left + expandDivRect.width + parseFloat(expandStyle.getPropertyValue('margin-right')) -
        contentAreaRect.left;

    controlArea.removeChild(expandDiv);

    result = {text: textContent, margin: margin};
  }

  if (findSearch) {
    result.search = [];

    var contentDiv = document.createElement('div');
    contentDiv.innerHTML = result.text;
    contentDiv.style.position = 'absolute';
    contentDiv.style.whiteSpace = 'pre';
    contentDiv.style.left = '0';

    controlArea.appendChild(contentDiv);

    var divLeft = Utils.getElementPos(contentDiv).x;

    result.contentWidth = contentDiv.scrollWidth + result.margin;

    var spans = contentDiv.querySelectorAll('span');
    [].forEach.call(spans, function(span) {
      if (span.className.indexOf('_search_str') >= 0) {
        var spanPos = Utils.getElementPos(span);
        result.search.push({left: spanPos.x - divLeft + result.margin, width: spanPos.width});
      }
    });

    controlArea.removeChild(contentDiv);
  }

  return result;
};

return {
  RowState: RowState,
  decoratePercent: decoratePercent,
  decorateValue: decorateValue,
  decorateValueEx: decorateValueEx,
  decorateMemorySize: decorateMemorySize,
  decorateTime: decorateTime,
  addSortingIndicator: addSortingIndicator,
  decorateCaption: decorateCaption,
  buildHeaderWithLegend: buildHeaderWithLegend,
  escapeHTML: escapeHTML,
  unEscapeHTML: unEscapeHTML,
  getLevelSize: getLevelSize,
  getLevelsStart: getLevelsStart,
  getExpandWidgetStyle: getExpandWidgetStyle,
  decorateSearchStr: decorateSearchStr,
  decorateSearchPositions: decorateSearchPositions,
  decorateExpand: decorateExpand,
  decorateMore: decorateMore,
  getContentLayout: getContentLayout
};

});

/*
 * Copyright (C) 2013 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

define('grid', ['./signal', './utils'], function(Signal, Utils) {

"use strict"

const CellTooltipAttr = 'data-cell-tooltip';

function loadGridStyles() {
  Utils.appendCSS(requirejs.toUrl('./grid_styles.css'));
}

function getLastParentByClass(elem, className) {
  var result = null;

  while (elem) {
    if (elem.className.indexOf &&
        elem.className.indexOf(className, 0) >= 0) {
      result = elem;
    } else if (result) {
      break;
    }
    elem = elem.parentElement;
  }

  return result;
}

function getFirstParentByClass(elem, className) {
  var result = null;

  while (elem) {
    if (elem.className.indexOf &&
        elem.className.indexOf(className, 0) >= 0) {
      result = elem;
      break;
    }
    elem = elem.parentElement;
  }

  return result;
}

const noRowHeight = 1;

//////////////////////////////////////////////////////////////////////////
//
//          VertScrollBar
//
//////////////////////////////////////////////////////////////////////////

function VertScrollBar(parentEl) {
  Utils.ScrolledHolder.call(this);

  this.scrollBody = document.createElement('div');
  this.scrollBody.className = 'idvcgrid_vertical_scroll';

  this.scrolledCoef = 1.;

  this.scrollSize = document.createElement('div');
  this.scrollSize.className = 'idvcgrid_scroll_size';

  this.scrollBody.appendChild(this.scrollSize);
  parentEl.appendChild(this.scrollBody);

  this.scrollBody.onscroll = processScroll.bind(this);
}

VertScrollBar.prototype = Object.create(Utils.ScrolledHolder.prototype);

function processScroll() {
  if (!this.scrollBody.offsetParent) return;

  this.setScrollTop(this.getScrollTop(), true);
}

const maxScrollSize = 10000000;

VertScrollBar.prototype.setScrollInfo = function(scrollSize, pageSize, scrollTop) {
  var scrollBarHeight = this.scrollBody.getBoundingClientRect().height;
  if (scrollBarHeight > 0) {
    if (scrollSize <= maxScrollSize) {
      this.scrolledCoef = pageSize / scrollBarHeight;
      this.scrollSize.style.height = this.fromArea(scrollSize) + 'px';
    } else {
      this.scrolledCoef = (scrollSize - pageSize) / (maxScrollSize - scrollBarHeight);
      this.scrollSize.style.height = maxScrollSize + 'px';
    }

    this.scrollBody.scrollTop = this.fromArea(scrollTop);
  }
};

VertScrollBar.prototype.toArea = function(size) {
  return size * this.scrolledCoef;
};

VertScrollBar.prototype.fromArea = function(size) {
  return size / this.scrolledCoef;
};

VertScrollBar.prototype.setScrollTop = function(val, noUpdateScrollBar) {
  this.syncScrollTop(val);

  if (!noUpdateScrollBar) {
    this._setScrollVal(val);
  }
};

VertScrollBar.prototype.getScrollTop = function() {
  return this.toArea(this.scrollBody.scrollTop);
};

VertScrollBar.prototype.hide = function() {
  return this.scrollBody.style.display = 'none';
};

VertScrollBar.prototype.show = function() {
  return this.scrollBody.style.display = 'block';
};

VertScrollBar.prototype.isVisible = function() {
  return this.scrollBody.style.display !== 'none';
};

VertScrollBar.prototype._setScrollVal = function(val) {
  if (this.endProcessTimeout) clearTimeout(this.endProcessTimeout);

  this.scrollBody.onscroll = undefined;

  this.scrollBody.scrollTop = this.fromArea(val);

  this.endProcessTimeout = setTimeout(function() {
    this.scrollBody.onscroll = processScroll.bind(this);
    delete this.endProcessTimeout;
  }.bind(this), 50);
};

//////////////////////////////////////////////////////////////////////////

//          GridColumn

//////////////////////////////////////////////////////////////////////////

var HeaderItemState = {
  collapsed: '+',
  expanded: '-',
  simple: ' '
};

function preventMouseAction(e) {
  e.stopPropagation();
  e.preventDefault();
}

function GridColumn(parentEl, dataIndex, state) {
  this.dataIndex = dataIndex;

  this.area = document.createElement('div');
  this.area.columnObj = this;

  this.header = document.createElement('div');
  this.scrollArea = document.createElement('div');
  this.footer = document.createElement('div');

  this.header.className = 'idvcgrid_header_section';
  this.scrollArea.className = 'idvcgrid_column_scrolled';
  this.footer.className = 'idvcgrid_footer_section';

  this.footer.onmousedown = preventMouseAction;
  this.footer.onmouseup = preventMouseAction;
  this.footer.onmousemove = preventMouseAction;

  this.setState(state);

  this.area.appendChild(this.scrollArea);
  this.area.appendChild(this.footer);
  this.area.appendChild(this.header);

  parentEl.appendChild(this.area);
}

GridColumn.prototype.setState = function(state) {
  if (!state) return;

  if (state !== HeaderItemState.simple &&
      !this.expandCollapse) {
    this.expandCollapse = document.createElement('div');
    this.header.appendChild(this.expandCollapse);
    this.header.classList.add('idvcgrid_header_expand_collapse');

    this.expandCollapse.onmousedown = function(ev) {
      ev.stopPropagation();
    };
    this.expandCollapse.onmouseup = this.expandCollapse.onmousedown;

    this.expandCollapse.onclick = function (ev) {
      ev.stopPropagation();

      var canceled = false;
      if (this.beforeExpandColumn) {
        this.beforeExpandColumn(this, {cancel: function() {canceled = true;}});
      }

      if (canceled) return;

      var visItem = this.visItem;
      if (visItem) {
        var state = visItem.getState();
        state = (state === HeaderItemState.expanded) ?
          HeaderItemState.collapsed :
          HeaderItemState.expanded;
        this.setState(state);
      }

      if (this.onExpandColumn) {
        this.onExpandColumn(this);
      }
    }.bind(this);
  }

  if (state === HeaderItemState.simple &&
      this.expandCollapse) {
    this.header.removeChild(this.expandCollapse);
    delete this.expandCollapse;
    this.header.classList.remove('idvcgrid_header_expand_collapse');
  }

  var className;
  if (this.expandCollapse) {
    className = 'idvcgrid_expand_collapse_column';
    if (state === HeaderItemState.expanded)
      className += ' idvcgrid_expanded_column';
    else
      className += ' idvcgrid_collapsed_column';
    this.expandCollapse.className = className;
  }

  var visItem = this.visItem;
  if (visItem) {
    visItem.setState(state);
  }

  className = 'idvcgrid_column';
  if (visItem && !visItem.isLeaf()) className += ' idvcgrid_inactive_column';
  this.area.className = className;

  this.resetLayout();
};

GridColumn.prototype.getState = function() {
  var result = HeaderItemState.simple;

  if (this.expandCollapse) {
    if (this.expandCollapse.classList.contains('idvcgrid_expanded_column'))
      result = HeaderItemState.expanded;
    else if (this.expandCollapse.classList.contains('idvcgrid_collapsed_column'))
      result = HeaderItemState.collapsed;
  }

  return result;
};

GridColumn.prototype.isLeaf = function() {
  var result = true;

  if (this.visItem) result = this.visItem.isLeaf();

  return result;
};

GridColumn.prototype.hide = function() {
  if (!this.canBeHidden()) return;

  var visItem = this.visItem;
  if (visItem) {
    visItem.setVisible(false);
  }
};

GridColumn.prototype.canBeHidden = function() {
  var visItem = this.visItem;
  if (visItem) {
    var visParent = visItem.getParent();
    if (visParent) {
      var count = 0;

      for (var visChild of visParent) {
        if (visChild.isVisible()) {
          count++;
        }
      }

      return count > 1;
    }
  }

  return false;
};

GridColumn.prototype.resetLayout = function() {
  if (this._columnLayout) this.setLayout(this._columnLayout);
};

var layoutPropertyPairs = [
  ['scrollArea', 'cells'],
  ['header', 'header'],
  ['area', 'column'],
  ['footer', 'footer']
];

GridColumn.prototype.setLayout = function(columnLayout) {
  function applyLayout(elem, layout) {
    if (!layout) return;

    var props = Object.getOwnPropertyNames(layout);
    props.forEach(function(prop) {
      if (prop !== 'className')
        elem.style[prop] = layout[prop];
      else
        elem.classList.add(layout.className);
    });
  }

  function clearLayout(elem, layout) {
    if (!layout) return;

    var props = Object.getOwnPropertyNames(layout);
    props.forEach(function(prop) {
      if (prop !== 'className')
        elem.style[prop] = '';
      else
        elem.classList.remove(layout.className);
    });
  }

  function process(funct) {
    if (!funct) return;

    layoutPropertyPairs.forEach(function(pair) {
      funct(this[pair[0]], this._columnLayout[pair[1]]);
    }.bind(this));
  }

  if (this._columnLayout) process.call(this, clearLayout);

  if (!columnLayout) {
    if (this._columnLayout) delete this._columnLayout;
  } else {
    this._columnLayout = columnLayout;

    process.call(this, applyLayout);

    // for compatibility
    if (columnLayout.width) {
      this.area.style.width = columnLayout.width;
    }

    if (columnLayout.textAlign) {
      this.scrollArea.style.textAlign = columnLayout.textAlign;
      this.footer.style.textAlign = columnLayout.textAlign;
    }
  }
};

GridColumn.prototype.setCaption = function(caption) {
  this.header.innerHTML = caption;
  if (this.expandCollapse) {
    this.header.appendChild(this.expandCollapse);
  }
};

GridColumn.prototype.setSortable = function(sortable) {
  if (sortable) Utils.removeClass(this.header, 'idvc_grid_notsortable');
  else Utils.addClass(this.header, 'idvc_grid_notsortable');
};

GridColumn.prototype.setScrollTop = function(val) {
  this.scrollArea.scrollTop = val;
};

GridColumn.prototype.getScrollTop = function() {
  return this.scrollArea.scrollTop;
};

GridColumn.prototype.getColumnWidth = function() {
  return this._columnWidth || this.area.offsetWidth;
  //return this.area.getBoundingClientRect().width;
};

GridColumn.prototype.setTop = function(top) {
  this._columnTop = top;
  this.area.style.top = top + 'px';
};

GridColumn.prototype.getTop = function() {
  return this._columnTop || 0;
};

GridColumn.prototype.setHeaderHeight = function(height) {
  this._columnHeaderHeight = height;

  var headerEl = this.header;
  headerEl.style.boxSizing = 'border-box';
  headerEl.style.height = height + 'px';
};

GridColumn.prototype.getHeaderHeight = function() {
  // return this.header.offsetHeight;

  if (!this._columnHeaderHeight) {
    this._columnHeaderHeight = this.header.getBoundingClientRect().height
  }

  return this._columnHeaderHeight;
};

GridColumn.prototype.getRowBufferSize = function() {
  return this.scrollArea.children.length;
};

function isHeaderIgnoredForAutoSize(ignore) {
  return ignore ||
    (this.visItem ? this.visItem.ignoreHeaderForAutoSize() : false);
}

GridColumn.prototype.prepare4OptimalWidth = function(ignoreHeader) {
  var maxContent = Utils.Consts.browserPrefix + 'max-content';

  this.scrollArea.style.width = maxContent;

  ignoreHeader = isHeaderIgnoredForAutoSize.call(this, ignoreHeader);

  if (!ignoreHeader) {
    this.header.style.width = maxContent;
  }

  this._prepared4OptimalWidth = true;
}

GridColumn.prototype.cleanUp4OptimalWidth = function(ignoreHeader) {
  ignoreHeader = isHeaderIgnoredForAutoSize.call(this, ignoreHeader);

  if (!ignoreHeader) {
    this.header.style.width = null;
  }
  this.scrollArea.style.width = null;

  if (this._prepared4OptimalWidth) delete this._prepared4OptimalWidth;
}

GridColumn.prototype.getOptimalWidth = function(extraSpace, ignoreHeader) {
  if (this._columnOptimalWidth) return this._columnOptimalWidth;

  extraSpace = extraSpace || Utils.em2px(1.25, this.area);

  ignoreHeader = isHeaderIgnoredForAutoSize.call(this, ignoreHeader);

  var result = 0;

  if (!this._prepared4OptimalWidth) {
    this.prepare4OptimalWidth(ignoreHeader);
    delete this._prepared4OptimalWidth;
  }

  if (!ignoreHeader) {
    result = this.header.offsetWidth;
  }

  result = Math.max(this.scrollArea.offsetWidth, result);

  if (!this._prepared4OptimalWidth) {
    this.cleanUp4OptimalWidth(ignoreHeader);
  }

  this._columnOptimalWidth = Math.round(result + extraSpace);

  return this._columnOptimalWidth;
};

GridColumn.prototype.clearRows = function() {
  Utils.removeAllChildren(this.scrollArea);
};

GridColumn.prototype.setLastColumnStyle = function() {
  var columnStyle = this.area.style;

  columnStyle.minWidth = columnStyle.width;
  columnStyle.width = 'auto';
  columnStyle.right = '0';
};

GridColumn.prototype.isLastColumn = function() {
  return this.area.style.width === 'auto';
};

GridColumn.prototype.clearLastColumnStyle = function() {
  if (!this.isLastColumn()) return;

  var columnStyle = this.area.style;

  columnStyle.width = columnStyle.minWidth;
  columnStyle.minWidth = '20px';
  columnStyle.right = 'auto';
};

GridColumn.prototype.getCell = function(index) {
  return this.scrollArea.children[index];
};

GridColumn.prototype.getRowsViewHeight = function() {
  return this.scrollArea.offsetHeight;
};

GridColumn.prototype.getRowsViewStart = function() {
  return this._scrollAreaTop || this.scrollArea.offsetTop;
};

GridColumn.prototype.getRowHeight = function(index) {
  index = index || 0;

  var child = this.getCell(index);
  return child ? child.getBoundingClientRect().height : noRowHeight;
};

GridColumn.prototype.updateRowStyle = function (row, childRow, selectionStyle, dataModel) {
  var isSelected = selectionStyle !== undefined;
  selectionStyle = selectionStyle || '';

  var rowElem = this.getCell(childRow);
  if (rowElem) fillCellStyle(rowElem, row, this.dataIndex, isSelected, selectionStyle, dataModel);
};

GridColumn.prototype.moveRows4Insert = function(to, count) {
  var area = this.scrollArea;
  var from = area.children.length - 1;

  if (from < 0) return;

  for (var j = 0; j < count; j++) {
    var cell = area.removeChild(area.children[from]);
    cell.className = 'idvcgrid_cell';
    area.insertBefore(cell, area.children[to]);
  }
};

GridColumn.prototype.moveRows4Remove = function(from, count, maxMoved) {
  // move to the end
  var moved = 0;
  var area = this.scrollArea;

  for (var j = 0; j < count; j++) {
    var elt = area.removeChild(area.children[from]);
    if (j < maxMoved) {
      area.appendChild(elt);
      moved++;
    }
  }

  return moved;
};

GridColumn.prototype.moveRows4ScrollBottom = function(count) {
  var area = this.scrollArea;

  for (var j = 0; j < count; j++) {
    var elt = area.removeChild(area.children[0]);
    area.appendChild(elt);
  }
};

GridColumn.prototype.moveRows4ScrollTop = function(count) {
  var area = this.scrollArea;
  var from = area.children.length - 1;

  if (from < 0) return;

  for (var j = 0; j < count; j++) {
    var elt = area.removeChild(area.children[from]);
    area.insertBefore(elt, area.children[0]);
  }
};

GridColumn.prototype.addRowsTop = function(count) {
  var area = this.scrollArea;

  for (var j = 0; j < count; j++) {
    var elt = document.createElement('div');
    area.insertBefore(elt, area.children[0]);
  }
};

GridColumn.prototype.addRowsBottom = function(count) {
  var area = this.scrollArea;

  for (var j = 0; j < count; j++) {
    var elt = document.createElement('div');
    area.appendChild(elt);
  }
};

GridColumn.prototype.fitChildrenCount = function(count) {
  var area = this.scrollArea;

  var removeCount = area.children.length - count;
  for (var j = 0; j < removeCount; j++) {
    area.removeChild(area.lastChild);
  }
};

GridColumn.prototype.fitRowsCount = function(count) {
  var currentCount = this.getRowBufferSize();
  if (count > currentCount) {
    this.addRowsBottom(count - currentCount);
  } else if (count < currentCount) {
    this.fitChildrenCount(count);
  }
};

function getCellStyle(isSelected, defSelectedStyle, row, col, dataModel) {
  var style;
  var defStyle = 'idvcgrid_cell';

  if (dataModel && dataModel.getCellStyle) {
    style = dataModel.getCellStyle(isSelected, defSelectedStyle, defStyle, row, col);
  }

  if (!style) {
    if (isSelected) style = defSelectedStyle;
    else style = defStyle;
  }

  return style;
}

function fillCellStyle(cell, row, col, isSelected, selectedStyle, dataModel) {
  let result = false;

  if (!cell) return result;

  var style = getCellStyle(isSelected, selectedStyle, row, col, dataModel);

  if (typeof style === 'string') {
    cell.className = style;
  } else if (typeof style === 'object') {
    if (style.className) {
      cell.className = style.className;
      delete style.className;
    }

    if (style.updateData) {
      fillCellData(cell, row, col, isSelected, dataModel);
      result = true;
      delete style.updateData
    }

    Utils.applyObjectProperties(cell, style);
  }

  return result;
}

function setCellContent(cell, newContent) {
  if (!cell) return;

  if (typeof newContent === 'string') {
    if (cell.isDirty) {
      cell.style.cssText = null;
      delete cell.isDirty;
    }

    cell.innerHTML = newContent;
  } else if (Utils.isHTMLNode(newContent)) {
    Utils.removeAllChildren(cell);
    cell.appendChild(newContent);
    cell.isDirty = true;
  } else if (typeof newContent === 'object') {
    Utils.applyObjectProperties(cell, newContent);
    cell.isDirty = true;
  }
}

function fillCellData(cell, row, col, isSelected, dataModel) {
  if (!cell || !dataModel) return;

  var newContent = dataModel.getCell(row, col, isSelected);

  if (typeof newContent === 'function') {
    newContent(cell, row, col, isSelected, setCellContent);
    cell.isDirty = true;
  } else {
    setCellContent(cell, newContent);
  }
}

function fillCell(cell, row, col, isSelected, selectedStyle, dataModel) {
  if (!cell) return;

  var updated = fillCellStyle(cell, row, col, isSelected, selectedStyle, dataModel);
  if (!updated) fillCellData(cell, row, col, isSelected, dataModel);
}

GridColumn.prototype.createRows = function(viewHeight, start, count, indexBefore,
    currentRow, selectedStyle, dataModel, getRowHeight) {
  if (this._columnOptimalWidth) delete this._columnOptimalWidth;

  if (dataModel) {
      //insert new rows into rows buffer
    var beforeEl = null;

    if (indexBefore !== undefined) {
      beforeEl = this.getCell(indexBefore);
    }

    var childrenLen = this.scrollArea.children.length;
    var rowCount = dataModel.getRowCount();
    if (childrenLen + count > rowCount) {
      count = rowCount - childrenLen;
    }

    var end = Math.min(start + count, rowCount);
    for (var j = start; j < end; j++) {
      var cell = document.createElement('div');
      fillCell(cell, j, this.dataIndex, j === currentRow,
        selectedStyle, dataModel);

      if (beforeEl) {
        this.scrollArea.insertBefore(cell, beforeEl);
      } else {
        this.scrollArea.appendChild(cell);
      }

      viewHeight -= getRowHeight(j);

      if (viewHeight < 0) {
        break;
      }
    }
  }
};

GridColumn.prototype.updateFooter = function(dataModel) {
  if (dataModel && dataModel.getFooter) {
    this.footer.innerHTML = dataModel.getFooter(this.dataIndex);
  }
};

GridColumn.prototype.regetRows = function(start, lastRow, from,
    currentRow, selectedStyle, dataModel) {
  if (this._columnOptimalWidth) delete this._columnOptimalWidth;

  if (dataModel) {
    for (var i = from, j = start; j <= lastRow; j++, i++) {
      var cell = this.getCell(i);
      fillCell(cell, j, this.dataIndex, j === currentRow,
        selectedStyle, dataModel);
    }
  }
};

GridColumn.prototype.updateRow = function(row, childIndex,
    currentRow, selectedStyle, dataModel) {
  if (this._columnOptimalWidth) delete this._columnOptimalWidth;

  if (dataModel) {
    var cellEl = this.getCell(childIndex);
    fillCell(cellEl, row, this.dataIndex, row === currentRow,
      selectedStyle, dataModel);
  }
};

GridColumn.prototype.updateLayout = function(dataModel) {
  var columnLayout = null;
  if (dataModel && dataModel.getColumnLayout) {
    columnLayout = dataModel.getColumnLayout(this.dataIndex);
  }

  this.setLayout(columnLayout);
};

GridColumn.prototype.updateColumn = function(rowFrom, lastRow,
    currentRow, selectedStyle, dataModel) {
  this.updateLayout(dataModel);

  if (dataModel) {
    this.regetRows(rowFrom, lastRow, 0, currentRow, selectedStyle, dataModel);
    this.updateFooter(dataModel);
  }
};

GridColumn.prototype.setWidth = function(width, isOptimal, noFitParent) {
  this._columnWidth = width;
  var widthStyle = width + 'px';

  if (this.isLastColumn()) {
    this.area.style.minWidth = widthStyle;
  } else {
    this.area.style.width = widthStyle;
  }

  var visItem = this.visItem;
  if (visItem &&
      visItem.isLeaf()) {
    if (isOptimal) {
      if (this.scrollArea.children.length) visItem.isOptimalWidth = true;
    } else {
      if (visItem.isOptimalWidth) delete visItem.isOptimalWidth;
    }

    visItem.setAutoSize(isOptimal);
    visItem.setWidth(widthStyle);
  } else if (!visItem) {
    if (isOptimal) {
      if (this.scrollArea.children.length) this.isOptimalWidth = true;
    } else {
      if (this.isOptimalWidth) delete this.isOptimalWidth;
    }
  }

  if (!noFitParent) this.fitParent();
};

GridColumn.prototype.setIsOptimalWidth =  function() {
  if (!this.scrollArea.children.length) return;

  var visItem = this.visItem;
  if (visItem &&
      visItem.isLeaf()) {
    visItem.isOptimalWidth = true;
  } else if (!visItem) {
    this.isOptimalWidth = true;
  }
};

GridColumn.prototype.setMaxAutoSizeWidth = function(maxWidth) {
  if (typeof maxWidth === 'number') {
    this._maxWidth = maxWidth;
  } else {
    delete this._maxWidth;
  }
};

GridColumn.prototype.getMaxAutoSizeWidth = function() {
  var result = this._maxWidth;
  if (!result && this.visItem) {
    var maxWidth = this.visItem.getMaxAutoSizeWidth();
    if (typeof maxWidth === 'string') {
      if (maxWidth.endsWith('em')) {
        result = Utils.em2px(parseInt(maxWidth), this.scrollArea);
      } else {
        result = parseInt(maxWidth);
      }
    } else if (typeof maxWidth === 'number') {
      result = maxWidth;
    }

    if (result) this.setMaxAutoSizeWidth(result);
  }

  return result;
};

GridColumn.prototype.setScrollAreaTop = function(top) {
  this.scrollArea.style.top = top + 'px';
  this._scrollAreaTop = top;
}

GridColumn.prototype.setLeft = function(left, noFitParent) {
  this.area.style.left = left + 'px';
  this._columnLeft = left;
  if (!noFitParent) this.fitParent();
};

GridColumn.prototype.getLeft = function() {
  return this._columnLeft !== undefined ? this._columnLeft : this.area.offsetLeft;
};

GridColumn.prototype.move = function(delta) {
  var left = this.getLeft();
  this.setLeft(left + delta);
};

GridColumn.prototype.getParentColumn = function() {
  var parent;
  if (this.visItem) {
    var visParent = this.visItem.getParent();
    if (visParent) parent = visParent.column;
  }
  return parent;
};

function getLastChildVisItem(parent) {
  var result;

  if (parent && !parent.isLeaf()) {
    for (let visChild of parent) {
      if (visChild.isVisible()) {
        result = visChild;
      }
    }
  }

  return result;
}

function getFirstChildVisItem(parent) {
  var result;

  if (parent && !parent.isLeaf()) {
    for (let visChild of parent) {
      if (visChild.isVisible()) {
        result = visChild;
        break;
      }
    }
  }

  return result;
}

GridColumn.prototype.getLastChild = function() {
  var result = this;
  var visItem = this.visItem;
  if (visItem && !visItem.isLeaf()) {
    var visChild = getLastChildVisItem(visItem);
    if (visChild) {
      result = visChild.column.getLastChild();
    }
  }
  return result;
};

GridColumn.prototype.getFirstChild = function() {
  var result = this;
  var visItem = this.visItem;
  if (visItem && !visItem.isLeaf()) {
    var visChild = getFirstChildVisItem(visItem);
    if (visChild) {
      result = visChild.column.getFirstChild();
    }
  }
  return result;
};

GridColumn.prototype.fitParent = function() {
  var visItem = this.visItem;
  if (visItem) {
    var visParent = visItem.getParent();
    if (visParent) {
      var parentColumn = visParent.column;
      if (parentColumn) {
        if (getFirstChildVisItem(visParent) === visItem) {
          var childLeft = this.getLeft();
          var parentLeft = parentColumn.getLeft();
          parentColumn.move(childLeft - parentLeft);
        }
        if (getLastChildVisItem(visParent) === visItem) {
          var childRight = this.getLeft() + this.getColumnWidth();
          var parentWidth = parentColumn.getColumnWidth();
          var parentRight = parentColumn.getLeft() + parentWidth;
          var delta = childRight - parentRight;
          parentColumn.setWidth(parentWidth + delta);

          if (this.isLastColumn() &&
              !parentColumn.isLastColumn()) {
            parentColumn.setLastColumnStyle();
          }
        }
      }
    }
  }
};

//////////////////////////////////////////////////////////////////////////

//          GridBody

//////////////////////////////////////////////////////////////////////////

function getContentStyle(elem) {
  var style = null;

  if (elem) {
    style = window.getComputedStyle(elem, null);
  } else {
    var button = document.createElement('button');
    button.style.position = 'absolute';
    button.style.left = '0';
    button.style.top = '0';
    document.body.appendChild(button);
    style = window.getComputedStyle(button, null);
    document.body.removeChild(button);
  }

  return style;
}

function GridBody(parentEl, tabIndex) {
  this.onDblClick = Signal.create();
  this.onContextMenu = Signal.create();
  this.onChangeCurrentRow = Signal.create();
  this.onFitHeaderHeight = Signal.create();
  this.beforeExpandColumn = Signal.create();
  this.onExpandColumn = Signal.create();
  this.onChangeColumnWidth = Signal.create();
  this.onShowTooltip = Signal.create();
  this.onSetFocus = Signal.create();
  this.onLostFocus = Signal.create();

  this.area = document.createElement('div');
  this.area.className = 'idvcgrid_body';

  if (tabIndex !== undefined) {
    this.area.setAttribute('tabindex', tabIndex);
  }

  this.connectedBody = null;
  parentEl.appendChild(this.area);

  this.columns = [];
  this.inActiveColumns = [];
  this.dataModel = null;

  this.scrolling = null;

  this._expandLastColumn = true;

  this.fitCentralColumnAsync = Utils.createAsyncCall(
    this.fitCentralColumnWidth, this, 100);

  this.currentChangeAsync = Utils.createAsyncCall(function(row, keyState) {
    this.onChangeCurrentRow.raise(row, keyState);
  }.bind(this), this, 50);

  this.S = Utils.createSequence();

  this._sizes = {};

  this.area.refreshSize = function(size) {
    if (this._needRefreshView) {
      delete this._needRefreshView;
      this.refreshView();
    } else {
      if (size && !size.width) return;

      this.fitCentralColumnHeader();
      this.fitLastColumn();

      this.refreshRowsContent();
    }
    return true;
  }.bind(this);

  var that = this;

  this._viewport = (function() {
    var topRow = 0;
    var bottomRow = 0;
    var scrollTop = 0;
    var currentRow = -1;
    var currentStyle = 'idvcgrid_cell idvcgrid_selected_row';

    function getVisibleColumnsRange() {
      return that.getVisibleColumnsRange() || {
        first: 0,
        last: that.columns.length - 1
      };
    }

    function forEachVisibleColumns(proc) {
      const range = getVisibleColumnsRange();

      for (let i = range.first; i <= range.last; i++) {
        let column = that.columns[i];
        if (column) proc(column, i)
      }
    }

    var getTopRow = function() {
      return topRow;
    };

    var getBottomRow = function() {
      return bottomRow;
    };

    var isExistingRow = function(row) {
      return row >= getTopRow() &&
        row <= getBottomRow();
    };

    function calcBottomRow() {
      var visibleRows = recalcVisibleRows();
      var visibleRowCount = visibleRows.count;
      if (!visibleRowCount) return;

      var result = topRow + visibleRowCount - 1;
      if (that.dataModel.getRowCount() <= result) {
        result = that.dataModel.getRowCount() - 1;
      }

      return result;
    }

    var updateBottomRow = function() {
      if (that.columns.length) {
        var newBottomRow = calcBottomRow();

        if (newBottomRow !== undefined) bottomRow = newBottomRow;
      }
    };

    var getChildIndex = function(row) {
      return row - getTopRow();
    };

    var getViewHeight = function() {
      return that.getRowsAreaHeight();
    };

    function getRowHeight(index) {
      return that.getRowHeight(index);
    }

    function getRowsHeight(from, to, rowHeight) {
      return that.getRowsHeight(from, to, rowHeight);
    }

    function getRowIndexByHeight(height, from, rowHeight) {
      return that.getRowIndexByHeight(height, from, rowHeight);
    }

    var updateRowStyle = function(inRow, style) {
      if (that.columns.length &&
          isExistingRow(inRow)) {
        var row = getChildIndex(inRow);

        if (row >= 0 &&
            row < that.getRowBufferSize()) {
          forEachVisibleColumns(function(column) {
            column.updateRowStyle(inRow, row, style, that.dataModel);
          });
        }
      }
    };

    var moveRows4Insert = function(to, count) {
      var chiledTo = getChildIndex(to);
      forEachVisibleColumns(function(column) {
        column.moveRows4Insert(chiledTo, count);
      });
    };

    var moveRows4Remove = function(from, count) {
      // move to the end
      var moved = 0;

      var rowCount = that.getRowBufferSize(true);
      var childFrom = getChildIndex(from);
      if (childFrom + count >= rowCount) {
        count = rowCount - childFrom;
      }

      var maxMoved = count - (topRow + rowCount -
                              that.dataModel.getRowCount());

      forEachVisibleColumns(function(column) {
        moved = column.moveRows4Remove(childFrom, count, maxMoved);
      });

      return moved;
    };

    var moveRows4ScrollBottom = function(count) {
      forEachVisibleColumns(function(column) {
        column.moveRows4ScrollBottom(count);
      });
    };

    var moveRows4ScrollTop = function(count) {
      forEachVisibleColumns(function(column) {
        column.moveRows4ScrollTop(count);
      });
    };

    var addRowsTop = function(count) {
      forEachVisibleColumns(function(column) {
        column.addRowsTop(count);
      });
    };

    var addRowsBottom = function(count) {
      forEachVisibleColumns(function(column) {
        column.addRowsBottom(count);
      });
    };

    function recalcVisibleRows(viewHeight, topRowInvisiblePart, newTopRow) {
      if (!that.getRowBufferSize()) return 0;

      if (newTopRow === undefined) newTopRow = getTopRow();

      viewHeight = viewHeight || getViewHeight();

      // add column scroll top (hidden part of the top row) to rows view height
      if (topRowInvisiblePart === undefined) {
        topRowInvisiblePart = getColumnsScrollTop();
      }

      var count = 0;
      viewHeight += topRowInvisiblePart;

      while (viewHeight > 0) {
        let rowHeight = getRowHeight(newTopRow + count) || 20;
        viewHeight -= rowHeight;
        count++;
      }

      if (that.dataModel.getRowCount() < count) {
        count = that.dataModel.getRowCount();
        if (count < 0) {
          count = 0;
        }
        viewHeight = 0;
      }

      return {
        count: count,
        isExtraCovered: viewHeight < 0
      };
    }

    var getVisibleRowCount = function(viewHeight, topRowInvisiblePart, newTopRow) {
      return recalcVisibleRows(viewHeight, topRowInvisiblePart, newTopRow).count;
    };

    function getColumnsScrollTop() {
      var result = 0;

      let column;
      if (column = that.getFirstColumn()) {
        result = column.getScrollTop();
      };

      return result;
    }

    return {
      clear: function() {
        topRow = 0;
        bottomRow = 0;
        scrollTop = 0;
        currentRow = -1;
      },
      setScrollTop: function(val) {
        if (val < 0) {
          val = 0;
        }

        function calcMaxScrollTop(rowCount, viewHeight) {
          var maxScrollTop = getRowsHeight(0, rowCount - 1) - viewHeight;
          if (maxScrollTop < 0) maxScrollTop = 0;
          return maxScrollTop;
        }

        var viewHeight = getViewHeight();

        var maxScrollTop = calcMaxScrollTop(that.dataModel.getRowCount(), viewHeight);
        if (val > maxScrollTop) {
          val = maxScrollTop;
        }

        if (val !== this.getScrollTop()) {
          scrollTop = val;

          var rowCount = that.getRowBufferSize();

          if (rowCount &&
              viewHeight) {
            var newTop = getRowIndexByHeight(val, 0);

            // scroll top for each column increase/decrease its size for one pixel
            // to make wholy visible top/bottom row
            var colScrollTop = val - getRowsHeight(0, newTop - 1);
            if (colScrollTop < 0) colScrollTop = 0;

            var regetStart = -1;
            var scrollCount = newTop - topRow;

            var neededChildren = getVisibleRowCount(viewHeight, colScrollTop, newTop);

            var newBottomRow = newTop + neededChildren - 1;
            if (that.dataModel.getRowCount() <= newBottomRow) {
              newBottomRow = that.dataModel.getRowCount() - 1;
            }

            if (rowCount < neededChildren &&
                -scrollCount > 0) {
              var addCount = neededChildren - rowCount;
              addRowsTop(addCount);
              topRow = newTop;
              bottomRow = newBottomRow;
              if (addCount === -scrollCount) {
                this.regetRows(topRow, -scrollCount);
              } else {
                this.regetRows(topRow);
              }
            } else {
              var oldBottom = bottomRow;

              this.fitRowsCount(neededChildren);
              bottomRow = newBottomRow;

              if (!scrollCount &&
                  oldBottom < bottomRow) {
                this.regetRows(oldBottom + 1, bottomRow - oldBottom);
              }

              topRow = newTop;

              if (Math.abs(scrollCount) < rowCount / 2) {
                if (scrollCount > 0) {
                  regetStart = oldBottom;
                  if (bottomRow === that.dataModel.getRowCount() - 1) {
                    regetStart--;
                  }
                  moveRows4ScrollBottom(scrollCount);
                  scrollCount = undefined;
                } else if (scrollCount < 0) {
                  regetStart = newTop;
                  scrollCount = -scrollCount;
                  moveRows4ScrollTop(scrollCount);
                }
              } else {
                regetStart = topRow;
                scrollCount = undefined;
              }

              if (regetStart >= 0) {
                this.regetRows(regetStart, scrollCount);
              }
            }

            this.setColumnsScrollTop(colScrollTop);
          }
        }
      },
      getScrollTop: function() {
        return scrollTop;
      },
      getTopRow: getTopRow,
      getBottomRow: getBottomRow,
      setColumnsScrollTop: function(columnsScrollTop) {
        this.columnsScrollTop = undefined;

        forEachVisibleColumns(function(column) {
          column.setScrollTop(columnsScrollTop);
        });

        var realColumnsScrollTop = this.getColumnsScrollTop();
        this.columnsScrollTop = realColumnsScrollTop;
        scrollTop += realColumnsScrollTop - columnsScrollTop;
      },
      getColumnsScrollTop: function() {
        if (this.columnsScrollTop === undefined) {
          this.columnsScrollTop = getColumnsScrollTop();
        }

        return this.columnsScrollTop;
      },
      updateCurrentRow: function(row) {
        currentRow = row;
      },
      setCurrentRow: function(newRow, oldRow) {
        var style = currentStyle;

        updateRowStyle(oldRow);

        currentRow = newRow;

        updateRowStyle(newRow, style);
      },
      getCurrentRow: function() {
        return currentRow;
      },
      getSelectionStyle: function() {
        return currentStyle;
      },
      insertRows: function(start, count) {
        // Rows buffer is completed; move rows from the end to inserted position
        if (start <= this.getCurrentRow()) {
          currentRow += count;
        }

        var rowCount = that.getRowBufferSize(true);
        var neededChildren = getVisibleRowCount();

        if (neededChildren <= rowCount) {
          if (start <= bottomRow &&
              start + count >= topRow) {
            if (start < topRow &&
                start + count >= topRow) {
              count -= topRow - start;
              start = topRow;
            }

            moveRows4Insert(start, count);
            this.regetRows(start);

            updateBottomRow();
          }
        } else {
          this.createRows(start, count, true);
        }
      },
      createRows: function(start, count, onlyVisible, isApplicableColumn) {
        if (that.dataModel) {
          //insert new rows into rows buffer
          var indexBefore;

          isApplicableColumn = isApplicableColumn || function() { return true; };

          if (start === undefined) {
            start = getTopRow();
          } else if (start <= getBottomRow()) {
            indexBefore = getChildIndex(start);
          }

          if (count === undefined) {
            count = that.dataModel.getRowCount();
          }

          var processStart, processCount;

          if (onlyVisible) {
            const visibleRange = getVisibleColumnsRange();

            processStart = visibleRange.from;
            processCount = visibleRange.last - visibleRange.from + 1;
          }

          var viewHeight = getViewHeight() + this.getColumnsScrollTop();

          that.S
          .do(function() {
            that.S.forEach(that.columns, function(column, index) {
              if (!column) return;

              if (isApplicableColumn(column, index)) column.createRows(viewHeight, start, count, indexBefore, this.getCurrentRow(),
                this.getSelectionStyle(), that.dataModel, getRowHeight);
            }.bind(this), thresholdColumnCount, processStart, processCount);
          }.bind(this))
          .do_(updateBottomRow);
        }
      },
      regetRows: function(start, count) {
        if (that.dataModel) {
          if (start === undefined ||
            start < getTopRow()) {
            start = getTopRow();
          }

          var lastRow = getBottomRow();
          if (count !== undefined &&
              start + count < lastRow) {
            lastRow = start + count - 1;
          }
          if (lastRow >= that.dataModel.getRowCount()) {
            lastRow = that.dataModel.getRowCount() - 1;
          }

          var startIndex = getChildIndex(start);

          forEachVisibleColumns(function(column) {
            column.regetRows(start, lastRow, startIndex,
              this.getCurrentRow(), this.getSelectionStyle(), that.dataModel);
          }.bind(this));
        }
      },
      removeRows: function(start, count) {
        if (!that.dataModel) return;

        if (that.getCurrentRow() >= start &&
            that.getCurrentRow() < start + count) {
          that.setCurrentRow(start - 1);
        }

        if (start < this.getCurrentRow()) {
          currentRow -= count;
        }

        if (start < topRow &&
            start + count >= topRow) {
          count -= topRow - start;
          start = topRow;
        }

        if (isExistingRow(start) &&
            that.columns.length > 0) {
          var moved = moveRows4Remove(start, count);
          updateBottomRow();
          if (moved > 0) {
            //this.regetRows(bottomRow - moved + 1, moved);
            this.regetRows(start);
          }

          this.refreshSize();
        }
      },
      updateRow: function(row) {
        if (isExistingRow(row) &&
            that.dataModel) {
          var index = getChildIndex(row);

          forEachVisibleColumns(function(column) {
            column.updateRow(row, index, this.getCurrentRow(),
              this.getSelectionStyle(), that.dataModel);
          }.bind(this));
        }
      },
      updateColumn: function(col) {
        var column = that.getColumnByDataIndex(col);
        if (that.dataModel &&
            column) {
          column.updateColumn(getTopRow(), getBottomRow(), this.getCurrentRow(),
              this.getSelectionStyle(), that.dataModel);
        }
      },
      updateCell: function(row, col) {
        if (isExistingRow(row)) {
          var column = that.getColumnByDataIndex(col);
          if (that.dataModel &&
              column) {
            var index = getChildIndex(row);
            column.updateRow(row, index, this.getCurrentRow(),
              this.getSelectionStyle(), that.dataModel);
          }
        }
      },
      refreshSize: function() {
        var newBottomRow = calcBottomRow();
        if (newBottomRow === undefined) return;

        if (newBottomRow > bottomRow) {
          var bottomCount = newBottomRow - bottomRow;
          addRowsBottom(bottomCount);
          bottomRow = newBottomRow;
          this.regetRows(bottomRow - bottomCount + 1, bottomCount);
        } else if (newBottomRow < bottomRow) {
          bottomRow = newBottomRow;
        }
      },
      getRowTop: function(row) {
        return getRowsHeight(topRow, row - 1);
      },
      calcExpand: function (expandedRow, expandedCount) {
        var result = scrollTop;
        if (expandedCount &&
            isExistingRow(expandedRow)) {
          var viewHeight = getViewHeight();

          var expandedHeight = getRowsHeight(expandedRow + 1, expandedRow + expandedCount);

          if (expandedHeight >= viewHeight) {
            result = getRowsHeight(0, expandedRow - 1);
          } else if (expandedRow + expandedCount > getBottomRow()) {
            result = getRowsHeight(0, expandedRow + expandedCount) - viewHeight;
          }
        }
        return result;
      },
      fitRowsCount: function(count) {
        count = count || getVisibleRowCount();

        forEachVisibleColumns(function(column) {
          column.fitRowsCount(count);
        });

        updateBottomRow();
      },
      refreshRowsContent: function(prevVisibleColumnsRange) {
        const createColumnsRows = (first, last) => {
          for (let i = first; i <= last; i++) {
            let column = that.columns[i];
            column.clearRows();
            column.createRows(viewHeight, start, count, undefined, this.getCurrentRow(),
                this.getSelectionStyle(), that.dataModel, getRowHeight);
          }
        }

        const scrollColumns = (first, last) => {
          for (let i = first; i <= last; i++) {
            that.columns[i].setScrollTop(columnsScrollTop);
          }
        }

        const clearColumnsRows = (first, last) => {
          for (let i = first; i <= last; i++) {
            that.columns[i].clearRows();
          }
        }

        prevVisibleColumnsRange = prevVisibleColumnsRange || {
          first: 0,
          last: that.columns.length - 1
        }

        const start = getTopRow();
        const columnsScrollTop = this.getColumnsScrollTop();
        const viewHeight = getViewHeight() + columnsScrollTop;

        const visibleColumnsRange = that.getVisibleColumnsRange();

        if (!visibleColumnsRange) return;

        const count = that.dataModel.getRowCount();

        if (visibleColumnsRange.first > prevVisibleColumnsRange.last ||
            visibleColumnsRange.last < prevVisibleColumnsRange.first) {
          if (prevVisibleColumnsRange.last >= 0) {
            clearColumnsRows(prevVisibleColumnsRange.first, prevVisibleColumnsRange.last);
          }

          createColumnsRows(visibleColumnsRange.first, visibleColumnsRange.last);
          scrollColumns(visibleColumnsRange.first, visibleColumnsRange.last);
        } else {
          if (prevVisibleColumnsRange.first !== visibleColumnsRange.first) {
            if (prevVisibleColumnsRange.first > visibleColumnsRange.first) {
              createColumnsRows(visibleColumnsRange.first, prevVisibleColumnsRange.first - 1);
              scrollColumns(visibleColumnsRange.first, prevVisibleColumnsRange.first - 1);
            } else {
              clearColumnsRows(prevVisibleColumnsRange.first, visibleColumnsRange.first - 1);
            }
          }

          if (prevVisibleColumnsRange.last !== visibleColumnsRange.last) {
            if (prevVisibleColumnsRange.last < visibleColumnsRange.last) {
              createColumnsRows(prevVisibleColumnsRange.last + 1, visibleColumnsRange.last);
              scrollColumns(prevVisibleColumnsRange.last + 1, visibleColumnsRange.last);
            } else {
              clearColumnsRows(visibleColumnsRange.last + 1, prevVisibleColumnsRange.last);
            }
          }
        }
      },
      getVisibleRowCount: getVisibleRowCount
    };
  })();

  var setTooltipPos = function(tooltipX, tooltipY, tooltip) {
    tooltip.classList.remove('idvcgrid_tooltip_ellipsis');

    if (tooltip.offsetWidth > 0.8 * window.innerWidth) {
      tooltip.style.height = 'auto';
      tooltip.style.width = Math.floor(0.8 * window.innerWidth) + 'px';
    }

    var cs = window.getComputedStyle(tooltip, null);

    var marginLeft = parseInt(cs.getPropertyValue('margin-left'), 10);
    var rightOffset = 6;
    var tooltipLeft = tooltipX;
    if (tooltipLeft < 0) tooltipLeft = 0;
    else if (tooltipLeft + marginLeft + tooltip.offsetWidth + rightOffset >
      document.documentElement.clientWidth + window.scrollX) {
      tooltipLeft = document.documentElement.clientWidth +
        window.scrollX - tooltip.offsetWidth - marginLeft - rightOffset;
    }

    var marginTop = parseInt(cs.getPropertyValue('margin-top'), 10);
    var bottomOffset = 2;
    var tooltipTop = tooltipY;
    if (tooltipTop < 0) tooltipTop = 0;
    if (tooltipTop + tooltip.offsetHeight + marginTop + bottomOffset >
      document.documentElement.clientHeight + window.scrollY) {
      tooltipTop = document.documentElement.clientHeight +
        window.scrollY - tooltip.offsetHeight - marginTop - bottomOffset;

      if (tooltipTop < 3) {
        tooltipTop = 0;
        tooltip.style.height = (document.documentElement.clientHeight -
          2 * (parseInt(cs.getPropertyValue('padding-top'), 10) + bottomOffset)) + 'px';
        tooltip.classList.add('idvcgrid_tooltip_ellipsis');
      }
    }

    tooltip.style.left = tooltipLeft + 'px';
    tooltip.style.top = tooltipTop + 'px';
  };

  var tooltipInfo = {
    elem: null,
    tooltip: null,
    timer: undefined
  };

  var hideTooltip = function() {
    if (tooltipInfo.tooltip) {
      document.body.removeChild(tooltipInfo.tooltip);
    }

    if (tooltipInfo.timer) {
      window.clearTimeout(tooltipInfo.timer);
      tooltipInfo.timer = undefined;
    }

    if (tooltipInfo.attrs) {
      delete tooltipInfo.attrs;
    }

    tooltipInfo.tooltip = null;
    tooltipInfo.elem = null;
  };

  function checkTooltipInfo() {
    const result = tooltipInfo.elem && tooltipInfo.elem.offsetParent;

    if (!result) hideTooltip();

    return result;
  }

  var showCellTooltip = function() {
    if (!tooltipInfo.elem) {
      return;
    }

    var tooltipText = tooltipInfo.elem.innerHTML;
    var tooltipMargin = 0;

    if (that.dataModel &&
        that.dataModel.getCellLayout) {
      var cellLayout = that.dataModel.getCellLayout(tooltipText);
      if (cellLayout) {
        tooltipText = cellLayout.text;
        tooltipMargin = cellLayout.margin;
      }
    }

    if (!tooltipText.length) {
      return;
    }

    if (!checkTooltipInfo()) return;

    var overCellPos = Utils.getElementPos(tooltipInfo.elem);
    overCellPos.x += tooltipMargin;

    if (overCellPos.x > lastMousePosition.x) {
      return;
    }

    var gridPos = Utils.getElementPos(that.area);
    var overWidth = tooltipInfo.elem.offsetWidth - tooltipMargin;
    if (overCellPos.x + overWidth > window.innerWidth + window.scrollX) {
      overWidth = window.innerWidth + window.scrollX -
        overCellPos.x;
    }
    if (overCellPos.x + overWidth > gridPos.x + that.area.clientWidth) {
      overWidth = gridPos.x + that.area.clientWidth - overCellPos.x;
    }

    var tooltip = document.createElement('div');
    tooltip.className = tooltipInfo.elem.className + ' idvcgrid_popup';
    tooltip.innerHTML = tooltipText;

    var elemStyle = window.getComputedStyle(tooltipInfo.elem, null);
    Utils.copyAppearance(tooltip, elemStyle);
    tooltip.style.lineHeight = elemStyle.lineHeight;
    tooltip.style.height = elemStyle.height;

    document.body.appendChild(tooltip);

    if ((tooltip.offsetWidth - 1 > overWidth) ||
        (gridPos.x > overCellPos.x)) {
      if (gridPos.x > overCellPos.x) {
        overCellPos.x = gridPos.x;
      }

      setTooltipPos(overCellPos.x, overCellPos.y, tooltip);
      tooltipInfo.tooltip = tooltip;
    } else {
      document.body.removeChild(tooltip);
    }
  };

  var showStdTooltip = function() {
    var attrs = tooltipInfo.attrs;

    if (!attrs ||
        !attrs.text.length) {
      return;
    }

    if (!checkTooltipInfo()) return;

    var complition = {completed: false};
    that.onShowTooltip.raise(attrs.x, attrs.y, attrs.text, attrs.row, attrs.col, tooltipInfo.elem, complition);
    if (complition.completed) return;

    var tooltip = document.createElement('div');
    tooltip.className = 'idvcgrid_popup idvcgrid_tooltip';
    tooltip.innerHTML = attrs.text;
    Utils.setFont(tooltip, getContentStyle(that.area));

    document.body.appendChild(tooltip);

    setTooltipPos(attrs.x, attrs.y, tooltip);
    tooltipInfo.tooltip = tooltip;
  };

  this.area.onmouseout = hideTooltip;

  var lastMousePosition = {x: -1, y: -1};
  var currentDrag = {
    x: -1,
    columnIndex: -1,
    columnObj: undefined,
    isStarted: false,
    isScrolling: false,
    scrollingInterval: undefined,
    areaPos: undefined,
    scrollingInfo: {
      scrollingDelta: 0
    },
    oldClassName: '',
    clear: function() {
      this.x = -1;
      this.columnIndex = -1;
      this.columnObj = undefined;
      this.isStarted = false;
      this.isScrolling = false;
      this.oldClassName = '';

      if (this.scrollingInterval) {
        clearInterval(this.scrollingInterval);
        this.scrollingInterval = undefined;
      }

      this.areaPos = undefined;

      if (this.columns) delete this.columns;
      if (this.maxX) delete this.maxX;
      if (this.oldClientHeight) delete this.oldClientHeight;
    }};

  this.area.onmousemove = function(e) {
    function showMousePosTooltip(tooltipText, row, col, elem) {
      tooltipInfo.elem = elem;
      tooltipInfo.attrs = {
        x: e.pageX,
        y: e.pageY,
        row: row,
        col: col,
        text: tooltipText
      };
      tooltipInfo.timer = window.setTimeout(showStdTooltip, 350);
    }

    e = e || event;
    if (e.pageX === lastMousePosition.x &&
        e.pageY === lastMousePosition.y) {
      return;
    }

    lastMousePosition.x = e.pageX;
    lastMousePosition.y = e.pageY;

    hideTooltip();

    if (!e.ctrlKey && !e.shiftKey &&
        that.columns.length > 0) {
      if (currentDrag.columnIndex < 0) {
        var res = that.hitTest(e);
        var row = res.row;
        if (row === -1) {
          if (res.columnObj) {
            var sepSize = 8;

            var column = res.columnObj;
            var columnPos = Utils.getElementPos(column.area);
            if (!column.isLastColumn() &&
                that.isResizable(column.dataIndex) &&
                e.pageX - columnPos.x > columnPos.width - sepSize) {
              column.header.style.cursor = 'ew-resize';
            } else {
              column.header.style.cursor = 'default';

              if (that.dataModel.getColumnDescription) {
                showMousePosTooltip(that.dataModel.getColumnDescription(column.dataIndex), row, res.columnObj.dataIndex, column.header);
              }
            }
          }
        } else if (row >= 0) {
          let target = e.target;

          if (target) {
            let cellTooltipText = target.getAttribute(CellTooltipAttr);

            let targetCell = getFirstParentByClass(target, 'cell');

            if (!cellTooltipText &&
                that.dataModel.getCellTooltip &&
                res.columnObj) {
              cellTooltipText = that.dataModel.getCellTooltip(row, res.columnObj.dataIndex);
            }

            if (cellTooltipText) {
              showMousePosTooltip(cellTooltipText, row, res.columnObj.dataIndex, targetCell);
            } else if (targetCell) {
              tooltipInfo.elem = targetCell;
              tooltipInfo.timer = window.setTimeout(showCellTooltip, 250);
            }
          }
        }
      }
    }
  };

  var resizingColumn = function(e) {
    function getScrollingDelta(posX, areaPos) {
      const maxDelta = 50;

      var delta = posX - (areaPos.x + areaPos.width);

      if (delta > maxDelta) delta = maxDelta;
      else if (delta < 0) delta = 0;

      return delta;
    }

    if (currentDrag.columnIndex >= 0 &&
        currentDrag.columnObj) {
      var delta = e.pageX - currentDrag.x;
      if (!currentDrag.isStarted) {
        if (Math.abs(delta) > 5) {
          currentDrag.isStarted = true;

          currentDrag.areaPos = Utils.getElementPos(that.area);

          globalCursor = document.createElement('div');
          globalCursor.className = 'idvcgrid_global_cursor';

          document.body.appendChild(globalCursor);

          if (that.isColumnResizingConstrainedByGridWidth()) {
            var bodyPos = Utils.getElementPos(that.area);
            var rightOffset = 20;
            currentDrag.maxX = bodyPos.x + bodyPos.width - rightOffset;
          }
        }
      } else {
        e.preventDefault();
        var resizingColumn = that.columns[currentDrag.columnIndex];
        if (resizingColumn) {
          var width = resizingColumn.getColumnWidth();
          delta = e.pageX - currentDrag.x;
          if ((currentDrag.maxX && currentDrag.maxX >= e.pageX || !currentDrag.maxX) &&
              width + delta >= 30) {
            let changer = {
              gridAreaChanged: false
            };

            that.setColumnWidth(currentDrag.columnIndex, Math.round(width + delta), false, changer);
            that.refreshRowsContent();

            currentDrag.x = e.pageX;

            if (changer.gridAreaChanged) {
              currentDrag.areaPos = Utils.getElementPos(that.area);
            }
          }
        }

        var scrollingDelta = getScrollingDelta(e.pageX, currentDrag.areaPos);

        currentDrag.scrollingInfo.scrollingDelta = scrollingDelta;

        if (!currentDrag.isScrolling &&
            scrollingDelta > 0) {
          currentDrag.isScrolling = true;

          currentDrag.scrollingInterval = setInterval(function(info) {
            var scrollingDelta = info.scrollingDelta;

            that.area.scrollLeft += scrollingDelta;

            currentDrag.x -= scrollingDelta;

            Utils.dispatchMouseEvent({x: currentDrag.x + scrollingDelta}, 'mousemove', window);
          }, 100, currentDrag.scrollingInfo);
        } else if (currentDrag.isScrolling &&
                    !scrollingDelta) {
          currentDrag.isScrolling = false;
          clearInterval(currentDrag.scrollingInterval);
          currentDrag.scrollingInterval = undefined;
        }
      }
    }
  };

  var forbidSelection = function() {
    return false;
  };

  var globalCursor = null;

  var stopResizingColumn = function() {
    if (currentDrag.columnObj) {
      currentDrag.columnObj.header.className = currentDrag.oldClassName;
    }

    var resized = currentDrag.isStarted;

    if (globalCursor) {
      document.body.removeChild(globalCursor);
      globalCursor = null;
    }

    window.removeEventListener('mousemove', resizingColumn, true);
    window.removeEventListener('mouseup', stopResizingColumn, true);
    window.removeEventListener('selectstart', forbidSelection, false);

    if (resized) {
      if (currentDrag.oldClientHeight !== that.area.clientHeight) {
        that.refreshLayout({height: true});
      }
      that.raiseColumnChanges(false);
    }

    currentDrag.clear();
  };

  var getToColumn = function(pageX, columns) {
    columns = columns || that.columns;

    var res = that.getColumnByPageX(pageX, columns);
    var result = res.columnIndex;

    if (result >= 0) {
      var column = columns[result];
      if (res.x > column.getColumnWidth() / 2) {
        result++;
      }
    } else {
      result = columns.length;
    }

    return result;
  };

  function getChildIndex(visItem) {
    var result = -1;

    if (!visItem) return result;

    var parentItem = visItem.getParent();
    if (parentItem) {
      for (let i = 0, len = parentItem.getChildrenCount(); i < len; i++) {
        let item = parentItem.getChild(i);
        if (item &&
            item.isVisible() &&
            item === visItem) {
          result = i;
          break;
        }
      }
    }

    return result;
  }

  var stopMovingColumn = function(e) {
    e = e || window.event;

    if (currentDrag.columnObj) {
      currentDrag.columnObj.header.className = currentDrag.oldClassName;
    }

    if (globalCursor) {
      that.area.removeChild(globalCursor.header);
      document.body.removeChild(globalCursor.pointer);
      globalCursor = null;
    }

    if (currentDrag.isStarted) {
      if (!that.columnsVisModel) {
        that.moveColumn(currentDrag.columnIndex, getToColumn(e.pageX), true);
      } else {
        let toIndex = -1;

        let afterLast = false;

        let toColumn = currentDrag.columns[getToColumn(e.pageX, currentDrag.columns)];
        if (!toColumn) {
          toColumn = currentDrag.columns[currentDrag.columns.length - 1];
          afterLast = true;
        }

        if (toColumn) {
          toIndex = getChildIndex(toColumn.visItem);
          if (afterLast) toIndex++;
        }

        if (toIndex >= 0) {
          let parentItem = currentDrag.columnObj.visItem.getParent();
          parentItem.moveChild(currentDrag.columnIndex, toIndex);
          that.refreshColumns();
        }
      }
    } else {
      that.clickColumnHeader(currentDrag.columnObj);
    }

    currentDrag.clear();

    window.removeEventListener('mousemove', movingColumn, true);
    window.removeEventListener('mouseup', stopMovingColumn, true);
    window.removeEventListener('selectstart', forbidSelection, false);
  };

  function buildMovingContent(column) {
    function createMovingSection(column, left, top) {
      if (!column || !column.area.offsetParent) return {section: undefined, left: 0, top:0, width: 0, height: 0};

      var section = document.createElement('div');
      section.className = 'idvcgrid_header_section idvcgrid_header_section_moving';
      section.innerHTML = column.header.innerHTML;

      var columnArea = column.area;
      section.style.left = left + 'px';
      section.style.top = top + 'px';
      section.style.width = columnArea.offsetWidth + 'px';
      var headerHeight = column.getHeaderHeight();
      section.style.height = headerHeight + 'px';

      return {
        section: section,
        left: columnArea.offsetLeft,
        top: columnArea.offsetTop,
        width: columnArea.offsetWidth,
        height: headerHeight
      };
    }

    function processVisItems(items, content, left, top) {
      if (!items) return;

      for (let item of items) {
        var sectionContent = undefined;

        if (item &&
            item.column &&
            item.isVisible()) {
          sectionContent = createMovingSection(item.column, left, top);

          var newBottom = content.top + top + sectionContent.height;
          if (content.bottom < newBottom) {
            content.bottom = newBottom;
          }

          if (sectionContent.section) content.childSections.push(sectionContent.section);
        }

        if (sectionContent) {
          processVisItems(item, content, left, top + sectionContent.height);
          left += sectionContent.width;
        }
      }
    }

    if (!column) return {section: undefined, left: 0, top: 0, height: 0};

    var sectionContainer = document.createElement('div');
    sectionContainer.className = 'idvcgrid_header_section_moving_container';

    var content = createMovingSection(column, 0, 0);
    if (column.visItem) {
      content.childSections = [];
      content.bottom = content.top + content.height;
      processVisItems(column.visItem, content, 0, content.height);
      content.height = content.bottom - content.top;
    }

    sectionContainer.style.left = content.left + 'px';
    sectionContainer.style.top = content.top + 'px';
    sectionContainer.style.height = content.height + 'px';
    sectionContainer.style.width = content.width + 'px';

    sectionContainer.appendChild(content.section);
    if (content.childSections) {
      content.childSections.forEach(sectionContainer.appendChild.bind(sectionContainer));
    }

    return {
      section: sectionContainer,
      left: content.left,
      top: content.top,
      height: content.height
    };
  }

  var movingColumn = function(e) {
    e = e || window.event;
    if (currentDrag.columnObj) {
      e.preventDefault();

      if (!that.isMovable(currentDrag.columnObj.dataIndex)) {
        return;
      }

      var delta = e.pageX - currentDrag.x;
      if (!currentDrag.isStarted) {
        if (Math.abs(delta) > 5) {
          let visItem = currentDrag.columnObj.visItem;
          if (visItem) {
            currentDrag.columnIndex = getChildIndex(visItem);

            var parentItem = visItem.getParent();
            currentDrag.columns = [];
            for (let item of parentItem) {
              if (item &&
                  item.isVisible()) {
                currentDrag.columns.push(item.column);
              }
            }

            if (currentDrag.columns.length <= 1) {
              delete currentDrag.columns;
              return;
            }
          }

          currentDrag.isStarted = true;

          var areaRect = Utils.getElementPos(that.area);

          var movingContent = buildMovingContent(currentDrag.columnObj);
          that.area.appendChild(movingContent.section);

          var arrow = document.createElement('div');
          arrow.className = 'idvcgrid_arrow_up';
          arrow.style.top = (areaRect.y + movingContent.top + movingContent.height) + 'px';
          arrow.style.left = (areaRect.x + movingContent.left - arrow.offsetWidth / 2) + 'px';
          document.body.appendChild(arrow);

          globalCursor = {header: movingContent.section, pointer: arrow, left: areaRect.x};
        }
      } else {
        var currentLeft = parseInt(globalCursor.header.style.left, 10);
        globalCursor.header.style.left = (currentLeft + delta) + 'px';
        currentDrag.x = e.pageX;
        var columns = currentDrag.columns || that.columns;

        var toColumn;
        var toIndex = getToColumn(e.pageX, currentDrag.columns);
        var leftDelta = 0;

        if (toIndex >= columns.length) {
          toColumn = columns[columns.length - 1];
          leftDelta = toColumn.area.offsetLeft + toColumn.area.offsetWidth - that.area.scrollLeft;
        } else if (toIndex >= 0) {
          toColumn = columns[toIndex];
          leftDelta = toColumn.area.offsetLeft - that.area.scrollLeft;
        }

        if (leftDelta < 0) leftDelta = 0;

        globalCursor.pointer.style.left = (globalCursor.left + leftDelta - globalCursor.pointer.offsetWidth / 2) + 'px';

        var calcScrollDelta = function() {
          var scrollDelta = 0;

          if (globalCursor && globalCursor.header) {
            if (globalCursor.header.offsetLeft +
                globalCursor.header.offsetWidth > that.area.scrollLeft +
                that.area.clientWidth) {
              scrollDelta = globalCursor.header.offsetLeft +
                globalCursor.header.offsetWidth -
                that.area.scrollLeft -
                that.area.clientWidth;
              scrollDelta = Math.min(scrollDelta, currentDrag.wholeWidth -
                                    that.area.scrollLeft - that.area.clientWidth);
              if (scrollDelta < 0) {
                scrollDelta = 0;
              }
            } else if (globalCursor.header.offsetLeft < that.area.scrollLeft &&
                    that.area.scrollLeft > 0) {
              scrollDelta = that.area.scrollLeft -
                globalCursor.header.offsetLeft;
              scrollDelta = Math.min(scrollDelta, that.area.scrollLeft);
              scrollDelta = -scrollDelta;
            }
          }

          return scrollDelta;
        };

        if (!globalCursor.isScrolling) {
          var scrollDelta = calcScrollDelta();
          if (scrollDelta !== 0) {
            globalCursor.isScrolling = true;

            var scrolling = setInterval(function() {
              var scrollDelta = calcScrollDelta();
              if (scrollDelta !== 0) {
                that.area.scrollLeft += scrollDelta;
                currentLeft = parseInt(globalCursor.header.style.left, 10);
                globalCursor.header.style.left =
                  (currentLeft + scrollDelta) + 'px';

                Utils.dispatchMouseEvent({x: currentDrag.x}, 'mousemove', window);
              } else {
                clearInterval(scrolling);
                if (globalCursor) globalCursor.isScrolling = false;
              }
            }, 200);
          }
        }
      }
    }
  };

  var onMouseDown = function(e) {
    e = e || event;
    if (that.columns.length > 0) {
      var res = that.hitTest(e);
      var row = res.row;
      if (row === -1) {
        if (res.columnObj &&
            e.button === 0 && !e.ctrlKey && !e.shiftKey) {
          currentDrag.x = e.pageX;
          var column = res.columnObj;
          currentDrag.columnObj = column;
          currentDrag.columnIndex = that.columns.indexOf(column);
          currentDrag.oldClassName = column.header.className;
          e.preventDefault();

          if (column.header.style.cursor === 'ew-resize') {
            column.header.className += ' idvcgrid_header_section_hover';
            currentDrag.isStarted = false;
            if (currentDrag.columnIndex < 0) {
              var lastChild = column.getLastChild();
              if (lastChild) {
                currentDrag.columnIndex = that.columns.indexOf(lastChild);
              }
            }

            currentDrag.oldClientHeight = that.area.clientHeight;

            window.addEventListener('mousemove', resizingColumn, true);
            window.addEventListener('mouseup', stopResizingColumn, true);
            window.addEventListener('selectstart', forbidSelection, false);
          } else if (that.columns.length > 0) {
            currentDrag.isStarted = false;
            currentDrag.wholeWidth = that.area.scrollWidth;

            if (that.isSortable(column.dataIndex)) {
              column.header.className += ' idvcgrid_header_section_active';
            }

            if (that.columns.length > 1) {
              window.addEventListener('mousemove', movingColumn, true);
            }
            window.addEventListener('mouseup', stopMovingColumn, true);
            window.addEventListener('selectstart', forbidSelection, false);
          }
        }
      } else if (row >= 0) {
        if (e.button !== 0 && that.isRowSelected(row)) return;

        that.setCurrentRow(row, false,
          {ctrl: e.ctrlKey, alt: e.altKey, shift: e.shiftKey, toggle: e.ctrlKey, isMouse: true});
        that.rowVisible(row);
      }
    }
  };

  this.area.addEventListener('mousedown', onMouseDown, false);

  function processMouseWheel(e) {
    e = e || window.event;

    var coef = Utils.Consts.engine === 'webkit' ? 200 : 3;
    var wheelDelta = e.deltaY / coef;

    if (!e.shiftKey) {
      var newScroll = this.getScrollTop() + wheelDelta * this.getRowHeight(0);
      this.setScrollTop(newScroll);
    } else {
      this.area.scrollLeft += wheelDelta * this.area.offsetWidth * 0.1;
      e.stopPropagation();
      e.preventDefault();
    }
  }

  this.area.addEventListener('wheel', processMouseWheel.bind(this), false);

  function makeEventParams(row) {
    return {
      isFooter: row === -2,
      isHeader: row === -1,
      isRow: row >= 0
    };
  }

  this.area.ondblclick = function(e) {
    e = e || event;
    if (!e.ctrlKey && !e.shiftKey &&
        that.columns.length > 0) {
      var res = that.hitTest(e);

      var row = res.row;
      var column = res.columnObj;
      if (column && row === -1 &&
          column.header.style.cursor === 'ew-resize') {
        column = column.getLastChild();
        that.fitColumnOptimalWidth(that.columns.indexOf(column), column.getMaxAutoSizeWidth());
        that.raiseColumnChanges(false);
        that.refreshRowsContent();
      } else {
        that.onDblClick.raise(row, column ? column.dataIndex : -1, makeEventParams(row));
      }
    }
  };

  this.area.oncontextmenu = function(e) {
    e = e || event;
    if (!e.ctrlKey && !e.shiftKey &&
        that.columns.length > 0) {
      var res = that.hitTest(e);

      var row = res.row;
      var column = res.columnObj;
      var propagation = {stopPropagation: true};
      that.onContextMenu.raise(row, column ? column.dataIndex : -1, propagation, makeEventParams(row), e.pageX, e.pageY, e);

      if (that.onContextMenu._subscribers.length && propagation.stopPropagation) {
        e.stopPropagation();
      }
    }
  };

  this.area.onselectstart = function() {
    return false;
  };

  this.area.onkeydown = function(e) {
    var keyState = {ctrl: e.ctrlKey, alt: e.altKey, shift: e.shiftKey, toggle: e.keyCode === 32};
    if (e.keyCode === 33) {
      that.currentPgUp(keyState);
      e.preventDefault();
    } else if (e.keyCode === 34) {
      that.currentPgDown(keyState);
      e.preventDefault();
    } else if (e.keyCode === 35) {
      that.currentEnd(keyState);
      e.preventDefault();
    } else if (e.keyCode === 36) {
      that.currentHome(keyState);
      e.preventDefault();
    } else if (e.keyCode === 38) {
      that.currentUp(keyState);
      e.preventDefault();
    } else if (e.keyCode === 40) {
      that.currentDown(keyState);
      e.preventDefault();
    } else if (e.keyCode === 13) {
      if (that.getCurrentRow() >= 0) {
        that.onDblClick.raise(that.getCurrentRow());
      }
    } else if (e.keyCode === 32) {
      that.setCurrentRow(that.getCurrentRow(), false, keyState);
    } else if (e.keyCode === 65 && (e.ctrlKey || e.metaKey)) {
      keyState.all = true;
      that.setCurrentRow(that.getCurrentRow(), false, keyState);
      e.preventDefault();
    }
  };

  this.area.onfocus = function() {
    this.processSetFocus(true);
  }.bind(this);

  this.area.onblur = function() {
    that.processLostFocus(true);
  };

  this.refreshRowsContentCall = Utils.createAsyncCall(
    this.refreshRowsContent, this, 50);

  this.area.onscroll = function() {
    this.fitCentralColumnHeader();

    this.refreshRowsContentCall.call();

    if (Utils.Consts.engine === 'moz') {
      if (!this.hoverEnableAsync) {
        this.hoverEnableAsync = Utils.createAsyncCall(
          Utils.enableHover, undefined, 250);
      }

      Utils.disableHover(this.area);

      this.hoverEnableAsync.call(this.area);
    }
  }.bind(this);
}

GridBody.prototype.hitTest = function(ev) {
  var row = -1;

  var column = getColumnByElement(ev.target);
  if (column &&
      this.getRowsAreaHeight()) {
    var elemPos = Utils.getElementPos(column.scrollArea);
    if (!elemPos.height && !elemPos.width) {
      row = -1; // inactive column
    } else {
      var top = ev.pageY - elemPos.y;
      if (top > elemPos.height) {
        row = -2;
      } else if (top > 0) {
        var targetCell = getLastParentByClass(ev.target, 'cell');
        if (targetCell) {
          row = 0;
          var sibling = targetCell;
          while (sibling = sibling.previousSibling) row++;
          row += this._viewport.getTopRow();
        } else {
          row = -3;
        }
      }
    }
  } else if (!column) {
    row = -3;
  }

  return {
    row: row,
    columnObj: column
  };
};

function getColumnByElement(element) {
  var columnObj;
  if (element) {
    var parent = element.parentElement;
    if (parent) {
      columnObj = parent.columnObj;
      if (!columnObj) {
        columnObj = getColumnByElement(parent);
      }
    }
  }

  return columnObj;
}

GridBody.prototype.getColumnByPageX = function(val, columns) {
  columns = columns || this.columns;
  var result = -1;
  var obj;

  var len = columns.length;
  if (len) {
    var elemPos = Utils.getElementPos(this.area);
    var left = val - elemPos.x + this.area.scrollLeft - columns[0].getLeft();

    for (var i = 0; i < len; i++) {
      left -= columns[i].getColumnWidth();
      if (left <= 0) {
        result = i;
        obj = columns[i];
        left = columns[i].getColumnWidth() + left;
        break;
      }
    }
  }

  return {
    columnIndex: result,
    x: left,
    columnObj: obj
  };
};

GridBody.prototype._isActive = function() {
  return this.columns.length > 0;
};

GridBody.prototype._updateScrollTop = function(val) {
  if (val !== this.getScrollTop()) {
    this._viewport.setScrollTop(val);

    if (this.isCentralColumn()) {
      this.fitCentralColumnAsync.call();
    }
  }
};

GridBody.prototype._endScroll = function() {
  if (this._isWaitingForDataUpdate) return;

  if (this.rowsScrollWaiter) {
    this.rowsScrollWaiter.resolve(true);
    delete this.rowsScrollWaiter;
  }
};

GridBody.prototype._updateVisibleScrollTop = function(val) {
  this._viewport.setColumnsScrollTop(val);
};

GridBody.prototype._getScrollState = function() {
  return {
    visibleScrollTop: this._viewport.getColumnsScrollTop(),
    topIndex: this._viewport.getTopRow()
  };
};

GridBody.prototype.setScrollTop = function(val) {
  if (val !== this.getScrollTop()) {
    const result = new Promise(resolve => {
      this.rowsScrollWaiter = {
        resolve
      };
    });

    this.scrolling.setScrollTop(val);

    return result;
  }

  return Promise.resolve(false)
};

GridBody.prototype.getScrollTop = function() {
  return this._viewport.getScrollTop();
};

GridBody.prototype.clearView = function() {
  this.setScrollTop(0);

  Utils.removeAllChildren(this.area);

  this.columns.length = 0;
  this.inActiveColumns.length = 0;
  this._viewport.clear();
};

GridBody.prototype.clearRows = function(currentRow) {
  this.setScrollTop(0);

  for(var i = 0, len = this.columns.length; i < len; i++) {
    this.columns[i].clearRows();
  }

  this._viewport.clear();

  if (currentRow === undefined) {
    this._viewport.updateCurrentRow(-1);
  } else {
    this._viewport.updateCurrentRow(currentRow);
  }
};

GridBody.prototype.refreshRowsContent = function() {
  const prevVisibleColumnsRange = this.visibleColumnsRange;
  this.visibleColumnsRange = undefined;

  this._viewport.refreshRowsContent(prevVisibleColumnsRange);

  if (this.columnsScrollWaiter) {
    this.columnsScrollWaiter.resolve(true);
    delete this.columnsScrollWaiter;
  }
};

GridBody.prototype.getFirstColumn = function() {
  if (!this.columns || !this.columns.length) return undefined;

  if (this.visibleColumnsRange) return this.columns[this.visibleColumnsRange.first];

  return this.columns[0];
};

GridBody.prototype.refreshDataModel = function() {
  if (this.dataModel &&
      this.dataModel.setElement !== undefined) {
    this.dataModel.setElement(this);
  }
};

GridBody.prototype.refreshRows = function(currentRow, rowCount) {
  var keepRows = (rowCount !== undefined &&
                  this.dataModel &&
                  this.dataModel.getRowCount() === rowCount);

  if (!keepRows) {
    this.clearRows(currentRow);
    this._viewport.createRows(undefined, undefined, true);
  } else {
    this._viewport.regetRows();
    this.setCurrentRow(currentRow);
  }

  this.refreshDataModel();
};

GridBody.prototype.refreshView = function() {
  Utils.removeAllChildren(this.area);

  this.visibleColumnsRange = undefined;

  this.columns.length = 0;
  this.inActiveColumns.length = 0;

  this.updateView();
};

GridBody.prototype.refreshColumns = function() {
  var scrollLeft = this.area.scrollLeft;
  var scrollTop = this._viewport.getColumnsScrollTop();
  if (scrollLeft) this.area.scrollLeft = 0;

  this.S
  .do_(this.refreshView.bind(this))
  .do_(function() {
    this.area.scrollLeft = scrollLeft;
    this.scrolling.updateVisibleScrollTop(scrollTop);

    this.raiseColumnChanges(true);
  }.bind(this));
};

GridBody.prototype.raiseColumnChanges = function(global) {
  if (this.columnsVisModel && this.columnsVisModel.onChange) {
    this.columnsVisModel.onChange.raise(global);
  }
};

GridBody.prototype.processExpandColumn = function(column) {
  this.refreshColumns();
  this.onExpandColumn.raise(column.dataIndex, column.visItem.getState());
};

GridBody.prototype.processBeforeExpandColumn = function(column, canceller) {
  this.beforeExpandColumn.raise(column, canceller);
};

GridBody.prototype.refreshScroll = function() {
  if (this.onRefreshScroll) this.onRefreshScroll();
};

GridBody.prototype.updateView = function() {
  if (!this.area.offsetParent) {
    this._needRefreshView = true;
    return;
  }

  var rowsCreation = { count: 0 }

  this._sizes = {};
  if (this.dataModel) {
    this.S
    .do_(this.insertColumns.bind(this))
    .do_(function() {
      this.refreshDataModel();
      this.updateFooter();

      if (this.isCentralColumn()) {
        this._viewport.createRows();
        rowsCreation.count = 1;
      } else {
        this.applyColumnAutoSize(undefined, rowsCreation);
      }
    }.bind(this))
    .do_(function() {
      if (rowsCreation.count) this.visibleColumnsRange = undefined;
      else this.visibleColumnsRange = { first: -1, last: -1 };

      this.refreshRowsContent();
    }.bind(this))
    .do_(function() {
      this._viewport.refreshSize();
      this.refreshScroll();
      if (this.isCentralColumn()) {
        this.fitCentralColumnAsync.call();
      }
    }.bind(this));
  }
};

function updateColumnCaption(column, state) {
  if (this.dataModel && column) {
    var params = {
      state: state || column.getState(),
      width: 300, // some default column width
      height: column.getHeaderHeight() || this.headerLevelHeight,
      levelHeight: this.headerLevelHeight
    };
    column.setCaption(this.dataModel.getColumnCaption(column.dataIndex, params));
  }
};


GridBody.prototype.insertColumns = function() {
  var processedColumns = 0;
  var isCanceled = false;

  function addColumn(dataIndex, state, isLeaf) {
    if (isCanceled) return;

    var column = new GridColumn(this.area, dataIndex, state);

    if (!this.isSortable(dataIndex)) {
      column.setSortable(false);
    }

    column.updateLayout(this.dataModel);

    if (isLeaf) this.columns.push(column);
    else this.inActiveColumns.push(column);

    return column;
  }

  var columnSizes = {
    headerBottom: 0,
    scrollTop: 0,
    levelHeight: 0
  };

  function enumVisItems(items, level, processItem, processItemRes, onInterrupt, onEnd, index) {
    processItem = processItem || function() {};
    processItemRes = processItemRes || function() {};
    onInterrupt = onInterrupt || function() {};
    onEnd = onEnd || function() {};

    index = index || 0;

    var result = 0;

    var checkCanceled = function() {
      var result = false;
      if (this.S.canceled()) {
        isCanceled = true;
        this.S.end();
        result = true;
      }

      return result;
    }.bind(this);

    processChildren.call(this, index, items);

    function processChildren(startIndex, items) {
      if (checkCanceled()) {
        return;
      }

      var iterator = items[Symbol.iterator](startIndex);
      var iterVal = iterator.next();
      while (iterVal.value && !iterVal.done) {
        if (checkCanceled()) {
          return;
        }

        var item = iterVal.value;
        if (item &&
            item.isVisible()) {
          result += processItem.call(this, item, level) || 0;
          processedColumns++;

          if (!item.isLeaf()) {
            var res = enumVisItems.call(this, item, level + 1, processItem,
              processItemRes, onInterrupt, onEnd);
            processItemRes.call(this, item, res);
          }
        }

        iterVal = iterator.next();

        if (level === 0 &&
            processedColumns >= thresholdColumnCount &&
            !iterVal.done) {
          onInterrupt.call(this);

          processedColumns = 0;
          this.S.nextStep(enumVisItems.bind(this, items, 0, processItem, processItemRes,
              onInterrupt, onEnd, iterVal.index));
          if (checkCanceled()) {
            return;
          }
          break;
        }
      }

      if (level === 0 &&
          iterVal.done) {
        onEnd.call(this);
        this.S.end();
      }
    }

    return result;
  }

  function createColumnsByVisItems(items) {
    processedColumns = 0;
    enumVisItems.call(this, items, 0, function(item) {
      var isLeafColumn = item.isLeaf();

      var columnState = item.getState();
      var column = addColumn.call(this, item.getDataIndex(), columnState, isLeafColumn);

      column.visItem = item;
      item.column = column;
      column.setState(columnState);
      column.onExpandColumn = this.processExpandColumn.bind(this);
      column.beforeExpandColumn = this.processBeforeExpandColumn.bind(this);

      if (isLeafColumn) {
        var styleWidth = item.getWidth();
        if (styleWidth) column.area.style.width = styleWidth;
      }
    });
  }

  function updateColumnCaptionsByVisItems(items) {
    processedColumns = 0;
    enumVisItems.call(this, items, 0, function(item) {
      updateColumnCaption.call(this, item.column, item.getState());
    });
  }

  function calcPositionsByVisItems(items) {
    processedColumns = 0;
    var bottomMap = new Map();
    enumVisItems.call(this, items, 0, function(item, level) {
      var headerBottom = 0;
      if (level === 0) {
        bottomMap.clear();
      } else {
        headerBottom = bottomMap.get(level - 1);
      }

      var column = item.column;

      var calculatedPos = {};

      var headerEl = column.header;
      var headerPos = headerEl.getBoundingClientRect();
      calculatedPos.headerPosTop = headerPos.top;
      calculatedPos.headerPosHeight = headerPos.height;
      if (!columnSizes.levelHeight) {
        columnSizes.levelHeight = headerPos.height;
        this.headerLevelHeight = columnSizes.levelHeight;
      }

      var scrollAreaTop = column.getRowsViewStart();
      calculatedPos.scrollTopDelta = this.keepScrollAreaOffset ?
            scrollAreaTop - headerPos.height :
            0;

      if (item.getHeight) {
        var columnHeight = item.getHeight();
        if (columnHeight > 1) {
          calculatedPos.headerPosHeight = columnHeight * columnSizes.levelHeight;
          calculatedPos.customHeaderPosHeight = calculatedPos.headerPosHeight;
        }
      }

      headerBottom += calculatedPos.headerPosHeight;
      bottomMap.set(level, headerBottom);

      if (columnSizes.headerBottom < headerBottom + 1) {
        columnSizes.headerBottom = headerBottom + 1;
        columnSizes.scrollTop = columnSizes.headerBottom + calculatedPos.scrollTopDelta;
      }

      if (item.isLeaf()) {
        calculatedPos.columnWidth = column.getColumnWidth();
      }

      item.calculatedPos = calculatedPos;
    });
  }

  function setColumnPositionsByVisItems(items) {
    processedColumns = 0;

    var topMap = new Map();
    var columnLeft = 0;
    enumVisItems.call(this, items, 0, function(item, level) {
      var columnTop = 0;
      if (level === 0) {
        topMap.clear();
      } else {
        columnTop = topMap.get(level);
      }

      var column = item.column;
      var calculatedPos = item.calculatedPos;

      topMap.set(level + 1, columnTop + calculatedPos.headerPosHeight);

      column.setTop(columnTop);
      column.setLeft(columnLeft, true);

      if (item.calculatedPos.customHeaderPosHeight) {
        column.setHeaderHeight(calculatedPos.customHeaderPosHeight);
      }

      var result = 0;
      if (calculatedPos.columnWidth) {
        result = calculatedPos.columnWidth;
        columnLeft += calculatedPos.columnWidth;
      }

      delete item.calculatedPos;

      return result;
    }, function(item, width) {
      item.column.area.style.width = width + 'px';
    }, function() {
      this.fitHeaderHeight(columnSizes);
    }, function() {
      this.fitHeaderHeight(columnSizes);
    });
  }

  if (this.dataModel) {
    var columnCount = this.dataModel.getColumnCount();

    this.S
    .do_(function createColumns() {
      if (!this.columnsVisModel) {
        var headerHeight;
        for (let i = 0; i < columnCount; i++) {
          let dataIndex = i;
          if (this.dataModel.getDataIndex) {
            dataIndex = this.dataModel.getDataIndex(dataIndex);
          }
          let column = addColumn.call(this, dataIndex, HeaderItemState.simple, true);
          if (!headerHeight) {
            headerHeight = column.getHeaderHeight();
            this.headerLevelHeight = headerHeight;
          }
          column.setHeaderHeight(headerHeight);
        }

        this.fitColumns();

        for (let i = 0, len = this.columns.length; i < len; i++) {
          updateColumnCaption.call(this, this.columns[i], HeaderItemState.simple);
        }
      } else {
        this.S
          .do(createColumnsByVisItems.bind(this, this.columnsVisModel))
          .do(calcPositionsByVisItems.bind(this, this.columnsVisModel))
          .do(setColumnPositionsByVisItems.bind(this, this.columnsVisModel))
          .do(updateColumnCaptionsByVisItems.bind(this, this.columnsVisModel))
      }
    }.bind(this))
    .do_(function expandLastColumn() {
      var columnLength = this.columns.length;
      if (columnLength >= 1 &&
          this.isLastColumnExpanded()) {
        this.columns[columnLength - 1].setLastColumnStyle();
      }
    }.bind(this));
  }
};

GridBody.prototype.fitColumns = function() {
  if (this.columnsVisModel) return;

  var len = this.columns.length;
  if (!len) return;

  var currentLeft = 0;
  var scrollTop;

  if (!this.keepScrollAreaOffset) {
    scrollTop = this.columns[0].getHeaderHeight();
  }

  var widths = [];
  widths.length = len;

  for (var i = 0; i < len; i++) {
    widths[i] = this.columns[i].getColumnWidth();
  }

  for (i = 0; i < len; i++) {
    var column = this.columns[i];
    column.setLeft(currentLeft);
    currentLeft += widths[i];

    if (scrollTop !== undefined) {
      column.setScrollAreaTop(scrollTop);
    }
  }
};

var thresholdColumnCount = 2000;

GridBody.prototype.applyColumnAutoSize = function(columns, rowsCreation) {
  function needAutoSize(column) {
    if (!column) return false;

    var visItem = column.visItem;
    if (visItem) {
      return (visItem.isAutoSize() || !parseInt(visItem.getWidth())) && !visItem.isOptimalWidth;
    } else {
      return !column.isOptimalWidth;
    }
  }

  columns = columns || this.columns;
  rowsCreation = rowsCreation || { count: 0 };

  var currentLeft = 0;
  var needClearSizes = false;

  var widths = [];
  widths.length = columns.length;

  var preparedColumns = [];
  preparedColumns.length = columns.length;

  var currentLeft = 0;

  this.S
  .do_(() => {
    this._viewport.createRows(undefined, undefined, false, column => {
      if (needAutoSize(column)) {
        rowsCreation.count++;
        return true;
      }

      return false;
    });
  })
  .do(() => {
    this.S.forEach(columns, (column, index) => {
      if (needAutoSize(column)) {
        column.prepare4OptimalWidth();
        preparedColumns[index] = true;
      }
    }, thresholdColumnCount);
  })
  .do(() => {
    this.S.forEach(columns, (column, index) => {
      if (!column) {
        return;
      }

      if ((needAutoSize(column)) &&
          (!this.dataModel ||
          !this.dataModel.beforeColumnAutoSize ||
          this.dataModel.beforeColumnAutoSize(column.dataIndex))) {
        var optimalWidth = column.getOptimalWidth();
        optimalWidth = Math.ceil(Math.min(optimalWidth, column.getMaxAutoSizeWidth() || optimalWidth));

        widths[index] = optimalWidth;
      } else {
        widths[index] = -column.getColumnWidth();
      }

    }, thresholdColumnCount);
  })
  .do(() => {
    this.S.forEach(columns, (column, index) => {
      if (!column) {
        return;
      }

      if (preparedColumns[index]) column.cleanUp4OptimalWidth();

    }, thresholdColumnCount);
  })
  .do(() => {
    this.S.forEach(columns, (column, index) => {
      if (!column) {
        return;
      }

      column.setLeft(currentLeft, true);

      var columnWidth = widths[index];
      if (columnWidth > 0) {
        column.setWidth(columnWidth, true, true);
        needClearSizes = true;
      }

      currentLeft += Math.abs(columnWidth);

    }, thresholdColumnCount);
  })
  .do(() => {
    this.S.forEach(columns, column => {
      if (!column) {
        return;
      }

      column.fitParent();

    }, thresholdColumnCount);
  })
  .do_(() => {
    const columnsCount = this.columns.length;
    if (rowsCreation.count && rowsCreation.count < columnsCount) {
      for(let i = 0; i < columnsCount; i++) {
        this.columns[i].clearRows();
      }

      rowsCreation.count = 0;
    }

    if (needClearSizes) this._sizes = {};
  });
};

GridBody.prototype.fitHeaderHeight = function(columnSizes, isSignal) {
  this._sizes = {};

  if (!this.columns.length) return;

  if (columnSizes) {
    this._sizes.headerBottom = columnSizes.headerBottom;

    for (var i = 0, len = this.columns.length; i < len; i++) {
      var column = this.columns[i];
      var columnTop = column.getTop();
      column.setHeaderHeight(columnSizes.headerBottom - columnTop);
      column.setScrollAreaTop(columnSizes.scrollTop - columnTop);
    }

    if (columnSizes.levelHeight &&
        !this.headerLevelHeight) {
      this.headerLevelHeight = columnSizes.levelHeight;
    }
  }

  if (this.isCentralColumn()) {
    this.updateColumnCaption(0);
  }

  if (!isSignal) this.onFitHeaderHeight.raise(columnSizes, true);
};

GridBody.prototype.fitExpand = function (expandedRow, expandedCount) {
  if (expandedCount) {
    var scrollTop = this._viewport.calcExpand(expandedRow, expandedCount);
    this.setScrollTop(scrollTop);
  }
}

GridBody.prototype.getColumnByDataIndex = function(index, all) {
  var result = this.columns[index];

  function isColumn(column) {
    if (column.dataIndex === index) {
      result = column;
      return true;
    }

    return false;
  }

  if (!result ||
      result.dataIndex !== index) {
    result = null;

    this.columns.some(isColumn);
    if (!result && all) this.inActiveColumns.some(isColumn);
  }

  return result;
};

GridBody.prototype.updateCells = function(updatedCells) {
  if (!updatedCells.length) {
    this._viewport.regetRows();
  }

  updatedCells.forEach(function(cell) {
    if (!cell) return;

    if (cell.col === undefined) {
      this._viewport.updateRow(cell.row);
    } else if (cell.row === undefined) {
      this._viewport.updateColumn(cell.col);
    } else if (cell.row === -1) {
      this.updateColumnCaption(cell.col);
    } else {
      this._viewport.updateCell(cell.row, cell.col);
    }
  }.bind(this));
};

GridBody.prototype.updateColumnCaption = function(col) {
  var column = this.getColumnByDataIndex(col, true);
  if (column) updateColumnCaption.call(this, column);
};

GridBody.prototype.getSelectionStyle = function() {
  return 'idvcgrid_cell idvcgrid_selected_row';
};

GridBody.prototype.setCurrentRow = function(row, stopPropagation, keyState) {
  if (isNaN(row)) {
    return;
  }

  var updateCurrentRow = function(row) {
    var result = false;

    var oldCurrent = this.getCurrentRow();
    if (oldCurrent !== row) {
      this._viewport.setCurrentRow(row, oldCurrent);
      result = true;
    }

    return result;
  }.bind(this);

  if (this.dataModel &&
      this.dataModel.setCurrentRow) {
    updateCurrentRow = function(row, keyState) {
      var result = false;

      this._viewport.updateCurrentRow(row);
      if (!stopPropagation) {
        result = this.dataModel.setCurrentRow(row, keyState);
      }

      return result;
    }.bind(this);
  }

  if (updateCurrentRow(row, keyState) &&
      !stopPropagation) {
    this.currentChangeAsync.call(row, keyState);

    if (this.connectedBody) {
      this.connectedBody.setCurrentRow(row, true, keyState, this);
    }
  }
};

GridBody.prototype.getCurrentRow = function() {
  if (this.dataModel &&
      this.dataModel.getCurrentRow) {
    return this.dataModel.getCurrentRow();
  }
  return this._viewport.getCurrentRow();
};

GridBody.prototype.isRowSelected = function(row) {
  if (this.dataModel &&
      this.dataModel.isRowSelected) {
    return this.dataModel.isRowSelected(row);
  }
  return this._viewport.getCurrentRow() === row;
};

GridBody.prototype.processSetFocus = function(propagate) {
  this.area.classList.add('idvcgrid_focused');

  if (this.connectedBody &&
      propagate) {
    this.connectedBody.processSetFocus(false, this);
  }

  if (propagate) this.onSetFocus.raise();
};

GridBody.prototype.processLostFocus = function(propagate) {
  this.area.classList.remove('idvcgrid_focused');

  if (this.connectedBody &&
        propagate) {
    this.connectedBody.processLostFocus(false, this);
  }

  if (propagate) this.onLostFocus.raise();
};

GridBody.prototype.getVisibleColumnsRange = function() {
  if (this.visibleColumnsRange) return this.visibleColumnsRange;

  var first = -1;
  var last = -1;

  const len = this.columns.length;
  if (len) {
    const screenWidth = window.screen.width;
    const viewWidth = this.area.offsetWidth;
    const scrollLeft = this.area.scrollLeft;

    const viewCenter = scrollLeft + viewWidth / 2;

    let vieportLeft = viewCenter - 1.5 * screenWidth;
    if (vieportLeft < 0) vieportLeft = 0;
    const vieportRight = vieportLeft + 3 * screenWidth;

    let columnsWidth = 0;
    for (let i = 0; i < len; i++) {
      columnsWidth += this.columns[i].getColumnWidth();
      if (columnsWidth > vieportLeft) {
        first = i;
        break;
      }
    }

    if (first < 0) first = 0;

    if (columnsWidth >= vieportRight) {
      last = first;
    } else {
      for (let i = first + 1; i < len; i++) {
        columnsWidth += this.columns[i].getColumnWidth();
        if (columnsWidth >= vieportRight) {
          last = i;
          break;
        }
      }

      if (last < 0) last = len - 1;
    }

    this.visibleColumnsRange = { first, last };
  }

  return this.visibleColumnsRange;
};

GridBody.prototype.getRowsRange = function() {
  return {
    first: this._viewport.getTopRow(),
    last: this._viewport.getBottomRow()
  };
};

GridBody.prototype.getVisibleRowsRange = function() {
  var first = this._viewport.getTopRow();
  if (Math.round(this._viewport.getColumnsScrollTop()) > 0) first++;

  var last = this._viewport.getBottomRow();
  if (Math.round(this.getScrollTop() + this.getRowsAreaHeight()) < Math.round(this.getRowsHeight(0, last))) last--;

  return {
    first,
    last
  };
};

GridBody.prototype.isRowVisible = function(row) {
  var range = this.getVisibleRowsRange();
  return row >= range.first && row <= range.last;
};

GridBody.prototype.getRowPos = function(row) {
  var result = 0;

  var range = this.getVisibleRowsRange();
  if (row < range.first) result = -1;
  else if (row > range.last) result = 1;

  return result;
};

GridBody.prototype.currentToVisibleTop = function() {
  return this.rowVisibleTop(this.getCurrentRow());
};

GridBody.prototype.currentToVisibleBottom = function() {
  return this.rowVisibleBottom(this.getCurrentRow());
};

GridBody.prototype.currentToVisibleCenter = function(scrollAllways) {
  return this.rowVisibleCenter(this.getCurrentRow(), scrollAllways);
};

GridBody.prototype.rowVisibleTop = function(row) {
  var newScrollTop = this.getRowsHeight(0, row - 1);
  if (newScrollTop < 0) {
    newScrollTop = 0;
  }

  return this.setScrollTop(newScrollTop);
};

GridBody.prototype.rowVisibleBottom = function(row) {
  var bodyHeight = this.getRowsAreaHeight();

  var newScrollTop = this.getRowsHeight(0, row) - bodyHeight;

  return this.setScrollTop(newScrollTop);
};

GridBody.prototype.rowVisibleCenter = function(row, scrollAllways) {
  if (!scrollAllways &&
      this.isRowVisible(row)) {
    return;
  }

  var bodyHeight = this.getRowsAreaHeight();

  var newScrollTop = this.getRowsHeight(0, row) - bodyHeight / 2;

  return this.setScrollTop(newScrollTop);
};

GridBody.prototype.columnToVisible = function(col, colOffset) {
  function updateScrollLeft(minOffset, maxOffset) {
    var area = this.area;
    var scrollLeft = area.scrollLeft;
    var clientWidth = area.clientWidth;

    var newScrollLeft = scrollLeft;

    if (maxOffset > newScrollLeft + clientWidth) {
      newScrollLeft = maxOffset - clientWidth;
    }

    if (minOffset < newScrollLeft) {
      newScrollLeft = minOffset;
    }

    if (newScrollLeft !== scrollLeft) {
      this.area.scrollLeft = newScrollLeft;
      return true;
    }

    return false;
  }

  function normalizeOffset(offset) {
    offset = offset || {};

    var result = {
      left: (offset.left || 0),
      width: (offset.width || 0),
      useScrollLeft: (offset.useScrollLeft || false),
      contentWidth: (offset.contentWidth || 0)
    }

    var padding = Utils.em2px(0.5, this.area);

    if (offset.left) {
      result.left -= padding;
      if (result.left < 0) result.left = 0;
    }

    if (offset.left) padding += (offset.left - result.left);
    if (result.width) result.width += padding;

    return result;
  }

  function applyScrollLeft(offset, columnWidth) {
    if (!offset || !offset.useScrollLeft) return offset;

    var contentWidth = offset.contentWidth || 0;
    var left = offset.left;
    if (left &&
        contentWidth &&
        contentWidth > columnWidth) {
      if (contentWidth - left >= columnWidth) {
        offset.left = 0;
      } else {
        offset.left -= (contentWidth - columnWidth);
        if (offset.left < 0) offset.left = 0;
      }
    }

    return offset;
  }

  var target = this.getColumnByDataIndex(col, true);
  if (!target) return Promise.resolve(false);

  target = target.getFirstChild();

  var minColumnOffset = 0;
  var maxColumnOffset = 0;
  this.columns.some(function(column) {
    if (column !== target) {
      minColumnOffset += column.getColumnWidth();
      return false;
    }

    maxColumnOffset = minColumnOffset + column.getColumnWidth();
    return true;
  });

  var normOffset = applyScrollLeft(normalizeOffset.call(this, colOffset),
    maxColumnOffset - minColumnOffset);

  var minContentOffset = normOffset.left;
  var maxContentOffset = minContentOffset + normOffset.width;

  if (maxContentOffset) {
      maxColumnOffset = Math.min(maxColumnOffset, minColumnOffset + maxContentOffset)
  }

  const result = new Promise(resolve => {
    this.columnsScrollWaiter = {
      resolve
    };
  });

  const scrolled = updateScrollLeft.call(this, minColumnOffset + minContentOffset, maxColumnOffset);
  if (!scrolled) {
    this.columnsScrollWaiter.resolve(false);
    delete this.columnsScrollWaiter;
  }

  return result;
};

GridBody.prototype.columnToExpanded = function(col) {
  function enumWholeHeaderModel(visItem, callback) {
    if (!visItem) return false;

    let res = false;

    if (visItem.getDataIndex() >= 0) {
      res = callback(visItem);
    }

    if (!res) {
      for (let i = 0, len = visItem.getChildrenCount(); i < len; i++) {
        res = enumWholeHeaderModel(visItem.getChild(i), callback);
        if (res) break;
      }
    }

    return res;
  }

  let expanded = false;

  let { columnsVisModel } = this;
  if (columnsVisModel) {
    let visItem;

    enumWholeHeaderModel(columnsVisModel, item => {
      if (item.getDataIndex() === col) {
        visItem = item

        return true;
      }

      return false;
    });

    while (visItem) {
      visItem = visItem.getParent();

      if (visItem &&
          visItem.getState() === HeaderItemState.collapsed) {
        visItem.setState(HeaderItemState.expanded);
        expanded = true;
      }
    }
  }

  if (expanded) {
    return this.S.wait().then(() => this.S.wait(this.refreshColumns.bind(this)));
  } else {
    return Promise.resolve(true);
  }
};

GridBody.prototype.currentRowVisible = function() {
  this.rowVisible(this.getCurrentRow());
};

GridBody.prototype.rowVisible = function(row) {
  var pos = this.getRowPos(row);
  if (pos < 0) this.rowVisibleTop(row);
  else if (pos > 0) this.rowVisibleBottom(row);
};

GridBody.prototype.currentUp = function(keyState) {
  var newCurrent = this.getCurrentRow() - 1;
  if (newCurrent >= 0) {
    this.setCurrentRow(newCurrent, false, keyState);
    this.rowVisible(newCurrent);
  }
};

GridBody.prototype.currentDown = function(keyState) {
  var newCurrent = this.getCurrentRow() + 1;
  if (this.dataModel &&
      newCurrent <= this.dataModel.getRowCount() - 1) {
    this.setCurrentRow(newCurrent, false, keyState);
    this.rowVisible(newCurrent);
  }
};

GridBody.prototype.getPageRowCount = function() {
  return this._viewport.getVisibleRowCount();
};

GridBody.prototype.currentPgUp = function(keyState) {
  if (this.getCurrentRow() > 0) {
    var newCurrent = this.getCurrentRow() - this.getPageRowCount();
    if (newCurrent < 0) {
      newCurrent = 0;
    }

    this.setCurrentRow(newCurrent, false, keyState);
    this.rowVisible(newCurrent);
  }
};

GridBody.prototype.currentPgDown = function(keyState) {
  if (this.dataModel &&
      this.getCurrentRow() < this.dataModel.getRowCount() - 1) {
    var newCurrent = this.getCurrentRow() + this.getPageRowCount();
    if (newCurrent >= this.dataModel.getRowCount()) {
      newCurrent = this.dataModel.getRowCount() - 1;
    }

    this.setCurrentRow(newCurrent, false, keyState);
    this.rowVisible(newCurrent);
  }
};

GridBody.prototype.currentHome = function(keyState) {
  if (this.getCurrentRow() > 0) {
    this.setCurrentRow(0, false, keyState);
    this.rowVisibleTop(0);
  }
};

GridBody.prototype.currentEnd = function(keyState) {
  var newCurrent = this.dataModel.getRowCount() - 1;
  if (this.dataModel &&
      this.getCurrentRow() < newCurrent) {
    this.setCurrentRow(newCurrent, false, keyState);
    this.rowVisibleBottom(newCurrent);
  }
};

GridBody.prototype.getCell = function(row, col) {
  if (this.getRowBufferSize()) {
    row -= this._viewport.getTopRow();
    return this.columns[col].getCell(row);
  }
};

GridBody.prototype.getVisibleRowHeight = function(index) {
  if (this.getRowBufferSize()) {
    if (index !== undefined) index -= this._viewport.getTopRow();

    let column;
    if (column = this.getFirstColumn()) {
      return column.getRowHeight(index);
    }
  }

  return noRowHeight;
};

GridBody.prototype.getRowHeight = function() {
  if (this._sizes.rowHeight !== undefined) return this._sizes.rowHeight;

  let column;
  if (column = this.getFirstColumn()) {
    this._sizes.rowHeight = column.getRowHeight(0);

    return this._sizes.rowHeight;
  }

  return noRowHeight;
};

GridBody.prototype.getRowsHeight = function(from, to, rowHeight) {
  from = from === undefined ? 0 : from;
  to = to === undefined ? this.dataModel && this.dataModel.getRowCount() || 0 : to;

  rowHeight = rowHeight || this.getRowHeight(0);

  return (to - from + 1) * rowHeight;
};

GridBody.prototype.getRowIndexByHeight = function(height, from, rowHeight) {
  rowHeight = rowHeight || this.getRowHeight(0);
  const epsilon = .05;

  if (rowHeight > 0) {
    return from + Math.floor(height / rowHeight + epsilon);
  }

  return 0;
};

GridBody.prototype.getRowTop = function(row) {
  return this._viewport.getRowTop(row);
};

GridBody.prototype.getViewHeight = function() {
  if (this.columns.length) {
    return this.getRowsAreaHeight();
  }

  return 1;
};

GridBody.prototype.getRowBufferSize = function(useLocal) {
  let column = !useLocal ?
    this.getFirstColumn() :
    GridBody.prototype.getFirstColumn.call(this);

  if (column) {
    return column.getRowBufferSize();
  }

  return 0;
};

GridBody.prototype.getHeaderHeight = function() {
  if (this.columns.length > 0) {
    return this.columns[0].getHeaderHeight();
  }

  return 0;
};

GridBody.prototype.getRowsAreaStart = function() {
  if (this.columns.length > 0) {
    var scrollPos = Utils.getElementPos(this.columns[0].scrollArea);
    var bodyPos = Utils.getElementPos(this.area);
    return scrollPos.y - bodyPos.y;
  }

  return 0;
};

GridBody.prototype.getRowsAreaHeight = function() {
  if (this._sizes.rowsAreaHeight !== undefined) return this._sizes.rowsAreaHeight;

  let column;
  if (column = this.getFirstColumn()) {
    var rowsAreaHeight = column.getRowsViewHeight();
    if (rowsAreaHeight > 0) {
      this._sizes.rowsAreaHeight = rowsAreaHeight;
    }

    return rowsAreaHeight;
  }

  return 0;
};

GridBody.prototype.setColumnWidth = function(col, width, isOptimal, changer) {
  if (this.onSetColumnWidth &&
      this.onSetColumnWidth(col, width)) {
    return;
  }

  if (width < 20) {
    return;
  }

  var column = this.columns[col];
  if (column) {
    var oldWidth = column.getColumnWidth();

    if (width !== oldWidth) {
      column.setWidth(width, isOptimal);
      this._sizes = {};
      width = column.getColumnWidth();
      var delta = width - oldWidth;
      for (var i = col + 1, len = this.columns.length; i < len; i++) {
        this.columns[i].move(delta);
      }

      this.onChangeColumnWidth.raise(col, width, changer);
    } else if (isOptimal) {
      column.setIsOptimalWidth();
    }
  }
};

GridBody.prototype.getColumnWidth = function(col) {
  var column = this.columns[col];
  if (column) {
    return column.getColumnWidth();
  }

  return 0;
};

GridBody.prototype.getColumnOptimalWidth = function(col) {
  var column = this.columns[col];
  if (column) {
    return column.getOptimalWidth();
  }

  return 0;
};

GridBody.prototype.fitColumnOptimalWidth = function(col, maxWidth) {
  var column = this.columns[col];
  if (column) {
    var optimalWidth = column.getOptimalWidth();
    if (maxWidth && optimalWidth > maxWidth) {
      optimalWidth = maxWidth;
    }
    this.setColumnWidth(col, optimalWidth, true);
  }
};

GridBody.prototype.isCentralColumn = function() {
  return this.columns.length === 1 && this.columns[0].isLastColumn();
};

GridBody.prototype.fitCentralColumnWidth = function() {
  if (this.isCentralColumn()) {
    var column = this.columns[0];
    var width = column.getOptimalWidth(1, true);

    column.setWidth(width);
    this._sizes = {};

    this.fitCentralColumnHeader();
  }
};

GridBody.prototype.fitLastColumn = function() {
  var columns = this.columns;
  if (this.isLastColumnExpanded() &&
      columns.length) {
    columns[columns.length - 1].fitParent();
  }
}

GridBody.prototype.isSortable = function(col) {
  if (this.dataModel &&
      this.dataModel.isColumnSortable) {
    return this.dataModel.isColumnSortable(col);
  }

  return true;
};

GridBody.prototype.isMovable = function(col) {
  if (this.dataModel &&
      this.dataModel.isColumnMovable) {
    return this.dataModel.isColumnMovable(col);
  }

  return true;
};

GridBody.prototype.isResizable = function(col) {
  if (this.dataModel &&
      this.dataModel.isColumnResizable) {
    return this.dataModel.isColumnResizable(col);
  }

  return true;
};

GridBody.prototype.clickColumnHeader = function(column) {
  if (this.dataModel &&
      this.dataModel.onSortColumn &&
      column &&
      this.isSortable(column.dataIndex)) {
    var sorted = this.dataModel.onSortColumn(
      column.dataIndex,
      this.getCurrentRow());

    if (sorted) {
      this.currentToVisibleCenter();
    }
  }
};

GridBody.prototype.moveColumn = function(from, to, anim) {
  var isLastColumnMoved = from === this.columns.length - 1 ||
                          to >= this.columns.length;
  if (from === to ||
      this.columns.length < 2 ||
      from < to && from === this.columns.length - 1) {
    return;
  }

  if (to > this.columns.length) {
    to = this.columns.length;
  } else if (to < 0) {
    to = 0;
  }

  if (from < to) {
    to--;
  }

  if (from === to) {
    return;
  }

  var column = this.columns[from];
  if (column) {
    if (isLastColumnMoved &&
        this.isLastColumnExpanded()) {
      this.columns[this.columns.length - 1].clearLastColumnStyle();
    }

    var fromWidth = column.getColumnWidth();
    var delta = (from > to) ? fromWidth : -fromWidth;
    var step = (from > to) ? -1 : 1;
    var fromDeltaLeft = 0;
    for (var i = from + step; i !== to + step; i += step) {
      fromDeltaLeft += this.columns[i].getColumnWidth();
    }

    fromDeltaLeft = (from > to) ? -fromDeltaLeft : fromDeltaLeft;

    var that = this;

    var finalMove = function() {
      for (var i = from + step; i !== to + step; i += step) {
        that.columns[i].setLeft(that.columns[i].getLeft() + delta, true);
      }

      that.columns[from].setLeft(that.columns[from].getLeft() + fromDeltaLeft, true);

      that.columns.splice(from, 1);
      that.columns.splice(to, 0, column);

      if (isLastColumnMoved &&
          that.isLastColumnExpanded()) {
        that.columns[that.columns.length - 1].setLastColumnStyle();
      }
    };

    if (anim) {
      var lastAnimStep = 8;
      var curAnimStep = 0;
      var moveStep = Math.floor(delta / (lastAnimStep + 1));
      var moveStep1 = Math.floor(fromDeltaLeft / (lastAnimStep + 1));

      var animation = setInterval(function() {
        for (var i = from + step; i !== to + step; i += step) {
          that.columns[i].setLeft(that.columns[i].getLeft() + moveStep, true);
        }

        that.columns[from].setLeft(that.columns[from].getLeft() + moveStep1, true);

        delta -= moveStep;
        fromDeltaLeft -= moveStep1;

        curAnimStep++;
        if (curAnimStep === lastAnimStep) {
          clearInterval(animation);
          finalMove();
        }
      }, 10);
    } else {
      finalMove();
    }
  }
};

GridBody.prototype.getColumnsInfo = function() {
  return this.columns.map(function(item) {
    return {index: item.dataIndex, width: item.area.style.width };
  });
};

GridBody.prototype.updateFooter = function() {
  for (var i = 0, len = this.columns.length; i < len; i++) {
    this.columns[i].updateFooter(this.dataModel);
  }
};

GridBody.prototype.fitCentralColumnHeader = function () {
  var area = this.area;

  if (!this.isCentralColumn()) return;

  var column = this.columns[0];

  var leftPadding = parseInt(column.header.style.paddingLeft);
  var rightPadding = parseInt(column.header.style.paddingRight);

  if (area.scrollWidth <= area.clientWidth &&
      leftPadding === rightPadding) return;

  column.header.style.paddingLeft = '0px';
  column.footer.style.paddingLeft = '0px';
  column.header.style.paddingRight = '0px';
  column.footer.style.paddingRight = '0px';

  var defPadding = 4;
  var paddingLeft = (defPadding + area.scrollLeft) + 'px';

  var scrollRight = area.scrollWidth - area.scrollLeft - area.clientWidth;
  if (scrollRight  < 0) scrollRight = 0;
  var paddingRight = (defPadding + scrollRight) + 'px';

  column.header.style.paddingLeft = paddingLeft;
  column.footer.style.paddingLeft = paddingLeft;
  column.header.style.paddingRight = paddingRight;
  column.footer.style.paddingRight = paddingRight;
}

GridBody.prototype.expandLastColumn = function(expand) {
  this._expandLastColumn = expand;
}

GridBody.prototype.isLastColumnExpanded = function() {
  return this._expandLastColumn;
}

GridBody.prototype.refreshLayout = function(mode) {
  mode = mode || {width: true, height: true};
  var refreshElem = this.area.idvcGridObject._parent;
  Utils.refreshSize(refreshElem, mode);
}

GridBody.prototype.setColumnResizingConstrainedByGridWidth = function(constrained) {
  if (constrained) this._isColumnResizingConstrainedByGridWidth = true;
  else delete this._isColumnResizingConstrainedByGridWidth;
}

GridBody.prototype.isColumnResizingConstrainedByGridWidth = function() {
  return !!this._isColumnResizingConstrainedByGridWidth;
}

GridBody.prototype.hideTooltip = function() {
  if (this.area.onmouseout) this.area.onmouseout();
};

GridBody.prototype.waitForDataUpdate = function(wait) {
  if (wait) {
    this._isWaitingForDataUpdate = true
  } else {
    delete this._isWaitingForDataUpdate;
    this._endScroll();
  }
};

//////////////////////////////////////////////////////////////////////////

//          Grid

//////////////////////////////////////////////////////////////////////////

function Grid(parent, tabIndex) {
  this._parent = Utils.getDomElement(parent);

  this._vertScroll = new VertScrollBar(this._parent);
  this.gridBody = new GridBody(this._parent, tabIndex);
  this.gridBody.scrolling = this._vertScroll;
  this._vertScroll.addScrolled(this.gridBody);
  this.gridBody.onRefreshScroll = this.recalcScroll.bind(this);

  this.gridBody.area.idvcGridObject = this;

  this.onUpdateViewComplete = Signal.create();

  function updateVertSize(size) {
    if (size && !size.height) return;

    this.gridBody._sizes = {};
    this.gridBody._viewport.refreshSize();

    this.recalcScroll();

    if (this.gridBody.isCentralColumn()) {
      this.gridBody.fitCentralColumnAsync.call();
    }
  }

  this._parent.refreshSize = updateVertSize.bind(this);

  this.windowResizeListener = Utils.addWindowResizeListener(this._parent);
}

Grid.prototype.destroy = function() {
  this.setDataModel(undefined, undefined);

  this.windowResizeListener.remove();
  delete this.windowResizeListener;

  delete this.gridBody.area.idvcGridObject;
};

Grid.prototype.updateView = function() {
  this.gridBody.S
  .cancel()
  .do_(this.gridBody.updateView.bind(this.gridBody))
  .do_(function() {
    this.onUpdateViewComplete.raise();
  }.bind(this));
};

Grid.prototype.processDataChanges = function(ev) {
  var fitFirstColumnWidth = true;
  if (ev &&
      ev.count !== undefined &&
      ev.start !== undefined) {
    if (ev.count > 0) {
      this.gridBody._viewport.insertRows(ev.start, ev.count);
    } else {
      this.gridBody._viewport.removeRows(ev.start, -ev.count);
    }

    this.recalcScroll();
  } else if (Array.isArray(ev)) {
    this.gridBody.updateCells(ev);
    fitFirstColumnWidth = false;
  } else if (ev &&
              (ev.currentRow !== undefined ||
              ev.rowCount !== undefined)) {
    this.gridBody.refreshRows(ev.currentRow, ev.rowCount);

    this.recalcScroll();
  } else if (ev &&
              (ev.regetStart !== undefined ||
              ev.regetCount !== undefined)) {
    if (ev.rowCountChanged) {
      this.gridBody._viewport.fitRowsCount();
    }
    this.gridBody._viewport.regetRows(ev.regetStart, ev.regetCount);
    if (ev.rowCountChanged) {
      this.recalcScroll();
    }
  } else {
    this.gridBody.clearView();
    this.updateView();
  }

  if (fitFirstColumnWidth && this.gridBody.isCentralColumn()) {
    this.gridBody.fitCentralColumnAsync.call();
  }
};

Grid.prototype.recalcScroll = function() {
  var scrollBody = this._vertScroll.scrollBody;
  var updated = false;

  var rowCount = this.gridBody.dataModel ?
    this.gridBody.dataModel.getRowCount() :
    0;

  if (rowCount) {
    if (this.gridBody.getRowBufferSize()) {
      var scrollTop = this.gridBody.getScrollTop();
      var scrollSize = this.gridBody.getRowsHeight(0, rowCount - 1);
      var pageSize = this.gridBody.getRowsAreaHeight();
      this._vertScroll.setScrollInfo(scrollSize, pageSize, scrollTop);

      if (scrollSize > pageSize) {
        if (scrollBody.style === undefined ||
            scrollBody.style.display === undefined ||
            scrollBody.style.display !== 'none') {
          scrollBody.style.visibility = 'visible';
          scrollBody.style.width = '2em'; //set fake width for scroll bar width calculation

          var scrollWidth = scrollBody.offsetWidth - scrollBody.clientWidth;
          if (scrollWidth <= 1) scrollWidth = 16; //fix for OSX where scrollbars may be invisible by default

          var bottomOffset = this.gridBody.area.offsetHeight - this.gridBody.area.clientHeight;
          if (bottomOffset < 0) bottomOffset = 0;

          if (Utils.Consts.engine === 'webkit') scrollWidth++; //Chrome hides scrollbar somehow
          scrollBody.style.width = scrollWidth + 'px';
          this.gridBody.area.style.right = scrollWidth + 'px';
          scrollBody.style.bottom = bottomOffset + 'px';

          this._vertScroll.setScrollInfo(scrollSize, pageSize, scrollTop);

          updated = true;
        }
      } else {
        scrollBody.style.visibility = 'hidden';
        this.gridBody.area.style.right = '0px';
        updated = true;
      }
    }
  } else {
    scrollBody.style.visibility = 'hidden';
    this.gridBody.area.style.right = '0px';
    updated = true;
  }

  if (updated) this.gridBody.fitLastColumn();
};

function updateGridBody(gridBody, dataModel, props) {
  props.forEach(function(prop) {
    if (dataModel && dataModel[prop]) {
      gridBody[prop] = dataModel[prop].bind(dataModel);
    } else if (gridBody.hasOwnProperty(prop)) {
      delete gridBody[prop];
    }
  });
}

Grid.prototype.setDataModel = function(dataModel, columnsVisModel) {
  var oldModel = this.gridBody.dataModel;
  if (oldModel) {
    if (oldModel.setElement) {
      oldModel.setElement(null);
    }
    this.gridBody.clearView();
    oldModel.changed.unsubscribe(this, this.processDataChanges);
  }

  this.gridBody.dataModel = dataModel;
  this.gridBody.columnsVisModel = columnsVisModel;

  updateGridBody(this.gridBody, dataModel, [
    'getRowHeight',
    'getRowsHeight',
    'getRowIndexByHeight',
    'getFirstColumn'
  ]);

  if (dataModel) {
    dataModel.changed.subscribe(this, this.processDataChanges);
    this.gridBody.refreshDataModel();
    this.updateView();
  }
};

Grid.prototype.refresh = function() {
  var dataModel = this.gridBody.dataModel;
  var columnsVisModel = this.gridBody.columnsVisModel;

  this.setDataModel(undefined, undefined);
  this.setDataModel(dataModel, columnsVisModel);
};

Grid.prototype.setCurrentRow = function(row) {
  this.gridBody.setCurrentRow(row);
};

Grid.prototype.getCurrentRow = function() {
  return this.gridBody.getCurrentRow();
};

Grid.prototype.hideVertScroll = function() {
  this._vertScroll.hide();
  this.gridBody.area.style.right = '0px';
};

Grid.prototype.showVertScroll = function() {
  this._vertScroll.show();
  this.gridBody.area.style.right = '';
};

Grid.prototype.isVertScrollVisible = function() {
  return this._vertScroll.isVisible();
};

Grid.prototype.setHorzScrollType = function(type) {
  this.gridBody.area.style.overflowX = type;
};

Grid.prototype.hideHeader = function() {
  Utils.addClass(this.gridBody.area, 'idvcgrid_hidden_header');
};

Grid.prototype.showFooter = function() {
  Utils.addClass(this.gridBody.area, 'idvcgrid_visible_footer');
  Utils.refreshSize(this._parent, {height: true});
};

Grid.prototype.hideFooter = function() {
  Utils.removeClass(this.gridBody.area, 'idvcgrid_visible_footer');
  Utils.refreshSize(this._parent, {height: true});
};

Grid.prototype.updateFooter = function() {
  this.gridBody.updateFooter();
};

Grid.prototype.getColumnByDataIndex = function(index, all) {
  return this.gridBody.getColumnByDataIndex(index, all);
};

Grid.prototype.hideColumn = function(index) {
  var column = this.getColumnByDataIndex(index, true);
  if (column) {
    column.hide();
    this.gridBody.refreshColumns();
  }
};

Grid.prototype.canColumnBeHidden = function(index) {
  var column = this.getColumnByDataIndex(index, true);
  if (column) {
    return column.canBeHidden();
  }

  return false;
};

Grid.prototype.hideTooltip = function() {
  this.gridBody.hideTooltip();
};

function processVisItems(items, callback) {
  if (!items) return;

  callback = callback || function() {};

  for (let item of items) {
    callback(item);
    processVisItems(item, callback);
  }
}

Grid.prototype.showAllColumns = function() {
  processVisItems(this.gridBody.columnsVisModel, function(item){
    if (item &&
        !item.isVisible()) {
      item.setVisible(true);
    }
  });
  this.gridBody.refreshColumns();
};

Grid.prototype.getInvisibleColumnsCount = function() {
  var result = 0;
  processVisItems(this.gridBody.columnsVisModel, function(item){
    if (item &&
        !item.isVisible()) {
      result++;
    }
  });
  return result;
};

Grid.prototype.getColumnsInfo = function() {
  return this.gridBody.getColumnsInfo();
};

Grid.prototype.columnToVisible = function(col, colOffset) {
  this.gridBody.columnToVisible(col, colOffset);
}

Grid.prototype.getRowCount = function() {
  var dataModel = this.gridBody.dataModel;
  if (dataModel &&
      dataModel.getRowCount) {
    return dataModel.getRowCount();
  }

  return 0;
}

function checkWidth(width) {
  if (typeof width === 'number') return width + 'px';

  return width;
}

function createHeaderVisModel() {
  function createItem(dataIndex, parent, inWidth, inState, filteredOut, id) {
    var children = [];
    var width = checkWidth(inWidth);
    var state = inState;
    var visible = true;
    var autoSize = false;
    var ignoreHeader = false;
    var maxWidth;
    var fixed = false;

    var visibleChildCount = 0;
    var visibleFixedChildCount = 0;

    function hasDataIndex(dataIndex, child) {
      return child.getDataIndex() === dataIndex;
    }

    return {
      getParent: function() {
        return parent;
      },
      getChildrenCount: function() {
        return children.length;
      },
      getChild: function(index) {
        return children[index];
      },
      getChildByDataIndex: function(dataIndex) {
        return children.find(hasDataIndex.bind(this, dataIndex));
      },
      addChild: function(dataIndex, width, state, filteredOut, id) {
        state = state || HeaderItemState.expanded;
        var item = createItem(dataIndex, this, width, state, filteredOut, id);
        children.push(item);

        if (!filteredOut) visibleChildCount++;

        return item;
      },
      remChild: function(dataIndex) {
        var index = children.findIndex(hasDataIndex.bind(this, dataIndex));

        if (index >= 0) {
          children.splice(index, 1);
        }
      },
      getDataIndex: function() {
        return dataIndex;
      },
      getWidth: function() {
        return width || '';
      },
      setWidth: function(inWidth) {
        width = checkWidth(inWidth);

        return this;
      },
      getMaxAutoSizeWidth: function() {
        return maxWidth;
      },
      setMaxAutoSizeWidth: function(inWidth) {
        maxWidth = inWidth;

        return this;
      },
      getState: function() {
        if (!visibleChildCount) return HeaderItemState.simple;

        return state;
      },
      setState: function(inState) {
        state = inState;

        return this;
      },
      setVisible: function(vis) {
        if (vis !== visible) {
          let oldVisible = this.isVisible();
          visible = vis;

          var parent = this.getParent();
          if (parent &&
              oldVisible !== this.isVisible()) {
            parent._onChangeChildVisibility(this);
          }
        }

        return this;
      },
      isVisible: function() {
        return visible && !filteredOut;
      },
      setFixed: function(fix) {
        if (fix !== fixed) {
          fixed = fix;

          var parent = this.getParent();
          if (parent) {
            parent._onChangeChildFixity(this);
          }
        }

        return this;
      },
      isFixed: function() {
        return fixed;
      },
      setAutoSize: function(as) {
        autoSize = as;

        return this;
      },
      isAutoSize: function() {
        return autoSize;
      },
      ignoreHeaderForAutoSize: function(val) {
        if (val !== undefined) ignoreHeader = !!val;

        return ignoreHeader;
      },
      moveChild: function(from, to) {
        var child = children.splice(from, 1)[0];
        if (child) {
          if (to > from) to--;
          children.splice(to, 0, child);
        }
      },
      isLeaf: function() {
        var result = true;

        if (state === HeaderItemState.collapsed) result = !visibleFixedChildCount;
        else result = !visibleChildCount;

        return result;
      },
      isFilteredOut() {
        return !!filteredOut;
      },
      getId() {
        return id || 0;
      },
      [Symbol.iterator]: function (startIndex) {
        function getNextFixedIndex(index) {
          var child;
          do {
            child = (index >= 0 && index < children.length) ? children[index++] : undefined;
          } while (child && !child.isFixed());

          return child ? index - 1 : -1;
        }

        var i = startIndex || 0;

        if (state !== HeaderItemState.collapsed)
          return {
            next: function () {
              var index = i;
              i++;

              return {
                value: children[index],
                done: i > children.length,
                index: index
              };
            }
          };
        else
          return {
            next: function () {
              var index = getNextFixedIndex(i);
              i = index + 1;

              return {
                value: (index >= 0) ? children[index] : undefined,
                done: index < 0,
                index: index
              };
            }
          };

        return {
          next: function () {
            return {
              value: undefined,
              done: true,
              index: -1
            };
          }
        };
      },
      _onChangeChildVisibility: function(child) {
        if (child.isVisible()) {
          visibleChildCount++;
          if (child.isFixed()) visibleFixedChildCount++;
        } else {
          visibleChildCount--;
          if (child.isFixed()) visibleFixedChildCount--;
        }
      },
      _onChangeChildFixity: function(child) {
        if (child.isFixed()) {
          visibleFixedChildCount++;
        } else {
          visibleFixedChildCount--;
        }
      }
    };
  }

  var model = createItem(-1, undefined, undefined, HeaderItemState.expanded);
  model.onChange = Signal.create();
  return model;
}

function createColumnConfigurator(parent, grid, visItem) {
  function culumnBuilderGen(parentObj) {
    function getCaptionText(html) {
      var templ = /<span class="idvcgrid_header_section_text"[^<>]*>([^<>]+)<\/span>/;
      var res = html.match(templ);

      return res && res[1];
    }

    var visItem = parentObj && parentObj.visItem || columnsVisModel;

    var children = [];

    for (let i = 0, len = visItem.getChildrenCount(); i < len; i++) {
      let child = visItem.getChild(i);

      let caption = getCaptionText(dataModel.getColumnCaption(child.getDataIndex(), {}));
      let childrentCount = child.getChildrenCount();

      let checked = child.isVisible() ? ' checked' : '';

      if (caption && !child.isFilteredOut()) {
        children.push({
          visItem: child,
          hasChildren: childrentCount > 0,
          expanded: childrentCount > 0,
          html: '<label><input type="checkbox"' + checked + '>' + caption + '</label>'
        });
      }
    }

    return {
      children: children
    }
  }

  var dataModel = grid.gridBody.dataModel;
  var columnsVisModel = visItem || grid.gridBody.columnsVisModel;

  var columnConfigurator = Utils.buildHierarchy(parent, culumnBuilderGen,
    function(html, obj) {
      html.expand.innerHTML = obj.html;
      html.expand.classList.add('idvc_grid_column_configurator_item');
      html.expand.children[0].children[0].idvcGridVisItem = obj.visItem;
    },
    function(_, __, target) {
      var tagName = target.tagName.toLowerCase();
      return tagName === 'input' || tagName === 'label';
    });

    columnConfigurator.addEventListener('change', function(e) {
    var visible = e.target.checked;
    var visItem = e.target.idvcGridVisItem;
    if (visItem) {
      if (!visible) {
        let column = visItem.column;
        if (!column.canBeHidden()) {
          e.target.checked = true;
          return;
        }
      }

      visItem.setVisible(visible);
      grid.gridBody.refreshColumns();
    }
  });
}

function saveVisItem(visItem) {
  var result = {
    dataIndex: visItem.getDataIndex()
  };

  if (!visItem.isVisible()) result.visible = false;

  if (!visItem.isAutoSize()) {
    var width = visItem.getWidth();
    if (width) result.width = width;
  }

  var state = visItem.getState();
  if (state === HeaderItemState.collapsed) result.state = state;

  if (visItem.isFixed()) result.fixed = true;
  if (visItem.isFilteredOut()) result.filteredOut = true;

  const maxWidth = visItem.getMaxAutoSizeWidth();
  if (maxWidth) result.maxWidth = maxWidth;

  var id = visItem.getId();
  if (id) result.id = id;

  var len = visItem.getChildrenCount();
  if (len) {
    result.children = [];

    for (let i = 0; i < len; i++) {
      result.children.push(saveVisItem(visItem.getChild(i)));
    }
  }

  return result;
}

function loadVisItem(parent, savedObj) {
  function columnWidth(width) {
    if (typeof width === 'number' && width > 0) return width + 'px'

    return width;
  }

  var child = parent.addChild(savedObj.dataIndex, columnWidth(savedObj.width),
    savedObj.state, savedObj.filteredOut, savedObj.id);

  if (savedObj.visible === false) child.setVisible(false);
  if (savedObj.fixed) child.setFixed(true);

  if (savedObj.maxWidth) child.setMaxAutoSizeWidth(savedObj.maxWidth);

  if (savedObj.children) {
    savedObj.children.forEach(function(savedChild) {
      loadVisItem(child, savedChild);
    });
  }
}

function saveHeaderVisModel(model) {
  var result = {
    children: []
  };

  for (let i = 0, len = model.getChildrenCount(); i < len; i++) {
    result.children.push(saveVisItem(model.getChild(i)));
  }

  return result;
}

function loadHeaderVisModel(savedObj) {
  var result = createHeaderVisModel();

  if (savedObj.children) {
    savedObj.children.forEach(function(savedChild) {
      loadVisItem(result, savedChild);
    });
  }

  return result;
}

function mergeSavedHeaderVisModels(resModel, tstModel) {
  let compareFn = (a, b) => (a.id || 0) - (b.id || 0);
  if (resModel.children.length && 
      typeof resModel.children[0].id === 'string') {
    compareFn = (a, b) => a.id.localeCompare(b.id);
  }

  function buildCompareData(model) {
    function walk(parent) {
      if (!parent) return;

      if (parent.id) result.push(parent);

      if (parent.children) {
        parent.children.forEach(walk);
      }
    }

    var result = [];
    walk(model);
    result.sort(compareFn);

    return result;
  }

  function compare(dataRes, dataTst) {
    function checkProp(res, tst, prop) {
      if (res[prop] !== tst[prop]) {
        if (tst[prop]) {
          res[prop] = tst[prop];
        } else {
          delete res[prop];
        }
      }
    }

    const checkableProps = [ 'filteredOut', 'maxWidth'];

    if (dataRes.length !== dataTst.length) return false;

    var compatible = true;
    for (let i = 0, len = dataRes.length; i < len; i++) {
      let res = dataRes[i];
      let tst = dataTst[i];

      if (res.id === tst.id) {
        checkableProps.forEach(checkProp.bind(this, res, tst));
      } else {
        compatible = false;
        break;
      }
    }

    return compatible;
  }

  if (!resModel || !tstModel) return;

  var compareDataRes = buildCompareData(resModel);
  var compareDataTst = buildCompareData(tstModel);

  if (!compare(compareDataRes, compareDataTst)) {
    // header vis models are not compatible
    resModel.children = tstModel.children;
  }
}

function enumHeaderVisModel(visItem, callback) {
  if (!visItem) return;

  if (visItem.getDataIndex() >= 0) callback(visItem);

  for (let item of visItem) {
    enumHeaderVisModel(item, callback);
  }
}

function getVisibleColumnsDataIndexes(grid) {
  var result = [];

  if (grid) {
    let columnsVisModel = grid.gridBody.columnsVisModel;
    if (columnsVisModel) {
      enumHeaderVisModel(columnsVisModel, item => {
        if (item.isLeaf()) {
          result.push(item.getDataIndex());
        }
      });
    } else {
      result = grid.gridBody.columns.map(column => column.dataIndex);
    }
  }

  return result;
}

function getRangeEndValue(range, prop, def) {
  if (range[prop] !== undefined) {
    let result = range[prop]
    if (result > def) result = def

    return result
  }

  return def
}

function getRangeStartValue(range, prop) {
  let result = range[prop] || 0
  if (result < 0) result = 0

  return result
}

async function enumCellsInRowsRange(process, grid, range) {
  if (!grid || !process) return;

  range = range || {}

  const firstColumnIndex = range.firstColumnIndex || 0

  function getVisibleColumnsRange(gridBody) {
    return gridBody.getVisibleColumnsRange() || {
      first: 0,
      last: gridBody.columns.length - 1
    }
  }

  function doProcess(gridBody, row, col) {
    const cell = gridBody.getCell(row, col)
    return cell && process(cell, row, firstColumnIndex + col)
  }

  let { gridBody } = grid
  if (!gridBody) return

  const colCount = gridBody.columns.length
  if (!colCount) return

  const startColumn = getRangeStartValue(range, 'startColumn')
  const endColumn = getRangeEndValue(range, 'endColumn', colCount - 1)

  if (startColumn >= colCount || endColumn < 0 || startColumn > endColumn) return

  const rowCount = gridBody.dataModel.getRowCount()
  if (!rowCount) return

  let startRow = getRangeStartValue(range, 'startRow')
  let endRow = getRangeEndValue(range, 'endRow', rowCount - 1)

  if (startRow > endRow) return

  const rowsRange = gridBody.getRowsRange()

  if (startRow > rowsRange.last || endRow < rowsRange.first) return

  if (rowsRange.first > startRow) startRow = rowsRange.first
  if (rowsRange.last < endRow) endRow = rowsRange.last

  const firstColumnDataIndex = gridBody.columns[startColumn].dataIndex
  await gridBody.columnToVisible(firstColumnDataIndex)

  let lastProcessedColumn = startColumn - 1
  do {
    const columnRange = getVisibleColumnsRange(gridBody)

    if (columnRange.first > lastProcessedColumn + 1) {
      throw("Invalid dumped columns range")
    }

    for (let j = lastProcessedColumn + 1, last = Math.min(columnRange.last, endColumn); j <= last; j++) {
      for (let i = startRow; i <= endRow; i++) {
        doProcess(gridBody, i, j)
      }

      lastProcessedColumn = j
    }

    if (lastProcessedColumn < endColumn) {
      const nextColumnDataIndex = gridBody.columns[lastProcessedColumn + 1].dataIndex;
      await gridBody.columnToVisible(nextColumnDataIndex)
    }
  } while (lastProcessedColumn < endColumn)

  return endRow
}

async function enumRowsRanges(process, grid, gridBody, range) {
  range = range || {}

  if (!grid || !gridBody || !process) return

  const rowCount = gridBody.dataModel.getRowCount()
  if (!rowCount) return

  const startRow = getRangeStartValue(range, 'startRow')
  const endRow = getRangeEndValue(range, 'endRow', rowCount - 1)

  if (startRow > endRow) return

  await gridBody.rowVisibleTop(startRow)

  let lastProcessedRow = startRow - 1
  do {
    range.startRow = lastProcessedRow + 1
    lastProcessedRow = await process(grid, range)

    if (lastProcessedRow === undefined) break

    const scrolled = await gridBody.rowVisibleTop(lastProcessedRow + 1)
    if (!scrolled) break
  } while (lastProcessedRow < endRow)

  return lastProcessedRow
}

async function enumAllCells(process, grid, range) {
  return enumRowsRanges(enumCellsInRowsRange.bind(this, process),
    grid, grid.gridBody, range)
}

async function dumpCells(grid, range, dumpElement, enumProc) {
  dumpElement = dumpElement || (el => el.innerText)
  enumProc = enumProc || enumAllCells

  let result = []

  if (!grid) return result

  let startRow = -1

  await enumProc((cell, row) => {
    if (startRow < 0) startRow = row

    const index = row - startRow

    if (result.length <= index) {
      const oldLength = result.length
      result.length = index + 1
      result.fill([], oldLength)
    }

    result[row - startRow].push(dumpElement(cell))
  }, grid, range)

  return result
}

Grid.prototype.enumCells = async function(process, range) {
  return enumAllCells(process, this, range)
}

Grid.prototype.dumpCells = async function(range, dumpElement, enumProc) {
  return dumpCells(this, range, dumpElement, enumProc)
}

return {
  create: function(parent, tabIndex) {
    return new Grid(parent, tabIndex);
  },
  CellTooltipAttr,
  loadCSS: loadGridStyles,
  HeaderItemState,
  createHeaderVisModel,
  createColumnConfigurator,
  saveHeaderVisModel,
  loadHeaderVisModel,
  mergeSavedHeaderVisModels,
  enumHeaderVisModel,
  getVisibleColumnsDataIndexes,
  dumpUtils: {
    enumCellsInRowsRange,
    enumAllCells,
    enumRowsRanges,
    getRangeStartValue,
    getRangeEndValue
  },
  dumpCells
};

});

/*
 * Copyright (C) 2013 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

define('hierarch_datamodel_virtual', ['./signal', './utils', './data_utils'], function(Signal, Utils, DataUtils) {

//Utils.consoleLog.enable();

var responceWaitInterval = 30000;
var refreshWaitInterval = 200;
var changeCurrentInterval = 200;
var defaultBufferSize = 400;

function HierarchicalDataModelVirtual(bufferSize) {
  bufferSize = bufferSize || defaultBufferSize;
  if (bufferSize & 1) bufferSize += 1;

  this._columnsInfo = [];
  this._visibleRows = [];

  this._visibleStart = 0;
  this._rowCount = 0;
  this._visibleMaxSize = bufferSize;

  this._selectionStart = 0;

  this._currentRow = -1;
  this._hierarchicalColumnIndex = 0;

  var model = this;
  this._waitBuffer = {
    reget: false,
    visibleStart: undefined,
    isInitialized: function() {
      return this.visibleStart !== undefined;
    },
    init: function(row, reget) {
      var isOverdrive = false;

      if (this.isInitialized()) {
        isOverdrive = true;
        Utils.consoleLog('%cWait Buffer overdrive', 'margin-left: 20px;', row);
        this.clear();
      }

      var maxSize = model._visibleMaxSize;
      var rowCount = model._rowCount;

      var startRow = row - maxSize / 2;
      if (startRow < 0) startRow = 0;

      if (startRow + maxSize > rowCount) startRow = rowCount - maxSize;
      if (startRow < 0) startRow = 0;

      if (isOverdrive || startRow !== this.visibleStart) {
        this.visibleStart = startRow;
        if (isOverdrive || reget) this.reget = true;
        Utils.consoleLog('Wait Buffer update', startRow, this.reget);
      }

      return this.isInitialized();
    },
    isRowIn: function(row, reget) {
      var result = false;

      if (this.isInitialized()) {
        result = row >= this.visibleStart &&
          row < this.visibleStart + model._visibleMaxSize;

        if (result &&
            reget &&
            !this.reget) {
          this.reget = true;
          Utils.consoleLog('%cisRowIn; set reget to true', 'margin-left: 20px;', row);
        }
      }

      return result;
    },
    clear: function() {
      Utils.consoleLog('Wait Buffer clear');
      this.visibleStart = undefined;
      this.reget = false;
    }
  };

  this.changed = Signal.create();

  this.grid = null;
  this.sorting = { col: -1, forward: true };

  this.processMouseDown = function (e) {
    e = e || window.event;

    var elem = e.target;
    if (this.grid && elem.className.indexOf('idvcgrid_widget') >= 0) {
      var row = this.grid.hitTest(e).row;
      if (elem.className.indexOf('idvcgrid_expand_collapse') >= 0) {
        this.expandRow(row, elem);
      }
      e.preventDefault();
      e.stopPropagation();
    }
  }.bind(this);

  var addHierarchicalLine = function (x, y, height) {
    if (!this.grid) return;

    var line = document.createElement('div');
    line.className = 'idvcgrid_hierarchical_line';

    line.style.left = x + 'px';
    line.style.top = y + 'px';
    line.style.height = height + 'px';

    line.onselectstart = function() {
      return false;
    };

    this.grid.columns[0].scrollArea.appendChild(line);

    return line;
  }.bind(this);

  var levelSize;
  var getLevelSize = function() {
    if (!levelSize) levelSize = DataUtils.getLevelSize(this.grid.area);
    return levelSize;
  }.bind(this);

  var getLevelByPageX = function(cell, pageX) {
    var pos = Utils.getElementPos(cell);
    var size = getLevelSize();

    return {
      index: Math.floor((pageX - pos.x) / size.offset),
      width: size.width,
      offset: size.offset
    };
  };

  var getRowRange = function(row, level) {
    row = this.rowToVisible(row);

    var topRow = 0;
    if (row > 0) {
      for (topRow = row - 1; topRow >= 0 && this._visibleRows[topRow].level > level; topRow--);
      topRow++;
    }

    var rowCount = this._visibleRows.length;
    var bottomRow = rowCount - 1;
    if (row < bottomRow) {
      for (bottomRow = row + 1; bottomRow < rowCount &&
            this._visibleRows[bottomRow].level > level; bottomRow++);
      bottomRow--;
    }

    topRow = this.visibleToRow(topRow);
    bottomRow = this.visibleToRow(bottomRow);

    return {top: topRow, bottom: bottomRow};
  }.bind(this);

  var lineInfo = {line: null, level: -1, row: -1, scrollTop: 0, scrollLeft: 0};
  lineInfo.clear = function() {
    if (this.line) {
      if (this.line.parentNode) {
        this.line.parentNode.removeChild(this.line);
      }
      this.line = null;
    }

    this.level = -1;
    this.row = -1;
  };

  this.processMouseMove = function (e) {
    if (!this.grid) return;

    e = e || window.event;
    if (this.grid.getScrollTop() !== lineInfo.scrollTop ||
        this.grid.area.scrollLeft !== lineInfo.scrollLeft) {
      lineInfo.scrollTop = this.grid.getScrollTop();
      lineInfo.scrollLeft = this.grid.area.scrollLeft;
      lineInfo.clear();
    }

    if (e.target.className.indexOf('idvcgrid_hierarchical_line') >= 0) {
      return;
    } else if (e.target.className.indexOf('idvcgrid_cell') === 0 &&
                e.target.childNodes[0] &&
                e.target.childNodes[0].className &&
                e.target.childNodes[0].className.indexOf('idvcgrid_expand_collapse') >= 0) {
      var levelStart = this.getLevelsStart();
      var level = getLevelByPageX(e.target, e.pageX - levelStart);
      var row = this.grid.hitTest(e).row;
      var visibleRow = this.getRowInfo(row);
      if (visibleRow) {
        var rowLevel = visibleRow.level;
        var rowRange = getRowRange(row, level.index);

        if (level.index !== lineInfo.level ||
            rowRange.top !== lineInfo.row) {
          lineInfo.clear();

          if (level.index >= 0 && level.index < rowLevel) {
            var top = this.grid.getRowTop(rowRange.top) + 3;
            var bottom = this.grid.getRowTop(rowRange.bottom + 1) - 5;
            var lineLeft = levelStart + level.offset * level.index + level.width / 2 - 1;
            lineInfo.line = addHierarchicalLine(lineLeft, top,
                                                bottom - top);
            lineInfo.level = level.index;
            lineInfo.row = rowRange.top;
          }
        }
      }
    } else {
      lineInfo.clear();
    }
  }.bind(this);

  this.processMouseOut = function(e) {
    e = e || window.event;
    if (e.relatedTarget !== lineInfo.line &&
        e.toElement !== lineInfo.line) {
      lineInfo.clear();
    }
  };

  this.processKeyPress = function (e) {
    if (!this.grid) return;

    e = e || window.event;
    var char = String.fromCharCode(e.keyCode || e.charCode);
    if (char === '+' ||
        char === '-') {
      var row = this.grid.getCurrentRow();
      if (row >= 0) {
        this.expandRow(row);
      }
    }
  }.bind(this);

  this.processKeyDown = function (e) {
    if (!this.grid) return;

    e = e || window.event;
    var processed = false;
    if (e.keyCode === 39) {
      processed = this.processArrowRight(e);
    } else if (e.keyCode === 37) {
      processed = this.processArrowLeft(e);
    } else if (e.keyCode === 8) {
      this.processBackspace(e);
    }

    if (processed) {
      e.stopPropagation();
      e.preventDefault();
    }
  }.bind(this);
}

HierarchicalDataModelVirtual.prototype.setColumnInfo = function(info) {
  this._columnsInfo = info;
};

HierarchicalDataModelVirtual.prototype.getColumnInfo = function() {
  return this._columnsInfo;
};

HierarchicalDataModelVirtual.prototype.setHierarchicalColumnIndex = function(index) {
  this._hierarchicalColumnIndex = index;
};

HierarchicalDataModelVirtual.prototype.getHierarchicalColumnIndex = function() {
  return this._hierarchicalColumnIndex;
};

HierarchicalDataModelVirtual.prototype.setColumnCaption = function(col, caption) {
  var columnInfo = this._columnsInfo[col];
  if (columnInfo) {
    columnInfo.caption = caption;
    this.changed.raise([{row: -1, col: col}]);
  }
};

HierarchicalDataModelVirtual.prototype.getColumnCount = function() {
  return this._columnsInfo.length;
};

HierarchicalDataModelVirtual.prototype.getRowCount = function() {
  return this._rowCount;
};

HierarchicalDataModelVirtual.prototype.getColumnCaption = function(val, params) {
  var caption = '';

  var columnInfo = this._columnsInfo[val];
  if (columnInfo) {
    if (val === this.sorting.col) {
      params.sortingForward = this.sorting.forward;
      params.sortingWait = this.sorting.wait;
    }

    caption = DataUtils.decorateCaption(columnInfo.caption, params);
  }

  return caption;
};

HierarchicalDataModelVirtual.prototype.getCell = function(row, col, isSelected) {
  var getCellData = this.getCellData || function(rowObj, columnInfo) { return rowObj.data[columnInfo.id]; };
  var escapeHTML = this.doEscapeHTML || DataUtils.escapeHTML;

  var columnInfo = this._columnsInfo[col];
  var visibleRow = this._visibleRows[this.rowToVisible(row)];
  if (visibleRow &&
      columnInfo) {
    this.checkRowsBuffer(row);
    this.checkWaitBuffer(row);

    var expand = '';

    if (col === this._hierarchicalColumnIndex) {
      expand = DataUtils.decorateExpand(visibleRow.id, visibleRow.state, visibleRow.level);
    }

    return expand + escapeHTML(getCellData(visibleRow, columnInfo, row, col), row, col, isSelected);
  } else if (!visibleRow && this.fetchRowsBuffer(row)) {
    return null;
  }

  return '';
};

HierarchicalDataModelVirtual.prototype.refresh = function() {
  this._visibleRows = [];

  this._visibleStart = 0;
  this._rowCount = 0;

  this._selectionStart = 0;

  this._currentRow = -1;
};

HierarchicalDataModelVirtual.prototype.waitForData = function (waitInterval) {
  Utils.consoleLog('Data waiting started');

  var visibleStart = this._waitBuffer.visibleStart;
  this.waitForDataTimer = setTimeout(function () {
    this.waitForDataTimer = undefined;
    Utils.consoleLog('Reget data after waiting');
    if (visibleStart === this._waitBuffer.visibleStart) this.refreshRowsBuffer(visibleStart);
  }.bind(this), waitInterval);
};

HierarchicalDataModelVirtual.prototype.getRowsBufferFrame = function() {
  return {
    startRow: this._waitBuffer.isInitialized() ? this._waitBuffer.visibleStart : this._visibleStart,
    rowCount: this._visibleMaxSize
  };
};

HierarchicalDataModelVirtual.prototype.checkRowsBuffer = function(row) {
  var borderSize = Math.min(this._visibleMaxSize / 4, 24);
  if (((row < this._visibleStart + borderSize &&
        this._visibleStart > 0) ||
      (row > this._visibleStart + this._visibleMaxSize - borderSize &&
        this._visibleStart + this._visibleRows.length < this._rowCount)) &&
        !this._waitBuffer.isRowIn(row) &&
        this._waitBuffer.init(row)) {
    this.refreshRowsBuffer(this._waitBuffer.visibleStart);
  }
};

HierarchicalDataModelVirtual.prototype.checkWaitBuffer = function(row) {
  if (this._waitBuffer.isInitialized() &&
      !this._waitBuffer.isRowIn(row) &&
      this._waitBuffer.init(row)) {
    this.refreshRowsBuffer(this._waitBuffer.visibleStart);
  }
};

HierarchicalDataModelVirtual.prototype.fetchRowsBuffer = function (row) {
  var needStartWaiting = false;
  var oldReget = this._waitBuffer.reget;

  if (!this._waitBuffer.isRowIn(row, true) &&
      this._waitBuffer.init(row, true)) {
    this.refreshRowsBuffer(this._waitBuffer.visibleStart, true);
    needStartWaiting = true;
  }

  needStartWaiting = needStartWaiting || oldReget !== this._waitBuffer.reget;
  if (needStartWaiting){
      var startWaiting = this.onStartWaitingForData || function () {
        this.setDisabled(true);
    }.bind(this);

    startWaiting();
    this.waitForDataUpdate(true);
  }

  return true;
};

HierarchicalDataModelVirtual.prototype.regetRowsBuffer = function (visibleStartRow) {
  this._waitBuffer.visibleStart = (visibleStartRow !== undefined) ? visibleStartRow : this._visibleStart;
  this._waitBuffer.reget = true;

  this.refreshRowsBuffer(this._waitBuffer.visibleStart);
}

HierarchicalDataModelVirtual.prototype.refreshRowsBuffer = function(row, wait) {
  if (this.waitForDataTimer) {
    clearTimeout(this.waitForDataTimer);
    this.waitForDataTimer = undefined;
    Utils.consoleLog('Data waiting canceled');
  }

  if (this.refreshTimer) {
    clearTimeout(this.refreshTimer);
    Utils.consoleLog('%cPrevious refresh canceled', 'color: red;');
  }

  var waitInterval = wait ? refreshWaitInterval : 0;
  Utils.consoleLog('%crefreshRowsBuffer', 'color: indigo', row, waitInterval);
  this.refreshTimer = setTimeout(function () {
    this.refreshTimer = undefined;
    if (this.onupdatevisible) this.onupdatevisible(row, this._visibleMaxSize);

    if (this.responceWaitTimer) clearTimeout(this.responceWaitTimer);
    this.responceWaitTimer = setTimeout(function () {
      this.responceWaitTimer = undefined;
      Utils.consoleLog('%cResponce wait ended', 'color: red;', row);

      if (row === this._waitBuffer.visibleStart) {
        this.forceUpdate(row);
        this.refreshRowsBuffer(row, wait);
      }
    }.bind(this), responceWaitInterval);
  }.bind(this), waitInterval);
};

HierarchicalDataModelVirtual.prototype.waitForDataUpdate = function(update) {
  const elems = [...(this.secondaryGrids || []), this.grid];

  elems.forEach(elem => {
    if (elem && elem.waitForDataUpdate) {
      elem.waitForDataUpdate(update);
    }
  })
};

HierarchicalDataModelVirtual.prototype.raiseChanged = function(gridUpdates) {
  this.changed.raise(gridUpdates)
  this.waitForDataUpdate(false);
};

HierarchicalDataModelVirtual.prototype.updateRowsBuffer = function(view, rows, isInitial) {
  if (!view) return false;

  if (view.waitTime !== undefined) {
      this.waitForData(view.waitTime);
      return false;
  }

  var start = view.startRow;
  var count = view.rowCount;
  var rowCountDelta = count - this._rowCount;
  if (view.forceUpdate) {
      this.forceUpdate(start);
  }

  var initial = isInitial || view.isInitial;

  var waitBufferVisibleStart = this._waitBuffer.visibleStart;
  var waitBufferReget = this._waitBuffer.reget;

  if (!initial &&
      start !== waitBufferVisibleStart) {
    Utils.consoleLog('%cupdateRowsBuffer rejected', 'color: red;', start, this._waitBuffer);
    return false;
  }

  Utils.consoleLog('%cupdateRowsBuffer', 'color: green;', start, waitBufferReget);
  this._waitBuffer.clear();

  var endWaiting = this.onEndWaitingForData || function () {
    this.setDisabled(false);
  }.bind(this);

  endWaiting();

  if (this.responceWaitTimer) {
    clearTimeout(this.responceWaitTimer);
    this.responceWaitTimer = undefined;
  }

  var rowCountChanged = false;
  if (this._rowCount !== count) {
    this._rowCount = count;
    rowCountChanged = true;
  }

  var expandedRow;
  var expanded = view.expanded;
  if (expanded) {
    expandedRow = this.getRowById(expanded.row, expanded.id);
    if (expandedRow >= 0) {
      clearVisibleRowTimeout(this._visibleRows[this.rowToVisible(expandedRow)]);
    }
  }

  this._visibleStart = start;
  this._visibleRows = rows;

  if (view.currentRow !== undefined) {
    this._currentRow = view.currentRow;
  }

  if (initial) {
    this.raiseChanged(null);
    if (view.scrollToCurrent) {
      // TODO: this is a quick fix for DPD200587201; it's suboptimal as the initial response does return the proper
      //       viewport containing the current row, but grid will still send scroll request to get the same viewport
      //       after rowVisibleCenter.
      setTimeout(function () {
        this.grid.rowVisibleCenter(this._currentRow);
        // setCurrentRow actually selects currentRow
        if (view.selectCurrent) this.grid.setCurrentRow(this._currentRow);
      }.bind(this), 10);
    }
  } else if (waitBufferReget) {
    var gridUpdates = {regetStart: this._visibleStart, rowCountChanged: rowCountChanged};

    if (view.scrollToCurrent) {
      this.grid.rowVisibleCenter(this._currentRow);
      setTimeout(function() {
        this.raiseChanged(gridUpdates);
      }.bind(this), 10);
    } else {
      this.raiseChanged(gridUpdates);
    }
  }

  if (expanded) {
    // _visibleRows variable is completelly changed; get expanded row index again
    expandedRow = this.getRowById(expanded.row, expanded.id);
    if (expandedRow >= 0)
      this.fitExpand(expandedRow, expanded.rowCount || rowCountDelta);
  }

  if (view.sorting) this.updateSorting(view.sorting);

  if (this.pendingSelection) {
    let { row, keyState } = this.pendingSelection;
    delete this.pendingSelection;

    this.setCurrentRow(row, keyState);
  }

  return true;
};

HierarchicalDataModelVirtual.prototype.updateSelection = function(view, rows) {
  if (!view || !rows) return false;

  var start = view.startRow;
  var visibleRowsLen = this._visibleRows.length;
  var rowsLen = rows.length;

  if (this._waitBuffer.visibleStart !== undefined ||
      this._visibleStart !== start ||
      visibleRowsLen !== rowsLen) {
    return false;
  }

  Utils.consoleLog('%cupdateSelection', 'color: green;', start);

  var updatedRows = [];
  var addedCurrentRow;

  if (view.currentRow !== undefined &&
      view.currentRow !== this._currentRow) {
    this._currentRow = view.currentRow;
    updatedRows.push({row: this._currentRow});
    addedCurrentRow = this._currentRow;
  }

  rows.forEach(function(row, index) {
    var newSelected = row.selected || false;
    var oldSelected = this._visibleRows[index].selected || false;
    if (newSelected !== oldSelected) {
      this._visibleRows[index].selected = newSelected;
      var rowIndex = this.visibleToRow(index);
      if (addedCurrentRow != rowIndex) {
        updatedRows.push({row: rowIndex});
      }
    }
  }.bind(this));

  if (updatedRows.length) this.changed.raise(updatedRows);

  return true;
};

HierarchicalDataModelVirtual.prototype.getCellLayout = function(cellText) {
  return DataUtils.getContentLayout(cellText, this.grid.area);
};

HierarchicalDataModelVirtual.prototype.isColumnSortable = function (col) {
  var columnInfo = this._columnsInfo[col];
  if (columnInfo && (columnInfo.sortable === false)) return false;

  return true;
};

HierarchicalDataModelVirtual.prototype.toString = function () {
  return '[object HierarchicalDataModelVirtual]';
};

HierarchicalDataModelVirtual.prototype.processArrowRight = function(e) {
  var processed = false;

  var row = this.grid.getCurrentRow();

  var visibleRow = this._visibleRows[this.rowToVisible(row)];
  if (visibleRow) {
    if (visibleRow.state === DataUtils.RowState.collapsed) {
      this.expandRow(row);
      processed = true;
    } else if (visibleRow.state === DataUtils.RowState.expanded) {
      var keyState = {ctrl: e.ctrlKey, alt: e.altKey, shift: e.shiftKey, toggle: false};
      this.grid.currentDown(keyState);
      processed = true;
    }
  }

  return processed;
};

function goToHierarchicalParent(visibleIndex, row) {
  function findHierachicalParent(visibleRows, level, startIndex, row) {
    var isFound = false;
    var currentIndex = startIndex - 1;
    for (; currentIndex >= 0; currentIndex--) {
      if (visibleRows[currentIndex].level < level) {
        isFound = true;
        break;
      }
    }

    if (!isFound) currentIndex = 0;

    return {
      isFound: isFound,
      row: row - (startIndex - currentIndex)
    }
  }

  var level = this._visibleRows[visibleIndex].level;
  if (level > 0) {
    var parentInfo = findHierachicalParent(this._visibleRows, level, visibleIndex, row);
    if (this.onGoToHierarchicalParent &&
        !parentInfo.isFound) {
      this.onGoToHierarchicalParent(row);
    } else {
      this.grid.setCurrentRow(parentInfo.row);
      this.grid.rowVisible(parentInfo.row);
    }
  }
}

HierarchicalDataModelVirtual.prototype.processArrowLeft = function(e) {
  var processed = false;

  var row = this.grid.getCurrentRow();

  var visibleIndex = this.rowToVisible(row);
  var visibleRow = this._visibleRows[visibleIndex];
  if (visibleRow) {
    if (visibleRow.state === DataUtils.RowState.expanded) {
      this.expandRow(row);
      processed = true;
    } else {
      goToHierarchicalParent.call(this, visibleIndex, row);
      processed = true;
    }
  }

  return processed;
};

HierarchicalDataModelVirtual.prototype.processBackspace = function(e) {
  var row = this.grid.getCurrentRow();

  var visibleIndex = this.rowToVisible(row);
  var visibleRow = this._visibleRows[visibleIndex];
  if (visibleRow) {
    goToHierarchicalParent.call(this, visibleIndex, row);
  }
};

HierarchicalDataModelVirtual.prototype.onSortColumn = function (col) {
  var prevColumn = this.sorting.col;
  var forward = this.sorting.forward;

  if (prevColumn === col) {
    forward = !forward;
  } else {
    forward = true;
    var columnInfo = this._columnsInfo[col];
    if (columnInfo &&
        columnInfo.backwardSorting) {
      forward = false;
    }
  }

  if (this.processSortColumn && this.processSortColumn(col, forward)) {
    this.updateSorting({col: col, wait: true});
  }
};

HierarchicalDataModelVirtual.prototype.expandRow = function (row, elem) {
  var visibleRow = this._visibleRows[this.rowToVisible(row)];
  if (!visibleRow || visibleRow.state === DataUtils.RowState.simple) return;

  if (this.processExpandRow) {
    elem = elem || document.getElementById(visibleRow.id);

    this.processExpandRow(row, visibleRow, visibleRow.state === DataUtils.RowState.collapsed);

    visibleRow.timeout = setTimeout((function() {
      visibleRow.state = DataUtils.RowState.waiting;
      this.updateExpandElement(visibleRow, elem);
      delete visibleRow.timeout;
    }).bind(this), 200);
  }
};

function clearVisibleRowTimeout(visibleRow) {
  if (visibleRow && visibleRow.timeout) {
    clearTimeout(visibleRow.timeout);
    delete visibleRow.timeout;
  }
}

HierarchicalDataModelVirtual.prototype.insertRows = function(row, id, rows) {
  var count = rows.length;
  var expandedRow = this.getRowById(row, id);
  if (expandedRow >= 0) {
    var visibleRow = this._visibleRows[this.rowToVisible(expandedRow)];
    clearVisibleRowTimeout(visibleRow);
    if (count > 0)
      visibleRow.state = DataUtils.RowState.expanded;
    else
      visibleRow.state = DataUtils.RowState.simple;
    this.updateExpandElement(visibleRow);

    var startRow = expandedRow + 1;
    var startIndex = this.rowToVisible(startRow);
    this._visibleRows.splice.apply(this._visibleRows,
      [startIndex, 0].concat(rows));
    this.changed.raise({start: startRow, count: count});

    if (expandedRow >= 0 &&
        count > 0) {
      this.fitExpand(expandedRow, count);
    }
  } else if (row === -1) {
    this._visibleRows = rows.concat(this._visibleRows);
    this.changed.raise({start: this._visibleStart, count: count});
  }

  this._rowCount += count;
};

HierarchicalDataModelVirtual.prototype.removeRows = function(row, id, count, rows) {
  var visibleIndex = -1;
  var startRow = -1;
  count = count || 0;
  var collapseRow = this.getRowById(row, id);
  if (collapseRow >= 0) {
    var visibleRow = this._visibleRows[this.rowToVisible(collapseRow)];
    clearVisibleRowTimeout(visibleRow);
    visibleRow.state = DataUtils.RowState.collapsed;
    this.updateExpandElement(visibleRow);

    startRow = collapseRow + 1;
    visibleIndex = this.rowToVisible(startRow);
  } else if (row === -1) {
    visibleIndex = 0;
    startRow = this._visibleStart;
  }

  if (visibleIndex >= 0 &&
      startRow >= 0) {
    this._visibleRows.splice(visibleIndex, count);
    if (rows) this._visibleRows.concat(rows);
    this.changed.raise({start: startRow, count: -count});
  }

  this._rowCount -= count;
};

HierarchicalDataModelVirtual.prototype.updateSorting = function(sorting) {
  if (!sorting) return;

  var updatedColumns = [];
  if (this.sorting.col === sorting.col) {
    this.sorting.forward = sorting.forward;
  } else {
    if (this.sorting.col >= 0) {
      updatedColumns.push({row: -1, col: this.sorting.col});
    }
    this.sorting.col = sorting.col;
    this.sorting.forward = sorting.forward;
  }

  this.sorting.wait = sorting.wait;

  updatedColumns.push({row: -1, col: sorting.col});
  this.changed.raise(updatedColumns);
};

HierarchicalDataModelVirtual.prototype.setElement = function (elem) {
  if (elem &&
      this.grid ||
      this.grid === elem) {
    return;
  }

  if (this.grid) {
    this.grid.area.removeEventListener('mousedown', this.processMouseDown, true);
    this.grid.area.removeEventListener('mousemove', this.processMouseMove, false);
    this.grid.area.removeEventListener('mouseout', this.processMouseOut, false);
    this.grid.area.removeEventListener('keypress', this.processKeyPress, false);
    this.grid.area.removeEventListener('keydown', this.processKeyDown, false);
  }

  this.grid = elem;

  if (this.grid) {
    this.grid.area.addEventListener('mousedown', this.processMouseDown, true);
    this.grid.area.addEventListener('mousemove', this.processMouseMove, false);
    this.grid.area.addEventListener('mouseout', this.processMouseOut, false);
    this.grid.area.addEventListener('keypress', this.processKeyPress, false);
    this.grid.area.addEventListener('keydown', this.processKeyDown, false);
  }
};

HierarchicalDataModelVirtual.prototype.addSecondaryElement = function (elem) {
  if (!elem) return;

  if (!this.secondaryGrids) this.secondaryGrids = [];

  if (this.secondaryGrids.indexOf(elem) >= 0) {
    return;
  }

  this.secondaryGrids.push(elem);

  elem.area.addEventListener('keypress', this.processKeyPress, false);
  elem.area.addEventListener('keydown', this.processKeyDown, false);
};

function removeSecondaryListeners(elem) {
  if (!elem || !elem.area) return;

  elem.area.removeEventListener('keypress', this.processKeyPress, false);
  elem.area.removeEventListener('keydown', this.processKeyDown, false);
}

HierarchicalDataModelVirtual.prototype.removeSecondaryElement = function (elem) {
  if (!elem || !this.secondaryGrids) return;

  var elemIndex = this.secondaryGrids.indexOf(elem);
  if (elemIndex < 0) {
    return;
  }

  this.secondaryGrids.splice(elemIndex, 1);

  removeSecondaryListeners.call(this, elem);

  elem.area.removeEventListener('keypress', this.processKeyPress, false);
  elem.area.removeEventListener('keydown', this.processKeyDown, false);
};

HierarchicalDataModelVirtual.prototype.removeAllSecondaryElements = function () {
  if (!this.secondaryGrids) return;

  this.secondaryGrids.forEach(removeSecondaryListeners.bind(this));

  this.secondaryGrids = [];
};

HierarchicalDataModelVirtual.prototype.rowToVisible = function (row) {
  return row - this._visibleStart;
};

HierarchicalDataModelVirtual.prototype.visibleToRow = function (vis) {
  return vis + this._visibleStart;
};

HierarchicalDataModelVirtual.prototype.findRowId = function(id) {
  var result = -1;

  this._visibleRows.some(function(val, index) {
    if (val.id === id) {
      result = this.visibleToRow(index);
      return true;
    }
    return false;
  }.bind(this));

  return result;
};

HierarchicalDataModelVirtual.prototype.getRowById = function(row, id) {
  var result = -1;
  var visibleRow = this._visibleRows[this.rowToVisible(row)];
  if (visibleRow) {
    if (visibleRow.id === id)
      result = row;
    else
      result = this.findRowId(id);
  }
  return result;
};

HierarchicalDataModelVirtual.prototype.forceUpdate = function (row) {
  this._waitBuffer.visibleStart = row;
  this._waitBuffer.reget = true;
  Utils.consoleLog('forceUpdate', this._waitBuffer);
};

HierarchicalDataModelVirtual.prototype.updateExpandElement = function(row, elem) {
  elem = elem || document.getElementById(row.id);
  if (elem) elem.className = 'idvcgrid_widget idvcgrid_expand_collapse' +
    DataUtils.getExpandWidgetStyle(row.state);
};

HierarchicalDataModelVirtual.prototype.setDisabled = function (disabled) {
  function getParentElement(grid) {
    return grid.area.parentElement;
  }

  var getParentElements = function () {
    var result = [];

    if (this.secondaryGrids) {
      result = this.secondaryGrids.map(getParentElement);
    }

    if (this.grid) {
      result.push(getParentElement(this.grid));
    }

    return result;
  }.bind(this);

  getParentElements().forEach(function(parentElement) {
    if (parentElement) {
      if (disabled) parentElement.classList.add('idvc_disable');
      else parentElement.classList.remove('idvc_disable');
    }
  });
};

HierarchicalDataModelVirtual.prototype.fitExpand = function (expandedRow, expandedCount) {
  if (this.grid) {
    this.grid.fitExpand(expandedRow, expandedCount);
  }
}

HierarchicalDataModelVirtual.prototype.getRowInfo = function (row) {
  return this._visibleRows[this.rowToVisible(row)];
}

HierarchicalDataModelVirtual.prototype.getCellStyle = function (isSelected, defSelectedStyle,
    defStyle, row, col) {
  var result = defStyle;

  var rowInfo = this.getRowInfo(row);
  if (rowInfo) {
    if (rowInfo.selected) {
      result = defSelectedStyle;
    } else {
      result = defStyle;
    }

    if (row === this._currentRow) {
      result += ' idvcgrid_current_row';
    }
  }

  return result;
}

function clearSelection(updatedRows) {
  this._visibleRows.forEach(function(row, index) {
    if (row.selected) {
      updatedRows.push({row: this.visibleToRow(index)});
      row.selected = false;
    }
  }.bind(this));
}

function addSelectionRange(from, to, updatedRows) {
  if (to < from) {
    var tmp = to;
    to = from;
    from = tmp;
  }

  updatedRows = updatedRows || [];

  var visibleFrom = this.rowToVisible(from);
  var visibleTo = this.rowToVisible(to);
  if (visibleFrom < 0) visibleFrom = 0;
  if (visibleTo >= this._visibleRows.length) visibleTo = this._visibleRows.length - 1;

  this._visibleRows.forEach(function(row, index) {
    var newSelected = index >= visibleFrom && index <= visibleTo;

    if (row.selected !== newSelected) {
      updatedRows.push({row: this.visibleToRow(index)});
      row.selected = newSelected;
    }
  }.bind(this));
}

var setCurrentRowWaitingClass = 'idvcgrid_set_current_waiting';

HierarchicalDataModelVirtual.prototype.setCurrentRow = function (row, keyState) {
  function currentRowChange2Updates(from, to, updates) {
    if (this.getRowInfo(from)) {
      updates.push({row: from});
    }

    if (from !== to) {
      updates.push({row: to});
    }
  }

  keyState = keyState || {};

  var result = false;
  var gridUpdates;

  var newRowInfo = this.getRowInfo(row);
  if (newRowInfo) {
    var currentRow = this._currentRow;
    this._currentRow = row;

    if (keyState.all) {
        addSelectionRange.call(this, 0, this.getRowCount() - 1);
        gridUpdates = {regetStart: 0};

        result = true;
    } else if (keyState.shift) {
      if (currentRow !== row) {
        gridUpdates = [];

        currentRowChange2Updates.call(this, currentRow, row, gridUpdates);

        addSelectionRange.call(this, this._selectionStart, row, gridUpdates);

        result = true;
      }
    } else {
      gridUpdates = [];

      if (keyState.ctrl || keyState.toggle) {
        if (keyState.isMouse || keyState.toggle) {
            newRowInfo.selected = !newRowInfo.selected;
        }
      } else {
        clearSelection.call(this, gridUpdates);
        newRowInfo.selected = true;
      }

      currentRowChange2Updates.call(this, currentRow, row, gridUpdates);

      this._selectionStart = row;

      result = gridUpdates.length > 0;
    }

    if (result) {
      if (gridUpdates) {
        // process grid changes in another event loop iteration
        setTimeout(function() {
          this.changed.raise(gridUpdates);
        }.bind(this), 0);
      }

      if (this.onChangeCurrentRow) {
        if (!keyState.toggle) {
          var delayedRow = row;
          if (this.onChangeCurrentRowTimeout) {
            clearTimeout(this.onChangeCurrentRowTimeout);
          } else {
            delayedRow = -1;
            this.onChangeCurrentRow(row, keyState);
            Utils.consoleLog('setCurrentRow', row);
          }

          this.onChangeCurrentRowTimeout = setTimeout(function() {
            if (delayedRow >= 0) {
              this.onChangeCurrentRow(delayedRow, keyState);
              if (this.grid) Utils.removeClass(this.grid.area, setCurrentRowWaitingClass);
              Utils.consoleLog('setCurrentRow delayed', delayedRow);
            }

            this.onChangeCurrentRowTimeout = undefined;
          }.bind(this), changeCurrentInterval);

          if (this.grid && delayedRow >= 0) {
            Utils.addClass(this.grid.area, setCurrentRowWaitingClass);
          }
        } else {
          this.onChangeCurrentRow(row, keyState);
          Utils.consoleLog('setCurrentRow toggle', row);
        }
      }
    }
  } else {
    // row is out of _visibleRows
    this.pendingSelection = {
      row: row,
      keyState: keyState
    };
  }

  return result;
}

HierarchicalDataModelVirtual.prototype.getCurrentRow = function () {
  return this._currentRow;
}

HierarchicalDataModelVirtual.prototype.isRowSelected = function (row) {
  var rowInfo = this.getRowInfo(row);
  if (rowInfo &&
      rowInfo.selected) {
    return true;
  }

  return false;
};

/**
 * @callback ModifyVisibleRowsCallback
 * @param {Object} rowInfo
 * @param {Number} [row]
 * @return {Boolean} - indicates if rowInfo has changed and row should be re-rendered
 */

/**
 * Iterates over visible rows and allows user to modify any row via a callback.
 * @param {ModifyVisibleRowsCallback} callback
   */
HierarchicalDataModelVirtual.prototype.modifyVisibleRows = function modifyVisibleRows(callback) {
  var modifiedRows = [];

  this._visibleRows.forEach(function forEach(rowInfo, index) {
    var row = this.visibleToRow(index);
    var modified = callback(rowInfo, row);
    if (modified) modifiedRows.push({row: row});
  }.bind(this));

  this.changed.raise(modifiedRows);
};

HierarchicalDataModelVirtual.prototype.getLevelsStart = function() {
  if (!this.grid) return 0;

  if (!this._levelsStart) {
    this._levelsStart = DataUtils.getLevelsStart(this.grid.area);
  }

  return this._levelsStart;
};

HierarchicalDataModelVirtual.prototype.testRows = function(tester) {
  return this._visibleRows.some(function(row) {
    return tester(row);
  });
};

return {
  create: function(bufferSize) {
    return new HierarchicalDataModelVirtual(bufferSize);
  }
};

});

/*
 * Copyright (C) 2013 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

define('signal', [], function() {

"use strict"

function Signal() {
  this._subscribers = [];
}

Signal.prototype.find = function (obj, funct) {
  for (var i = 0, len = this._subscribers.length; i < len; i++) {
    if ((obj === this._subscribers[i].obj) &&
        ((funct === this._subscribers[i].funct) ||
          (funct === undefined && this._subscribers[i].funct === undefined))) {
      return i;
    }
  }

  return -1;
};

Signal.prototype.subscribe = function (obj, funct) {
  var index = this.find(obj, funct);
  if (index < 0) {
    this._subscribers.push({'obj': obj, 'funct': funct});
  }
};

Signal.prototype.unsubscribe = function (obj, funct) {
  var index = this.find(obj, funct);
  if (index >= 0) {
    this._subscribers.splice(index, 1);
  }
};

Signal.prototype.unsubscribeAll = function (obj) {
  for (var i = this._subscribers.length - 1; i >= 0; i--) {
    if (obj === this._subscribers[i].obj) {
      this._subscribers.splice(i, 1);
    }
  }
};

Signal.prototype.raise = function() {
  for (var i = 0; i < this._subscribers.length; i++) {
    var obj = this._subscribers[i].obj || window;
    if (this._subscribers[i].funct) {
      this._subscribers[i].funct.apply(obj, arguments);
    } else if (obj.raise) {
      obj.raise.apply(obj, arguments);
    }
  }
};

Signal.prototype.clear = function() {
  this._subscribers.length = 0;
};

return {
  create: function() {
    return new Signal();
  }
};

});

/*
 * Copyright (C) 2013 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

define('timeline_ctrl', ['./signal','./utils', './data_utils'], function(Signal, Utils, DataUtils) {

"use strict"

//Utils.consoleLog.enable();

const _updateWaitingInterval = 150;

var dataId_ = (function createId() {
  var startId = 0;
  var currentId = startId;
  return {
    getNext: function() {
      ++currentId;
      if (currentId < 0) currentId = ++startId;
      return currentId;
    }
  };
})();

function applyStyle(elem, style) {
  if (!style) return;

  if (typeof style === 'string') {
    Utils.addClass(elem, style);
  } else {
    Utils.applyObjectProperties(elem, style);
  }
}

function setHorzResizingCursor(elem) {
  setCursor(elem, 'ew-resize');
}

function setVertResizingCursor(elem) {
  setCursor(elem, 'ns-resize');
}

function clearResizingCursors(elem) {
  if (!elem) return;

  let cursor = getCursor(elem);

  if (cursor === 'ns-resize' ||
      cursor === 'ew-resize') {
    setCursor(elem, '');
  }
}

function processResizeTempl(inParam, getNewWidth, process, checking) {
  if (!process || !getNewWidth) return false;

  var forceProcessResize = false;
  var newWidth;
  if (this.oldWidth &&
      inParam &&
      inParam.graphWidthDelta !== undefined) {
    newWidth = this.oldWidth + inParam.graphWidthDelta;
    forceProcessResize = true;
  } else {
    newWidth = getNewWidth();
  }

  if (forceProcessResize ||
      this.oldWidth === undefined ||
      Math.abs(newWidth - this.oldWidth) > 1) {
    var param = {
      graphWidth: newWidth
    };

    Utils.applyObjectProperties(param, inParam);

    process(param);

    if (checking) {
      var realWidth = getNewWidth();
      if (realWidth !== newWidth) {
        console.log('%cwrong width', 'color: red;', newWidth, realWidth);
      } else {
        console.log('%ccorrect width', 'color: green;', newWidth);
      }
    }

    this.oldWidth = newWidth;
  }

  return forceProcessResize;
}

var dataIdAttribute = 'data-id';
var dataIndexAttribute = 'data-index';

var dragProcessing = (function() {
  var resizing = 'resizing';
  var selection = 'selection';
  var none = 'none';

  var state = none;

  function checkState(newState) {
    if (state !== none &&
        newState !== state) {
      console.warn('state !== none; current state === %s, new state === %s', state, newState);
    }
  }

  function setState(newState) {
    if (newState !== state) {
      var oldState = state;
      state = newState;
      this.onChangeState.raise(newState, oldState);
    }
  }

  return {
    startSelection: function() {
      checkState(selection);
      setState.call(this, selection);
    },
    startResizing: function() {
      checkState(resizing);
      setState.call(this, resizing);
    },
    clear: function() {
      setState.call(this, none);
    },
    isSelection: function() {
      return state === selection;
    },
    isResizing: function() {
      return state === resizing;
    },
    onChangeState: Signal.create()
  }
})();

function Selection(parent, builder) {
  const ratio = window.devicePixelRatio || 1;
  const lineGap = 0.5;

  function setSelectionPos(selection, left, width, mousePos, label) {
    function getFont(label) {
      if (label.font) return label.font;

      var fontStyle = label.fontStyle ? label.fontStyle : '';
      var fontVariant = label.fontVariant ? ' ' + label.fontVariant : '';
      var fontWeight = label.fontWeight ? ' ' + label.fontWeight : '';
      var fontSize = label.fontSize ? ' ' + Math.round(label.fontSize * ratio) + 'px' : '';
      var fontFamily = label.fontFamily ?  ' ' + label.fontFamily : ' caption';

      return fontStyle + fontVariant + fontWeight + fontSize + fontFamily;
    };

    function getHeight(label) {
      if (label.height) return label.height;

      var font = context.font;
      var res = font.match(/(\d+)px/);

      if (res.length > 1) {
        var result = parseInt(res[1]);
        var padding = 0.1 * result;
        return Math.floor(result + 2 * padding);
      }

      return 20;
    }

    if (!selection) return;

    var context = selection.getContext('2d');

    context.setTransform(1, 0, 0, 1, 0, 0);

    var selectionWidth = selection.width;
    var selectionHeight = selection.height;

    context.clearRect(0, 0, selectionWidth, selectionHeight);

    if (width > 0) {
      context.globalAlpha = 0.4;
      context.fillStyle = selectionColor;
      context.fillRect(Math.round(left * ratio), 0, Math.round(width * ratio), selectionHeight);
    }

    if (mousePos !== undefined) {
      var top = Math.round(((label && label.top !== undefined) ? label.top : 2) * ratio) + lineGap;

      mousePos = Math.round(mousePos * ratio) + lineGap;

      context.globalAlpha = 1;
      context.beginPath();
      context.moveTo(mousePos, top);
      context.lineTo(mousePos, selectionHeight);
      context.strokeStyle = '#000';
      context.lineWidth = 1;
      context.stroke();

      if (label && label.text) {
        context.font = getFont(label);

        var text = label.text;
        var measure = context.measureText(text);
        var width = Math.ceil(measure.width) + 10;
        var height = getHeight(label);

        var bkColor = label.bkColor || 'InfoBackground';
        var textColor = label.textColor || 'InfoText';

        context.fillStyle = bkColor;
        context.strokeStyle = textColor;

        var rightPadding = 1;
        var halfWidth = Math.floor(width / 2);
        var leftPos = mousePos - halfWidth;

        if (leftPos < 0) {
          leftPos = lineGap;
        } else if (leftPos + width + rightPadding > selectionWidth) {
          leftPos = Math.floor(selectionWidth - width - rightPadding) + lineGap;
        }

        context.fillRect(leftPos, top, width, height);
        context.strokeRect(leftPos, top, width, height);

        context.fillStyle = textColor;
        context.textAlign = 'center';
        context.textBaseline = 'middle';
        context.fillText(text, leftPos + halfWidth, top + Math.ceil(height / 2));
      }
    }
  }

  function showSelection(selection) {
    if (!selection ||
        !selection.offsetParent ||
        !selection.needRefreshSize) {
      return;
    }

    var clientWidth = selection.offsetParent.clientWidth;
    var width = Math.round(clientWidth * ratio);

    if (width && selection.width !== width) {
      selection.width = width;
      selection.style.width = clientWidth + 'px';
      selection.needRefreshSize = false;
    }

    var clientHeight = selection.offsetParent.clientHeight;
    var height = Math.round(clientHeight * ratio);

    if (height && selection.height !== height) {
      selection.height = height;
      selection.style.height = clientHeight + 'px';
      selection.needRefreshSize = false;
    }
  }

  function hideSelection(selection) {
    if (!selection) return;

    setSelectionPos(selection, -100, 50, -50);
  }

  function clearSelection() {
    hideSelection(this.selection);

    var selectionFinisher = this._selectionFinisher;

    if (!selectionFinisher || !selectionFinisher.isPostponed()) {
      window.removeEventListener('mousemove', processMouseMove, false);
      window.removeEventListener('mouseup', processMouseUp, false);
    }
    parent.addEventListener('mousemove', processMouseMove, false);

    dragProcessing.clear();
    this.selectionStart = undefined;
    this.parentPos = undefined;

    if (selectionFinisher) delete this._selectionFinisher;

    if (this.builder &&
        this.builder.onClearSelection) {
      this.builder.onClearSelection();
    }
  }

  parent = Utils.getDomElement(parent);

  this.onSelect = Signal.create();

  this.selection = document.createElement('canvas');
  this.selection.className = 'idvc_timeline_ctrl_selection_overlay';
  this.selection.needRefreshSize = true;

  showSelection(this.selection);
  hideSelection(this.selection);

  parent.appendChild(this.selection);

  this.clientOffsets = { left: 0, right: 0 };
  this.builder = builder;
  var selectionColor = 'Highlight';
  if(this.builder.getSelectionColor) {
    var builderSelectionColor = this.builder.getSelectionColor();
    if(builderSelectionColor) selectionColor = builderSelectionColor;
  }

  dragProcessing.onChangeState.subscribe(this, function() {
    if (dragProcessing.isResizing()) {
      setSelectionPos(this.selection, 0, 0);
    }
  });

  this.selectionStart = undefined;
  this.parentPos = undefined;
  parent.addEventListener('mousedown', function(e) {
    if (dragProcessing.isResizing()) return;

    e = e || window.event;

    if (checkMousePos.call(this, e.pageX)) {
      this.selectionStart = e.pageX - this.parentPos.x;

      showSelection(this.selection);

      if (builder &&
          builder.onStartSelection) {
        builder.onStartSelection(this.selectionStart, e.button === 2);
      }

      dragProcessing.startSelection();

      parent.removeEventListener('mousemove', processMouseMove, false);
      window.addEventListener('mousemove', processMouseMove, false);
      window.addEventListener('mouseup', processMouseUp, false);
    }
  }.bind(this), false);

  function calcSelection(start, end, offsets, wholeWidth) {
    var width = end - start;
    var resStart = start;
    var resEnd = end;
    if (width < 0) {
      resStart = end;
      resEnd = resStart - width;
    }

    if (resStart < offsets.left) resStart = offsets.left;
    var parentEnd = wholeWidth - offsets.right;
    if (resEnd > parentEnd) resEnd = parentEnd;

    return {
      start: resStart,
      width: resEnd - resStart,
      mousePos: (width > 0) ? resEnd : resStart
    };
  }

  function checkMousePos(pos) {
    if (!this.parentPos) {
      this.parentPos = Utils.getElementPos(parent);
    }

    var localPos = pos - this.parentPos.x;
    return localPos >= this.clientOffsets.left &&
        localPos <= this.parentPos.width - this.clientOffsets.right;
  }

  function getLabel(mousePos) {
    var label;
    if (builder &&
        builder.onGetLabel) {
      var offset = this.clientOffsets.left;
      label = builder.onGetLabel(mousePos - offset,
        this.selectionStart !== undefined ? this.selectionStart - offset : undefined);

      this.lastLabelMousePos = mousePos;
    }

    return label;
  }

  var processMouseMove = function(e) {
    if (dragProcessing.isResizing()) return;

    e = e || window.event;

    if (checkMousePos.call(this, e.pageX) ||
        this.selectionStart !== undefined) {
      var selectionCur = e.pageX - this.parentPos.x;

      showSelection(this.selection);

      if (e.buttons === 0 &&
          this.selectionStart !== undefined) {
        clearSelection.call(this);
      }

      let selectioStartPos = selectionCur;
      let selectionWidth = 0;
      let mousePos = selectionCur;

      if (this.selectionStart !== undefined) {
        clearResizingCursors(e.target);

        let params = calcSelection(this.selectionStart, selectionCur,
                                   this.clientOffsets, this.parentPos.width);

        selectioStartPos = params.start;
        selectionWidth = params.width;
        mousePos = params.mousePos;

        e.preventDefault();
      }

      setSelectionPos(this.selection, selectioStartPos, selectionWidth, mousePos, getLabel.call(this, mousePos));
    } else if (this.selectionStart === undefined) {
      setSelectionPos(this.selection, undefined, 0);
    }
  }.bind(this);

  parent.addEventListener('mousemove', processMouseMove, false);
  parent.addEventListener('mouseleave', function(e) {
    if (this.lastLabelMousePos) delete this.lastLabelMousePos;

    if (this.selectionStart === undefined) {
      setSelectionPos(this.selection, 0, 0);
    }

    if (builder &&
        builder.onMouseLeaveContent) {
      builder.onMouseLeaveContent();
    }
  }.bind(this), false);
  parent.addEventListener('mouseenter', function(e) {
    if(builder &&
       builder.onMouseEnterContent) {
      builder.onMouseEnterContent();
    }
  }, false);

  var processMouseUp = function(e) {
    e = e || window.event;

    if (!this.parentPos) return;

    var minPixelDistance = 1;

    var selectionCur = e.pageX - this.parentPos.x;
    var left = 0;
    var width = 0;
    var wholeWidth = 0;

    var selectionFinisher = {
      _selectionObj: this,
      _isPostponed: false,
      process: function() {
        this.raise();
        this.clear();
      },
      raise: function() {
        this._selectionObj.onSelect.raise(left, width, wholeWidth);
      },
      clear: function() {
        clearSelection.call(this._selectionObj);
      },
      postpone: function() {
        this._isPostponed = true;
        window.removeEventListener('mousemove', processMouseMove, false);
        window.removeEventListener('mouseup', processMouseUp, false);
      },
      isPostponed: function() {
        return this._isPostponed;
      },
      isRightMouseButton: function() {
        return e.button === 2;
      }
    };

    this._selectionFinisher = selectionFinisher;

    if (Math.abs(this.selectionStart - selectionCur) > minPixelDistance) {
      var params = calcSelection(this.selectionStart, selectionCur,
        this.clientOffsets, this.parentPos.width);

      var left = params.start - this.clientOffsets.left;
      var width = params.width;
      var wholeWidth = this.parentPos.width - this.clientOffsets.left - this.clientOffsets.right;

      if (builder &&
          builder.beforeProcessSelection) {
        builder.beforeProcessSelection(left, width, wholeWidth, selectionFinisher, e);
      }

      if (!selectionFinisher.isPostponed()) {
        selectionFinisher.raise();
      }
    }

    if (!selectionFinisher.isPostponed()) {
      selectionFinisher.clear();
    }
  }.bind(this);

  this.processScroll = function() {
    var mousePos = this.lastLabelMousePos;
    if (mousePos !== undefined) {
      var offset = this.clientOffsets.left;
      if (mousePos >= offset) {
        setSelectionPos(this.selection, mousePos, 0, mousePos, getLabel.call(this, mousePos));
      }
    }
  }.bind(this);
}

Selection.prototype.destroy = function() {
  dragProcessing.onChangeState.unsubscribeAll(this);
};

Selection.prototype.setClientOffsets = function(offsets) {
  this.parentPos = undefined;
  this.clientOffsets = offsets;
};

Selection.prototype.processResize = function() {
  this.parentPos = undefined;

  if (this._selectionFinisher) {
    this._selectionFinisher.clear();

    if (this.builder &&
        this.builder.onClearSelection) {
      this.builder.onClearSelection();
    }
  }

  this.selection.needRefreshSize = true;
};

/////////////////////////////////////////////////////////////////////////
///   Ruler
/////////////////////////////////////////////////////////////////////////

function Ruler(parent, rulerBuilder, style) {
  parent = Utils.getDomElement(parent);

  this.area = document.createElement('div');
  this.area.className = 'idvc_timeline_ctrl_ruler';
  applyStyle(this.area, style);
  this.area.idvcTimelineRuler = this;

  this.corner = document.createElement('div');
  this.corner.className = 'idvc_timeline_ctrl_ruler_corner';

  this.content = document.createElement('div');
  this.content.className = 'idvc_timeline_ctrl_ruler_content';

  this.area.appendChild(this.corner);
  this.area.appendChild(this.content);

  parent.appendChild(this.area);

  this.rulerBuilder = rulerBuilder;

  if (rulerBuilder &&
      rulerBuilder.onUpdate) {
    this._onUpdateCall = Utils.createAsyncCall(
        rulerBuilder.onUpdate, rulerBuilder, _updateWaitingInterval);
  }
}

Ruler.prototype.destroy = function() {
  delete this.area.idvcTimelineRuler;
};

Ruler.prototype.setHeight = function(height) {
  this.area.style.height = height + 'px';
};

Ruler.prototype.setClientOffsets = function(offsets) {
  this.corner.style.width = offsets.left + 'px';
  this.content.style.right = offsets.right + 'px';
  this.content.style.left = offsets.left + 'px';
};

Ruler.prototype.processScroll = function(left, page, delta, param) {
  if (this.rulerBuilder &&
      this.rulerBuilder.onScroll) {
    this.rulerBuilder.onScroll(left, page, delta, param);
  }

  if ((!param || !param.skipScrollUpdate) &&
      this._onUpdateCall) {
    this._onUpdateCall.call(true, param);
  }
};

Ruler.prototype.processResize = function(param) {
  var processed = processResizeTempl.call(this, param,
    function() {
      return this.content.offsetWidth;
    }.bind(this),
    function(param) {
      if (this.rulerBuilder) {
        if (this.rulerBuilder.onHorzResize) {
          this.rulerBuilder.onHorzResize(this.content, this.corner, param);
        }

        if (this.rulerBuilder.onScroll) {
          this.rulerBuilder.onScroll(undefined, undefined, 0, param);
        }
      }

      if (this._onUpdateCall) this._onUpdateCall.call(false, param);
    }.bind(this)
  );

  if (!processed &&
      this.rulerBuilder &&
      this.rulerBuilder.onVertResize) {
    param = param || {};
    param.graphWidth = this.oldWidth;
    this.rulerBuilder.onVertResize(this.content, this.corner, param);
  }
};

Ruler.prototype.refreshContent = function() {
  if (this.rulerBuilder &&
      this.rulerBuilder.onFillRuler) {
    this.rulerBuilder.onFillRuler(this.content, this.corner);
  }
}

/////////////////////////////////////////////////////////////////////////
///   HorzScrollBar
/////////////////////////////////////////////////////////////////////////

function showScrollbar(scrollbar) {
  if (!scrollbar.idvcScrollAlwaysVisible) {
    scrollbar.style.height = '2em'; //set fake width for scroll bar width calculation
    scrollbar.style.overflowX = 'scroll';

    var scrollHeight = scrollbar.offsetHeight - scrollbar.clientHeight;
    if (scrollHeight <= 1) scrollHeight = 16; //fix for OSX where scrollbars may be invisible by default
    //if (Utils.Consts.engine === 'webkit') scrollHeight++; //Chrome hides scrollbar somehow
    scrollbar.style.height = scrollHeight + 'px';

    scrollbar.style.overflowX = '';
  }
}

function hideScrollbar(scrollbar) {
  if (!scrollbar.idvcScrollAlwaysVisible) {
    scrollbar.style.height = '';
    scrollbar.style.overflowX = 'hidden';
  }
}

function HorzScrollBar(parentEl, scrollProcessor) {
  function processScrollValue(val) {
    return Math.ceil(val);
  }

  var zoomed = false;

  scrollProcessor = scrollProcessor || { onProcessScrollValue: processScrollValue };

  parentEl = Utils.getDomElement(parentEl);

  this.scrollBody = document.createElement('div');
  this.scrollBody.className = 'idvc_timeline_ctrl_horz_scroll';

  this.scrollSize = document.createElement('div');
  this.scrollSize.className = 'idvc_timeline_ctrl_scroll_size';

  this.scrollBody.appendChild(this.scrollSize);
  parentEl.appendChild(this.scrollBody);

  this.onScroll = Signal.create();
  this.onResize = Signal.create();

  this._sizes = {};

  this.skipScrollUpdate = true;

  var _scrollLeft = 0;
  Object.defineProperty(this, 'scrollLeft', {
    get: function() {
      return _scrollLeft;
    },
    set: function(val) {
      _scrollLeft = val;
      var rawScrollLeft = Math.round(val);
      if (this.scrollBody.scrollLeft !== rawScrollLeft) {
        this.scrollBody.scrollLeft = rawScrollLeft;
      }
    }
  })

  this.sendFinalScroll = function() {
    var processValue = scrollProcessor.onProcessScrollValue || processScrollValue;

    if (scrollProcessor.onScroll) {
      scrollProcessor.onScroll(processValue(this.getScrollPageFrom()), processValue(this.getScrollPageSize()),
        0, true);
    }
  };

  this.onScroll.subscribe(this, function(left, page, delta) {
    if (scrollProcessor.onScroll) scrollProcessor.onScroll(left, page, delta, false);
  });

  this.onResize.subscribe(this, function() {
    if (scrollProcessor.onResize) scrollProcessor.onResize();
  });

  function getScrollBodyWidth() {
    if (!this._sizes.scrollBodyWidth) {
      this._sizes.scrollBodyWidth = this.scrollBody.getBoundingClientRect().width;
    }

    return this._sizes.scrollBodyWidth;
  }

  function getScrollSizeWidth() {
    if (!this._sizes.scrollSizeWidth) {
      this._sizes.scrollSizeWidth = this.scrollSize.getBoundingClientRect().width;
    }

    return this._sizes.scrollSizeWidth;
  }

  function sendOnScroll(oldPageFrom) {
    var processValue = scrollProcessor.onProcessScrollValue || processScrollValue;

    var newPageFrom = this.getScrollPageFrom();
    var needSignaling = oldPageFrom === undefined ||
                        newPageFrom !== oldPageFrom;

    if (needSignaling) {
      if (oldPageFrom === undefined) oldPageFrom = newPageFrom;

      var param;
      if (this.skipScrollUpdate) {
        param = param || {};
        param.skipScrollUpdate = this.skipScrollUpdate;
        delete this.skipScrollUpdate;
      }
      if (this.showHideScrollBar) {
        param = param || {};
        param.showHideScrollBar = this.showHideScrollBar;
        delete this.showHideScrollBar;
      }

      var signalFrom = processValue(newPageFrom);
      var signalPageSize = processValue(this.getScrollPageSize());
      var signalDelta = processValue(newPageFrom - oldPageFrom);

      setTimeout(function() {
        this.onScroll.raise(signalFrom, signalPageSize, signalDelta, param);

        if (param && param.showHideScrollBar) {
          this.onResize.raise(param);
        }
      }.bind(this), 0);
    }
  }

  var refreshScroll = function(param) {
    var isResize = !param || !param.onlyUpdateScroll;

    this.scrollBody.onscroll = undefined;

    if (isResize) {
      this._sizes.scrollBodyWidth = undefined;
    }

    if (this.endProcessTimeout) clearTimeout(this.endProcessTimeout);

    var showHideScrollBar = 0;

    var newScrollLeft = 0;
    var scrollSizeStyleWidth = 1;

    var oldScrollSize = getScrollSizeWidth.call(this);
    var bodyWidth = getScrollBodyWidth.call(this);
    var newScrollSize = bodyWidth;

    if (this.scrollPage && this.scrollRange) {
      let scrolledRight = this.scrollPage.to === this.scrollRange.to;
      let logicalFrom = this.scrollPage.from - this.scrollRange.from;

      var l2pCoef = bodyWidth / this.getScrollPageSize();

      newScrollSize = this.getScrollSize() * l2pCoef;

      scrollSizeStyleWidth = newScrollSize;

      if (!scrolledRight) {
        newScrollLeft = logicalFrom * l2pCoef;
      } else {
        newScrollLeft = newScrollSize - bodyWidth;
      }
    }

    if (oldScrollSize <= bodyWidth && newScrollSize > bodyWidth) {
      showHideScrollBar = 1;
    } else if (oldScrollSize > bodyWidth && newScrollSize <= bodyWidth) {
      showHideScrollBar = -1;
    }

    if (showHideScrollBar) this.showHideScrollBar = showHideScrollBar;
    this.scrollSize.style.width = scrollSizeStyleWidth + 'px';
    this._sizes.scrollSizeWidth = scrollSizeStyleWidth;

    if (showHideScrollBar === 1) {
      showScrollbar(this.scrollBody);
    } else if (showHideScrollBar === -1) {
      hideScrollbar(this.scrollBody);
    }

    this.scrollLeft = newScrollLeft;

    sendOnScroll.call(this);

    this.endProcessTimeout = setTimeout(function() {
      this.scrollBody.onscroll = processScroll;
      delete this.endProcessTimeout;
    }.bind(this), 50);

    if (isResize) {
      this.onResize.raise();
    }

    if (!zoomed && showHideScrollBar === 1) {
      zoomed = true;

      let message = scrollProcessor.getScrollTip && scrollProcessor.getScrollTip();
      if (message) {
        setTimeout(function() {
          Utils.toast(message, this.scrollBody.previousSibling, {bottom: '0.3em'});
        }.bind(this), 50);
      }
    }
  }.bind(this);

  var processScroll = function() {
    _scrollLeft = this.scrollBody.scrollLeft;

    if (!this.scrollRange) return;

    let oldPageFrom = this.getScrollPageFrom();

    if (this.scrollPage) {
      let physicalScrollSize = getScrollSizeWidth.call(this);
      let p2lCoef = this.getScrollSize() / physicalScrollSize;

      let newPageFrom = this.scrollRange.from + _scrollLeft * p2lCoef;
      this.scrollPage.to += newPageFrom - this.scrollPage.from;
      this.scrollPage.from = newPageFrom;

      if (this.scrollPage.to > this.scrollRange.to) {
        let delta = this.scrollPage.to - this.scrollRange.to;

        this.scrollPage.to -= delta;
        this.scrollPage.from -= delta;
      }
    }

    sendOnScroll.call(this, oldPageFrom);
  }.bind(this);

  this.scrollBody.refreshSize = refreshScroll;
  this.scrollBody.onscroll = processScroll;

  this.onAccept = function() { return true; }

  this.zoomManager = (function(scrollInfo) {
    var zoom = {};

    function checkZoomFrame(frame) {
      var result = true;

      var currentZoomFrame = getCurrentZoomFrame();
      if (currentZoomFrame.from !== undefined &&
          currentZoomFrame.to !== undefined) {
        result = currentZoomFrame.to - currentZoomFrame.from !== frame.to - frame.from;
      }

      return result;
    }

    function addZoomFrame(frame) {
      if (!zoom) return;

      if (!zoom.frames) zoom.frames = [];

      if (zoom.index === undefined) zoom.index = -1;
      else if (zoom.index < zoom.frames.length - 1) {
        zoom.frames.splice(zoom.index + 1);
      }

      zoom.frames.push(frame);
      zoom.index++;

      if (zoom.frames.length > 1000) {
        zoom.frames.shift()
        zoom.index--;
      }
    }

    function getCurrentZoomFrame() {
      var frame;
      if (zoom &&
          zoom.frames &&
          zoom.index >= 0) {
        frame = zoom.frames[zoom.index];
      } else {
        frame = {
          from: undefined,
          to: undefined
        };
      }

      return frame;
    }

    function applyZoom(frame) {
      frame = frame || getCurrentZoomFrame();
      scrollInfo._setScrollPage(frame.from, frame.to);
    }

    function calcZoomFrame(coef) {
      if (!coef) return undefined;

      var page;
      var left;

      var scrollPage = scrollInfo.scrollPage;
      var scrollRange = scrollInfo.scrollRange;

      if (scrollRange) {
        if (scrollPage) {
          page = scrollPage.to - scrollPage.from;
          left = scrollPage.from;
        } else {
          page = scrollRange.to - scrollRange.from;
          left = scrollRange.from;
        }

        var newPage = page / coef;
        var newLeft = left + (page - newPage) / 2;

        if (newLeft < scrollRange.from) newLeft = scrollRange.from;
        if (newLeft + newPage > scrollRange.to) newPage = scrollRange.to - newLeft;

        return {
          from: newLeft,
          to: newLeft + newPage
        }
      }

      return undefined;
    }

    return {
      setZoom: function(param1, param2) {
        if (param1 === undefined) return;

        var frame;

        if (param2 === undefined) frame = calcZoomFrame(param1);
        else frame = {from: param1, to: param2};

        if (frame &&
            scrollInfo.onAccept(frame.from, frame.to - frame.from)) {
          if (checkZoomFrame(frame)) addZoomFrame(frame);
          applyZoom(frame);
        }
      },
      undoZoom: function() {
        if (zoom &&
            zoom.frames &&
            zoom.index >= 0) {
          zoom.index--;
          applyZoom();
        }
      },
      redoZoom: function() {
        if (zoom &&
            zoom.frames &&
            zoom.index < zoom.frames.length - 1) {
          zoom.index++;
          applyZoom();
        }
      },
      cancelZoom: function() {
        if (zoom &&
            zoom.frames) {
          zoom.frames.length = 0;
          zoom.index = -1;
          applyZoom();
        }
      },
      getZoomState: function(coef) {
        var set = false;
        var undo = false;
        var redo = false;
        var cancel = false;

        var testFrame = calcZoomFrame(coef);
        set = testFrame &&
            scrollInfo.onAccept(testFrame.from, testFrame.to - testFrame.from);

        var curFrame = getCurrentZoomFrame();
        var wholeView = curFrame.from === undefined ||
                        curFrame.to === undefined ||
                        curFrame.from <= scrollInfo.scrollRange.from &&
                        curFrame.to >= scrollInfo.scrollRange.to;

        if (zoom) {
          undo = zoom.index >= 0;
          if (zoom.frames) {
            redo = zoom.index < zoom.frames.length - 1;
            cancel = zoom.frames.length > 0;
          }
        }

        return {
          set: set,
          undo: undo,
          redo: redo,
          cancel: cancel,
          wholeView: wholeView
        };
      }
    };
  })(this);
}

HorzScrollBar.prototype.connect = function(selection, onAccept) {
  this.onAccept = onAccept || this.onAccept;

  selection.onSelect.subscribe(this, function(from, width, wholeWidth) {
    var oldScrollFrom, oldScrollTo, oldPageSize;
    var newScrollFrom, newScrollTo, newPageSize;

    if (!this.scrollRange) return;

    var scrolledRight = Math.abs(from + width - wholeWidth) < 0.0001 &&
                        (!this.scrollPage || this.scrollPage.to === !this.scrollRange.to);

    var coefWidth = width / wholeWidth;
    var coefFrom = from / wholeWidth;

    if (!this.scrollPage) {
      oldPageSize = this.scrollRange.to - this.scrollRange.from;
      oldScrollFrom = this.scrollRange.from;
      oldScrollTo = this.scrollRange.to;
    } else {
      oldPageSize = this.scrollPage.to - this.scrollPage.from;
      oldScrollFrom = this.scrollPage.from;
      oldScrollTo = this.scrollPage.to;
    }

    newPageSize = oldPageSize * coefWidth;
    if (!scrolledRight) {
      newScrollFrom = oldScrollFrom + oldPageSize * coefFrom;
      newScrollTo = newScrollFrom + newPageSize;
    } else {
      newScrollFrom = oldScrollTo - newPageSize;
      newScrollTo = oldScrollTo;
    }

    this.zoomManager.setZoom(newScrollFrom, newScrollTo);
  });
};

HorzScrollBar.prototype.setAlwaysVisible = function() {
  showScrollbar(this.scrollBody);
  this.scrollBody.style.overflowX = 'scroll';

  this.scrollBody.idvcScrollAlwaysVisible = true;
};

HorzScrollBar.prototype.setScrollInfo = function(wholeFrom, wholeTo, pageFrom, pageTo) {
  this.scrollRange = {
    from: wholeFrom,
    to: wholeTo
  };

  if (pageTo) this.setScrollPage(pageFrom, pageTo);
  else this.refreshScrollInfo();
};

HorzScrollBar.prototype.setScrollPage = function(pageFrom, pageTo) {
  this.zoomManager.setZoom(pageFrom, pageTo);
};

HorzScrollBar.prototype._setScrollPage = function(pageFrom, pageTo) {
  if (pageFrom === undefined ||
      pageTo === undefined) {
    this.scrollPage = undefined;
  } else {
    this.scrollPage = {
      from: pageFrom,
      to: pageTo
    };
  }

  this.refreshScrollInfo();
};

HorzScrollBar.prototype.updateScrollPage = function(pageFrom, pageTo) {
  var isNewScroll = !this.scrollPage ||
                    this.scrollPage.from !== pageFrom ||
                    this.scrollPage.to !== pageTo;

  if (isNewScroll) {
    this.skipScrollUpdate = true;
    this.setScrollPage(pageFrom, pageTo);
  }
};

HorzScrollBar.prototype.refreshScrollInfo = function() {
  this.scrollBody.refreshSize({onlyUpdateScroll: true});
};

HorzScrollBar.prototype.getScrollSize = function() {
  var scrollRange = this.scrollRange;
  return scrollRange ? scrollRange.to - scrollRange.from : 1;
};

HorzScrollBar.prototype.getScrollPageSize = function() {
  var scrollPage = this.scrollPage;
  return scrollPage ? scrollPage.to - scrollPage.from : this.getScrollSize();
};

HorzScrollBar.prototype.getScrollPageFrom = function() {
  if (!this.scrollRange) return 0;

  var scrollPage = this.scrollPage;
  return scrollPage ? scrollPage.from : this.scrollRange.from;
};

HorzScrollBar.prototype.setScrollLeft = function(val) {
  var scrollPage = this.scrollPage;
  if (scrollPage) {
    let scrollSize = this.getScrollSize();
    this.setScrollPage(val, val + scrollSize);
  }
};

HorzScrollBar.prototype.getScrollLeft = function() {
  var val = this.scrollLeft;
  return val / this.scrollSize.offsetWidth * this.getScrollSize();
};

HorzScrollBar.prototype.getRawScrollLeft = function() {
  var val = this.scrollBody.scrollLeft;
  return val / this.scrollSize.offsetWidth * this.getScrollSize();
};

HorzScrollBar.prototype.setClientOffsets = function(offsets) {
  this.scrollBody.style.marginRight = offsets.right + 'px';
  this.scrollBody.style.marginLeft = offsets.left + 'px';

  this._sizes = {};
};


/////////////////////////////////////////////////////////////////////////
///   TimelineArea
/////////////////////////////////////////////////////////////////////////

function AreaScrolling(area) {
  Utils.ScrolledHolder.call(this);

  this.area = area;
}

AreaScrolling.prototype = Object.create(Utils.ScrolledHolder.prototype);

AreaScrolling.prototype.setScrollTop = function(val) {
  this.area.scrolled.scrollTop = val;
};

AreaScrolling.prototype._updateVisibleScrollTop = function(val) {
  if (val !== undefined && val !== this.visibleScrollTop) {
    let delta = val - this.visibleScrollTop;

    area.bandHolder.style.paddingTop = (this.topOffset + delta) + 'px';
    area.bandHolder.style.paddingBottom = (this.bottomOffset - delta) + 'px';
  }
};

AreaScrolling.prototype._isActive = function() {
  return true;
};

AreaScrolling.prototype._getScrollState = function() {
  return {
    visibleScrollTop: this.visibleScrollTop,
    topIndex: this.topIndex
  };
};

AreaScrolling.prototype._updateScrollTop = function(val) {
  // do nothing
};

AreaScrolling.prototype.update = function(val) {
  val = val || {};

  this.visibleScrollTop = val.visibleScrollTop;
  this.topIndex = val.topIndex;

  this.topOffset = val.topOffset;
  this.bottomOffset = val.bottomOffset;

  this.syncScrollTop(val.scrollTop)
};

function TimelineArea(parent, bandBuilder, style, tabIndex) {
  parent = Utils.getDomElement(parent);

  var area = document.createElement('div');
  area.className = 'idvc_timeline_ctrl_area';
  applyStyle(area, style);

  area.idvcTimelineArea = this;

  var captionPadding = document.createElement('div');
  captionPadding.className = 'idvc_timeline_ctrl_caption_padding';

  area.appendChild(captionPadding);

  var caption = document.createElement('div');
  caption.className = 'idvc_timeline_ctrl_caption';

  var padding = document.createElement('div');
  padding.className = 'idvc_timeline_ctrl_padding';

  var container = document.createElement('div');
  container.className = 'idvc_timeline_ctrl_band_container';

  var scrolled = document.createElement('div');
  scrolled.className = 'idvc_timeline_ctrl_scrolled';
  scrolled.tabIndex = '-1';

  scrolled.appendChild(container);

  area.appendChild(caption);
  area.appendChild(scrolled);
  area.appendChild(padding);

  parent.appendChild(area);

  this.content = area;
  this.scrolled = scrolled;
  this.bandHolder = container;
  this.caption = caption;
  this.padding = padding;
  this.captionPadding = captionPadding;

  var captionPos;
  var resizingBandObj;
  var lastRefreshBandHeight;

  function getResizeTarget(elem) {
    if (!elem) return;

    if (!elem.classList) return getResizeTarget(elem.parentNode);

    var result;
    if (elem.classList.contains('idvc_timeline_ctrl_band_caption')) {
      result = elem;
    }

    return result || getResizeTarget(elem.parentNode);
  }

  var readyForResize = false;

  this.content.addEventListener('mousemove', function(e) {
    function isResizingPos(isBottomUpAlign, pos, height) {
      if (isBottomUpAlign) return pos <= borderSize;
      else return pos >= height - borderSize;
    }

    if (dragProcessing.isSelection()) return;

    e = e || event;

    dragProcessing.clear();

    var target = e.target;
    var resizeTarget = getResizeTarget(target);

    if (resizeTarget &&
        !e.ctrlKey && !e.shiftKey) {
      var borderSize = 6;

      captionPos = Utils.getElementPos(resizeTarget);

      var posY = e.pageY - captionPos.y;

      if (isResizingPos(!!this._isBottomUpAlign, posY, captionPos.height)) {
        setVertResizingCursor(target);
        readyForResize = true;
        dragProcessing.startResizing();
      } else {
        clearResizingCursors(target);
        readyForResize = false;
      }

      if (readyForResize) e.stopPropagation();
    }
  }.bind(this), false);

  Utils.createResizeProcess(this.content, {
    accepted: function(target) {
      if (dragProcessing.isSelection()) return false;

      var result = getCursor(target) === 'ns-resize' && readyForResize;
      if (result) {
        var resizeTarget = getResizeTarget(target);
        if (resizeTarget) {
          resizingBandObj = createBandObject.call(this, resizeTarget.parentNode);
          lastRefreshBandHeight = captionPos.height;
        } else {
          result = false;
        }
      }
      return result;
    },
    getDelta: function(newPos, oldPos, startPos) {
      if (this._isBottomUpAlign) return startPos.y - newPos.y;
      else return newPos.y - startPos.y;
    },
    getCursor: function() {
      return 'ns-resize';
    },
    onProcess: function(delta) {
      var minHeight = this.heights.defHeight;
      var newHeight = captionPos.height + delta;

      if (newHeight < minHeight) {
        newHeight = minHeight;
      }

      if (resizingBandObj) {
        resizingBandObj.setHeight(newHeight, true);

        if (newHeight < lastRefreshBandHeight) {
          lastRefreshBandHeight = newHeight;
          if (this._getBandCount() > 1) {
            this._refreshBands();
          } else {
            refreshTimelineSize(resizingBandObj.getDiv());
          }
        }
      }
    },
    onEnd: function() {
      resizingBandObj = undefined;
      lastRefreshBandHeight = 0;
      readyForResize = false;
      dragProcessing.clear();
    }/*,
    onDblClick: function() {
      var optimalHeight = resizingBandObj.getOptimalHeight();
      if (resizingBandObj.getHeight() !== optimalHeight) {
        resizingBandObj.setHeight(optimalHeight);
      }
    }*/
  }, this);

  var height2Index = bandBuilder.getBandIndexByHeight ?
    function(height) {
      var index = bandBuilder.getBandIndexByHeight(height, 0);
      var restHeight = height - this.getWholeHeight(index, true);

      return {
        index,
        restHeight,
        height: this.getHeight(index)
      }
    } :
    undefined;

  var getBandsHeight = bandBuilder.getBandsHeight ?
    function(count) {
      return bandBuilder.getBandsHeight(0, count - 1);
    } :
    undefined;

  this.heights = {
    map: new Map(),
    sortedHeights: undefined,
    defHeight: 20,
    wholeHeight: 0,
    getWholeHeight: getBandsHeight || function(count, onlyCalc) {
      var result = this.wholeHeight;
      if (!result || onlyCalc) {
        result = count * this.defHeight;
        this.map.forEach(function(height, index) {
          if (index < count) {
            result += (height - this.defHeight);
          }
        }.bind(this));

        if (!onlyCalc) this.wholeHeight = result;
      }
      return result;
    },
    getSortedHeights: function() {
      if (!this.sortedHeights) {
        this.sortedHeights = Array.from(this.map.entries());
        this.sortedHeights.sort(function(a, b) {
          return a[0] - b[0];
        });
      }
      return this.sortedHeights;
    },
    height2Index: height2Index || function(height) {
      var sorted = this.getSortedHeights();
      var result;
      var calcHeight = 0;
      var calcCount = 0;
      var defHeight = this.defHeight;

      function calcByDefHeight() {
        var resIndex = Math.floor((height - calcHeight) / defHeight);
        var restHeight = height - calcHeight - resIndex * defHeight;
        return {
          index: resIndex + calcCount,
          restHeight: restHeight,
          height: defHeight
        };
      }

      sorted.every(function(val) {
        var index = val[0];
        var tmpHeight = calcHeight + (index - calcCount) * defHeight;
        if (tmpHeight > height) {
          result = calcByDefHeight();
        } else {
          calcHeight = tmpHeight;
          calcCount = index;
          tmpHeight += val[1];
          if (tmpHeight > height) {
            result = {
              index: index,
              restHeight: height - calcHeight,
              height: val[1]
            };
          } else {
            calcHeight = tmpHeight;
            calcCount += 1;
          }
        }
        return !result;
      }.bind(this));

      if (!result) {
        result = calcByDefHeight();
      }
      return result;
    },
    getHeight: bandBuilder.getBandHeight || function(index) {
      var result = this.map.get(index);
      if (result === undefined) result = this.defHeight;
      return result;
    },
    setHeight: function(index, height) {
      if (height !== this.defHeight) {
        this.map.set(index, height);
      } else {
        this.map.delete(index);
      }
      this._clearCalculated();
    },
    insert: function(index, count) {
      this._shiftIndexes(this.map, index, count);
      this._shiftIndexes(this.infoMap, index, count);
      this._clearCalculated();
    },
    remove: function(index, count) {
      this._unshiftIndexes(this.map, index, count);
      this._unshiftIndexes(this.infoMap, index, count);
      this._clearCalculated();
    },
    setInfo: function(index, info, prop) {
      prop = prop || 'userInfo';

      if (!this.infoMap) {
        this.infoMap = new Map();
      }

      var storedInfo = this.infoMap.get(index);
      if (!storedInfo) {
        storedInfo = {};
        this.infoMap.set(index, storedInfo);
      }

      storedInfo[prop] = info;
    },
    getInfo: function(index, prop) {
      prop = prop || 'userInfo';

      var storedInfo;
      if (this.infoMap) {
        storedInfo = this.infoMap.get(index);
      }

      if (storedInfo) {
        return storedInfo[prop];
      }

      return undefined;
    },
    clear: function() {
      this._clearCalculated();
      this.map.clear();
      if(this.infoMap) delete this.infoMap;
    },
    _clearCalculated: function() {
      this.sortedHeights = undefined;
      this.wholeHeight = 0;
    },
    _shiftIndexes: function(map, index, count) {
      if (!map) return;

      var keys = Array.from(map.keys());
      keys.forEach(function(key) {
        if (key > index) {
          var value = map.get(key);
          map.delete(key);
          map.set(key + count, value);
        }
      });
    },
    _unshiftIndexes: function(map, index, count) {
      if (!map) return;

      var keys = Array.from(map.keys());
      keys.forEach(function(key) {
        if (key > index &&
            key <= index + count) {
          map.delete(key);
        }
      });

      keys.forEach(function(key) {
        if (key > index + count) {
          var value = map.get(key);
          map.delete(key);
          map.set(key - count, value);
        }
      });
    }
  };

  var origDefHeight = Utils.em2px(2, this.scrolled);

  this.heights.defHeight = bandBuilder && bandBuilder.getDefHeight && bandBuilder.getDefHeight() || origDefHeight;
  this.heights.origDefHeight = origDefHeight;

  this.scrolled.onscroll = function() {
    this.isScrolling = true;
    var res = this._refreshBands(true);
    res.scrollTop = this.scrolled.scrollTop;
    if (this.externalScrolling) this.externalScrolling.update(res);
    this.isScrolling = undefined;
  }.bind(this);

  this.rebuild(bandBuilder);
}

TimelineArea.prototype.getExternalScrolling = function() {
  if (!this.externalScrolling) this.externalScrolling = new AreaScrolling(this);

  this.content.addEventListener('wheel', function(e) {
    e = e || window.event;

    if (e.altKey) {
      var coef = Utils.Consts.engine === 'webkit' ? 200 : 3;
      var wheelDelta = e.deltaY / coef;

      var newScroll = this.scrolled.scrollTop + wheelDelta * this.heights.defHeight;
      this.externalScrolling.setScrollTop(newScroll);
      e.preventDefault();
    }
  }.bind(this), false);

  return this.externalScrolling;
};

TimelineArea.prototype.destroy = function() {
  delete this.content.idvcTimelineArea;
};

TimelineArea.prototype.rebuild = function(bandBuilder) {
  function copyScrollInfo(fromBandBuilder, toBandBuilder) {
    if (fromBandBuilder &&
        fromBandBuilder.updater &&
        toBandBuilder &&
        toBandBuilder.updater &&
        toBandBuilder.updater.from === undefined &&
        toBandBuilder.updater.to === undefined &&
        toBandBuilder.updater.scrollDelta === undefined ) {
      toBandBuilder.updater.from = fromBandBuilder.updater.from;
      toBandBuilder.updater.to = fromBandBuilder.updater.to;
      toBandBuilder.updater.scrollDelta = fromBandBuilder.updater.scrollDelta;
    }
  }

  bandBuilder = bandBuilder || this.bandBuilder;

  if (this.bandHolder.innerHTML) {
    var scrolledHandler = this.scrolled.onscroll;
    this.scrolled.onscroll = undefined;

    this._releaseContent();
    this.bandHolder.style.paddingTop = '';
    this.bandHolder.style.paddingBottom = '';
    this.scrolled.scrollTop = 0;

    this.scrolled.onscroll = scrolledHandler;
  }

  if (this.caption.innerHTML) {
    Utils.removeAllChildren(this.caption);
    this.caption.style.display = 'none';
  }

  copyScrollInfo(this.bandBuilder, bandBuilder);

  this.heights.clear();

  this.bandBuilder = bandBuilder;
  if (bandBuilder) {
    bandBuilder.area = this;

    if (bandBuilder.getAreaName) {
      var areaName = bandBuilder.getAreaName();
      if (areaName) {
        var text = document.createElement('span');
        text.className = 'idvc_timeline_ctrl_caption_text';
        text.innerHTML = DataUtils.escapeHTML(areaName);
        this.caption.appendChild(text);
        this.caption.style.display = 'block';

        this.content.style.minHeight = (bandBuilder.captionWidthAsAreaMinHeight && bandBuilder.captionWidthAsAreaMinHeight()) ?
          (Utils.getTextWidth(this.caption, areaName) + 'px') : '';
      } else {
        this.content.style.minHeight = '';
      }
    }
  }

  var bandCount = this.bandBuilder.getBandCount();

  if (bandCount > 1) {
    this.content.style.flexGrow = bandCount;
  }

  if (bandCount) {
    this._refreshBands();
    if (this.onChangeClientOffsets) this.onChangeClientOffsets();
  }
};

function createBandObject(band, index) {
  if (!band) return;

  if (band.classList.contains('hidden')) return;

  var bandIndex = (index !== undefined) ? index : this.getBandIndex(band);
  if (bandIndex < 0) return;

  var currentArea = this;

  function getBorderSize() {
    if (currentArea.bandBorderSize === undefined) {
      currentArea.bandBorderSize = band.offsetHeight - band.clientHeight;
    }

    return currentArea.bandBorderSize;
  }

  return {
    caption: band.firstChild,
    graph: band.lastChild,
    setHeight: function(height, isResizing) {
      var styleHeight;
      var bandHeight;
      var isBandVisible = !!band.offsetParent;
      switch (typeof height) {
        case 'string': styleHeight = height; break;
        case 'number': {
          styleHeight = height + 'px';
          bandHeight = height;
          break;
        }
      }

      var oldHeight = this.getHeight();

      var oldStyleHeight = band.style.height;
      band.style.height = this.caption.style.lineHeight = styleHeight;

      if (oldStyleHeight !== band.style.height) {
        currentArea.heights.setHeight(bandIndex, bandHeight || band.getBoundingClientRect().height);
      }

      if (currentArea.bandBuilder.onChangeBandHeight) {
        currentArea.bandBuilder.onChangeBandHeight(this);
      }

      if (currentArea.onChangeClientOffsets) currentArea.onChangeClientOffsets();

      if (isBandVisible && !isResizing && this.getHeight() < oldHeight && currentArea.getBandCount() > 1) {
        currentArea._refreshBands();
      }

      if (!isResizing) {
        this.setInfo(this.getHeight(), '_optimalHeight_');
      }
    },
    getHeight: function() {
      return currentArea.heights.getHeight(bandIndex);
    },
    setClientHeight: function(height) {
      this.setHeight(height + getBorderSize());
    },
    getClientHeight: function() {
      return this.getHeight() - getBorderSize();
    },
    getDefHeight: function() {
      return currentArea.heights.defHeight;
    },
    getClientDefHeight: function() {
      return this.getDefHeight() - getBorderSize();
    },
    getOptimalHeight: function() {
      return this.getInfo('_optimalHeight_') || this.getDefHeight();
    },
    getArea: function() {
      return currentArea;
    },
    getIndex: function() {
      return bandIndex;
    },
    getDataId: function() {
      return band.getAttribute(dataIdAttribute);
    },
    setInfo: function(info, prop) {
      currentArea.setInfo(bandIndex, info, prop);
    },
    getInfo: function(prop) {
      return currentArea.getInfo(bandIndex, prop);
    },
    getDiv: function() {
      return band;
    }
  };
}

TimelineArea.prototype.revertSizing = function() {
  this.content.style.maxHeight = '';

  if (this.bandBuilder.getAreaName) {
    var areaName = this.bandBuilder.getAreaName();
    if (areaName) {
      var minHeight = Utils.getTextWidth(this.caption, areaName);
      this.content.style.minHeight = minHeight + 'px';
    }
  }

  refreshTimelineSize(this.content);
  this._refreshBands();
  if (this.onChangeClientOffsets) this.onChangeClientOffsets();
}

TimelineArea.prototype.setInfo = function(bandIndex, info, prop) {
  this.heights.setInfo(bandIndex, info, prop);
};

TimelineArea.prototype.getInfo = function(bandIndex, prop) {
  return this.heights.getInfo(bandIndex, prop);
};

TimelineArea.prototype.getBand = function(dataId, index) {
  var band = this.bandHolder.querySelector(':scope > *[' + dataIdAttribute + '="' +
    dataId + '"]');
  return createBandObject.call(this, band, index);
};

TimelineArea.prototype.getBandIndex = function(band) {
  var childIndex = [].indexOf.call(this.bandHolder.children, band);

  if (childIndex >= 0) return this.oldTopIndex + childIndex;

  return -1;
};

TimelineArea.prototype.getTop = function() {
  return this.content.offsetTop;
};

TimelineArea.prototype.getHeight = function() {
  return this.content.offsetHeight;
};

TimelineArea.prototype.getBottom = function() {
  return this.getTop() + this.getHeight();
};

TimelineArea.prototype.getClientOffsets = function(holder) {
  function roundClientOffsets(offsets) {
    var scrollBarWidth = Utils.getScrollbarSize().width;
    offsets.right = Math.round(offsets.right / scrollBarWidth) * scrollBarWidth;
  };

  holder = holder || this.content;

  var holderPos = holder.getBoundingClientRect();
  var areaPos = this.content.getBoundingClientRect();

  var bandHolder = this.bandHolder;
  var leftOffset = Math.floor(this.caption.offsetWidth);
  var clientWidth = leftOffset + this.scrolled.clientWidth + this.padding.offsetWidth;
  if (bandHolder.children.length) {
    var bandCaption = bandHolder.firstChild.firstChild;
    leftOffset += bandCaption.offsetWidth;
  } else {
    clientWidth = this.content.clientWidth;
  }

  var right = holderPos.right - areaPos.right + (areaPos.width - clientWidth);
  if (right < 0) right = 0;
  var result = {
    left: areaPos.left - holderPos.left + leftOffset,
    right: right
  };

  roundClientOffsets(result);

  return result;
};

function setBandCaptionWidth(caption, width, hideCaption) {
  if (!caption || width === undefined) return;

  if (width < 0) width = 0;

  caption.style.width = width + 'px';

  if (hideCaption) caption.style.display = 'none';
}

TimelineArea.prototype.setClientOffsets = function(offsets, holder) {
  var bandHolder = this.bandHolder;

  function resizeBandCaptions(deltaWidth) {
    if (this.bandCaptionWidth) this.bandCaptionWidth += deltaWidth;

    for (var i = 0, len = bandHolder.children.length; i < len; i++) {
      var bandCaption = bandHolder.children[i].firstChild;
      if (this.bandCaptionWidth === undefined) {
        this.bandCaptionWidth = bandCaption.offsetWidth + deltaWidth;
      }

      setBandCaptionWidth(bandCaption, this.bandCaptionWidth, this.hideBandsCaptions);
    }
  }

  if (offsets.hideCaptions) this.hideBandsCaptions = true;

  if (offsets.leftDelta !== undefined &&
      this.bandCaptionWidth) {
    if (offsets.leftDelta) {
      this.captionPadding.style.width = offsets.left + 'px';
      resizeBandCaptions.call(this, offsets.leftDelta);
    }
  } else {
    var curOffsets = this.getClientOffsets(holder);

    var paddingRight = offsets.right - curOffsets.right;
    if (paddingRight > 0) {
      this.padding.style.display = 'block';
      this.padding.style.width = (paddingRight +
        (Utils.Consts.engine === 'moz' ? 1 : 0.5)) + 'px';
    } else {
      this.padding.style.display = 'none';
      this.padding.style.width = '0';
    }

    if (offsets.left > 0)
      this.captionPadding.style.width = offsets.left + 'px';
    else
      this.captionPadding.style.display = 'none';

    var deltaWidth = offsets.left - curOffsets.left;
    if (bandHolder.children.length) {
      resizeBandCaptions.call(this, deltaWidth);
    } else {
      this.bandCaptionWidth = deltaWidth;
    }
  }
};

TimelineArea.prototype.updateCtrlScroll = function(top, bottom) {
  var areaTop = this.getTop();
  var areaBottom = this.getBottom();
  if (bottom <= areaTop ||
      top >= areaBottom) {
    this._releaseContent();
  } else {
    var topDelta = Math.max(top - areaTop, 0);
    var bottomDelta = Math.max(areaBottom - bottom, 0);
    this._updateContent(topDelta, bottomDelta);
  }
};

TimelineArea.prototype.refresh = function() {
  disableBandNotification(this.bandBuilder);
  this.scrolled.onscroll();
  enableBandNotification(this.bandBuilder);
};

TimelineArea.prototype.refreshContent = function() {
  var fromIndex = this.oldTopIndex;

  [].forEach.call(this.bandHolder.children, function(child, index) {
    var dataId = child.getAttribute(dataIdAttribute);

    if (this.bandBuilder &&
        this.bandBuilder.onFillBand) {
      this.bandBuilder.onFillBand(fromIndex + index, dataId, this.isScrolling);
    }
  }.bind(this));
};

TimelineArea.prototype.isVisible = function() {
  return !this.content.classList.contains('hidden');
}

function refreshTimelineSize(el) {
  var timelineDiv = Utils.getParentByClass(el, 'idvc_timeline_ctrl');
  if (timelineDiv) Utils.refreshSize(timelineDiv.idvcTimeline.holder, {height: true});
}

TimelineArea.prototype.hide = function() {
  this.content.classList.add('hidden');

  this._releaseContent();
  refreshTimelineSize(this.content);
  if (this.onChangeClientOffsets) this.onChangeClientOffsets();
}

TimelineArea.prototype.show = function() {
  this.content.classList.remove('hidden');

  refreshTimelineSize(this.content);
  this._refreshBands();
  if (this.onChangeClientOffsets) this.onChangeClientOffsets();
}

TimelineArea.prototype.hideBand = function(index) {
  if (index >= 0 && index < this._getBandCount()) {
    this.heights.setHeight(index, 0);
    var band = this.bandHolder.children[index - this.oldTopIndex];
    if (band) {
      band.classList.add('hidden');
      this._refreshBands();
      if (this.onChangeClientOffsets) this.onChangeClientOffsets();
    }
  }
};

TimelineArea.prototype.isBandVisible = function(index) {
  if (index >= 0 && index < this._getBandCount()) {
    return this.heights.getHeight(index) > 0;
  }

  return false;
};

TimelineArea.prototype.showBand = function(index) {
  if (index >= 0 && index < this._getBandCount() &&
      this.heights.getHeight(index) === 0) {
    this.heights.setHeight(index, this.height.defHeight);
    var band = this.bandHolder.children[index - this.oldTopIndex];
    if (band) {
      band.classList.remove('hidden');
      this._refreshBands();
      if (this.onChangeClientOffsets) this.onChangeClientOffsets();
    }
  }
};

TimelineArea.prototype.showAllBands = function() {
  for (var i = 0, count = this._getBandCount(); i < count; i++) {
    this.showBand(i);
  }
};

TimelineArea.prototype.setTopBand = function(index) {
  if (index >= 0 && index < this._getBandCount()) {
    var scrollTop = 0;

    if (index) {
      scrollTop = this.heights.getWholeHeight(index, true);
    }

    this.scrolled.scrollTop = scrollTop;
  }
};

TimelineArea.prototype.insertBands = function(index, count) {
  var insertBands = function(index, count, before) {
    while (count > 0) {
      this._addBand(index, before);
      index++;
      count--;
    }
  }.bind(this);

  var insertCount = 0;
  var insertIdx = index + 1;
  var beforeEl;

  var defHeight = this.heights.defHeight;

  var beforeHeight = this.heights.getWholeHeight(index + 1, true);
  var crtlHeight = this._getSelfScrollTop() + this._getCtrlBottom();

  if (index < this.oldTopIndex) {
    var topHeight = this._getSelfScrollTop() + this._getCtrlTop();

    var invisibleCount = Math.floor((topHeight - beforeHeight) / defHeight);
    if (invisibleCount < 0) invisibleCount = 0;

    var gap = defHeight - (topHeight - beforeHeight - invisibleCount * defHeight);
    if (gap < 0) gap = 0;

    insertCount = Math.min(count - invisibleCount,
      Math.ceil((crtlHeight - topHeight - gap) / defHeight) + 1);
    if (insertCount > 0) {
      insertIdx = this.oldTopIndex;
      beforeEl = this.bandHolder.children[0];
    }
  } else if (index <= this.oldBottomIndex && crtlHeight > beforeHeight) {
    insertCount = Math.min(count, Math.ceil((crtlHeight - beforeHeight) / defHeight));
    if (insertCount > 0) {
      beforeEl = this.bandHolder.children[index - this.oldTopIndex + 1];
    }
  }

  this.heights.insert(index, count);
  insertBands(insertIdx, insertCount, beforeEl);
  this.oldBottomIndex += insertCount;
  this._refreshBands();
  if (this.onChangeClientOffsets) this.onChangeClientOffsets();
};

TimelineArea.prototype.removeBands = function(index, count) {
  var removeBands = function(index, count, from) {
    while (count > 0 && from) {
      var nextFrom = from.nextSibling;
      this._removeBand(from, index);
      from = nextFrom;
      index++;
      count--;
    }
  }.bind(this);

  var removeCount = 0;
  var removeIdx = index + 1;
  var removeEl;
  if (index < this.oldTopIndex) {
    removeCount = index + count - this.oldTopIndex + 1;
    if (removeCount > 0) {
      removeIdx = this.oldTopIndex;
      removeEl = this.bandHolder.children[0];
    }
  } else if (index <= this.oldBottomIndex) {
    removeCount = Math.min(count, this.oldBottomIndex - index);
    if (removeCount > 0) {
      removeEl = this.bandHolder.children[index - this.oldTopIndex + 1];
    }
  }

  this.heights.remove(index, count);
  removeBands(removeIdx, removeCount, removeEl);
  this.oldBottomIndex -= removeCount;
  this._refreshBands();
  if (this.onChangeClientOffsets) this.onChangeClientOffsets();
};

function updateVisibleBands(param) {
  if (this.bandBuilder.onUpdateBands) {
    var fromIndex = this.oldTopIndex;

    var bandObjects = [];

    [].forEach.call(this.bandHolder.children, function(band, index) {
      var dataId = dataId_.getNext().toString();
      band.setAttribute(dataIdAttribute, dataId);
      bandObjects.push(createBandObject.call(this, band, index + fromIndex));
    }.bind(this));

    if (bandObjects.length) {
      this.bandBuilder.onUpdateBands(bandObjects, param);
    }
  }
}

TimelineArea.prototype.updateVisibleBands = function(param) {
  updateVisibleBands.call(this, param);

  if (this.bandBuilder &&
      this.bandBuilder.updater) {
    return this.bandBuilder.updater.processAllRequests();
  }
}

TimelineArea.prototype.getVisibleBands = function() {
  var bandObjects = [];
  var fromIndex = this.oldTopIndex;
  [].forEach.call(this.bandHolder.children, function(band, index) {
    var bandObj = createBandObject.call(this, band, fromIndex + index);
    if (bandObj) bandObjects.push(bandObj);
  }.bind(this));

  return bandObjects;
}

TimelineArea.prototype.processScroll = function(left, page, delta, param) {
  if (this.bandBuilder) {
    if (this.bandBuilder.onScroll) {
      this.bandBuilder.onScroll(left, page, delta, param);
    }

    if (this.isVisible() &&
        (!param || !param.skipScrollUpdate)) {
      updateVisibleBands.call(this, param);
    }
  }
}

TimelineArea.prototype.processResize = function(param) {
  function getGraphWidth(bandHolder) {
    if (bandHolder.children.length) {
      var bandCaption = bandHolder.firstChild.firstChild;
      var bandGraph = bandCaption.nextSibling;
      return bandGraph.offsetWidth;
    }

    return 0;
  }

  processResizeTempl.call(this, param,
    getGraphWidth.bind(this, this.bandHolder),
    function(param) {
      if (this.bandBuilder.onChangeAreaWidth) {
        this.bandBuilder.onChangeAreaWidth(this.getVisibleBands(), param);
      }

      if (this.bandBuilder &&
          this.bandBuilder.onResize) {
        this.bandBuilder.onResize(param);
      }

      updateVisibleBands.call(this, param);
    }.bind(this)
  );
};

TimelineArea.prototype._releaseContent = function() {
  var fromIndex = this.oldTopIndex;
  var toIndex = this.oldBottomIndex;
  for (var index = fromIndex; index <= toIndex; index++) {
    this._removeBand(this.bandHolder.firstChild, index);
  }

  Utils.removeAllChildren(this.bandHolder);

  var height = this.heights.getWholeHeight(this._getBandCount());

  this.bandHolder.style.paddingTop = height + 'px';
  this.bandHolder.style.paddingBottom = '0px';
};

TimelineArea.prototype._updateContent = function(top, bottom) {
  this._ctrlTop = top;
  this._ctrlBottom = bottom;
  this._refreshBands();
};

/* addBand function with additional check of band existance for particular index
TimelineArea.prototype._addBand = function(index, before) {
  function getBandByIndex(index) {
    return this.bandHolder.querySelector(':scope > *[' + dataIndexAttribute + '="' +
      index + '"]');
  }

  if (index >= this._getBandCount()) return;

  var dataId = dataId_.getNext().toString();

  var band = getBandByIndex.call(this, index);
  if (band) {
    console.log('Band is created: ', index);
  } else {
    band = document.createElement('div');
    band.className = 'idvc_timeline_ctrl_band';
    band.setAttribute(dataIndexAttribute, index);

    var caption = document.createElement('div');
    caption.className = 'idvc_timeline_ctrl_band_caption';

    var graph = document.createElement('div');
    graph.className = 'idvc_timeline_ctrl_band_graph';

    var height = this.heights.getHeight(index);
    if (height && height !== this.heights.defHeight) {
      band.style.height = height + 'px';
      caption.style.lineHeight = height + 'px';
    }

    if (!height) {
      band.classList.add('hidden');
    }

    band.appendChild(caption);
    band.appendChild(graph);

    if (!before) this.bandHolder.appendChild(band);
    else this.bandHolder.insertBefore(band, before);

    if (this.bandCaptionWidth) {
      caption.style.width = this.bandCaptionWidth + 'px';
    }
  }

  band.setAttribute(dataIdAttribute, dataId);

  if (this.bandBuilder &&
      this.bandBuilder.onFillBand) {
    this.bandBuilder.onFillBand(index, dataId, this.isScrolling);
  }
};
*/

TimelineArea.prototype._addBand = function(index, before) {
  if (index >= this._getBandCount()) return;

  var dataId = dataId_.getNext().toString();

  var band = document.createElement('div');
  band.className = 'idvc_timeline_ctrl_band';
  band.setAttribute(dataIndexAttribute, index);
  band.setAttribute(dataIdAttribute, dataId);

  var caption = document.createElement('div');
  caption.className = 'idvc_timeline_ctrl_band_caption';

  var graph = document.createElement('div');
  graph.className = 'idvc_timeline_ctrl_band_graph';

  var height = this.heights.getHeight(index);
  if (height && height !== this.heights.origDefHeight) {
    band.style.height = height + 'px';
    caption.style.lineHeight = height + 'px';
  }

  if (!height) {
    band.classList.add('hidden');
  }

  band.appendChild(caption);
  band.appendChild(graph);

  if (!before) this.bandHolder.appendChild(band);
  else this.bandHolder.insertBefore(band, before);

  setBandCaptionWidth(caption, this.bandCaptionWidth, this.hideBandsCaptions);

  if (this.bandBuilder &&
      this.bandBuilder.onFillBand) {
    this.bandBuilder.onFillBand(index, dataId, this.isScrolling);
  }
};

TimelineArea.prototype._removeBand = function(band, index) {
  if (!band) return;

  var dataId = band.getAttribute(dataIdAttribute);
  this.bandHolder.removeChild(band);

  if (this.bandBuilder &&
      this.bandBuilder.onRemoveBand) {
    this.bandBuilder.onRemoveBand(index, dataId);
  }
};

TimelineArea.prototype._refreshBands = function() {
  if(!this.isVisible()) return;

  var addAllBands = function(from, to, count) {
    for (var i = from; i < count && i <= to; i++) {
      this._addBand(i);
    }
  }.bind(this);

  if (this._refreshCallCount) {
    this._refreshCallCount++;
    return;
  }

  this._refreshCallCount = 1;

  var bandCount = this._getBandCount();

  var top = this._getSelfScrollTop() + this._getCtrlTop();
  var topPos = this.heights.height2Index(top);

  var bottom = this._getSelfScrollTop() + this._getCtrlBottom();
  var bottomPos = this.heights.height2Index(bottom);

  var fromIndex = topPos.index;
  var toIndex = bottomPos.index;

  var offsetTop = top - topPos.restHeight;

  var offsetBottom = 0;
  if (bottomPos.index < bandCount - 1) {
    offsetBottom = this.heights.getWholeHeight(bandCount) - bottom - (bottomPos.height - bottomPos.restHeight);
    if (offsetBottom < 0) offsetBottom = 0;
  } else {
    toIndex = bandCount - 1;
  }

  this.bandHolder.style.paddingTop = offsetTop + 'px';
  this.bandHolder.style.paddingBottom = offsetBottom + 'px';

  var result = {
    visibleScrollTop: topPos.restHeight,
    topIndex: fromIndex,
    topOffset: offsetTop,
    bottomOffset: offsetBottom
  };

  /*this.caption.style.paddingTop = (fromIndex < toIndex) ?
    (offsetTop + topPos.restHeight + Utils.em2px(0.4)) + 'px' : '0px';*/

  if (!this.bandHolder.children.length) {
    addAllBands(fromIndex, toIndex, bandCount);
  } else {
    var changedTop = fromIndex - this.oldTopIndex;
    var changedBottom = this.oldBottomIndex - toIndex;

    var curIndex = this.oldTopIndex;
    if (changedTop > 0) {
      while (this.bandHolder.firstChild && changedTop) {
        this._removeBand(this.bandHolder.firstChild, curIndex);
        curIndex++;
        changedTop--;
      }

      changedTop = 0;
    }

    curIndex = this.oldBottomIndex;
    if (changedBottom > 0) {
      while (this.bandHolder.lastChild && changedBottom) {
        this._removeBand(this.bandHolder.lastChild, curIndex);
        curIndex--;
        changedBottom--;
      }

      changedBottom = 0;
    }

    if (!this.bandHolder.children.length) {
      addAllBands(fromIndex, toIndex, bandCount);
    } else {
      if (changedTop < 0) {
        var topIndex = this.oldTopIndex;
        while (changedTop < 0 && topIndex >= 0) {
          topIndex--;
          this._addBand(topIndex, this.bandHolder.firstChild);
          changedTop++;
        }
      }

      if (changedBottom < 0) {
        var bottomIndex = this.oldBottomIndex;
        while (changedBottom < 0 && bottomIndex < bandCount - 1) {
          bottomIndex++;
          this._addBand(bottomIndex);
          changedBottom++;
        }
      }
    }
  }

  this.oldTopIndex = fromIndex;
  this.oldBottomIndex = toIndex;

  this._refreshCallCount--;

  if (this._refreshCallCount) {
    this._refreshCallCount = 0;
    //Temporary removed it seems it works
    //this._refreshBands();
  }

  return result;
};

TimelineArea.prototype._getSelfScrollTop = function() {
  return this.scrolled.scrollTop;
};

TimelineArea.prototype._getCtrlTop = function() {
  return this._ctrlTop || 0;
};

TimelineArea.prototype._getCtrlBottom = function() {
  return this.getHeight() - (this._ctrlBottom || 0);
};

TimelineArea.prototype._getBandCount = function() {
  if (this.bandBuilder &&
      this.bandBuilder.getBandCount) {
    return this.bandBuilder.getBandCount();
  }

  return 0;
};

TimelineArea.prototype.getBandCount = function() {
  return this._getBandCount();
};

TimelineArea.prototype.setBottomAlign = function() {
  this.content.classList.add('idvc_timeline_bottom_align');

  this._isBottomUpAlign = true;
};

TimelineArea.prototype.clearBottomAlign = function() {
  this.content.classList.remove('idvc_timeline_bottom_align');

  delete this._isBottomUpAlign;
};

/////////////////////////////////////////////////////////////////////////
///   TimelineAreaHolder
/////////////////////////////////////////////////////////////////////////

function TimelineAreaHolder(parent, style, tabIndex) {
  this._parent = Utils.getDomElement(parent);

  this.content = document.createElement('div');
  this.content.className = 'idvc_timeline_ctrl_area_holder';
  applyStyle(this.content, style);
  this._parent.appendChild(this.content);

  this.areas = [];

  function refreshScrollArea(ev) {
    var top = this._getSelfScrollTop();
    var bottom = top + this.content.offsetHeight;

    this.forEachVisibleAreas(function(area) {
      if (!ev) disableBandNotification(area.bandBuilder);
      area.updateCtrlScroll(top, bottom);
      if (!ev) enableBandNotification(area.bandBuilder);
    });
  }

  this.content.onscroll = refreshScrollArea.bind(this);
  this.content.refreshSize = refreshScrollArea.bind(this, {});
};

TimelineAreaHolder.prototype.addArea = function(bandBuilder, style, tabIndex) {
  var area = new TimelineArea(this.content, bandBuilder, style, tabIndex);
  this.areas.push(area);
  area._releaseContent();
  area.onChangeClientOffsets = this.onChangeClientOffsets;
  return area;
};

TimelineAreaHolder.prototype.getArea = function(index) {
  if (index >= 0 &&
      index < this.getAreaCount()) {
    return this.areas[index];
  }

  return undefined;
};

TimelineAreaHolder.prototype.getAreaCount = function() {
  return this.areas.length;
};

TimelineAreaHolder.prototype.refresh = function() {
  this.content.onscroll();
};

TimelineAreaHolder.prototype.refreshContent = function() {
  this.forEachVisibleAreas(function(area) {
    area.refreshContent();
  });
};

TimelineAreaHolder.prototype.getClientOffsets = function(holder) {
  var offsets = {
    right: 0,
    left: 0
  };

  this.forEachVisibleAreas(function(area) {
    if (area.getClientOffsets) {
      var newOffsets = area.getClientOffsets(holder);
      if (newOffsets.right > offsets.right) offsets.right = newOffsets.right;
      if (newOffsets.left > offsets.left) offsets.left = newOffsets.left;
    }
  });

  return offsets;
};

TimelineAreaHolder.prototype.setClientOffsets = function(offsets, holder) {
  this.forEachVisibleAreas(function(area) {
    if (area.setClientOffsets) {
      area.setClientOffsets(offsets, holder);
    }
  });
};

TimelineAreaHolder.prototype._getSelfScrollTop = function() {
  return this.content.scrollTop;
};

TimelineAreaHolder.prototype.processScroll = function(left, page, delta, param) {
  this.areas.forEach(function(area) {
    if (area.processScroll) {
      area.processScroll(left, page, delta, param);
    }
  });
};

TimelineAreaHolder.prototype.processResize = function(param) {
  this.forEachVisibleAreas(function(area) {
    if (area.processResize) {
      area.processResize(param);
    }
  });
};

TimelineAreaHolder.prototype.forEachVisibleAreas = function(process) {
  this.areas.forEach(function(area) {
    if (area.isVisible()) {
      process(area);
    }
  });
};

/////////////////////////////////////////////////////////////////////////
///   Timeline
/////////////////////////////////////////////////////////////////////////

function Timeline(parent, tabIndex, rightPadding) {
  this._parent = Utils.getDomElement(parent);

  this.holder = document.createElement('div');
  this.holder.className = 'idvc_timeline_ctrl'

  this.content = document.createElement('div');
  this.content.className = 'idvc_timeline_ctrl_content';

  this.holder.appendChild(this.content);
  this._parent.appendChild(this.holder);

  this.elements = [];

  if (rightPadding === undefined) rightPadding = 2;

  this.maxPaddingWidth = rightPadding * Utils.getScrollbarSize().width;
  this.holder.style.right = this.maxPaddingWidth + 'px';

  this.holder.idvcTimeline = this;

  this.holder.refreshSize = function(param) {
    if (this._needSetClientOffset) {
      setClientOffsets.call(this);
      delete this._needSetClientOffset;
    }
  }.bind(this);

  function getResizeTarget(elem) {
    if (!elem) return;

    if (!elem.classList) return getResizeTarget(elem.parentNode);

    var result;
    if (elem.classList.contains('idvc_timeline_ctrl_area')) {
      result = elem;
    }

    return result || getResizeTarget(elem.parentNode);
  }

  var contentPos;
  var resizedArea;

  function refreshContentPos() {
    if (!contentPos) {
      contentPos = Utils.getElementPos(this.content);
      this.contentHeight = contentPos.height;
    }
  }

  function refreshRulerHeight() {
    if (this.rulerHeight === undefined) {
      var ruler = this.content.querySelector('.idvc_timeline_ctrl_ruler');
      if (ruler) this.rulerHeight = ruler.offsetHeight;
      else this.rulerHeight = 0;
    }
  }

  this.content.refreshSize = function(param) {
    contentPos = undefined;
    if (!param ||
        param.height) {
      this.contentHeight = undefined;
    }
  }.bind(this);

  this.content.addEventListener('mousemove', function(e) {
    function clearResizingAttrs() {
      resizedArea = undefined;

      clearResizingCursors(target);
    }

    if (dragProcessing.isSelection()) return;

    if (e.ctrlKey || e.shiftKey) return;

    e = e || event;

    var target = e.target;
    var borderSize = 6;

    refreshRulerHeight.call(this);
    refreshContentPos.call(this);
    var posX = e.pageX - contentPos.x;
    var posY = e.pageY - contentPos.y;

    var captionsWidth = this.getCaptionsWidth();

    if (posY <= this.rulerHeight) {
      dragProcessing.clear();
      clearResizingAttrs();
    } else if (posX >= captionsWidth - borderSize &&
               posX < captionsWidth) {
      setHorzResizingCursor(target);
      resizedArea = undefined;
      dragProcessing.startResizing();
    } else if (!dragProcessing.isResizing() &&
               posX < captionsWidth) {
      dragProcessing.clear();
      clearResizingAttrs();

      var areaResizeTarget = getResizeTarget(target);
      if (areaResizeTarget) {
        var areaPos = Utils.getElementPos(areaResizeTarget);

        var posY = e.pageY - areaPos.y;

        if (posY <= borderSize) {
          var prevElement = areaResizeTarget.previousSibling;
          while (prevElement && prevElement.classList.contains('idvc_timeline_ctrl_area')) {
            if (prevElement.offsetParent) {
              resizedArea = prevElement;
              break;
            } else {
              prevElement = prevElement.previousSibling;
            }
          }
        } else {
          clearResizingAttrs();
        }

        if (resizedArea) {
          if (resizedArea.idvcTimelineArea &&
              resizedArea.idvcTimelineArea._getBandCount() < 2 &&
              resizedArea.classList.contains('idvc_timeline_ctrl_area_fixed')) {
            resizedArea = undefined;
          }
        }

        if (resizedArea) {
          setVertResizingCursor(target);
          dragProcessing.startResizing();
        }
      }
    } else {
      clearResizingAttrs();
    }
  }.bind(this), false);

  Utils.createResizeProcess(this.content, {
    accepted: function(target) {
      if (dragProcessing.isSelection()) return false;

      if (!resizedArea) {
        return getCursor(target) === 'ew-resize';
      }

      return resizedArea !== undefined;
    },
    getDelta: function(newPos, oldPos) {
      if (!resizedArea)
        return newPos.x - contentPos.x - this.clientOffsets.left;

      return newPos.y - oldPos.y;
    },
    getCursor: function() {
      if (!resizedArea)
        return 'ew-resize';

      return 'ns-resize';
    },
    onProcess: function(delta) {
      if (!resizedArea) {
        var minOffsetLeft = 70;
        var maxOffsetLeft = contentPos.width / 2;

        var newOffsetLeft = this.clientOffsets.left + delta;

        if (newOffsetLeft < minOffsetLeft &&
            this.clientOffsets.left > minOffsetLeft) {
          newOffsetLeft = minOffsetLeft;
        }

        if (newOffsetLeft > maxOffsetLeft &&
            this.clientOffsets.left < maxOffsetLeft) {
          newOffsetLeft = maxOffsetLeft;
        }

        if (newOffsetLeft >= minOffsetLeft &&
            newOffsetLeft <= maxOffsetLeft) {
          setClientOffsets.call(this, {
            left: newOffsetLeft,
            right: this.clientOffsets.right,
            leftDelta: newOffsetLeft - this.clientOffsets.left
          });
        }
      } else {
        var height = resizedArea.offsetHeight + delta;
        resizedArea.style.minHeight = height + 'px';
        resizedArea.style.maxHeight = height + 'px';

        if (delta > 0) resizedArea.idvcTimelineArea._refreshBands();

        Utils.refreshSize(this.holder, {height: true});
        this.processResize();
      }
    },
    onEnd: function() {
      if (!resizedArea) this.horzScroll.refreshScrollInfo();
      else resizedArea.idvcTimelineArea._refreshBands();

      resizedArea = undefined;
      dragProcessing.clear();
    }
  }, this);

  var captionExpansion = Utils.addExpansion(this.content, function(el) {
    if (dragProcessing.isSelection()) return undefined;

    if (el &&
        el.classList.contains('idvc_timeline_ctrl_band_caption') &&
        getCursor(el) !== 'ew-resize') {
      return el;
    }

    return undefined;
  });

  var groupingExpansion = Utils.addExpansion(this.content, function(el) {
    if (dragProcessing.isSelection()) return undefined;

    if (el &&
        el.classList.contains('idvc_timeline_ctrl_caption') &&
        getCursor(el) !== 'ns-resize') {
      return el;
    }

    return undefined;
  }, undefined, true);

  groupingExpansion.afterCreateTooltip = function(tooltip) {
    if (!tooltip) return;

    var text = tooltip.firstChild;
    if (text) {
      tooltip.style.height = (text.offsetWidth + Utils.em2px(0.6, text)) + 'px';
    }
  }

  this.updateCaptionExpansion = function() {
    captionExpansion.update();
  };

  this.setExpansionContentProcessor = function(processor) {
    captionExpansion.processContent = processor;
  };

  var tooltipInfo = {
    elem: undefined,
    elemPos: undefined
  };
  var clickInfo;

  function findAreaAndBand(el) {
    var areaObj;
    var bandIndex = -1;
    var bandObj;

    var areaDiv = Utils.getParentByClass(el, 'idvc_timeline_ctrl_area');
    if (areaDiv &&
        (areaObj = areaDiv.idvcTimelineArea)) {
      var bandDiv = Utils.getParentByClass(el, 'idvc_timeline_ctrl_band');
      if (bandDiv) {
        bandIndex = areaObj.getBandIndex(bandDiv);
        bandObj = createBandObject.call(areaObj, bandDiv, bandIndex);
      }
    }

    return {
      areaObj: areaObj,
      bandObj: bandObj,
      bandIndex: bandIndex
    };
  }

  function findTimelineElement(el) {
    var areaAndBand = findAreaAndBand(el);

    if (areaAndBand.areaObj) {
      var bandEmptySpace = false;

      if (!areaAndBand.bandObj) {
        var paddingDiv = Utils.getParentByClass(el, 'idvc_timeline_ctrl_padding');
        if (!paddingDiv) {
          bandEmptySpace = true;
        }
      }

      return {
        areaObj: areaAndBand.areaObj,
        bandObj: areaAndBand.bandObj,
        bandIndex: areaAndBand.bandIndex,
        bandEmptySpace: bandEmptySpace
      };
    } else {
      var rulerDiv = Utils.getParentByClass(el, 'idvc_timeline_ctrl_ruler');
      var rulerObj;
      if (rulerDiv) rulerObj = rulerDiv.idvcTimelineRuler;

      return {
        rulerObj: rulerObj
      };
    }
  }

  function getElementPos(el) {
    return tooltipInfo.elemPos || Utils.getElementPos(el);
  }

  function sendBandNotification(el, notification, x, y, ev, findResult) {
    var elemPos = getElementPos(el);
    if (!elemPos) return undefined;

    var areaAndBand = findResult || findAreaAndBand(el);
    var areaObj = areaAndBand.areaObj;
    var bandObj = areaAndBand.bandObj;
    if (areaObj &&
        areaObj.bandBuilder &&
        areaObj.bandBuilder[notification] &&
        bandObj) {
      return areaObj.bandBuilder[notification](bandObj, areaAndBand.bandIndex, el,
        x - elemPos.x, y - elemPos.y, ev);
    }
  }

  function sendElementNotification(el, bandNotification, rulerNotification, x, y, ev) {
    var elemPos = getElementPos(el);
    if (!elemPos) return undefined;

    var findResult = findTimelineElement(el);
    var areaObj = findResult.areaObj;
    var rulerObj = findResult.rulerObj;
    if (areaObj &&
        findResult.bandObj) {
      return sendBandNotification(el, bandNotification, x, y, ev, findResult);
    } else if (findResult.bandEmptySpace &&
               areaObj.bandBuilder[bandNotification]) {
      return areaObj.bandBuilder[bandNotification](undefined, -1, el,
                x - elemPos.x, y - elemPos.y, ev);
    } else if (rulerObj &&
               rulerNotification &&
               rulerObj.rulerBuilder &&
               rulerObj.rulerBuilder[rulerNotification]) {
      return rulerObj.rulerBuilder[rulerNotification](el,
               x - elemPos.x, y - elemPos.y, ev);
    }

    return undefined;
  }

  this.content.addEventListener('mousedown', function(e) {
    if (e.button === 0) {
      clickInfo = {
        elem: e.target,
        pos: {
          x: e.pageX,
          y: e.pageY
        }
      };
    } else if (e.button === 2) {
      sendElementNotification(e.target, 'onBandContextMenu', 'onContextMenu', e.pageX, e.pageY, e);
    }
  }, false);

  this.content.addEventListener('mouseup', function(e) {
    if (clickInfo) {
      var clickDistance = 1;

      var el = e.target;
      var x = clickInfo.pos.x;
      var y = clickInfo.pos.y;

      if (clickInfo.elem === el &&
          Math.abs(x - e.pageX) <= clickDistance &&
          Math.abs(y - e.pageY) <= clickDistance) {
        sendElementNotification(el, 'onBandClick', undefined, x, y, e);
      }

      clickInfo = undefined;
    }
  }, false);

  this.content.addEventListener('dblclick', function(e) {
    sendBandNotification(e.target, 'onBandDblClick', e.pageX, e.pageY, e);
  }, false);

  var tooltip = Utils.addTooltip(this.content, function(el, x, y) {
    if (tooltipInfo.elem !== el) {
      tooltipInfo.elemPos = Utils.getElementPos(el);
      tooltipInfo.elem = el;
    }

    if (el) return sendElementNotification(el, 'onBandTooltip', 'onTooltip', x, y);

    return undefined;
  }, function() {
    return true;
  });

  this.showTooltip = function(el, text, x, y, autoHideDelay) {
    function getElemPos() {
      if (!elemPos) {
        if (el === tooltipInfo.elem) {
          elemPos = tooltipInfo.elemPos;
        } else {
          elemPos = Utils.getElementPos(el);
        }
      }

      return elemPos;
    }

    if (!el ||
        tooltipInfo.elem !== el) {
      return;
    }

    var elemPos;

    if (x !== undefined) x += getElemPos().x;
    if (y !== undefined) y += getElemPos().y;

    tooltip.show(el, text, x, y, autoHideDelay);
  }

  this.hideTooltip = function(hideInterval) {
    hideInterval = hideInterval || 1000;
    tooltip.hide(hideInterval);

    captionExpansion.hide();
    groupingExpansion.hide();
  };

  var scrollStep = .1;

  function getCurrentFrame(horzScroll) {
    var result = {
      from: 0,
      to: 0
    };

    if (horzScroll) {
      if (horzScroll.scrollPage) {
        result.from = horzScroll.scrollPage.from;
        result.to = horzScroll.scrollPage.to;
      } else if (horzScroll.scrollRange) {
        result.from = horzScroll.scrollRange.from;
        result.to = horzScroll.scrollRange.to;
      }
    }

    return result;
  }

  function checkFrame(frame, range) {
    var result = {
      from: frame.from,
      to: frame.to
    };

    var pageSize = Math.min(frame.to - frame.from, range.to - range.from);

    if (result.from < range.from) {
      result.from = range.from;
      result.to = result.from + pageSize;
    }

    if (result.to > range.to) {
      result.to = range.to;
      result.from = result.to - pageSize;
    }

    return result;
  }

  function calcNewFrame(from, to, step) {
    return {
      from: from + step,
      to: to + step
    };
  }

  this.holder.addEventListener('keydown', function(e) {
    var scrollRange = this.horzScroll && this.horzScroll.scrollRange;

    if (scrollRange) {
      var currentFrame = getCurrentFrame(this.horzScroll);
      var frameSize = currentFrame.to - currentFrame.from;
      var step;
      var newFrame;

      if (e.keyCode === 37) {
        step = -frameSize * scrollStep;
      } else if (e.keyCode === 39) {
        step = frameSize * scrollStep;
      } else if (e.ctrlKey && e.keyCode === 36) {
        newFrame = {
          from: 0,
          to: frameSize
        };
      }  else if (e.ctrlKey && e.keyCode === 35) {
        newFrame = {
          from: scrollRange.to - frameSize,
          to: scrollRange.to
        };
      }

      if (step) {
        newFrame = checkFrame(calcNewFrame(currentFrame.from, currentFrame.to, step), scrollRange);
      }

      if (newFrame &&
          (newFrame.from !== currentFrame.from ||
           newFrame.to !== currentFrame.to)) {
        this.horzScroll.setScrollPage(newFrame.from, newFrame.to);
      }
    }
  }.bind(this), false);

  this.holder.addEventListener('wheel', function(e) {
    function calcNewZoomFrame(from, to, pos, coef) {
      var before = pos - from;
      var after = to -pos;

      return {
        from: pos - before * coef,
        to: pos + after * coef
      };
    }

    e = e || window.event;

    if (e.altKey) {
      // default vertical scrolling
    } else if (this.horzScroll &&
        this.horzScroll.scrollRange) {
      var currentFrame = getCurrentFrame(this.horzScroll);
      var newFrame;
      if (e.ctrlKey) {
        // horizontal scrolling
        var frameSize = currentFrame.to - currentFrame.from;
        var coef = Utils.Consts.engine === 'webkit' ? 200 : 3;
        var step = e.deltaY / coef * frameSize * scrollStep;
        newFrame = calcNewFrame(currentFrame.from, currentFrame.to, step);
      } else {
        var scrollPos = Utils.getElementPos(this.horzScroll.scrollBody);
        if (e.pageX > scrollPos.x &&
            e.pageX < scrollPos.x + scrollPos.width) {
          var coef = e.deltaY > 0 ? 2 : 0.5;
          var pos = currentFrame.from + (e.pageX - scrollPos.x) / scrollPos.width * (currentFrame.to - currentFrame.from);

          newFrame = calcNewZoomFrame(currentFrame.from, currentFrame.to, pos, coef);
        }
      }

      if (newFrame) {
        newFrame = checkFrame(newFrame, this.horzScroll.scrollRange);
        if (newFrame.from !== currentFrame.from ||
            newFrame.to !== currentFrame.to) {
          this.horzScroll.setScrollPage(newFrame.from, newFrame.to);
        }
      }

      e.preventDefault();
    }
  }.bind(this), false);

  this.content.addEventListener('mouseenter', scrollingCursors.addTimeline.bind(scrollingCursors, this));
  this.content.addEventListener('mouseleave', scrollingCursors.removeTimeline.bind(scrollingCursors, this));

  this.windowResizeListener = Utils.addWindowResizeListener(this.holder);
}

var scrollingCursors = (function(){
  var _timeline;
  var keyUpIsOn = false;

  document.addEventListener('keydown', function(e) {
    if (updateScrollingCursors(_timeline, e) > 0) {
      add(_timeline);
    }
  });

  function add(timeline) {
    _timeline = timeline;

    if (!keyUpIsOn) {
      keyUpIsOn = true;
      document.addEventListener('keyup', removeScrollingCursors);
    }
  }

  function updateScrollingCursors(timeline, e) {
    if (!timeline || !e) return 0;

    var result = 0;

    if (e.altKey) {
      timeline.setCursor('ns-resize');
      e.stopPropagation();
      e.preventDefault();

      result = 1;
    } else if (e.ctrlKey) {
      timeline.setCursor('ew-resize');

      result = 1;
    } else {
      timeline.setCursor('');

      result = -1;
    }

    return result;
  }

  function removeScrollingCursors(e) {
    if (updateScrollingCursors(_timeline, e) <= 0) {
      document.removeEventListener('keyup', removeScrollingCursors);
      keyUpIsOn = false;
    }
  }

  return {
    addTimeline: function(timeline, e) {
      var result = updateScrollingCursors(timeline, e);

      if (result > 0) {
        add(timeline);
      } else if (result < 0) {
        _timeline = timeline;
      }
    },
    removeTimeline: function(timeline) {
      if (timeline) {
        timeline.setCursor('');

        if (timeline === _timeline) _timeline = undefined;
      }
    }
  };
})();

Timeline.prototype.destroy = function() {
  this.windowResizeListener.remove();

  scrollingCursors.removeTimeline(this);

  this.elements.forEach(function(elem) {
    if (elem.destroy) {
      elem.destroy();
    }
  });

  delete this.windowResizeListener;

  delete this.holder.idvcTimeline;
};

Timeline.prototype.getContentHeight = function() {
  return this.contentHeight;
};

Timeline.prototype.getRulerHeight = function() {
  return this.rulerHeight;
};

function getClientOffsets() {
  var holder = this.holder;

  var offsets = {
    right: 0,
    left: 0
  };

  this.elements.forEach(function(elem) {
    if (elem.getClientOffsets) {
      var newOffsets = elem.getClientOffsets(holder);
      if (newOffsets.right > offsets.right) offsets.right = newOffsets.right;
      if (newOffsets.left > offsets.left) offsets.left = newOffsets.left;
    }
  });

  return offsets;
}

function updateRightPadding(offsets) {
  var paddingWidth = this.maxPaddingWidth - offsets.right;
  if (paddingWidth < 0) paddingWidth = 0;

  if (this.legendDiv) {
    if (this.legendWidth === undefined) this.legendWidth = this.legendDiv.offsetWidth;
    paddingWidth += this.legendWidth;
  }

  this.holder.style.right = paddingWidth + 'px';

  if (this.splitterDiv) {
    if (!this._splitterDivWidth) {
      this._splitterDivWidth = this.splitterDiv.offsetWidth;
    }

    paddingWidth -= this._splitterDivWidth;
    this.splitterDiv.style.right = paddingWidth + 'px';
  }

  if (this.legendDiv) {
    this.legendDiv.style.width = paddingWidth + 'px';
  }
}

function updateSize(param, offsets) {
  offsets = offsets || getClientOffsets.call(this);

  updateRightPadding.call(this, offsets);

  if (offsets.leftDelta !== undefined) {
    param = param || {};
    param.graphWidthDelta = -offsets.leftDelta;
  }

  notifyResize.call(this, param);
}

function setClientOffsets(offsets, param) {
  var holder = this.holder;

  offsets = offsets || getClientOffsets.call(this);

  this.clientOffsets = {
    left: offsets.left,
    right: offsets.right
  };

  if (this._hideCaptions) {
    offsets.hideCaptions = true;
  }

  this.elements.forEach(function(elem) {
    if (elem.setClientOffsets) {
      elem.setClientOffsets(offsets, holder);
    }
  });

  updateSize.call(this, param, offsets);
}

Timeline.prototype.getCaptionsWidth = function() {
  var offsets = this.clientOffsets || getClientOffsets.call(this);

  return offsets.left;
};

Timeline.prototype.setCaptionsWidth = function(width) {
  var offsets = this.clientOffsets || getClientOffsets.call(this);

  setClientOffsets.call(this, {
    left: width,
    right: offsets.right
  });
};

Timeline.prototype.hideCaptions = function() {
  this._hideCaptions = true;

  var offsets = this.clientOffsets || getClientOffsets.call(this);

  setClientOffsets.call(this, {
    left: 0,
    right: offsets.right
  });
};

Timeline.prototype.onChangeClientOffsets = function() {
  if (!this.content.offsetParent) this._needSetClientOffset = true;
  else setClientOffsets.call(this);
};

Timeline.prototype.refresh = function() {
  if (!this.content.offsetParent) return;

  this.elements.forEach(function(elem) {
    if (elem.refresh) {
      elem.refresh();
    }
  });

  setClientOffsets.call(this);
  this.refreshContent();
};

Timeline.prototype.refreshContent = function() {
  this.elements.forEach(function(elem) {
    if (elem.refreshContent) {
      elem.refreshContent();
    }
  });
};

Timeline.prototype.processScroll = function(left, page, delta, param) {
  if (!param || !param.skipScrollUpdate) this.isScrollUpdated = true;

  this.elements.forEach(function(elem) {
    if (elem.processScroll) {
      elem.processScroll(left, page, delta, param);
    }
  });

  this.hideTooltip(1);
};

Timeline.prototype.processBandRequest = function(bandBuilder, reqs) {
  if (this.isScrollUpdated) {
    this.horzScroll.sendFinalScroll();
    this.isScrollUpdated = false;
  }

  if (bandBuilder &&
      bandBuilder.onProcessRequest) {
    return bandBuilder.onProcessRequest(reqs);
  }
};

function notifyResize(param) {
  if (this.resizeTimeout) {
    if (param &&
        param.graphWidthDelta !== undefined) {
      if (this.resizeParam) {
        if (!this.resizeParam.graphWidthDelta) {
          this.resizeParam.graphWidthDelta = 0;
        }

        this.resizeParam.graphWidthDelta += param.graphWidthDelta;
      } else {
        this.resizeParam = param;
      }
    }

    return;
  }

  this.resizeParam = param;

  this.resizeTimeout = setTimeout(function() {
    delete this.resizeTimeout;

    if (!this.content.offsetParent) return;

    var outParam = this.resizeParam;

    var contentHeight = this.getContentHeight();
    if (contentHeight) {
      outParam = outParam || {};
      outParam.contentHeight = contentHeight;
    }

    var rulerHeight = this.getRulerHeight();
    if (rulerHeight) {
      outParam = outParam || {};
      outParam.rulerHeight = rulerHeight;
    }

    this.elements.forEach(function(elem) {
      if (elem.processResize) {
        elem.processResize(outParam);
      }
    });

    if (this.resizeParam) delete this.resizeParam;
  }.bind(this), 0);
}

Timeline.prototype.processResize = function(param) {
  if (param && param.showHideScrollBar) {
    this.content.refreshSize();
  }

  updateSize.call(this, param);
};

Timeline.prototype.createAreaHolder = function(style, tabIndex) {
  var elem = new TimelineAreaHolder(this.content, style, tabIndex);
  this.elements.push(elem);
  elem.onChangeClientOffsets = this.onChangeClientOffsets.bind(this);

  if (this._setBottomAlign) {
    this.elements.forEach(function(elem) {
      if (elem.clearBottomAlign) {
        elem.clearBottomAlign();
      }
    });
  } else {
    this._setBottomAlign = true;
  }

  return elem;
};

Timeline.prototype.createArea = function(bandBuilder, style, tabIndex) {
  var elem = new TimelineArea(this.content, bandBuilder, style, tabIndex);
  this.elements.push(elem);
  elem.onChangeClientOffsets = this.onChangeClientOffsets.bind(this);

  if (this._setBottomAlign) elem.setBottomAlign();

  return elem;
};

Timeline.prototype.createHorzScroll = function(scrollProcessor) {
  if (!this.horzScroll) {
    var elem = new HorzScrollBar(this.holder, scrollProcessor);
    this.elements.push(elem);
    this.horzScroll = elem;
    elem.onScroll.subscribe(this, this.processScroll);
    elem.onResize.subscribe(this, this.processResize);
  }

  return this.horzScroll;
};

Timeline.prototype.createSelection = function(builder) {
  var elem = new Selection(this.content, builder);
  this.elements.push(elem);
  return elem;
};

Timeline.prototype.createRuler = function(rulerBuilder, style) {
  var elem = new Ruler(this.content, rulerBuilder, style);
  this.elements.push(elem);

  this.topLeftCorner = elem.corner;

  return elem;
};

Timeline.prototype.createLegendHolder = function(width, style) {
  this.legendDiv = document.createElement('div');
  this.legendDiv.className = 'idvc_timeline_ctrl_legend';
  applyStyle(this.legendDiv, style);

  this.legendDiv.style.width = width || '8em';

  this.splitterDiv = document.createElement('div');
  this.splitterDiv.className = 'idvc_timeline_ctrl_splitter';

  this._parent.appendChild(this.legendDiv);
  this._parent.appendChild(this.splitterDiv);

  var startLegendWidth;
  var startClientOffsets;
  Utils.createResizeProcess(this.splitterDiv, {
    accepted: function(target) {
      startLegendWidth = this.legendWidth || this.legendDiv.offsetWidth;
      startClientOffsets = getClientOffsets.call(this);
      this.splitterDiv.classList.add('idvc_timeline_ctrl_splitter_hover');
      return true;
    },
    getDelta: function(newPos, oldPos, startPos) {
      return newPos.x - startPos.x;
    },
    getCursor: function() {
      return 'ew-resize';
    },
    onProcess: function(delta) {
      var minLegendWidth = 70;
      var maxLegendWidth = this._parent.offsetWidth / 3;

      var newLegendWidth = startLegendWidth - delta;

      if (newLegendWidth < minLegendWidth) {
        newLegendWidth = minLegendWidth;
      }

      if (newLegendWidth > maxLegendWidth) {
        newLegendWidth = maxLegendWidth;
      }

      if (newLegendWidth !== this.legendWidth) {
        var oldLegendWidth = this.legendWidth;
        this.legendWidth = newLegendWidth;

        updateRightPadding.call(this, startClientOffsets);
        notifyResize.call(this, {graphWidthDelta: -(newLegendWidth - oldLegendWidth)});

        Utils.refreshSize(this.legendDiv, {width: true});
      }
    },
    onEnd: function() {
      startLegendWidth = startClientOffsets = undefined;
      this.splitterDiv.classList.remove('idvc_timeline_ctrl_splitter_hover');
      Utils.refreshSize(this.holder, {width: true});
    }
  }, this);

  return this.legendDiv;
};

function setCursor(elem, cursor, noKeepOld) {
  if (!elem) return;

  if (cursor) {
    if (!noKeepOld && !elem.idvcTimelineOldCursor) {
      let oldCursor = getCursor(elem);
      if (oldCursor) elem.idvcTimelineOldCursor = oldCursor;
    }

    elem.style.cursor = cursor;
  } else {
    if (elem.idvcTimelineOldCursor) {
      elem.style.cursor = elem.idvcTimelineOldCursor;
      delete elem.idvcTimelineOldCursor;
    } else {
      elem.style.cursor = '';
    }
  }
};

function getCursor(elem) {
  if (!elem) return '';

  return elem.style.cursor;
};

Timeline.prototype.setCursor = function(cursor) {
  setCursor(this.content, cursor, true);
};

Timeline.prototype.getCursor = function() {
  return getCursor(this.content);
};

function onFillBandDef(bandIndex, dataId, isScrolling) {
  function processFillBand() {
    var completed;
    if (this.onFillEmptyBand) {
      completed = this.onFillEmptyBand(band, bandIndex, isScrolling);
    }

    if (!completed) {
      this.updater.addRequest({
        id: dataId,
        bandIndex: bandIndex,
      });
    }
  }

  var band = this.getBand(dataId, bandIndex);

  if (!band) return;

  if (!isScrolling) {
    processFillBand.call(this);
  } else {
    setTimeout(processFillBand.bind(this), 0);
  }
}

function onRemoveBandDef(bandIndex, dataId) {
  this.updater.removeRequest(bandIndex, dataId);
}

function onUpdateBandsDef(bandObjects, param) {
  if (!bandObjects) return;

  var fromDelta = this.updater.fromDelta || 0;
  var scrollDelta = this.updater.scrollDelta || 0;

  param = param || {};

  if (this.onScrollBands) {
    this.onScrollBands(bandObjects,
      this.updater.from, this.updater.to - this.updater.from, scrollDelta, fromDelta, param);
  }

  bandObjects.forEach(function(band) {
    if (!band) return;

    this.updater.addRequest({
      id: band.getDataId(),
      bandIndex: band.getIndex(),
      fromDelta: fromDelta,
      scrollDelta: scrollDelta
    });
  }.bind(this));
}

function disableBandNotification(bandBuilder) {
  if (!bandBuilder) return;

  bandBuilder.onFillBand = undefined;
  bandBuilder.onRemoveBand = undefined;
  bandBuilder.onUpdateBands = undefined;
}

function enableBandNotification(bandBuilder) {
  if (!bandBuilder) return;

  bandBuilder.onFillBand = onFillBandDef;
  bandBuilder.onRemoveBand = onRemoveBandDef;
  bandBuilder.onUpdateBands = onUpdateBandsDef;
}

function createStdBandBuilder(baseObj, ownerTimeline) {
  var bandBuilder = Object.create(baseObj);
  bandBuilder.updater = {
    from: undefined,
    to: undefined,
    scrollDelta: undefined,
    fromDelta: undefined,
    queue: new Map(),
    waitTimeout: undefined,
    addRequest: function(req) {
      req.from = this.from;
      req.to = this.to;
      this.queue.set(req.bandIndex, req);

      Utils.consoleLog('%cAdd request', 'color: green;', req);

      this._updateTimeout();
    },
    removeRequest: function(bandIndex, dataId) {
      this.queue.delete(bandIndex);
    },
    checkRequest: function(req) {
      return (req &&
        req.from === this.from &&
        req.to === this.to);
    },
    setRange: function(from, to, delta) {
      this.fromDelta = from - (this.from ? this.from : 0);
      this.from = from;
      this.to = to;
      this.scrollDelta = delta;

      this.clearQueue();
    },
    clearQueue: function() {
      this.queue.clear();
    },
    processAllRequests: function() {
      var result;
      if (this.queue.size &&
          this.processRequest) {
        result = this.processRequest([...this.queue.values()]);
        this.clearQueue();
      }

      return result;
    },
    _updateTimeout: function() {
      if (this.waitTimeout) {
        clearTimeout(this.waitTimeout);
      }

      this.waitTimeout = setTimeout(function() {
        this.waitTimeout = undefined;
        this.processAllRequests();
      }.bind(this), _updateWaitingInterval);
    }
  };
  bandBuilder.area = undefined;
  bandBuilder.getBand = function(dataId, index) {
    return this.area ? this.area.getBand(dataId, index) : undefined;
  };
  enableBandNotification(bandBuilder);
  bandBuilder.onScroll = function(left, page, delta) {
    this.updater.setRange(left, left + page, delta);
  };
  bandBuilder.onResize = function() {
    this.updater.clearQueue();
  };
  bandBuilder.checkRequest = function(req) {
    return this.updater.checkRequest(req);
  };

  if (ownerTimeline &&
      ownerTimeline.processBandRequest) {
    bandBuilder.updater.processRequest = ownerTimeline.processBandRequest.bind(ownerTimeline, bandBuilder);
  } else if (bandBuilder.onProcessRequest) {
    bandBuilder.updater.processRequest = bandBuilder.onProcessRequest.bind(bandBuilder);
  }

  return bandBuilder;
}

function createCustomBandBuilder(baseObj, ownerTimeline) {
  var bandCount = 0;
  var bandBuilder = createStdBandBuilder(baseObj, ownerTimeline);
  bandBuilder.getBandCount = function() {
    return bandCount;
  }

  bandBuilder.insertBand = function(index) {
    if (index === undefined) index = bandCount - 1;
    else if (index > bandCount) index = bandCount;
    else index -= 1;

    bandCount++;
    if (this.area) this.area.insertBands(index, 1);
  }

  bandBuilder.removeBand = function(index) {
    if (index >= 0 &&
        index < bandCount) {
      index -= 1;
    }

    bandCount--;
    if (this.area) this.area.removeBands(index, 1);
  }

  return bandBuilder;
}

function getTimeline4Elem(elem) {
  if (!elem) return;

  var timelineDiv = Utils.getParentByClass(elem, 'idvc_timeline_ctrl');
  if (timelineDiv) return timelineDiv.idvcTimeline;

  return undefined;
}

function getArea4Elem(elem) {
  if (!elem) return;

  var areaDiv = Utils.getParentByClass(elem, 'idvc_timeline_ctrl_area');
  if (areaDiv) return areaDiv.idvcTimelineArea;

  return undefined;
}

return {
  create: function(parent, tabIndex, rightPadding) {
    return new Timeline(parent, tabIndex, rightPadding);
  },
  createStdBandBuilder: createStdBandBuilder,
  createCustomBandBuilder: createCustomBandBuilder,
  getTimelineObj: getTimeline4Elem,
  getAreaObj: getArea4Elem,
  createBandObj: function(el) {
    var result = undefined;

    var areaObj = this.getAreaObj(el);
    if (areaObj) {
      var bandDiv = Utils.getParentByClass(el, 'idvc_timeline_ctrl_band');
      if (bandDiv) {
        result = createBandObject.call(areaObj, bandDiv);
      }
    }

    return result;
  },
  getElementPosition: function(el) {
    var timelineDiv = Utils.getParentByClass(el, 'idvc_timeline_ctrl');
    if (timelineDiv && el) {
      var elPos = el.getBoundingClientRect();
      var timelinePos = timelineDiv.getBoundingClientRect();

      return {
        x: elPos.left - timelinePos.left,
        y: elPos.top - timelinePos.top
      };
    }

    return undefined;
  }
};

});

/*
 * Copyright (C) 2013 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

define('utils', [], function() {

"use strict"

var Consts = {
  browserPrefix: '-webkit-',
  engine: 'webkit',
  os: 'win'
};

(function initConsts() {
  if (navigator.userAgent.indexOf('WebKit') === -1) {
    Consts.engine = 'moz';
    Consts.browserPrefix = '-moz-';
    var versionStart = navigator.userAgent.lastIndexOf('/');
    if (versionStart !== -1) {
      var version = parseInt(navigator.userAgent.substr(versionStart + 1));
      if (version >= 54) Consts.browserPrefix = '';
    }
  }

  const platform = navigator.platform.toLowerCase();
  if (platform.indexOf('lin') >= 0) Consts.os = 'lin';
  else if (platform.indexOf('mac') >= 0) Consts.os = 'mac';
})();

function appendCSS(path) {
  var cssElt = document.createElement('link');
  if (cssElt) {
    cssElt.setAttribute('href', path);
    cssElt.setAttribute('rel', 'stylesheet');
    cssElt.setAttribute('type', 'text/css');
  }
  var head = document.getElementsByTagName('head')[0];
  head.appendChild(cssElt);
}

function appendScript(path, loaded, failed) {
  var head = document.getElementsByTagName('head')[0];
  var scriptElt = document.createElement('script');
  scriptElt.setAttribute('src', path);
  scriptElt.setAttribute('type', 'text/javascript');

  if (loaded) {
    scriptElt.onload = loaded;
  }

  if (failed) {
    scriptElt.onerror = failed;
  }

  head.appendChild(scriptElt);
}

function applyUAFont(elem) {
  var target = getDomElement(elem) || document.body;

  var button = document.createElement('button');
  button.style.position = 'absolute';
  button.style.left = '0';
  button.style.top = '0';

  target.appendChild(button);
  var UAStyle = window.getComputedStyle(button, null);
  setFont(target, UAStyle, true);
  target.removeChild(button);
}

var getScrollbarSize = (function() {
  var result;
  return function(update) {
    if (!result ||
        update) {
      var div = document.createElement('div');

      div.style.overflow = 'scroll';
      div.style.position = 'absolute';
      div.style.top = '0';
      div.style.left = '0';
      div.style.width = '100px';
      div.style.height = '100px';

      document.body.appendChild(div);

      result = {
        width: div.offsetWidth - div.clientWidth,
        height: div.offsetHeight - div.clientHeight
      };

      document.body.removeChild(div);
    }

    return result;
  }
})();

function doMath(val, prec, fn) {
  var num = Math.pow(10, prec);
  return Math[fn](val * num) / num;
}

function round(val, prec) {
  return doMath(val, prec, 'round');
}

function floor(val, prec) {
  return doMath(val, prec, 'floor');
}

function ceil(val, prec) {
  return doMath(val, prec, 'ceil');
}

function roundExp(val, prec) {
  if (val <= 10 && val >= 1 ||
      val <= -1 && val >= -10) return round(val, prec);

  return val.toExponential(prec);
}

function disableDragging(el) {
  function disabledRoutine(e) {
    e.preventDefault();
    return false;
  }

  if (!el) return;

  el.addEventListener('dragstart', disabledRoutine);
  el.addEventListener('drop', disabledRoutine);
}

function createElement(template, className, id, parent, type, customize) {
  type = type || 'div';

  var result = document.createElement(type);
  disableDragging(result);

  if (className) result.className = className;
  if (id) result.id = id;
  if (template) result.innerHTML = template;

  if (customize) customize(result);

  if (parent) parent.appendChild(result);

  return result;
}

var getElementById = (function() {
  var elements = {};

  return function(id, renew) {
    var result;
    if (renew || !elements[id]) {
      result = document.getElementById(id);
      if (result) elements[id] = result;
    } else {
      result = elements[id];
    }

    return result;
  }
})();

function getElementPos(elem) {
  if (!elem) return undefined;

  var rect = elem.getBoundingClientRect();

  var docElement = document.documentElement;
  var scrollTop = window.pageYOffset || docElement.scrollTop;
  var scrollLeft = window.pageXOffset || docElement.scrollLeft;

  var top  = rect.top + scrollTop - docElement.clientTop;
  var left = rect.left + scrollLeft - docElement.clientLeft;

  return { x: left, y: top, width: rect.width, height: rect.height };
}

function defaultCompareItems(v1, v2) {
  if (typeof v1 === 'number' ||
      typeof v1 === 'boolean') {
    return v2 - v1;
  } else if (typeof v1 === 'string') {
    return v2.localeCompare(v1);
  }

  return 0;
}

function format() {
  var format_args = arguments;
  var str = arguments[0];
  if (!str) return;

  str = str.replace(/\{\d+\}/g, function(val) {
    return format_args[parseInt(val.slice(1, -1), 10)];
  });
  return str;
}

function curryThis(func, thisArg) {
  var args = [].slice.call(arguments, 2);
  return function () {
    return func.apply(thisArg, args.concat([].slice.call(arguments)));
  };
}

function curry(func) {
  var args = [].slice.call(arguments, 1);
  args.unshift(undefined);
  args.unshift(func);
  return curryThis.apply(undefined, args);
}

function composeThis(thisArg) {
  var args = [].slice.call(arguments, 1);
  return function() {
    var callArgs = [].slice.call(arguments);
    var startCall = args.shift();
    if (startCall) {
      var result = startCall.apply(thisArg, callArgs);
      args.forEach(function(func) {
        result = func.call(thisArg, result);
      });
      return result;
    }
  };
}

function compose() {
  var args = [].slice.call(arguments);
  args.unshift(undefined);
  return composeThis.apply(undefined, args);
}

function getDomElement(val) {
  if (typeof val === 'string') {
    return document.getElementById(val);
  }
  return val;
}

var stopRefreshSizeAttr = 'data-stopRefreshSize';

function refreshSize(elem, size) {
  if (!elem) return;

  var stopped;

  if (elem.hasAttribute(stopRefreshSizeAttr)) {
    stopped = true;
  } else if (elem.refreshSize) {
    stopped = elem.refreshSize(size);
  }

  if (!stopped && elem.children) {
    for (var i = 0, len = elem.children.length; i < len; i++) {
      refreshSize(elem.children[i], size);
    }
  }
}

function createResizeProcess(parentEl, traits, thisObj, mouseButton) {
  thisObj = thisObj || traits;
  if (mouseButton === undefined) mouseButton = 0;

  var currentDrag = {};
  var parent = getDomElement(parentEl);

  var dblClickHandle;

  parent.addEventListener('mousedown', function(e) {
    if (e.button !== mouseButton) return;

    if (traits.accepted.call(thisObj, e.target)) {
      currentDrag.isStarted = false;
      currentDrag.startMousePos = {x: e.pageX, y: e.pageY};
      currentDrag.mousePos = {x: e.pageX, y: e.pageY};

      currentDrag.globalCursor = document.createElement('div');
      currentDrag.globalCursor.className = 'idvcgrid_global_cursor';
      currentDrag.globalCursor.style.cursor = traits.getCursor();

      document.body.appendChild(currentDrag.globalCursor);

      window.addEventListener('mousemove', doResizing, true);
      window.addEventListener('mouseup', stopResizing, true);

      e.preventDefault();
      e.stopPropagation();
      if (traits.onDblClick &&
          !dblClickHandle) {
        dblClickHandle = {
          count: 0,
          timeout: setTimeout(function() {
            dblClickHandle = undefined;
          }, 800)
        };
      }
    }
  }, false);

  function doResizing(e) {
    var deltaThreshold = 5;
    if (traits.getDeltaThreshold) deltaThreshold = traits.getDeltaThreshold();

    var curMousePos = {x: e.pageX, y: e.pageY};
    var delta = traits.getDelta.call(thisObj, curMousePos,
      currentDrag.mousePos, currentDrag.startMousePos);
    if (!currentDrag.isStarted) {
      if (Math.abs(delta) > deltaThreshold) {
        currentDrag.isStarted = true;
      }
    } else {
      e.preventDefault();

      traits.onProcess.call(thisObj, delta);
      currentDrag.mousePos = curMousePos;
    }
  }

  function stopResizing(e) {
    if (currentDrag.globalCursor) {
      document.body.removeChild(currentDrag.globalCursor);
      currentDrag.globalCursor = null;
    }

    currentDrag = {};

    window.removeEventListener('mousemove', doResizing, true);
    window.removeEventListener('mouseup', stopResizing, true);

    if (dblClickHandle) {
      dblClickHandle.count++;
      if (dblClickHandle.count > 1) {
        traits.onDblClick.call(thisObj);

        clearTimeout(dblClickHandle.timeout);
        dblClickHandle = undefined;
      }
    }

    traits.onEnd.call(thisObj);

    if (Consts.engine === 'moz') {
      dispatchMouseEvent({x: e.pageX, y: e.pageY}, 'mousemove');
    }
  }
}

function createSplitter(parent, minSize, defSplitterSize, traits) {
  defSplitterSize = defSplitterSize || 0.3;

  function getDiv(prop, className) {
    var result;
    var div = parent[prop];
    if (div) {
      result = getDomElement(div);
      result.classList.add(className);
      parentEl = result.offsetParent || parentEl;
    }

    return result;
  }

  function createDiv(className) {
    var div = document.createElement('div');
    div.className = className;
    div.idvcCreatedBySplitter = true;
    parentEl.appendChild(div);

    return div;
  }

  if (!parent) return;

  var parentEl;
  var primaryDiv = getDiv('primaryDiv', traits.primaryDivClass);
  var splitterDiv = getDiv('splitterDiv', traits.splitterDivClass);
  var secondaryDiv = getDiv('secondaryDiv', traits.secondaryDivClass);

  var isPercentSize = false;

  if (!parentEl) parentEl = getDomElement(parent);

  if (!parentEl) return;

  if (!primaryDiv) primaryDiv = createDiv(traits.primaryDivClass);
  if (!splitterDiv) splitterDiv = createDiv(traits.splitterDivClass);
  if (!secondaryDiv) secondaryDiv = createDiv(traits.secondaryDivClass);

  splitterDiv.onselectstart = function() {
    return false;
  };

  function refreshSplitterSize(isUserAction) {
    if (!isUserAction && traits.refreshSizeByUserAction) return;

    var size = {};
    size[traits.size] = true;

    refreshSize(primaryDiv, size);
    refreshSize(secondaryDiv, size);

    if (splitterObject && splitterObject.onRefreshSize) {
      splitterObject.onRefreshSize();
    }
  }

  function clearSizes() {
    function clear(div) {
      div.style[traits.size] = '';
      div.style[traits.pos] = '';
      div.style[traits.posBackward] = '';
    }

    clear(primaryDiv);
    clear(splitterDiv);
    clear(secondaryDiv);
  }

  function setPercentSize(size, splitterSize, isUserAction) {
    clearSizes();

    if (size === '100%') {
      size = Consts.browserPrefix + 'calc(100% - ' +
        splitterSize + 'px)';
      primaryDiv.style[traits.size] = size;
      splitterDiv.style[traits.pos] = size;
      secondaryDiv.style[traits.pos] = '100%';
    } else if (size === '0%') {
      size = '0';
      primaryDiv.style[traits.size] = size;
      splitterDiv.style[traits.pos] = size;
      secondaryDiv.style[traits.pos] = splitterSize + 'px';
    } else {
      primaryDiv.style[traits.size] = size;
      splitterDiv.style[traits.pos] = size;
      var secondaryPosStr = Consts.browserPrefix + 'calc(' + size + ' + ' +
        splitterSize + 'px)';
      secondaryDiv.style[traits.pos] = secondaryPosStr;
    }

    refreshSplitterSize(isUserAction);
  }

  function setPixelSize(size, splitterSize, isUserAction) {
    function primaryProp() {
      return !traits.backward ? traits.size : traits.posBackward;
    }

    function splitterProp() {
      return !traits.backward ? traits.pos : traits.posBackward;
    }

    function secondaryProp() {
      return !traits.backward ? traits.pos : traits.size;
    }

    function primarySize() {
      return !traits.backward ? size : size + splitterSize;
    }

    function secondarySize() {
      return !traits.backward ? size + splitterSize : size;
    }

    clearSizes();

    primaryDiv.style[primaryProp()] = primarySize() + 'px';
    splitterDiv.style[splitterProp()] = size + 'px';
    secondaryDiv.style[secondaryProp()] = secondarySize() + 'px';

    refreshSplitterSize(isUserAction);
  }

  var setPrimaryDivSize = function(primarySize, wholeSize, isUserAction) {
    var splitterSize = getSplitterSize();

    if (traits.backward) primarySize = wholeSize - primarySize;

    if (isPercentSize) {
      let primarySizeStr = (primarySize * 100 / (wholeSize + splitterSize)) + '%';
      setPercentSize(primarySizeStr, splitterSize, isUserAction);
    } else {
      setPixelSize(primarySize, splitterSize, isUserAction);
    }
  };

  var currentDrag = {
    pos: -1,
    dragged: false,
    primarySize: 0,
    secondarySize: 0
  };

  var globalCursor = null;

  var updateSplitter = function(e) {
    if (primarySize4Folded) return;

    if (currentDrag.dragged) {
      minSize = minSize || 100;

      let eventPos = e[traits.eventPos];

      let { primarySize, secondarySize } = currentDrag;
      const wholeSize = primarySize + secondarySize;

      let delta = eventPos - currentDrag.pos;
      if (primarySize + delta < minSize) {
        delta = minSize - primarySize;
        eventPos = delta + currentDrag.pos;
      } else if (secondarySize - delta < minSize) {
        delta = secondarySize - minSize;
        eventPos = delta + currentDrag.pos;
      }

      primarySize += delta;
      setPrimaryDivSize(primarySize, wholeSize, true);

      currentDrag.pos = eventPos;
      currentDrag.primarySize = primarySize;
      currentDrag.secondarySize = wholeSize - primarySize;
    }
  };

  var stopUpdateSplitter = function() {
    document.body.removeChild(globalCursor);
    globalCursor = null;

    currentDrag.pos = -1;
    currentDrag.dragged = false;
    currentDrag.primarySize = 0;
    currentDrag.secondarySize = 0;

    splitterDiv.classList.remove('idvcgrid_splitter_hover');

    window.removeEventListener('mousemove', updateSplitter, true);
    window.removeEventListener('mouseup', stopUpdateSplitter, true);

    if (splitterObject && splitterObject.onStopUpdateSplitter) {
      splitterObject.onStopUpdateSplitter();
    }
  };

  splitterDiv.onmousedown = function(e) {
    e = e || event;

    if (primarySize4Folded) {
      splitterObject.unfold(true);
      return;
    }

    if (!e.ctrlKey && !e.shiftKey &&
        e.target === splitterDiv) {
      e.preventDefault();

      globalCursor = document.createElement('div');
      globalCursor.className = 'idvcgrid_global_cursor';
      var splitterStyle = window.getComputedStyle(splitterDiv, null);
      globalCursor.style.cursor = splitterStyle.cursor;

      document.body.appendChild(globalCursor);

      splitterDiv.classList.add('idvcgrid_splitter_hover');

      currentDrag.pos = e[traits.eventPos];
      currentDrag.dragged = true;

      currentDrag.primarySize = primaryDiv[traits.offsetSize];
      currentDrag.secondarySize = secondaryDiv[traits.offsetSize];

      window.addEventListener('mousemove', updateSplitter, true);
      window.addEventListener('mouseup', stopUpdateSplitter, true);

      if (splitterObject && splitterObject.onStartUpdateSplitter) {
        splitterObject.onStartUpdateSplitter();
      }
    }
  };

  function beforeFold() {
    if (primarySize4Folded) return;

    primarySize4Folded = primaryDiv.style[traits.size];
    splitterDiv.classList.add('idvcsplitter_folded');

    if (currentDrag.dragged) {
      stopUpdateSplitter();
    }
  }

  var splitterMinVisibleSize = em2px(defSplitterSize);

  function getSplitterSize() {
    function isSplitterHidden() {
      return splitterDiv && splitterDiv.style.display === 'none';
    }

    var splitterSize = splitterDiv[traits.offsetSize];
    if (splitterSize < 0) {
      splitterSize = 0;
    }

    if (!isSplitterHidden() &&
        !splitterSize) {
      // splitter parent is hidden
      splitterSize = splitterMinVisibleSize;
    }

    return splitterSize;
  }

  var primarySize4Folded;

  var splitterObject = {
    primaryDiv: primaryDiv,
    splitterDiv: splitterDiv,
    secondaryDiv: secondaryDiv,
    setSize: function(size, userAction) {
      if (primarySize4Folded) {
        primarySize4Folded = size;
        return;
      }

      var splitterSize = getSplitterSize();

      isPercentSize = size[size.length - 1] === '%';

      if (isPercentSize) {
        setPercentSize(size, splitterSize, userAction);
      } else {
        setPixelSize(parseInt(size, 10), splitterSize, userAction);
      }
    },
    getSize: function() {
      if (isPercentSize || !traits.backward) {
        return primaryDiv.style[traits.size];
      }

      return secondaryDiv.style[traits.size];
    },
    foldPrimary: function(hideSplitter, userAction) {
      beforeFold();
      primaryDiv.style.display = 'none';
      primaryDiv.setAttribute(stopRefreshSizeAttr, '1');
      secondaryDiv.style.display = '';
      secondaryDiv.removeAttribute(stopRefreshSizeAttr);
      if (hideSplitter) splitterDiv.style.display = 'none';
      setPercentSize('0%', getSplitterSize(), userAction);

      if (this.afterFoldPrimary) {
        this.afterFoldPrimary(userAction);
      }
    },
    isPrimaryFolded: function() {
      return primaryDiv.style.display === 'none';
    },
    foldSecondary: function(hideSplitter, userAction) {
      beforeFold();
      secondaryDiv.style.display = 'none';
      secondaryDiv.setAttribute(stopRefreshSizeAttr, '1');
      primaryDiv.style.display = '';
      primaryDiv.removeAttribute(stopRefreshSizeAttr);
      if (hideSplitter) splitterDiv.style.display = 'none';
      else splitterDiv.style.display = 'block';
      setPercentSize('100%', getSplitterSize(), userAction);

      if (this.afterFoldSecondary) {
        this.afterFoldSecondary(userAction);
      }
    },
    isSecondaryFolded: function() {
      return secondaryDiv.style.display === 'none';
    },
    isSplitterHidden: function() {
      return splitterDiv.style.display === 'none';
    },
    unfold: function(userAction) {
      if (!primarySize4Folded) return false;

      primaryDiv.style.display = '';
      primaryDiv.removeAttribute(stopRefreshSizeAttr);
      secondaryDiv.style.display = '';
      secondaryDiv.removeAttribute(stopRefreshSizeAttr);
      splitterDiv.style.display = 'block';
      var size = primarySize4Folded;
      primarySize4Folded = undefined;
      this.setSize(size, userAction);

      splitterDiv.classList.remove('idvcsplitter_folded');

      if (this.afterUnfold) {
        this.afterUnfold(userAction);
      }

      return true;
    },
    hideSecondary: function() {
      secondaryDiv.style.display = 'none';
      secondaryDiv.setAttribute(stopRefreshSizeAttr, '1');
      splitterDiv.style.display = 'none';
    },
    showSecondary: function() {
      secondaryDiv.style.display = '';
      secondaryDiv.removeAttribute(stopRefreshSizeAttr);
      splitterDiv.style.display = 'block';

      this.setSize(primaryDiv.style[traits.size]);
    },
    addSash: function(isPrimary, className) {
      className = className || 'idvcsplitter_sash';

      var sash = document.createElement('div');
      sash.className = className;

      sash.onmousedown = function(e) {
        if (!primarySize4Folded) {
          if (isPrimary) this.foldPrimary(false, true);
          else this.foldSecondary(false, true);
        } else {
          this.unfold(true);
        }

        e.stopPropagation();
        e.preventDefault();
      }.bind(this);

      splitterDiv.appendChild(sash);
      this.sashDiv = sash;

      return sash;
    },
    removeSash: function() {
      removeAllChildren(splitterDiv);
      delete this.sashDiv;
    },
    remove: function() {
      primaryDiv.classList.remove(traits.primaryDivClass);
      secondaryDiv.classList.remove(traits.secondaryDivClass);
      splitterDiv.classList.remove(traits.splitterDivClass);

      clearSizes();

      checkRemoveDiv(primaryDiv);
      checkRemoveDiv(secondaryDiv);
      checkRemoveDiv(splitterDiv);

      function checkRemoveDiv(elem) {
        if (elem.idvcCreatedBySplitter) elem.parentNode.removeChild(elem);
      }
    }
  };

  splitterObject.setSize('50%');

  return splitterObject;
}

function createVertSplitter(parent, minSize, params) {
  params = params || {};

  var traits = {
    primaryDivClass: params.primaryDivClass || 'idvcsplitter_vert_primary',
    splitterDivClass: params.splitterDivClass || 'idvcsplitter_vert_splitter',
    secondaryDivClass: params.secondaryDivClass || 'idvcsplitter_vert_secondary',
    size: 'height',
    pos: 'top',
    posBackward: 'bottom',
    offsetSize: 'offsetHeight',
    eventPos: 'pageY',
    backward: params.backward,
    refreshSizeByUserAction: params.refreshSizeByUserAction
  };

  return createSplitter(parent, minSize, params.splitterSize, traits);
}

function createHorzSplitter(parent, minSize, params) {
  params = params || {};

  var traits = {
    primaryDivClass: params.primaryDivClass || 'idvcsplitter_horz_primary',
    splitterDivClass: params.splitterDivClass || 'idvcsplitter_horz_splitter',
    secondaryDivClass: params.secondaryDivClass || 'idvcsplitter_horz_secondary',
    size: 'width',
    pos: 'left',
    posBackward:  'right',
    offsetSize: 'offsetWidth',
    eventPos: 'pageX',
    backward: params.backward,
    refreshSizeByUserAction: params.refreshSizeByUserAction
  };

  return createSplitter(parent, minSize, params.splitterSize, traits);
}

function processTooltip(el, tooltipObj, delay) {
  if (!tooltipObj) return;

  var mainTooltipElement = getDomElement(el);
  if (!mainTooltipElement) return;

  var tooltipInfo = {tooltip: null, timer: null};
  var lastMouseEvent;
  var hideTimeout;
  var ignoreMouseOut;

  var hideNextTooltip;
  var clearHideNextTooltip;

  var hideTooltip = function(hideInterval) {
    if (hideTimeout) {
      clearTimeout(hideTimeout);
      hideTimeout = undefined;
    }

    if (tooltipInfo.tooltip) {
      document.body.removeChild(tooltipInfo.tooltip);
    }

    if (hideInterval) {
      hideNextTooltip = true;
      if (!clearHideNextTooltip) {
        clearHideNextTooltip = createAsyncCall(function() {
          hideNextTooltip = undefined;
        }, window, hideInterval);
      }

      clearHideNextTooltip.call();
    }

    if (tooltipInfo.timer) {
      window.clearTimeout(tooltipInfo.timer);
      tooltipInfo.timer = null;
    }

    tooltipInfo.tooltip = null;
    clearTooltipElement();
  };

  var hideTooltipByMouseOut = function() {
    if (!ignoreMouseOut)
      hideTooltip();
    tooltipObj.getInfo(undefined, 0, 0);
  };

  var showTooltip = function(text, x, y, style) {
    if (hideNextTooltip) return;

    tooltipInfo.timer = undefined;

    var tooltip = tooltipObj.create(this, text);
    if (!tooltip) return;

    applyObjectProperties(tooltip.style, style);

    setTooltipPos(tooltip, x, y);
    tooltipInfo.tooltip = tooltip;
  };

  function setTooltipElement(element) {
    if (tooltipInfo.elem &&
        tooltipInfo.elem !== element &&
        tooltipInfo.elem !== mainTooltipElement) {
      clearTooltipElement();
    }

    if (element &&
        element !== tooltipInfo.elem &&
        element !== mainTooltipElement) {
      element.addEventListener('mouseout', hideTooltipByMouseOut, false);
    }

    tooltipInfo.elem = element;
  }

  function clearTooltipElement() {
    if (tooltipInfo.elem &&
        tooltipInfo.elem !== mainTooltipElement) {
      tooltipInfo.elem.removeEventListener('mouseout', hideTooltipByMouseOut, false);
      tooltipInfo.elem = null;
    }
  }

  function setAutoHide(autoHideDelay) {
    if (hideTimeout) clearTimeout(hideTimeout);

    hideTimeout = undefined;
    if (autoHideDelay > 0) {
      hideTimeout = setTimeout(function() {
        hideTimeout = undefined;
        hideTooltip();
      }, autoHideDelay);
    }
  }

  var setTooltipPos = function(tooltip, tooltipX, tooltipY) {
    if (tooltipX === undefined) tooltipX = lastMouseEvent.pageX;
    if (tooltipY === undefined) tooltipY = lastMouseEvent.pageY;

    if (!tooltipObj.simpleMode) {
      tooltip.classList.remove('idvcgrid_tooltip_ellipsis');
      tooltip.style.height = '';
    }

    if (tooltip.offsetWidth > 0.8 * window.innerWidth) {
      tooltip.style.height = 'auto';
      tooltip.style.width = Math.floor(0.8 * window.innerWidth) + 'px';
    }

    var cs = window.getComputedStyle(tooltip, null);

    var marginLeft = parseInt(cs.getPropertyValue('margin-left'), 10);
    var rightOffset = 6;
    var tooltipLeft = tooltipX;
    if (tooltipLeft < 0) tooltipLeft = 0;
    else if (tooltipLeft + marginLeft + tooltip.offsetWidth + rightOffset >
        window.innerWidth + window.scrollX) {
      tooltipLeft = window.innerWidth +
        window.scrollX - tooltip.offsetWidth - marginLeft - rightOffset;
    }

    var marginTop = parseInt(cs.getPropertyValue('margin-top'), 10);
    var bottomOffset = 2;
    var tooltipTop = tooltipY;
    if (tooltipTop < 0) tooltipTop = 0;
    if (tooltipTop + tooltip.offsetHeight + marginTop + bottomOffset >
        window.innerHeight + window.scrollY) {
      tooltipTop = window.innerHeight +
        window.scrollY - tooltip.offsetHeight - marginTop - bottomOffset;

      if (tooltipTop < 1) {
        tooltipTop = 1;
        var newHeight = window.innerHeight - tooltipTop -
          2 * (parseInt(cs.getPropertyValue('padding-top'), 10) + bottomOffset);
        tooltip.style.height = newHeight + 'px';
        if (!tooltipObj.simpleMode) tooltip.classList.add('idvcgrid_tooltip_ellipsis');

        if (tooltip.style.lineHeight) {
          var lineHeight = parseInt(tooltip.style.lineHeight);
          if (lineHeight > newHeight) {
            tooltip.style.lineHeight = tooltip.style.height;
          }
        }
      }
    }

    if (!tooltipObj.ignoreOverlapping &&
        tooltipLeft < tooltipX - 0.5 * marginLeft &&
        tooltipTop < tooltipY - 0.5 * marginTop) {
      var altTooltipLeft = tooltipX - rightOffset - 1.5 * marginLeft - tooltip.offsetWidth;
      if (altTooltipLeft >= 0) {
        tooltipLeft = altTooltipLeft;
      } else {
        var altTooltipTop = tooltipY - bottomOffset - 1.5 * marginTop - tooltip.offsetHeight;
        if (altTooltipTop >= 0) {
          tooltipTop = altTooltipTop;
        }
      }
    }

    tooltip.style.left = tooltipLeft + 'px';
    tooltip.style.top = tooltipTop + 'px';
  };

  function processTooltip(e) {
    var needUpdateTooltip = tooltipObj.needUpdate ||
                            function(curEl, newEl) { return curEl !== newEl;};

    e = e || event;

    lastMouseEvent = e;

    if (!needUpdateTooltip(tooltipInfo.elem, e.target, e.pageX, e.pageY)) {
      return;
    }

    var info = tooltipObj.getInfo(e.target, e.pageX, e.pageY);
    if (info) {
      if (info.keepCurrent && tooltipInfo.tooltip) {
        if (info.tracking) {
          if (info.text) {
            tooltipInfo.tooltip.innerHTML = info.text;
          }
          setTooltipPos(tooltipInfo.tooltip, e.pageX, e.pageY);
        }
      } else {
        hideTooltip();

        if (info.text) {
          var tooltipDelay = info.delay !== undefined ?
                            info.delay :
                            delay;

          setTooltipElement(info.target);
          tooltipInfo.timer = window.setTimeout(
            showTooltip.bind(info.target, info.text, info.x, info.y, info.style),
            tooltipDelay);
        }
      }

      if (tooltipInfo.tooltip) {
        applyObjectProperties(tooltipInfo.tooltip.style, info.style);
      }

      if (!hideTimeout) setAutoHide(info.autoHideDelay);
      ignoreMouseOut = info.ignoreMouseOut;
    } else if (tooltipInfo.tooltip) {
      hideTooltip();
    }
  }

  mainTooltipElement.addEventListener('mouseout', hideTooltipByMouseOut, false);
  mainTooltipElement.addEventListener('mousemove', processTooltip, false);

  return {
    show: function(el, text, x, y, autoHideDelay, style) {
      setTooltipElement(el);
      if (tooltipInfo.tooltip) {
        tooltipInfo.tooltip.innerHTML = text;
        applyObjectProperties(tooltipInfo.tooltip.style, style);
        setTooltipPos(tooltipInfo.tooltip, x, y);
      } else {
        showTooltip(text, x, y, style);
      }

      setAutoHide(autoHideDelay);
    },
    hide: function(hideInterval) {
      hideTooltip(hideInterval);
    },
    update: function() {
      if (tooltipInfo.tooltip) {
        setTimeout(processTooltip.bind(this, lastMouseEvent), 0);
      }
    },
    getTooltipDOMElement: function() {
      if (tooltipInfo) return tooltipInfo.tooltip;

      return undefined;
    },
    getParentDOMElement: function() {
      if (tooltipInfo) return tooltipInfo.elem;

      return undefined;
    }
  };
}

function createTooltipDiv(text, style) {
  if (!text || !text.length) {
    return undefined;
  }

  var tooltip = document.createElement('div');
  tooltip.className = 'idvcgrid_popup idvcgrid_tooltip';
  if (style) tooltip.classList.add(style);
  tooltip.innerHTML = text;

  document.body.appendChild(tooltip);

  return tooltip;
}

function addTooltip(el, tooltipText, needUpdateTooltip, style) {
  return processTooltip(el, {
    needUpdate: function(curElem, newElem, x, y) {
      needUpdateTooltip = needUpdateTooltip ||
                          function(curEl, newEl) { return curEl !== newEl;};

      return needUpdateTooltip(curElem, newElem);
    },
    getInfo: function(el, x, y) {
      var result;
      var text;
      if (typeof tooltipText === 'function') {
        text = tooltipText(el, x, y);
      } else {
        text = tooltipText;
      }

      if (text) {
        if (typeof text === 'object') {
          result = {};
          applyObjectProperties(result, text);
        } else {
          result = {
            text: text,
            target: el
          };
        }
      }

      return result;
    },
    create: function(elem, text) {
      var tooltip = createTooltipDiv(text, style);
      var cs = window.getComputedStyle(document.body, null);
      setFont(tooltip, cs);

      return tooltip;
    }
  }, 250);
}

function copyAppearance(tooltip, elemStyle, background) {
  function convertWhiteSpace(whiteSpace) {
    if (whiteSpace === 'pre') {
      return 'pre-wrap';
    } else if (whiteSpace === 'nowrap') {
      return 'normal';
    }

    return whiteSpace;
  }

  if (!tooltip || !elemStyle) return;

  setFont(tooltip, elemStyle);

  tooltip.style.paddingLeft = elemStyle.paddingLeft;
  tooltip.style.paddingTop = elemStyle.paddingTop;
  tooltip.style.paddingRight = elemStyle.paddingRight;
  tooltip.style.paddingBottom = elemStyle.paddingBottom;

  tooltip.style.borderTopStyle = elemStyle.borderTopStyle;
  tooltip.style.borderTopWidth = elemStyle.borderTopWidth;
  tooltip.style.borderRightStyle = elemStyle.borderRightStyle;
  tooltip.style.borderRightWidth = elemStyle.borderRightWidth;
  tooltip.style.borderBottomStyle = elemStyle.borderBottomStyle;
  tooltip.style.borderBottomWidth = elemStyle.borderBottomWidth;
  tooltip.style.borderLeftStyle = elemStyle.borderLeftStyle;
  tooltip.style.borderLeftWidth = elemStyle.borderLeftWidth;

  tooltip.style.color = elemStyle.color;
  tooltip.style.borderColor = background || elemStyle.backgroundColor;
  tooltip.style.backgroundColor = background || elemStyle.backgroundColor;
  tooltip.style.backgroundImage = elemStyle.backgroundImage;
  tooltip.style.backgroundRepeat = elemStyle.backgroundRepeat;
  tooltip.style.backgroundPosition = elemStyle.backgroundPosition;
  tooltip.style.backgroundOrigin = elemStyle.backgroundOrigin;

  tooltip.style.whiteSpace = convertWhiteSpace(elemStyle.whiteSpace);
}

function copyContent(elem, tooltip, background, process, isVerical) {
  if (!elem || !tooltip) return;

  process = process || function(from, to) {
    to.innerHTML = from.innerHTML;
  };

  process(elem, tooltip);
  var elemStyle = window.getComputedStyle(elem, null);

  tooltip.style.boxSizing = elemStyle.boxSizing;
  tooltip.style.lineHeight = elemStyle.lineHeight;

  if (!isVerical) tooltip.style.height = elemStyle.height;
  else tooltip.style.width = elemStyle.width;

  copyAppearance(tooltip, elemStyle, background);
}

function addExpansion(el, processTarget, background, expandVerical) {
  processTarget = processTarget || function(el) {return el;}

  var sizeChecker;
  if (!expandVerical) {
    sizeChecker = {
      check: function(elem, tooltip) {
        if (!elem || !tooltip) return false;

        var overPos = getElementPos(elem);
        var overWidth = elem.offsetWidth;
        if (overPos.x + overWidth > window.innerWidth + window.scrollX) {
          overWidth = window.innerWidth + window.scrollX - overPos.x;
        }

        return tooltip.offsetWidth - 1 > overWidth;
      }
    };
  } else {
    sizeChecker = {
      check: function(elem, tooltip) {
        if (!elem || !tooltip) return false;

        var overPos = getElementPos(elem);
        var overHeight = elem.offsetHeight;
        if (overPos.y + overHeight > window.innerHeight + window.scrollY) {
          overHeight = window.innerHeight + window.scrollY - overPos.y;
        }

        return tooltip.offsetHeight - 1 > overHeight;
      }
    }
  }

  var tooltipProcessor = processTooltip(el, {
    ignoreOverlapping: true,
    simpleMode: true,
    needUpdate: function(curElem, newElem) {
      return curElem !== newElem;
    },
    getInfo: function(elem, x, y) {
      elem = processTarget(elem, x, y);

      if (elem) {
        var elemPos = getElementPos(elem);

        return {
            x: elemPos.x,
            y: elemPos.y,
            target: elem,
            text: 'exp'
          };
      }

      return undefined;
    },
    create: function(elem) {
      var tooltipText = elem.innerHTML;
      if (!tooltipText.length) {
        return undefined;
      }

      var tooltip = document.createElement('div');
      tooltip.className = 'idvcgrid_popup';

      copyContent(elem, tooltip, background, result.processContent, expandVerical);

      document.body.appendChild(tooltip);

      if (result.afterCreateTooltip) result.afterCreateTooltip(tooltip);

      if (sizeChecker.check(elem, tooltip)) {
        return tooltip;
      } else {
        document.body.removeChild(tooltip);
        return undefined;
      }
    }
  }, 50);

  var result = {
    update: function() {
      copyContent(tooltipProcessor.getParentDOMElement(), tooltipProcessor.getTooltipDOMElement());
    },
    hide: function() {
      tooltipProcessor.hide();
    }
  };

  return result;
}


function buildHierarchy(parentEl, hierarchGen, processHTML, beforeExpand, afterExpand) {
  var HierarchyContainerClass = 'idvc_hierarchy_container';
  var HierarchyCaptionClass = 'idvc_hierarchy_caption';
  var HierarchyLeafClass = 'idvc_hierarchy_leaf';
  var HierarchyBodyClass = 'idvc_hierarchy_body';
  var HierarchyExpanded = 'idvc_hierarchy_expanded';

  function buildHierarchyLevel(parent, parentObj, processHTML) {
    function addHierarchyHTML(parent, hasChildren) {
      var expandItem = document.createElement('div');

      var body, container, root;

      if (hasChildren) {
        expandItem.className = HierarchyCaptionClass;

        container = document.createElement('div');
        container.className = HierarchyContainerClass;
        container.appendChild(expandItem);

        body = document.createElement('div');
        body.className = HierarchyBodyClass;
        container.appendChild(body);

        root = container;
      } else {
        expandItem.className = HierarchyLeafClass;

        root = expandItem;
      }

      parent.appendChild(root);

      return { expand: expandItem, body: body, container: container };
    }

    if (!parent || !parentObj) return;

    var objs = parentObj.children;
    if (!Array.isArray(objs)) return;

    objs.forEach(function(obj) {
      var html = addHierarchyHTML(parent, obj.hasChildren || obj.children && obj.children.length);
      var ignoreDefault = processHTML(html, obj, parentObj);

      var expand = html.expand;

      if (!ignoreDefault) {
        if (obj.expanded) expand.classList.add(HierarchyExpanded);
        if (obj.className) expand.classList.add(obj.className);
      }

      if (obj.expanded) buildHierarchyLevel(html.body, getObj(obj), processHTML);
      else expand.hierarchicalObj = obj;
    });
  }

  function addHierarchyEventsProcessing(parent, beforeExpand, afterExpand) {
    beforeExpand = beforeExpand || function(){ return false; };
    afterExpand = afterExpand || function(){};
    parent = parent || document;

    parent.addEventListener('click', function(e) {
      e = e || window.event;
      var element = e.target;
      while (element &&
              element.classList &&
              !element.classList.contains(HierarchyCaptionClass)) {
        element = element.parentNode;
      }

      if (element &&
          element.classList &&
          element.classList.contains(HierarchyCaptionClass) &&
          !beforeExpand(element, element.classList.contains(HierarchyExpanded), e.target)) {
        if (element.hierarchicalObj) {
          buildHierarchyLevel(element.nextElementSibling, getObj(element.hierarchicalObj), processHTML);
          delete element.hierarchicalObj;
        }
        element.classList.toggle(HierarchyExpanded);
        afterExpand(element);
      }
    });
  }

  var parent = getDomElement(parentEl);

  processHTML = processHTML || function(html, obj){
    if (!html || !html.expand || !obj) return;

    var expand = html.expand;

    if (obj.html || obj.displayName) expand.innerHTML = obj.html || obj.displayName;
  };

  function getObj(obj) {
    var result = obj;

    if (typeof hierarchGen === 'function') {
      result = hierarchGen(obj !== hierarchGen ? obj : undefined);
    }

    return result;
  }

  var obj = getObj(hierarchGen);
  if (Array.isArray(obj)) {
    obj = {
      children: obj
    };
  }

  buildHierarchyLevel(parent, obj, processHTML);
  addHierarchyEventsProcessing(parent, beforeExpand, afterExpand);

  return parent;
}

function getTextWidth(elem, text) {
  var testDiv = document.createElement('div');
  testDiv.className = 'idvcgrid_popup';

  copyContent(elem, testDiv);

  document.body.appendChild(testDiv);

  var result = testDiv.offsetWidth;

  document.body.removeChild(testDiv);

  return result;
}

function disableElement(elem, text, needDisable, reuseDisabler) {
  needDisable = needDisable || function() {return true;}
  reuseDisabler = reuseDisabler || false;

  if (!needDisable()) {
    return {
      disabler: disabler,
      disabled: disabled,
      end: function() {}
    };
  }

  var disabler = null;
  var disabled = getDomElement(elem);
  if (disabled) {
    if (reuseDisabler) {
      disabler = disabled.getElementsByClassName('idvcgrid_disabler')[0];
    }

    if (!disabler) {
      disabler = document.createElement('div');
      disabler.className = 'idvcgrid_disabler';
      disabled.appendChild(disabler);
    }

    if (text) {
      disabler.innerText = text;
    }

    disabler.getBoundingClientRect(); //recalculate layout
  }

  return {
    disabler: disabler,
    disabled: disabled,
    end: function() {
      if (disabler.parentNode) {
        disabler.parentNode.removeChild(disabler);
      }
    }
  };
}

function makeCaption(str1, str2) {
  var result = '<b>' + str1 + '</b>';
  if (str2) {
    result += ' (' + str2 + ')';
  }
  return result;
}

function setFont(elem, styleSet, tuneSize) {
  if (elem && styleSet) {
    if (tuneSize) {
      // Round font size to the nearest smaller even value
      var size = parseInt(styleSet.fontSize);
      size = Math.floor(size / 2) * 2;
      elem.style.fontSize = size + 'px';
    } else {
      elem.style.fontFamily = styleSet.fontFamily;
      elem.style.fontSize = styleSet.fontSize;
    }

    elem.style.fontStyle = styleSet.fontStyle;
    elem.style.fontWeight = styleSet.fontWeight;
    elem.style.fontVariant = styleSet.fontVariant;
  }
}

function changeCursor(cursor) {
  var parent = document.createElement('div');
  parent.style.overflow = 'hidden';
  parent.style.position = 'absolute';
  parent.style.left = '0';
  parent.style.top = '0';
  parent.style.width = '100%';
  parent.style.height = '100%';

  var child = document.createElement('div');
  child.style.width = '120%';
  child.style.height = '120%';

  parent.appendChild(child);
  document.body.appendChild(parent);

  child.style.cursor = cursor;
  parent.scrollLeft = 1;
  parent.scrollLeft = 0;

  document.body.removeChild(parent);
}

function em2px(val, element) {
  element = element || document.documentElement;
  return val * parseFloat(getComputedStyle(element).fontSize);
}

function addClass(elem, className) {
  if (!elem || !className) return;

  var sep = ' ';
  var exist = elem.className.split(sep);
  var addition = className.split(sep).filter(function(name) {
    return (exist.indexOf(name) === -1);
  }).join(sep);
  if (addition.length) elem.className += ' ' + addition;
}

function removeClass(elem, className) {
  if (!elem || !className) return;

  var sep = ' ';
  var removing = className.split(sep);
  elem.className = elem.className.split(sep).filter(function(name) {
    return (removing.indexOf(name) === -1);
  }).join(sep);
}

function getParentByClass(elem, className) {
  while (elem && elem.classList && !elem.classList.contains(className)) {
    elem = elem.parentElement;
  }

  return elem;
}

function animate(onAnimationProcess, onEndAnimation) {
  var requestAnimationFrame = window.requestAnimationFrame ||
                              window.mozRequestAnimationFrame ||
                              window.webkitRequestAnimationFrame ||
                              window.msRequestAnimationFrame;

  if (!requestAnimationFrame) return;

  var startTimestamp;
  function step(timestamp) {
    if (!startTimestamp) {
      startTimestamp = timestamp;
      requestAnimationFrame(step);
    } else {
      var progress = timestamp - startTimestamp;
      if (onAnimationProcess(progress)) {
        requestAnimationFrame(step);
      } else if (onEndAnimation) {
        onEndAnimation(progress);
      }
    }
  }

  requestAnimationFrame(step);
}

function dispatchMouseEvent(pos, name, elem, params) {
  name = name || 'click';
  pos = pos || {x: 0, y: 0};
  if (pos.x === undefined) pos.x = 0;
  if (pos.y === undefined) pos.y = 0;

  var eventInit = {
    'view': window,
    'bubbles': true,
    'cancelable': true,
    'screenX': pos.x,
    'screenY': pos.y,
    'clientX': pos.x,
    'clientY': pos.y
  };

  applyObjectProperties(eventInit, params);

  elem = elem || document.elementFromPoint(pos.x, pos.y) || window;

  var event;

  try {
    event = new MouseEvent(name, eventInit);
  }
  catch(e) {
    event = document.createEvent('MouseEvents');
    event.initMouseEvent(name, true, true, window, 0, pos.x, pos.y, pos.x, pos.y, false, false, false, false, 0, null);
  }

  event.simulation = true;

  elem.dispatchEvent(event);
}

function dispatchElementMouseEvent(name, elem) {
  if (!elem) return;

  var elemPos = getElementPos(elem);

  var pos = {
    x: elemPos.x + elemPos.width / 2,
    y: elemPos.y + elemPos.height / 2
  };

  dispatchMouseEvent(pos, name, elem);
}

var disableHoverClass = 'idvc_disable_hover';

function disableHover(elem) {
  if (elem &&
      !elem.classList.contains(disableHoverClass)) {
    elem.classList.add(disableHoverClass);
  }
}

function enableHover(elem) {
  if (elem ) {
    elem.classList.remove(disableHoverClass);
  }
}

function createAsyncCall(func, thisArg, delay) {
  delay = delay || 0;
  thisArg = thisArg || window;
  func = func || function() {};

  var timeout;
  return {
    call: function() {
      this.cancel();

      var args = arguments;
      timeout = setTimeout(function() {
        func.apply(thisArg, args);
        timeout = undefined;
      }, delay);
    },
    cancel: function() {
      if (timeout) {
        clearTimeout(timeout);
        timeout = undefined;
      }
    },
    isCalled: function() {
      return !!timeout;
    },
    setFunction: function(f) {
      func = f || function() {};
    },
    setThisArg: function(arg) {
      thisArg = arg || window;
    },
    setDelay: function(newDelay) {
      delay = newDelay;
    }
  };
}


var disabledLogging = function () { };
var _log;

function consoleLog() {
  if (typeof _log === 'function') {
    _log.apply(this, arguments);
  }
}

consoleLog.enable = function() {
  _log = console.log.bind(console);
};
consoleLog.disable = function() {
  _log = disabledLogging;
};
consoleLog.set = function(log) {
  _log = log;
};

consoleLog.disable();

function applyObjectProperties(elem, obj) {
  if (!elem || !obj || (typeof obj !== 'object')) return;

  Object.getOwnPropertyNames(obj).forEach(function(prop) {
    if (typeof obj[prop] === 'object' &&
        !isHTMLNode(obj[prop])) {
      if (!elem[prop]) {
        if (Array.isArray(obj[prop])) elem[prop] = [];
        else elem[prop] = {};
      }
      applyObjectProperties(elem[prop], obj[prop]);
    } else if (prop === 'className' && elem.classList) {
      elem.classList.add(obj.className);
    } else {
      elem[prop] = obj[prop];
    }
  });
}

function copyObject(from, to, mapProp) {
  if (!from || !to || (typeof to !== 'object')) return;

  mapProp = mapProp || function(p) { return p; };

  Object.getOwnPropertyNames(from).forEach(function(name) {
    let value = from[name];
    if (isSystemProp(name) || typeof value === 'function') return;

    value = mapProp(value, name);

    if (typeof value === 'object') {
      if (value === null || isHTMLNode(value)) return;

      if (!to[name]) {
        if (Array.isArray(value)) to[name] = [];
        else to[name] = {};
      }
      copyObject(value, to[name], mapProp);
    } else {
      to[name] = value;
    }
  });
}

function isHTMLNode(obj) {
  return obj instanceof Node &&
    obj.nodeType !== undefined &&
    obj.nodeName !== undefined;
}

function isSystemProp(prop) {
  var len = prop.length;
  if (len > 4) return prop[0] === '_' && prop[1] === '_' && prop[len - 1] === '_' && prop[len - 2] === '_';

  return false;
}

function createSequence() {
  var index = -1;
  var doIndex = -1;
  var isCall = false;
  var isCanceled = false;
  var steps = [];
  var tmpResult;

  var defTimeout = 0;

  var doFinally = undefined;

  function clear() {
    steps.length = 0;
    index = doIndex = -1;
    isCanceled = false;
    tmpResult = undefined;
    isCall = false;
    doFinally = undefined;
  }

  function doCall(proc) {
    if (!proc) return;

    isCanceled = false;
    isCall = true;
    index++;
    doIndex = index;

    if (tmpResult !== undefined)
      proc(tmpResult);
    else
      proc();
  }

  function simpleFunc(func) {
    this.end(func());
  }

  return {
    do: function(proc) {
      if (!proc) return;

      steps.splice(doIndex + 1, 0, proc);
      doIndex++;
      if (!isCall) {
        doCall(proc);
      }

      return this;
    },
    do_: function(proc) {
      return this.do(simpleFunc.bind(this, proc));
    },
    end: function(result) {
      if (!isCall) return;

      tmpResult = result;
      isCall = false;

      if (index < steps.length - 1) {
        doCall(steps[index + 1]);
      } else {
        // the last step is done
        if (doFinally && !isCanceled) {
          doFinally();
        }

        clear();
      }

      return this;
    },
    end_: function(result) {
      setTimeout(this.end.bind(this, result), defTimeout);
      return this;
    },
    nextStep: function(proc) {
      if (isCanceled) {
        return this.end();
      } else {
        setTimeout(proc, defTimeout);
        return this;
      }
    },
    cancel: function() {
      if (isCall) {
        isCanceled = true;
        steps.splice(index + 1);
      } else {
        clear();
      }

      return this;
    },
    canceled: function() {
      return isCanceled;
    },
    isActive: function() {
      return !!steps.length
    },
    forEach: function(array, proc, stepCount) {
      if (!Array.isArray(array)) return;

      stepCount = stepCount || array.length;
      var restCount = array.length;
      var startIndex = 0;

      if (restCount > stepCount) processItems.call(this);
      else {
        processStep();
        this.end();
      }

      function processItems() {
        if (isCanceled) {
          this.end();
          return;
        }

        processStep();

        restCount -= stepCount;
        startIndex += stepCount;

        if (restCount > 0) this.nextStep(processItems.bind(this));
        else this.end();
      }

      function processStep() {
        for (var i = startIndex, len = startIndex + Math.min(restCount, stepCount); i < len; i++) {
          proc(array[i], i, array);
        }
      }
    },
    wait(proc) {
      if (proc) proc();

      return new Promise(resolve => {
        function resolvePromise(prevDoFinally) {
          if (prevDoFinally) prevDoFinally();

          resolve(true);
        }

        if (this.isActive()) {
          doFinally = resolvePromise.bind(undefined, doFinally);
        } else {
          resolve(true);
        }
      })
    }
  };
};

function createCounter(start) {
  start = start || 0;

  var current = start;

  return {
    next: function() {
      ++current;
      if (current < start) current = ++start;
      return current;
    },
    get: function() {
      return current;
    }
  }
}

function removeAllChildren(el) {
  if (!el) return;

  while (el.firstChild) {
    el.removeChild(el.firstChild);
  }
}

function addWindowResizeListener(elem, param) {
  param = param || {height: true, width: true};

  var processWindowResize = refreshSize.bind(undefined, elem, param);

  window.addEventListener('resize', processWindowResize, false);

  return {
    remove: function() {
      window.removeEventListener('resize', processWindowResize, false);
      processWindowResize = undefined;
    }
  };
}

function toast(msg, parent, params) {
  parent = parent || document.body;
  params = params || {top: '1em'};

  var content = createElement(msg, 'idvc_toast', '');
  applyObjectProperties(content.style, params);

  var parent = disableElement(parent, '', undefined, true);
  parent.disabler.classList.add('idvc_toast_holder');
  parent.disabler.appendChild(content);

  var isMouseOn = false;
  var isTimeout = false;

  function hideToast(forced) {
    if (!forced && (isMouseOn || !isTimeout)) return;

    content.classList.remove('toast-show');

    setTimeout(function() {
      if (parent) {
        parent.end();
        parent = undefined;
      }
    }, 400);
  }

  setTimeout(function() {
    content.classList.add('toast-show');
  }, 0);

  var hideTimeout = setTimeout(function() {
    isTimeout = true;
    hideToast()
  }, 5400);

  content.addEventListener('mouseenter', function() {
    isMouseOn = true;
  }, false);

  content.addEventListener('mouseleave', function() {
    isMouseOn = false;
    hideToast();
  }, false);

  parent.disabler.addEventListener('click', function(e) {
    clearTimeout(hideTimeout);
    hideToast(true);
    e.stopPropagation();
    e.preventDefault();
  }, false);
}

function ScrolledHolder() {
  this.scrolled = [];
}

ScrolledHolder.prototype.updateVisibleScrollTop = function(scrollTop) {
  this.scrolled.forEach(function(scrolled) {
    scrolled._updateVisibleScrollTop(scrollTop);
  });
};

ScrolledHolder.prototype.addScrolled = function(scrolled, front) {
  if (front) this.scrolled.unshift(scrolled);
  else this.scrolled.push(scrolled);
};

ScrolledHolder.prototype.removeAllScrolled = function() {
  this.scrolled.length = 0;
};

ScrolledHolder.prototype.syncScrollTop = function(val) {
  var visibleScrollTop = 0;
  var topIndex = 0;
  var needUpdateVisibleScrollTop = false;
  var topIndexesEqual = true;

  var isFirstActiveScroll = true;

  this.scrolled.forEach(function(scrolled) {
    if (!scrolled._isActive()) return;

    scrolled._updateScrollTop(val);
    var scrollState = scrolled._getScrollState();
    var newColumnScrollTop = scrollState.visibleScrollTop;
    if (isFirstActiveScroll) {
      visibleScrollTop = newColumnScrollTop;
      topIndex = scrollState.topIndex;
      isFirstActiveScroll = false;
    } else if (newColumnScrollTop !== visibleScrollTop) {
      visibleScrollTop = Math.min(visibleScrollTop, newColumnScrollTop);
      needUpdateVisibleScrollTop = true;
      topIndexesEqual = topIndexesEqual && (topIndex === scrollState.topIndex);
    }
  });

  if (topIndexesEqual && needUpdateVisibleScrollTop) this.updateVisibleScrollTop(visibleScrollTop);

  this.scrolled.forEach(function(scrolled) {
    if (!scrolled._isActive() || !scrolled._endScroll) return;

    scrolled._endScroll();
  });
};

return {
  Consts: Consts,
  appendCSS: appendCSS,
  appendScript: appendScript,
  applyUAFont: applyUAFont,
  getScrollbarSize: getScrollbarSize,
  disableDragging: disableDragging,
  createElement: createElement,
  getElementById: getElementById,
  round: round,
  floor: floor,
  ceil: ceil,
  roundExp: roundExp,
  getElementPos: getElementPos,
  defaultCompareItems: defaultCompareItems,
  format: format,
  curry: curry,
  curryThis: curryThis,
  compose: compose,
  composeThis: composeThis,
  getDomElement: getDomElement,
  refreshSize: refreshSize,
  createResizeProcess: createResizeProcess,
  createVertSplitter: createVertSplitter,
  createHorzSplitter: createHorzSplitter,
  addTooltip: addTooltip,
  copyAppearance: copyAppearance,
  addExpansion: addExpansion,
  buildHierarchy: buildHierarchy,
  getTextWidth: getTextWidth,
  disableElement: disableElement,
  makeCaption: makeCaption,
  setFont: setFont,
  changeCursor: changeCursor,
  em2px: em2px,
  addClass: addClass,
  removeClass: removeClass,
  getParentByClass: getParentByClass,
  animate: animate,
  dispatchMouseEvent: dispatchMouseEvent,
  dispatchElementMouseEvent: dispatchElementMouseEvent,
  disableHover: disableHover,
  enableHover: enableHover,
  createAsyncCall: createAsyncCall,
  consoleLog: consoleLog,
  applyObjectProperties: applyObjectProperties,
  copyObject: copyObject,
  isHTMLNode: isHTMLNode,
  createSequence: createSequence,
  createCounter: createCounter,
  removeAllChildren: removeAllChildren,
  addWindowResizeListener: addWindowResizeListener,
  toast: toast,
  ScrolledHolder: ScrolledHolder
};

});

/*
 * Copyright (C) 2013 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

define('vtgrid', ['./signal', './grid', './utils'], function(Signal, Grid, Utils) {

"use strict"

function ProxyDataModel() {
  this.dataModel = null;
  this.changed = Signal.create();
  this.startIndex = 0;
  this.columnCount = -1;
}

ProxyDataModel.prototype.getRowCount = function() {
  if (this.dataModel)
    return this.dataModel.getRowCount();

  return -1;
};

ProxyDataModel.prototype.getColumnCaption = function(val, params) {
  if (this.dataModel) {
    return this.dataModel.getColumnCaption(val, params);
  }

  return '';
};

ProxyDataModel.prototype.getColumnDescription = function(val) {
  if (this.dataModel &&
      this.dataModel.getColumnDescription) {
    return this.dataModel.getColumnDescription(val);
  }

  return '';
};

ProxyDataModel.prototype.getCell = function(row, col, isSelected) {
  if (this.dataModel) {
    return this.dataModel.getCell(row, col, isSelected);
  }

  return '';
};

ProxyDataModel.prototype.getFooter = function(col) {
  if (this.dataModel && this.dataModel.getFooter) {
    return this.dataModel.getFooter(col);
  }

  return '';
};

ProxyDataModel.prototype.getCellLayout = function(text) {
  if (this.dataModel &&
      this.dataModel.getCellLayout) {
    return this.dataModel.getCellLayout(text);
  }

  return null;
};

ProxyDataModel.prototype.getCellStyle = function(isSelected, defSelectedStyle, defStyle, row, col) {
  if (this.dataModel &&
      this.dataModel.getCellStyle) {
    return this.dataModel.getCellStyle(isSelected, defSelectedStyle, defStyle, row, col);
  }

  return null;
};

ProxyDataModel.prototype.isColumnSortable = function (col) {
  if (this.dataModel &&
      this.dataModel.isColumnSortable) {
    return this.dataModel.isColumnSortable(col);
  }

  return false;
};

ProxyDataModel.prototype.onSortColumn = function (col, current) {
  if (this.dataModel &&
      this.dataModel.onSortColumn) {
    return this.dataModel.onSortColumn(col, current);
  }

  return false;
};

ProxyDataModel.prototype.processDataChanges = function(ev) {
  this.changed.raise(ev);
};

ProxyDataModel.prototype.setDataModel = function(dataModel) {
  var oldModel = this.dataModel;
  if (oldModel) {
    oldModel.changed.unsubscribe(this, this.processDataChanges);
  }

  this.dataModel = dataModel;

  if (dataModel) {
    dataModel.changed.subscribe(this, this.processDataChanges);
  }

  this.changed.raise(null);
};

ProxyDataModel.prototype.getColumnLayout = function (col) {
  if (this.dataModel &&
      this.dataModel.getColumnLayout !== undefined) {
    return this.dataModel.getColumnLayout(col);
  }

  return 0;
};

ProxyDataModel.prototype.setStartColumnIndex = function (index) {
  this.startIndex = index;
};

ProxyDataModel.prototype.setColumnCount = function (count) {
  this.columnCount = count;
};

ProxyDataModel.prototype.getColumnCount = function() {
  if (this.columnCount >= 0) {
    return this.columnCount;
  } else if (this.dataModel) {
    return this.dataModel.getColumnCount() - this.startIndex;
  }

  return 0;
};

ProxyDataModel.prototype.getDataIndex = function(index) {
  return index + this.startIndex;
};

function CentralProxyDataModel() {
  ProxyDataModel.call(this);
}

CentralProxyDataModel.prototype = Object.create(ProxyDataModel.prototype);


CentralProxyDataModel.prototype.setElement = function (elem) {
  if (this.dataModel &&
      this.dataModel.setElement !== undefined) {
    this.dataModel.setElement(elem);
  }
};

function GridBodyProxy() {
  this.bodies = [];
}

GridBodyProxy.prototype.addBody = function(body) {
  this.bodies.push(body);
};

GridBodyProxy.prototype.forEachBodies = function(process, source) {
  this.bodies.forEach(function(body) {
    if (body !== source) process(body);
  });
};

GridBodyProxy.prototype.setCurrentRow = function(row, stopPropagation, keyState, source) {
  this.forEachBodies(function(body) {
    body.setCurrentRow(row, stopPropagation, keyState);
  }, source);
};

GridBodyProxy.prototype.processSetFocus = function(propagate, source) {
  this.forEachBodies(function(body) {
    body.processSetFocus(propagate);
  }, source);
};

GridBodyProxy.prototype.processLostFocus = function(propagate, source) {
  this.forEachBodies(function(body) {
    body.processLostFocus(propagate);
  }, source);
};

////////////////////////////////////////////////////////////////////////////

//////////////               VtGrid

////////////////////////////////////////////////////////////////////////////

const ResizeMode = {
  rmRight: 1,
  rmCenter: 2
};

function VtGrid(parent, tabIndex, resizeMode) {
  function createSubGrid(parent, scrolType, needVertScroll) {
    var grid = Grid.create(parent, tabIndex);
    if (!needVertScroll) grid.hideVertScroll();
    grid.setHorzScrollType(scrolType);
    grid.gridBody.onDblClick.subscribe(this.onDblClick);
    grid.gridBody.onContextMenu.subscribe(this.onContextMenu);
    grid.gridBody.onChangeCurrentRow.subscribe(this.onChangeCurrentRow);
    grid.gridBody.onShowTooltip.subscribe(this.onShowTooltip);
    grid.gridBody.onSetFocus.subscribe(this.onSetFocus);
    grid.gridBody.onLostFocus.subscribe(this.onLostFocus);

    return grid;
  }

  resizeMode = resizeMode || ResizeMode.rmRight;

  this.onDblClick = Signal.create();
  this.onContextMenu = Signal.create();
  this.onChangeCurrentRow = Signal.create();
  this.onExpandColumn = Signal.create();
  this.beforeExpandColumn = Signal.create();
  this.onChangeCentralColumnWidth = Signal.create();
  this.onShowTooltip = Signal.create();
  this.onSetFocus = Signal.create();
  this.onLostFocus = Signal.create();

  var parentEl = Utils.getDomElement(parent);

  this.leftGridHolder = document.createElement('div');
  this.leftGridHolder.className = 'idvcgrid_leftgrid_holder';
  parentEl.appendChild(this.leftGridHolder);

  this.splitterHolder = document.createElement('div');
  this.splitterHolder.className = 'idvcgrid_splitter_holder';
  parentEl.appendChild(this.splitterHolder);

  this.splitter = Utils.createHorzSplitter(this.splitterHolder, 10, {
    primaryDivClass: 'idvcgrid_central_column',
    splitterDivClass: 'idvcgrid_splitter',
    secondaryDivClass: 'idvcgrid_right_grid',
    backward: resizeMode === ResizeMode.rmCenter
  });
  this.splitter.setSize('40%');

  this.splitter.onStopUpdateSplitter = function() {
    this.onChangeCentralColumnWidth.raise(this.getCentralColumnWidth());
  }.bind(this);

  this.leftGrid = createSubGrid.call(this, this.leftGridHolder, 'hidden');
  this.leftGrid.gridBody.expandLastColumn(false);

  this.centralGrid = createSubGrid.call(this, this.splitter.primaryDiv, 'scroll');

  this.rightGrid = createSubGrid.call(this, this.splitter.secondaryDiv, 'scroll', true);
  this.rightGrid.gridBody.onExpandColumn.subscribe(this.onExpandColumn);
  this.rightGrid.gridBody.beforeExpandColumn.subscribe(this.beforeExpandColumn);
  this.rightGrid.onUpdateViewComplete.subscribe(this, () => {
    Utils.refreshSize(parentEl);
  });

  this.rightGrid.gridBody.onFitHeaderHeight.subscribe(this, fitHeaderHeightByRight);
  this.leftGrid.gridBody.onChangeColumnWidth.subscribe(this, fitLeftGridWidth);
  this.leftGrid.onUpdateViewComplete.subscribe(this, () => {
    fitLeftGridWidth.call(this);
    fitLeftGridHeight.call(this);
  });

  this.leftDataModel = new ProxyDataModel();
  this.centralDataModel = new CentralProxyDataModel();
  this.rightDataModel = new ProxyDataModel();

  connectRightScrolling.call(this);

  var gridBodyProxy = new GridBodyProxy();
  gridBodyProxy.addBody(this.centralGrid.gridBody);
  gridBodyProxy.addBody(this.rightGrid.gridBody);
  gridBodyProxy.addBody(this.leftGrid.gridBody);

  this.centralGrid.gridBody.connectedBody = gridBodyProxy;
  this.rightGrid.gridBody.connectedBody = gridBodyProxy;
  this.leftGrid.gridBody.connectedBody = gridBodyProxy;

  this.splitter.primaryDiv.idvcVTGridObject = this;

  parentEl.onselectstart = function() {
    return false;
  };
}

function proxyPropImp(propName) {
  var dataModel = this.dataModel;
  if (dataModel &&
      typeof dataModel[propName] === "function") {
    var args = Array.prototype.slice.call(arguments, 1);
    return dataModel[propName].apply(dataModel, args);
  }

  return undefined;
}

function updateProxyDataModel(proxyDataModel, dataModel, props) {
  props.forEach(function(prop) {
    if (dataModel && dataModel[prop]) {
      proxyDataModel[prop] = proxyPropImp.bind(proxyDataModel, prop);
    } else if (proxyDataModel.hasOwnProperty(prop)) {
      delete proxyDataModel[prop];
    }
  });
}

var proxyProperties = [
  'setCurrentRow',
  'getCurrentRow',
  'getCellTooltip',
  'isRowSelected',
  'isColumnMovable',
  'beforeColumnAutoSize',
  'getRowHeight',
  'getRowsHeight',
  'getRowIndexByHeight'
];

function updateRowsHeightRoutines(destModel, srcBody, dataModel) {
  const props = [
    'getRowHeight',
    'getRowsHeight',
    'getRowIndexByHeight',
    'getFirstColumn'
  ];

  props.forEach(function(prop) {
    if (dataModel && !dataModel[prop] &&
        srcBody && srcBody[prop]) {
      destModel[prop] = srcBody[prop].bind(srcBody);
    }
  });
}

VtGrid.prototype.setDataModel = function(dataModel, columnVisModel, centralColumnIndex) {
  centralColumnIndex = centralColumnIndex || 0;

  updateProxyDataModel.call(this, this.leftDataModel, dataModel, proxyProperties);
  updateProxyDataModel.call(this, this.centralDataModel, dataModel, proxyProperties);
  updateProxyDataModel.call(this, this.rightDataModel, dataModel, proxyProperties);

  if (this.mainDataModel) {
    if (this.mainDataModel) {
      this.mainDataModel.setElement(null);
    }

    if (this.mainDataModel.removeAllSecondaryElements) {
      this.mainDataModel.removeAllSecondaryElements();
    }
  }

  this.mainDataModel = dataModel;
  this.columnVisModel = columnVisModel;
  this.centralColumnIndex = centralColumnIndex;

  if (dataModel) {
    updateRowsHeightRoutines(this.leftDataModel, this.centralGrid.gridBody, dataModel);
    updateRowsHeightRoutines(this.rightDataModel, this.centralGrid.gridBody, dataModel);

    if (dataModel.getColumnCount() <= centralColumnIndex) {
      centralColumnIndex = dataModel.getColumnCount() - 1;
    }

    if (dataModel.setHierarchicalColumnIndex) {
      dataModel.setHierarchicalColumnIndex(centralColumnIndex);
    }

    if (dataModel.addSecondaryElement) {
      dataModel.addSecondaryElement(this.leftGrid.gridBody);
      dataModel.addSecondaryElement(this.rightGrid.gridBody);
    }

    clearLeftGridWidth.call(this);

    if (dataModel.getColumnCount() > centralColumnIndex + 1) _setRightUpdater.call(this);
    else _setCentralUpdater.call(this);

    this.centralDataModel.setColumnCount(1);
    this.centralDataModel.setStartColumnIndex(centralColumnIndex);
    this.centralDataModel.setDataModel(dataModel);
    this.centralGrid.setDataModel(this.centralDataModel);

    this.leftDataModel.setColumnCount(centralColumnIndex);
    this.leftDataModel.setDataModel(dataModel);
    this.leftGrid.setDataModel(this.leftDataModel);

    this.rightDataModel.setColumnCount(-1);
    this.rightDataModel.setStartColumnIndex(centralColumnIndex + 1);
    this.rightDataModel.setDataModel(dataModel);
    this.rightGrid.setDataModel(this.rightDataModel, columnVisModel);
  } else {
    _setRightUpdater.call(this);

    this.leftDataModel.setColumnCount(-1);
    this.leftDataModel.setDataModel(undefined);
    this.leftGrid.setDataModel(undefined);

    this.rightDataModel.setColumnCount(-1);
    this.rightDataModel.setStartColumnIndex(0);
    this.rightDataModel.setDataModel(undefined);
    this.rightGrid.setDataModel(undefined);

    this.centralDataModel.setColumnCount(-1);
    this.centralDataModel.setStartColumnIndex(0);
    this.centralDataModel.setDataModel(undefined);
    this.centralGrid.setDataModel(undefined);

    fitLeftGridWidth.call(this);
  }
};

VtGrid.prototype.refresh = function() {
  const dataModel = this.mainDataModel;
  const columnVisModel = this.columnVisModel;
  const centralColumnIndex = this.centralColumnIndex;

  this.setDataModel(undefined, undefined, -1);
  this.setDataModel(dataModel, columnVisModel, centralColumnIndex);
};

VtGrid.prototype.setCurrentRow = function(row) {
  this.centralGrid.setCurrentRow(row);
};

VtGrid.prototype.getCurrentRow = function() {
  return this.centralGrid.getCurrentRow();
};

VtGrid.prototype.hideHeader = function() {
  this.leftGrid.hideHeader();
  this.centralGrid.hideHeader();
  this.rightGrid.hideHeader();
};

VtGrid.prototype.showFooter = function() {
  this.leftGrid.showFooter();
  this.centralGrid.showFooter();
  this.rightGrid.showFooter();

  this.leftGridHolder.style.borderBottomColor = 'transparent';
};

VtGrid.prototype.hideFooter = function() {
  this.leftGrid.hideFooter();
  this.centralGrid.hideFooter();
  this.rightGrid.hideFooter();

  this.leftGridHolder.style.borderBottomColor = 'ThreeDShadow';
};

VtGrid.prototype.updateFooter = function() {
  this.leftGrid.updateFooter();
  this.centralGrid.updateFooter();
  this.rightGrid.updateFooter();
};

VtGrid.prototype.setCentralColumnWidth = function(width) {
  this.splitter.setSize(width);
};

VtGrid.prototype.getCentralColumnWidth = function() {
  return this.splitter.getSize();
};

VtGrid.prototype.getColumnByDataIndex = function(index, all) {
  var result = this.rightGrid.getColumnByDataIndex(index, all);
  if (!result) result = this.centralGrid.getColumnByDataIndex(index, all);
  if (!result) result = this.leftGrid.getColumnByDataIndex(index, all);

  return result;
};

VtGrid.prototype.hideColumn = function(index) {
  this.rightGrid.hideColumn(index);
};

VtGrid.prototype.canColumnBeHidden = function(index) {
  return this.rightGrid.canColumnBeHidden(index);
};

VtGrid.prototype.showAllColumns = function() {
  this.rightGrid.showAllColumns();
}

VtGrid.prototype.getInvisibleColumnsCount = function() {
  return this.rightGrid.getInvisibleColumnsCount();
}

VtGrid.prototype.getColumnsInfo = function() {
  return this.leftGrid.getColumnsInfo().concat(
            this.centralGrid.getColumnsInfo().concat(
              this.rightGrid.getColumnsInfo()));
};

VtGrid.prototype.expandLastColumn = function(expand) {
  this.rightGrid.gridBody.expandLastColumn(expand);
}

VtGrid.prototype.isLastColumnExpanded = function() {
  return this.rightGrid.gridBody.isLastColumnExpanded();
}

VtGrid.prototype.rowToVisible = function(row) {
  this.centralGrid.gridBody.rowVisible(row);
}

VtGrid.prototype.currentRowToVisible = function() {
  this.centralGrid.gridBody.currentRowVisible();
}

VtGrid.prototype.columnToVisible = function(col, colOffset) {
  var result = this.rightGrid.gridBody.columnToVisible(col, colOffset);
  if (!result) result = this.centralGrid.gridBody.columnToVisible(col, colOffset);
  if (!result) result = this.leftGrid.gridBody.columnToVisible(col, colOffset);
}

VtGrid.prototype.columnToExpanded = function(col) {
  return this.rightGrid.gridBody.columnToExpanded(col);
}

VtGrid.prototype.getRowCount = function() {
  return this.centralGrid.getRowCount();
}

VtGrid.prototype.recalcScroll = function() {
  if (this.splitter.isSecondaryFolded()) {
    this.centralGrid.recalcScroll();
  } else {
    this.rightGrid.recalcScroll();
  }
}

VtGrid.prototype.hideTooltip = function() {
  this.rightGrid.hideTooltip();
  this.centralGrid.hideTooltip();
  this.leftGrid.hideTooltip();
}

VtGrid.prototype.destroy = function() {
  unConnectScrolling.call(this);

  this.setDataModel(undefined, undefined, -1);

  this.rightGrid.destroy();
  this.centralGrid.destroy();
  this.leftGrid.destroy();

  delete this.splitter.primaryDiv.idvcVTGridObject;
}

function _setRightUpdater() {
  if (!this.splitter.isSecondaryFolded()) return;

  this.splitter.unfold();
  connectRightScrolling.call(this);
}

function _setCentralUpdater() {
  if (this.splitter.isSecondaryFolded()) return;

  connectCentralScrolling.call(this);
  this.splitter.foldSecondary(true);
}

function connectScrolling(vertScroll, gridBody, front) {
  vertScroll.addScrolled(gridBody, front);
  gridBody.scrolling = vertScroll;
}

function unConnectScrolling() {
  function setDefScrolling(grid) {
    grid._vertScroll.removeAllScrolled();
    grid._vertScroll.addScrolled(grid.gridBody);
    grid.gridBody.scrolling = grid._vertScroll;
  }

  setDefScrolling(this.leftGrid);
  setDefScrolling(this.centralGrid);
  setDefScrolling(this.rightGrid);
}

function connectRightScrolling() {
  unConnectScrolling.call(this);

  this.centralGrid.hideVertScroll();
  this.rightGrid.showVertScroll();

  connectScrolling(this.rightGrid._vertScroll, this.centralGrid.gridBody, true);
  connectScrolling(this.rightGrid._vertScroll, this.leftGrid.gridBody);
}

function connectCentralScrolling() {
  unConnectScrolling.call(this);

  this.rightGrid.hideVertScroll();
  this.centralGrid.showVertScroll();

  connectScrolling(this.centralGrid._vertScroll, this.leftGrid.gridBody);
  connectScrolling(this.centralGrid._vertScroll, this.rightGrid.gridBody);
}

function connectExternalScrolling(grid, scrolling) {
  unConnectScrolling.call(grid);

  grid.rightGrid.hideVertScroll();
  grid.centralGrid.hideVertScroll();

  connectScrolling(scrolling, grid.centralGrid.gridBody, true);
  connectScrolling(scrolling, grid.leftGrid.gridBody);
  connectScrolling(scrolling, grid.rightGrid.gridBody);
}

function fitLeftGridWidth(_, __, changer) {
  var wholeWidth = 0;
  this.leftGrid.gridBody.columns.forEach(function(column) {
    wholeWidth += column.getColumnWidth();
  });

  if (!wholeWidth) {
    this.leftGridHolder.style.display = 'none';
  } else {
    this.leftGridHolder.style.display = 'block';
  }
  this.leftGridHolder.style.width = this.splitterHolder.style.left = wholeWidth + 'px';

  if (changer) changer.gridAreaChanged = true;
}

function fitLeftGridHeight() {
  var leftBottomDelta = this.centralGrid.gridBody.area.offsetHeight -
                        this.centralGrid.gridBody.area.clientHeight;

  this.leftGridHolder.style.bottom = (leftBottomDelta - 1) + 'px';
  Utils.refreshSize(this.leftGridHolder, {height:true});
}

function clearLeftGridWidth() {
  this.leftGridHolder.style.display = '';
  this.leftGridHolder.style.width = this.splitterHolder.style.left = '';
}

function fitHeaderHeightByRight(columnSize, isSignal) {
  this.centralGrid.gridBody.fitHeaderHeight(columnSize, isSignal);
  this.leftGrid.gridBody.fitHeaderHeight(columnSize, isSignal);

  fitLeftGridHeight.call(this);
}

async function enumCellsInRowsRange(process, grid, ranges) {
  function doProcess(type, cell, row, col) {
    process(cell, row, col, type)
  }

  if (!grid || !process) return

  ranges = ranges || {}

  let { rangesLeft, rangesCenter, rangesRight } = ranges

  if (rangesLeft && rangesCenter && rangesRight && ranges.startRow) {
    rangesLeft.startRow = rangesCenter.startRow = rangesRight.startRow = ranges.startRow
  }

  const leftLastProcessedRow = await Grid.dumpUtils.enumCellsInRowsRange(
    doProcess.bind(this, 'left'), grid.leftGrid, rangesLeft)
  const centralLastProcessedRow = await Grid.dumpUtils.enumCellsInRowsRange(
    doProcess.bind(this, 'central'), grid.centralGrid, rangesCenter)
  const rightLastProcessedRow = await Grid.dumpUtils.enumCellsInRowsRange(
    doProcess.bind(this, 'right'), grid.rightGrid, rangesRight)

  return rightLastProcessedRow !== undefined ? rightLastProcessedRow :
    (centralLastProcessedRow !== undefined ? centralLastProcessedRow :
      leftLastProcessedRow)
}

async function enumAllCells(process, grid, ranges) {
  function updateColumnRange(range, count) {
    function updateRangeVal(prop) {
      if (range[prop] !== undefined) {
        range[prop] -= count
      }
    }

    updateRangeVal('startColumn')
    updateRangeVal('endColumn')

    range.firstColumnIndex = count
  }

  ranges = ranges || {}

  let rangesLeft = Object.assign({}, ranges)
  let rangesCenter = Object.assign({}, ranges)
  let rangesRight = Object.assign({}, ranges)

  let { centralColumnIndex } = grid

  const leftColumnCount = centralColumnIndex
  const centralColumnCount = 1

  updateColumnRange(rangesCenter, leftColumnCount)
  updateColumnRange(rangesRight, leftColumnCount + centralColumnCount)

  ranges.rangesLeft = rangesLeft
  ranges.rangesCenter = rangesCenter
  ranges.rangesRight = rangesRight

  return Grid.dumpUtils.enumRowsRanges(enumCellsInRowsRange.bind(this, process),
    grid, grid.centralGrid.gridBody, ranges)
}

async function dumpCells(grid, range, dumpElement, enumProc) {
  enumProc = enumProc || enumAllCells
  return Grid.dumpCells(grid, range, dumpElement, enumProc)
}

VtGrid.prototype.enumCells = async function(process, range) {
  return enumAllCells(process, this, range)
}

VtGrid.prototype.dumpCells = async function(range, dumpElement, enumProc) {
  return dumpCells(this, range, dumpElement, enumProc)
}

return {
  ResizeMode,
  create: function(parent, tabIndex, resizeMode) {
    return new VtGrid(parent, tabIndex, resizeMode);
  },
  connectExternalScrolling,
  dumpUtils: {
    enumCellsInRowsRange,
    enumAllCells,
  },
  dumpCells
};

});


//# sourceMappingURL=idvcjs-631030.js.map
