/*******************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
*   Content : Intel(R) oneAPI Math Kernel Library (oneMKL) Sparse BLAS C OpenMP
*             offload example for mkl_sparse_s_trsv with async execution
*
********************************************************************************
*
* Consider the matrix A (see 'Sparse Storage Formats for Sparse BLAS Level 2
* and Level 3 in the  Intel oneMKL Reference Manual')
*
*                 |   1       -1      0   -3     0   |
*                 |  -2        5      0    0     0   |
*   A    =        |   0        0      4    6     4   |,
*                 |  -4        0      2    7     0   |
*                 |   0        8      0    0    -5   |
*
*  The matrix A is represented in a zero-based compressed sparse row (CSR) storage
*  scheme with three arrays (see 'Sparse Matrix Storage Schemes' in the
*   Intel oneMKL Reference Manual) as follows:
*
*         values  =  ( 1 -1 -3 -2  5  4  6  4 -4  2  7  8 -5 )
*         columns =  ( 0  1  3  0  1  2  3  4  0  2  3  1  4 )
*         rowIndex = ( 0        3     5        8       11    13 )
*
*  The test performs the operation
*
*       alpha * U^{-1}*b = x, using mkl_sparse_s_trsv
*
*  Here A is a general sparse matrix
*        U is the upper triangular part of matrix A with unit diagonal
*        b, b2, x, and z are vectors
*
********************************************************************************
*/
#include <stdio.h>
#include <assert.h>
#include <math.h>

#include "common_for_sparse_examples.h"
#include "mkl.h"
#include "mkl_omp_offload.h"

int main() {
    //*******************************************************************************
    //     Declaration and initialization of parameters for sparse representation of
    //     the matrix A in the compressed sparse row format:
    //*******************************************************************************
#define M     5     // nRows of A == length of b, if opA == non-transpose
#define N     5     // nCols of A == length of x, if opA == non-transpose
#define NNZ   13
#define ALIGN 64

    //*******************************************************************************
    //    Declaration of local variables:
    //*******************************************************************************
    // Descriptor of main sparse matrix properties
    struct matrix_descr descrA;

    // Structure with sparse matrix stored in CSR format
    sparse_matrix_t csrA = NULL;

    sparse_index_base_t indexing = SPARSE_INDEX_BASE_ZERO;
    sparse_operation_t opA       = SPARSE_OPERATION_NON_TRANSPOSE;

    float alpha = 2.0f;
    MKL_INT i;

    float *values      = (float *)mkl_malloc(sizeof(float) * NNZ, ALIGN);
    MKL_INT *columns   = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * NNZ, ALIGN);
    MKL_INT *row_index = (MKL_INT *)mkl_malloc(sizeof(MKL_INT) * (M + 1), ALIGN);
    float *b           = (float *)mkl_malloc(sizeof(float) * M, ALIGN);
    float *b2          = (float *)mkl_malloc(sizeof(float) * M, ALIGN);

    float *x  = (float *)mkl_malloc(sizeof(float) * N, ALIGN);
    float *z1 = (float *)mkl_malloc(sizeof(float) * N, ALIGN);
    float *z2 = (float *)mkl_malloc(sizeof(float) * N, ALIGN);

    const int num_pointers = 8;
    void *pointer_array[num_pointers];
    pointer_array[0] = values;
    pointer_array[1] = columns;
    pointer_array[2] = row_index;
    pointer_array[3] = b;
    pointer_array[4] = b2;
    pointer_array[5] = x;
    pointer_array[6] = z1;
    pointer_array[7] = z2;

    if (!values || !columns || !row_index || !b || !x || !z1 || !z2 || !b2) {
        free_allocated_memories(pointer_array, num_pointers);
        return 1;
    }

    //*******************************************************************************
    //    Sparse representation of the matrix A
    //*******************************************************************************
    float init_values[NNZ] = {1.0f, -1.0f, -3.0f, -2.0f, 5.0f, 4.0f, 6.0f,
                              4.0f, -4.0f, 2.0f,  7.0f,  8.0f, -5.0f};
    MKL_INT init_columns[NNZ]     = {0, 1, 3, 0, 1, 2, 3, 4, 0, 2, 3, 1, 4};
    MKL_INT init_row_index[M + 1] = {0, 3, 5, 8, 11, 13};
    float init_b[M]               = {1.0f, 5.0f, 1.0f, 4.0f, 1.0f};

    for (i = 0; i < NNZ; i++) {
        values[i]  = init_values[i];
        columns[i] = init_columns[i];
    }
    for (i = 0; i < M + 1; i++) {
        row_index[i] = init_row_index[i];
    }

    for (i   = 0; i < M; i++) {
        b[i] = init_b[i];
        b2[i] = init_b[i];
    }
    for (i = 0; i < N; i++) {
        x[i]  = 0.0f;
        z1[i] = 0.0f;
        z2[i] = 0.0f;
    }

    printf("\n EXAMPLE PROGRAM FOR mkl_sparse_s_trsv omp_offload \n");
    printf("-----------------------------------------------------\n");
    printf("\n");
    printf("   INPUT DATA FOR mkl_sparse_s_trsv omp offload async \n");
    printf("   UPPER TRIANGULAR SPARSE MATRIX \n");
    printf("   WITH UNIT DIAGONAL             \n");
    printf("   ALPHA = %4.1f \n", alpha);
    printf("   SPARSE_OPERATION_NON_TRANSPOSE \n");
    printf("   Input vector                   \n");
    for ( i = 0; i < M; i++ ) {
        printf("%7.1f\n", b[i]);
    }
    printf("---------------------------------------------------\n");
    fflush(stdout);

    // Create matrix descriptor
    descrA.type = SPARSE_MATRIX_TYPE_TRIANGULAR;
    descrA.mode = SPARSE_FILL_MODE_UPPER;
    descrA.diag = SPARSE_DIAG_UNIT;

    sparse_status_t ie_status;

    // Create handle with matrix stored in CSR format
    ie_status = mkl_sparse_s_create_csr(&csrA, indexing,
                                        M, // number of rows
                                        N, // number of cols
                                        row_index, row_index + 1, columns, values);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_s_create_csr: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    ie_status = mkl_sparse_set_sv_hint(csrA, opA, descrA, 1);
    if (ie_status != SPARSE_STATUS_SUCCESS && ie_status != SPARSE_STATUS_NOT_SUPPORTED) {
        printf(" Error in set_sv_hint: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    ie_status = mkl_sparse_optimize(csrA);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_optimize: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    // Compute x = alpha * A^{-1} * b
    ie_status = mkl_sparse_s_trsv(opA, alpha, csrA, descrA, b, x);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_s_trsv: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    // Release matrix handle and deallocate matrix
    ie_status = mkl_sparse_destroy(csrA);
    if (ie_status != SPARSE_STATUS_SUCCESS) {
        printf(" Error in mkl_sparse_destroy: %d\n", ie_status);
        free_allocated_memories(pointer_array, num_pointers);
        return ie_status;
    }

    printf( "                                   \n" );
    printf( "   OUTPUT DATA FOR mkl_sparse_s_trsv \n" );

    // x should be equal { 18.0, 5.0, -27.0, 4.0, 1.0 }
    for ( i = 0; i < N; i++ )
    {
        printf( "%7.4f\n", x[i] );
    };
    printf("---------------------------------------------------\n");
    fflush(stdout);

    const int devNum = 0;

    sparse_matrix_t csrA_gpu = NULL;

    sparse_status_t status_create;
    sparse_status_t status_hint;
    sparse_status_t status_opt;
    sparse_status_t status_trsv1;
    sparse_status_t status_trsv2;
    sparse_status_t status_destroy;

// call create_csr/set_sv_hint/optimize/trsv/destroy via omp_offload.
#pragma omp target data map(to:row_index[0:M+1],columns[0:NNZ],values[0:NNZ],b[0:M],b2[0:M])          \
                        map(tofrom:z1[0:N],z2[0:N]) device(devNum)
    {
        printf("Create CSR matrix via omp_offload\n");

#pragma omp dispatch device(devNum)
        status_create = mkl_sparse_s_create_csr(&csrA_gpu, indexing, M, N,
                                                row_index, row_index + 1, columns, values);

        printf("mkl_sparse_set_sv_hint() ... \n");
#pragma omp dispatch device(devNum) nowait
        status_hint = mkl_sparse_set_sv_hint(csrA_gpu, opA, descrA, 1);
#pragma omp taskwait

        printf("mkl_sparse_optimize() ... \n");
        // NOTE: users can run mkl_sparse_optimize offload functions in async execution,
        // but users should also handle the dependency between the optimize() call and
        // the computation calls, i.e. trsv() here.
#pragma omp dispatch device(devNum) nowait
        status_opt = mkl_sparse_optimize(csrA_gpu);
#pragma omp taskwait
        printf("Compute mkl_sparse_s_trsv via omp_offload\n");

#pragma omp dispatch device(devNum) nowait
        status_trsv1 = mkl_sparse_s_trsv(opA, alpha, csrA_gpu, descrA, b, z1);

#pragma omp dispatch device(devNum) nowait
        status_trsv2 = mkl_sparse_s_trsv(opA, alpha, csrA_gpu, descrA, b2, z2);
#pragma omp taskwait
        printf("Destroy the CSR matrix via omp_offload\n");

#pragma omp dispatch device(devNum)
        status_destroy = mkl_sparse_destroy(csrA_gpu);

        printf("End of mkl_sparse_destroy omp offload call.\n");
    }
    printf("End all offload function calls.\n"); fflush(stdout);

    int flps_per_value = 2 * ((NNZ / M) + 1);
    int status1        = 0;
    int status2        = 0;

    int status_offload = status_create | status_hint | status_opt | status_trsv1 | status_trsv2 | status_destroy;
    if (status_offload != 0) {
        printf("\tERROR: status_create = %d, status_hint = %d, status_opt = %d, "
               "status_trsv1 = %d, status_trsv2 = %d, status_destroy = %d\n",
               status_create, status_hint, status_opt, status_trsv1, status_trsv2, status_destroy);
        fflush(stdout);

        goto cleanup;
    }

    printf("   OUTPUT DATA FOR mkl_sparse_s_trsv_omp_offload async execution\n");
    // z should be equal { 18.0, 5.0, -27.0, 4.0, 1.0 }
    for (i = 0; i < N; i++) {
        printf("%7.4f\n", z1[i]);
    }
    printf("---------------------------------------------------\n");
    fflush(stdout);

    status1 = validation_result_float(x, z1, N, flps_per_value);
    status2 = validation_result_float(x, z2, N, flps_per_value);

cleanup:
    free_allocated_memories(pointer_array, num_pointers);

    const int status_all = status1 | status2 | status_offload;
    printf("Test %s\n", status_all == 0 ? "PASSED" : "FAILED");
    fflush(stdout);

    return status_all;
}
