# Copyright (C) 2025 Intel Corporation
# SPDX-License-Identifier: MIT

import os.path
import platform
import sys
import subprocess
from abc import ABC

from tools import project_root_path
from tools.common.command import Command, CommandException
from tools.common.python_version import PythonVersion, UnsupportedPythonVersionError


class Installer (ABC):
    SUPPORTED_OSES = ['Windows', 'Linux']
    RUSTY_WHEEL = 'pyrust_xlsxwriter'
    RUSTY_WHEEL_VERSION = '0.1.0'
    RUSTY_WHEELR_PREFIX = f'{RUSTY_WHEEL}-{RUSTY_WHEEL_VERSION}'
    BIN_PATH = os.path.join(project_root_path, 'bin')
    EXPECTED_OPTIONAL_DEPENDENCIES = ['tdigest']

    def __init__(self, verbose=True):
        self.__os = self._get_os()
        self.__python_version = PythonVersion.get_version()
        self.__verbose = verbose
        self.__python_path: str = sys.executable
        self._additional_installation_args = ['.']

    @property
    def verbose(self):
        return self.__verbose

    @property
    def python_path(self):
        return self.__python_path

    @staticmethod
    def pip_list():
        return Command.run(['pip', 'list'])

    def update_python_path(self, python_path: str):
        self.__python_path = python_path
        if platform.system() == 'Windows':
            os.environ['PATH'] = f'{os.path.dirname(self.python_path)};{os.environ["PATH"]}'

    def _validate_optional_dependencies(self, optional_dependencies):
        if not all([dep in self.EXPECTED_OPTIONAL_DEPENDENCIES for dep in optional_dependencies]):
            raise ValueError(f'Invalid optional dependencies. Expected: {self.EXPECTED_OPTIONAL_DEPENDENCIES}. Got: {optional_dependencies}')

    @staticmethod
    def _get_os():
        os_type = platform.system()
        if os_type in Installer.SUPPORTED_OSES:
            return os_type
        raise ValueError(f'Unsupported OS. Supported OSes: {Installer.SUPPORTED_OSES}')

    def _get_bin_directory(self):
        bin_directory = os.path.join(self.BIN_PATH, self.__os)
        if not os.path.exists(bin_directory):
            raise FileNotFoundError(f'No wheel binaries found for pyrust_xlsxwriter. ')
        return bin_directory

    def _get_rusty_wheel(self):
        bin_directory = self._get_bin_directory()
        python_version = self.__python_version.replace('.', '')
        for file in os.listdir(bin_directory):
            if file.startswith(self.RUSTY_WHEELR_PREFIX) and file.endswith('.whl') and python_version in file:
                return os.path.abspath(os.path.join(bin_directory, file))
        raise FileNotFoundError(f'Wheel file for {self.RUSTY_WHEEL} not found for python version {self.__python_version} on {self.__os}')

    def _install_rusty_wheel(self):
        try:
            Command.run_with_real_time_output([self.python_path, '-m', 'pip', 'install', '--force-reinstall', self._get_rusty_wheel()])
        except (subprocess.CalledProcessError, FileNotFoundError, UnsupportedPythonVersionError, CommandException) as e:
            print(f'Failed to install {self.RUSTY_WHEEL}. Error: {e}.')
            print(f'The metric post processor will use the slower Python-based Excel writer instead of the faster Rust-based Excel writer. '
                  f'This will affect post processor performance, but not the results. Please contact the development team with the details of this error for assistance.')

    def _install_dependencies(self):
        import certifi
        old_certifi_path = self._get_saved_certifi_path()
        os.environ['SSL_CERT_FILE'] = certifi.where()
        cmd = [self.python_path, '-m', 'pip', 'install'] + self._additional_installation_args
        print(f'Installing dependencies with command: {" ".join(cmd)}')
        Command.run_with_real_time_output(cmd)
        self.restore_certifi_path(old_certifi_path)

    @staticmethod
    def _get_saved_certifi_path():
        return os.environ.get('SSL_CERT_FILE')

    @staticmethod
    def restore_certifi_path(old_certifi_path):
        if old_certifi_path:
            os.environ['SSL_CERT_FILE'] = old_certifi_path
        else:
            del os.environ['SSL_CERT_FILE']

    def install(self):
        from tools.common.virtual_environment import VirtualEnvironment
        venv_path = VirtualEnvironment.get_active_virtualenv_path()
        print(f'Installing dependencies in {venv_path}...')
        self._install_rusty_wheel()
        self._install_dependencies()
        print(f'Dependencies installed successfully in {venv_path}.')
        print(f'Use \'{VirtualEnvironment.get_venv_activate_command(venv_path)}\' to activate the virtual environment if it not already active.')
        print(f'Use \'deactivate\' to deactivate an active virtual environment when you are done.')

class BasicProductionInstaller(Installer):
    def __init__(self, verbose=True):
        super().__init__(verbose)


class FullProductionInstaller(Installer):
    def __init__(self, verbose=True):
        super().__init__(verbose)
        self._additional_installation_args = ['.[percentile]']


class InstallerFactory:
    @staticmethod
    def create_installer(support_percentile=False, verbose=True):
        if support_percentile:
            return FullProductionInstaller(verbose)
        return BasicProductionInstaller(verbose)


