﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sagemaker/SageMakerRequest.h>
#include <aws/sagemaker/SageMaker_EXPORTS.h>
#include <aws/sagemaker/model/MonitoringAlertHistorySortKey.h>
#include <aws/sagemaker/model/MonitoringAlertStatus.h>
#include <aws/sagemaker/model/SortOrder.h>

#include <utility>

namespace Aws {
namespace SageMaker {
namespace Model {

/**
 */
class ListMonitoringAlertHistoryRequest : public SageMakerRequest {
 public:
  AWS_SAGEMAKER_API ListMonitoringAlertHistoryRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListMonitoringAlertHistory"; }

  AWS_SAGEMAKER_API Aws::String SerializePayload() const override;

  AWS_SAGEMAKER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of a monitoring schedule.</p>
   */
  inline const Aws::String& GetMonitoringScheduleName() const { return m_monitoringScheduleName; }
  inline bool MonitoringScheduleNameHasBeenSet() const { return m_monitoringScheduleNameHasBeenSet; }
  template <typename MonitoringScheduleNameT = Aws::String>
  void SetMonitoringScheduleName(MonitoringScheduleNameT&& value) {
    m_monitoringScheduleNameHasBeenSet = true;
    m_monitoringScheduleName = std::forward<MonitoringScheduleNameT>(value);
  }
  template <typename MonitoringScheduleNameT = Aws::String>
  ListMonitoringAlertHistoryRequest& WithMonitoringScheduleName(MonitoringScheduleNameT&& value) {
    SetMonitoringScheduleName(std::forward<MonitoringScheduleNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of a monitoring alert.</p>
   */
  inline const Aws::String& GetMonitoringAlertName() const { return m_monitoringAlertName; }
  inline bool MonitoringAlertNameHasBeenSet() const { return m_monitoringAlertNameHasBeenSet; }
  template <typename MonitoringAlertNameT = Aws::String>
  void SetMonitoringAlertName(MonitoringAlertNameT&& value) {
    m_monitoringAlertNameHasBeenSet = true;
    m_monitoringAlertName = std::forward<MonitoringAlertNameT>(value);
  }
  template <typename MonitoringAlertNameT = Aws::String>
  ListMonitoringAlertHistoryRequest& WithMonitoringAlertName(MonitoringAlertNameT&& value) {
    SetMonitoringAlertName(std::forward<MonitoringAlertNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The field used to sort results. The default is <code>CreationTime</code>.</p>
   */
  inline MonitoringAlertHistorySortKey GetSortBy() const { return m_sortBy; }
  inline bool SortByHasBeenSet() const { return m_sortByHasBeenSet; }
  inline void SetSortBy(MonitoringAlertHistorySortKey value) {
    m_sortByHasBeenSet = true;
    m_sortBy = value;
  }
  inline ListMonitoringAlertHistoryRequest& WithSortBy(MonitoringAlertHistorySortKey value) {
    SetSortBy(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sort order, whether <code>Ascending</code> or <code>Descending</code>, of
   * the alert history. The default is <code>Descending</code>.</p>
   */
  inline SortOrder GetSortOrder() const { return m_sortOrder; }
  inline bool SortOrderHasBeenSet() const { return m_sortOrderHasBeenSet; }
  inline void SetSortOrder(SortOrder value) {
    m_sortOrderHasBeenSet = true;
    m_sortOrder = value;
  }
  inline ListMonitoringAlertHistoryRequest& WithSortOrder(SortOrder value) {
    SetSortOrder(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If the result of the previous <code>ListMonitoringAlertHistory</code> request
   * was truncated, the response includes a <code>NextToken</code>. To retrieve the
   * next set of alerts in the history, use the token in the next request.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListMonitoringAlertHistoryRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to display. The default is 100.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListMonitoringAlertHistoryRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only alerts created on or before the specified
   * time.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTimeBefore() const { return m_creationTimeBefore; }
  inline bool CreationTimeBeforeHasBeenSet() const { return m_creationTimeBeforeHasBeenSet; }
  template <typename CreationTimeBeforeT = Aws::Utils::DateTime>
  void SetCreationTimeBefore(CreationTimeBeforeT&& value) {
    m_creationTimeBeforeHasBeenSet = true;
    m_creationTimeBefore = std::forward<CreationTimeBeforeT>(value);
  }
  template <typename CreationTimeBeforeT = Aws::Utils::DateTime>
  ListMonitoringAlertHistoryRequest& WithCreationTimeBefore(CreationTimeBeforeT&& value) {
    SetCreationTimeBefore(std::forward<CreationTimeBeforeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that returns only alerts created on or after the specified time.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTimeAfter() const { return m_creationTimeAfter; }
  inline bool CreationTimeAfterHasBeenSet() const { return m_creationTimeAfterHasBeenSet; }
  template <typename CreationTimeAfterT = Aws::Utils::DateTime>
  void SetCreationTimeAfter(CreationTimeAfterT&& value) {
    m_creationTimeAfterHasBeenSet = true;
    m_creationTimeAfter = std::forward<CreationTimeAfterT>(value);
  }
  template <typename CreationTimeAfterT = Aws::Utils::DateTime>
  ListMonitoringAlertHistoryRequest& WithCreationTimeAfter(CreationTimeAfterT&& value) {
    SetCreationTimeAfter(std::forward<CreationTimeAfterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that retrieves only alerts with a specific status.</p>
   */
  inline MonitoringAlertStatus GetStatusEquals() const { return m_statusEquals; }
  inline bool StatusEqualsHasBeenSet() const { return m_statusEqualsHasBeenSet; }
  inline void SetStatusEquals(MonitoringAlertStatus value) {
    m_statusEqualsHasBeenSet = true;
    m_statusEquals = value;
  }
  inline ListMonitoringAlertHistoryRequest& WithStatusEquals(MonitoringAlertStatus value) {
    SetStatusEquals(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_monitoringScheduleName;

  Aws::String m_monitoringAlertName;

  MonitoringAlertHistorySortKey m_sortBy{MonitoringAlertHistorySortKey::NOT_SET};

  SortOrder m_sortOrder{SortOrder::NOT_SET};

  Aws::String m_nextToken;

  int m_maxResults{0};

  Aws::Utils::DateTime m_creationTimeBefore{};

  Aws::Utils::DateTime m_creationTimeAfter{};

  MonitoringAlertStatus m_statusEquals{MonitoringAlertStatus::NOT_SET};
  bool m_monitoringScheduleNameHasBeenSet = false;
  bool m_monitoringAlertNameHasBeenSet = false;
  bool m_sortByHasBeenSet = false;
  bool m_sortOrderHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_creationTimeBeforeHasBeenSet = false;
  bool m_creationTimeAfterHasBeenSet = false;
  bool m_statusEqualsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SageMaker
}  // namespace Aws
