/* Copyright (C) 2020 Google, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <assert.h>
#include <stdbool.h>
#include <stdint.h>
#include <util/bitset.h>
#include <util/log.h>



#define BITMASK_WORDS BITSET_WORDS(128)

typedef struct {
    BITSET_WORD bitset[BITMASK_WORDS];
} bitmask_t;

static inline uint64_t
bitmask_to_uint64_t(bitmask_t mask)
{
    return ((uint64_t)mask.bitset[1] << 32) | mask.bitset[0];
}

static inline bitmask_t
uint64_t_to_bitmask(uint64_t val)
{
    bitmask_t mask = {
        .bitset[0] = val & 0xffffffff,
        .bitset[1] = (val >> 32) & 0xffffffff,
    };

    return mask;
}

static inline void
store_instruction(BITSET_WORD *dst, bitmask_t instr)
{
    *(dst + 0) = instr.bitset[0];
    *(dst + 1) = instr.bitset[1];
    *(dst + 2) = instr.bitset[2];
    *(dst + 3) = instr.bitset[3];
}

/**
 * Opaque type from the PoV of generated code, but allows state to be passed
 * thru to the hand written helpers used by the generated code.
 */
struct encode_state;

/**
 * Allows to use gpu_id in expr functions
 */
#define ISA_GPU_ID() s->gen

struct bitset_params;

static bitmask_t
pack_field(unsigned low, unsigned high, int64_t val, bool is_signed)
{
   bitmask_t field, mask;

   if (is_signed) {
      /* NOTE: Don't assume val is already sign-extended to 64b,
       * just check that the bits above the valid range are either
       * all zero or all one:
       */
      assert(!(( val & ~BITFIELD64_MASK(1 + high - low)) &&
               (~val & ~BITFIELD64_MASK(1 + high - low))));
   } else {
      assert(!(val & ~BITFIELD64_MASK(1 + high - low)));
   }

   BITSET_ZERO(field.bitset);

   if (!val)
      return field;

   BITSET_ZERO(mask.bitset);
   BITSET_SET_RANGE(mask.bitset, 0, high - low);

   field = uint64_t_to_bitmask(val);
   BITSET_AND(field.bitset, field.bitset, mask.bitset);
   BITSET_SHL(field.bitset, low);

   return field;
}

/*
 * Forward-declarations (so we don't have to figure out which order to
 * emit various encoders when they have reference each other)
 */

static bitmask_t encode__instruction_dst(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_dst * src);
static bitmask_t encode__instruction(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static bitmask_t encode__src_swizzle(struct encode_state *s, const struct bitset_params *p, const uint8_t src);
static bitmask_t encode__instruction_src(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static bitmask_t encode__left_shift(struct encode_state *s, const struct bitset_params *p, const unsigned int src);


/*
 * Expression evaluators:
 */

struct bitset_params {
   int64_t DST_USE;
   int64_t COMPS;
   int64_t SRC_REG;
   int64_t SRC_AMODE;
   int64_t SRC_RGROUP;
};






static inline int64_t __instruction_dst_expr_anon_0(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_dst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr___instruction_has_src0_src1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr___instruction_has_src0(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr___instruction_has_src2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr___instruction_has_src0(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr___instruction_has_src0_src1_src2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr___instruction_has_src0(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr___instruction_has_src0_src1_src2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_14(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_15(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_14(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_15(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src);
static inline int64_t __instruction_src_expr_anon_4(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static inline int64_t __instruction_src_expr_anon_5(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static inline int64_t __instruction_src_expr_anon_6(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static inline int64_t __instruction_src_expr_anon_7(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static inline int64_t __instruction_src_expr_anon_8(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static inline int64_t __instruction_src_expr_anon_9(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static inline int64_t __instruction_src_expr_anon_10(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static inline int64_t __instruction_src_expr_anon_11(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static inline int64_t __instruction_src_expr_anon_3(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src);
static inline int64_t __left_shift_expr_anon_13(struct encode_state *s, const struct bitset_params *p, const unsigned int src);



          
static inline int64_t
__instruction_dst_expr_anon_0(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_dst * src)
{
    int64_t DST_USE;

          DST_USE = p->DST_USE;
    return DST_USE == 0;
}



          
static inline int64_t
__instruction_expr_anon_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
    int64_t LOW_HALF;
    int64_t HIGH_HALF;

          LOW_HALF = (src->thread & 0x1);

          HIGH_HALF = (src->thread & 0x2) > 1;
    return HIGH_HALF << 1 | LOW_HALF;
}


          
static inline int64_t
__instruction_expr_anon_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
    int64_t TYPE_BIT2;
    int64_t TYPE_BIT01;

          TYPE_BIT2 = (src->type & 0x4) > 2;

          TYPE_BIT01 = src->type & 0x3;
    return TYPE_BIT2 << 2 | TYPE_BIT01;
}


          
static inline int64_t
__instruction_expr___instruction_has_src0_src1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
    int64_t SRC1_USE;
    int64_t SRC0_USE;

          SRC1_USE = src->src[1].use;

          SRC0_USE = src->src[0].use;
    return (SRC0_USE != 0) && (SRC1_USE != 0);
}


          
static inline int64_t
__instruction_expr___instruction_has_src0(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
    int64_t SRC1_USE;
    int64_t SRC0_USE;

          SRC1_USE = src->src[1].use;

          SRC0_USE = src->src[0].use;
    return (SRC0_USE != 0) && (SRC1_USE == 0);
}


          
static inline int64_t
__instruction_expr___instruction_has_src2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
    int64_t SRC2_USE;

          SRC2_USE = src->src[2].use;
    return (SRC2_USE != 0);
}


          
static inline int64_t
__instruction_expr_anon_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
    int64_t LOW_HALF;
    int64_t HIGH_HALF;

          LOW_HALF = (src->thread & 0x1);

          HIGH_HALF = (src->thread & 0x2) > 1;
    return HIGH_HALF << 1 | LOW_HALF;
}


          
static inline int64_t
__instruction_expr___instruction_has_src0_src1_src2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
    int64_t SRC2_USE;
    int64_t SRC1_USE;
    int64_t SRC0_USE;

          SRC2_USE = src->src[2].use;

          SRC1_USE = src->src[1].use;

          SRC0_USE = src->src[0].use;
    return (SRC0_USE != 0) && (SRC1_USE != 0)  && (SRC2_USE != 0);
}


          
static inline int64_t
__instruction_expr_anon_14(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
    int64_t LOW_HALF;
    int64_t HIGH_HALF;

          LOW_HALF = (src->thread & 0x1);

          HIGH_HALF = (src->thread & 0x2) > 1;
    return HIGH_HALF << 1 | LOW_HALF;
}


          
static inline int64_t
__instruction_expr_anon_15(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
    int64_t LOW_HALF;
    int64_t HIGH_HALF;

          LOW_HALF = (src->thread & 0x1);

          HIGH_HALF = (src->thread & 0x2) > 1;
    return HIGH_HALF << 1 | LOW_HALF;
}




          
static inline int64_t
__instruction_src_expr_anon_4(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
    int64_t SRC_RGROUP;
    int64_t IMMED_TYPE;

          SRC_RGROUP = p->SRC_RGROUP;

          IMMED_TYPE = __instruction_src_expr_anon_3(s, p, src);
    return ((SRC_RGROUP == 7) & (IMMED_TYPE == 0));
}


          
static inline int64_t
__instruction_src_expr_anon_5(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
    int64_t SRC_ABS;
    int64_t SRC_REG;
    int64_t SRC_NEG;
    int64_t SRC_SWIZ;
    int64_t SRC_AMODE;

          SRC_ABS = src->abs;

          SRC_REG = p->SRC_REG;

          SRC_NEG = src->neg;

          { 
 struct bitset_params bp = {
 };

          const bitmask_t tmp = encode__src_swizzle(s, &bp, src->swiz);
          SRC_SWIZ = bitmask_to_uint64_t(tmp);
          }

          SRC_AMODE = p->SRC_AMODE;
    return ((((SRC_AMODE & 0x1) << 19) |
				   (SRC_ABS << 18) |
				   (SRC_NEG << 17) |
				   (SRC_SWIZ << 9) |
				   SRC_REG) << 12);
}


          
static inline int64_t
__instruction_src_expr_anon_6(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
    int64_t SRC_RGROUP;
    int64_t IMMED_TYPE;

          SRC_RGROUP = p->SRC_RGROUP;

          IMMED_TYPE = __instruction_src_expr_anon_3(s, p, src);
    return ((SRC_RGROUP == 7) & (IMMED_TYPE == 1));
}


          
static inline int64_t
__instruction_src_expr_anon_7(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
    int64_t SRC_ABS;
    int64_t SRC_REG;
    int64_t SRC_NEG;
    int64_t SRC_SWIZ;
    int64_t SRC_AMODE;

          SRC_ABS = src->abs;

          SRC_REG = p->SRC_REG;

          SRC_NEG = src->neg;

          { 
 struct bitset_params bp = {
 };

          const bitmask_t tmp = encode__src_swizzle(s, &bp, src->swiz);
          SRC_SWIZ = bitmask_to_uint64_t(tmp);
          }

          SRC_AMODE = p->SRC_AMODE;
    return (((SRC_AMODE & 0x1) << 19) |
				  (SRC_ABS << 18) |
				  (SRC_NEG << 17) |
				  (SRC_SWIZ << 9) |
				   SRC_REG);
}


          
static inline int64_t
__instruction_src_expr_anon_8(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
    int64_t SRC_RGROUP;
    int64_t IMMED_TYPE;

          SRC_RGROUP = p->SRC_RGROUP;

          IMMED_TYPE = __instruction_src_expr_anon_3(s, p, src);
    return ((SRC_RGROUP == 7) & (IMMED_TYPE == 2));
}


          
static inline int64_t
__instruction_src_expr_anon_9(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
    int64_t SRC_ABS;
    int64_t SRC_REG;
    int64_t SRC_NEG;
    int64_t SRC_SWIZ;
    int64_t SRC_AMODE;

          SRC_ABS = src->abs;

          SRC_REG = p->SRC_REG;

          SRC_NEG = src->neg;

          { 
 struct bitset_params bp = {
 };

          const bitmask_t tmp = encode__src_swizzle(s, &bp, src->swiz);
          SRC_SWIZ = bitmask_to_uint64_t(tmp);
          }

          SRC_AMODE = p->SRC_AMODE;
    return (((SRC_AMODE & 0x1) << 19) |
				  (SRC_ABS << 18) |
				  (SRC_NEG << 17) |
				  (SRC_SWIZ << 9) |
				   SRC_REG);
}


          
static inline int64_t
__instruction_src_expr_anon_10(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
    int64_t SRC_RGROUP;
    int64_t IMMED_TYPE;

          SRC_RGROUP = p->SRC_RGROUP;

          IMMED_TYPE = __instruction_src_expr_anon_3(s, p, src);
    return ((SRC_RGROUP == 7) & (IMMED_TYPE == 3));
}


          
static inline int64_t
__instruction_src_expr_anon_11(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
    int64_t SRC_ABS;
    int64_t SRC_REG;
    int64_t SRC_NEG;
    int64_t SRC_SWIZ;
    int64_t SRC_AMODE;

          SRC_ABS = src->abs;

          SRC_REG = p->SRC_REG;

          SRC_NEG = src->neg;

          { 
 struct bitset_params bp = {
 };

          const bitmask_t tmp = encode__src_swizzle(s, &bp, src->swiz);
          SRC_SWIZ = bitmask_to_uint64_t(tmp);
          }

          SRC_AMODE = p->SRC_AMODE;
    return (((SRC_AMODE & 0x1) << 19) |
				  (SRC_ABS << 18) |
				  (SRC_NEG << 17) |
				  (SRC_SWIZ << 9) |
				   SRC_REG);
}


          
static inline int64_t
__instruction_src_expr_anon_3(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
    int64_t SRC_AMODE;

          SRC_AMODE = p->SRC_AMODE;
    return (SRC_AMODE >> 1);
}



          
static inline int64_t
__left_shift_expr_anon_13(struct encode_state *s, const struct bitset_params *p, const unsigned int src)
{
    int64_t LEFT_SHIFT;

          LEFT_SHIFT = src;
    return LEFT_SHIFT != 0;
}



/*
 * The actual encoder definitions
 */



static bitmask_t
snippet__instruction_dst_0(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_dst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    if (__instruction_dst_expr_anon_0(s, p, src)) {


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    {


           
    {

             fld = src->reg;
             const bitmask_t packed = pack_field(3, 9, fld, false);  /* REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->amode;
             const bitmask_t packed = pack_field(0, 2, fld, false);  /* AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = p->COMPS;
             const bitmask_t packed = pack_field(10, 13, fld, false);  /* COMPS */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}


static bitmask_t
encode__instruction_dst(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_dst * src)
{

      bitmask_t val = uint64_t_to_bitmask(0x0);
      BITSET_OR(val.bitset, val.bitset, snippet__instruction_dst_0(s, p, src).bitset);
      return val;
}


static bitmask_t
snippet__instruction_0(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst_full;
             const bitmask_t packed = pack_field(127, 127, fld, false);  /* DST_FULL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}



static bitmask_t
snippet__instruction_1(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->dst_full;
             const bitmask_t packed = pack_field(127, 127, fld, false);  /* DST_FULL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->sat;
             const bitmask_t packed = pack_field(11, 11, fld, false);  /* SAT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->skphp;
             const bitmask_t packed = pack_field(39, 39, fld, false);  /* SKPHP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->unk;
             const bitmask_t packed = pack_field(27, 27, fld, false);  /* UNK */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->rounding;
             const bitmask_t packed = pack_field(32, 33, fld, false);  /* RMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC2_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC2_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC2_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(110, 119, fld, false);  /* SRC2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(100, 108, fld, false);  /* SRC2_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(121, 123, fld, false);  /* SRC2_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(124, 126, fld, false);  /* SRC2_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}



static bitmask_t
snippet__instruction_2(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->dst_full;
             const bitmask_t packed = pack_field(127, 127, fld, false);  /* DST_FULL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->sat;
             const bitmask_t packed = pack_field(11, 11, fld, false);  /* SAT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->skphp;
             const bitmask_t packed = pack_field(39, 39, fld, false);  /* SKPHP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->unk;
             const bitmask_t packed = pack_field(27, 27, fld, false);  /* UNK */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->rounding;
             const bitmask_t packed = pack_field(32, 33, fld, false);  /* RMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[2].reg,  /* SRC2_REG */
    .SRC_AMODE = src->src[2].amode,  /* SRC2_AMODE */
    .SRC_RGROUP = src->src[2].rgroup,  /* SRC2_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[2]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(110, 119, fld, false);  /* SRC2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].reg;
             const bitmask_t packed = pack_field(100, 108, fld, false);  /* SRC2_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].amode;
             const bitmask_t packed = pack_field(121, 123, fld, false);  /* SRC2_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].rgroup;
             const bitmask_t packed = pack_field(124, 126, fld, false);  /* SRC2_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}



static bitmask_t
snippet__instruction_3(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->dst_full;
             const bitmask_t packed = pack_field(127, 127, fld, false);  /* DST_FULL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->sat;
             const bitmask_t packed = pack_field(11, 11, fld, false);  /* SAT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->skphp;
             const bitmask_t packed = pack_field(39, 39, fld, false);  /* SKPHP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->unk;
             const bitmask_t packed = pack_field(27, 27, fld, false);  /* UNK */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->rounding;
             const bitmask_t packed = pack_field(32, 33, fld, false);  /* RMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}







static bitmask_t
snippet__instruction_4(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->dst_full;
             const bitmask_t packed = pack_field(127, 127, fld, false);  /* DST_FULL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->sat;
             const bitmask_t packed = pack_field(11, 11, fld, false);  /* SAT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->skphp;
             const bitmask_t packed = pack_field(39, 39, fld, false);  /* SKPHP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->unk;
             const bitmask_t packed = pack_field(27, 27, fld, false);  /* UNK */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->rounding;
             const bitmask_t packed = pack_field(32, 33, fld, false);  /* RMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC2_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC2_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC2_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(110, 119, fld, false);  /* SRC2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(100, 108, fld, false);  /* SRC2_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(121, 123, fld, false);  /* SRC2_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(124, 126, fld, false);  /* SRC2_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}











static bitmask_t
snippet__instruction_5(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->imm;
             const bitmask_t packed = pack_field(103, 117, fld, false);  /* TARGET */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}





static bitmask_t
snippet__instruction_6(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->imm;
             const bitmask_t packed = pack_field(103, 117, fld, false);  /* TARGET */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}



static bitmask_t
snippet__instruction_7(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->imm;
             const bitmask_t packed = pack_field(103, 117, fld, false);  /* TARGET */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}



static bitmask_t
snippet__instruction_8(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    if (__instruction_expr___instruction_has_src0_src1(s, p, src)) {


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->rounding;
             const bitmask_t packed = pack_field(32, 33, fld, false);  /* RMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].use;
             const bitmask_t packed = pack_field(70, 70, fld, false);  /* SRC1_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].use;
             const bitmask_t packed = pack_field(43, 43, fld, false);  /* SRC0_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    if (__instruction_expr___instruction_has_src0(s, p, src)) {


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->rounding;
             const bitmask_t packed = pack_field(32, 33, fld, false);  /* RMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].use;
             const bitmask_t packed = pack_field(70, 70, fld, false);  /* SRC1_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].use;
             const bitmask_t packed = pack_field(43, 43, fld, false);  /* SRC0_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    {


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->rounding;
             const bitmask_t packed = pack_field(32, 33, fld, false);  /* RMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst_full;
             const bitmask_t packed = pack_field(127, 127, fld, false);  /* DST_FULL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].use;
             const bitmask_t packed = pack_field(70, 70, fld, false);  /* SRC1_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].use;
             const bitmask_t packed = pack_field(43, 43, fld, false);  /* SRC0_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}



static bitmask_t
snippet__instruction_9(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    if (__instruction_expr___instruction_has_src2(s, p, src)) {


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->tex.id;
             const bitmask_t packed = pack_field(27, 31, fld, false);  /* TEX_ID */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->tex.swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(35, 42, fld, false);  /* TEX_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[2].reg,  /* SRC2_REG */
    .SRC_AMODE = src->src[2].amode,  /* SRC2_AMODE */
    .SRC_RGROUP = src->src[2].rgroup,  /* SRC2_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[2]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(110, 119, fld, false);  /* SRC2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].use;
             const bitmask_t packed = pack_field(99, 99, fld, false);  /* SRC2_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].reg;
             const bitmask_t packed = pack_field(100, 108, fld, false);  /* SRC2_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].amode;
             const bitmask_t packed = pack_field(121, 123, fld, false);  /* SRC2_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].rgroup;
             const bitmask_t packed = pack_field(124, 126, fld, false);  /* SRC2_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    {


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->tex.id;
             const bitmask_t packed = pack_field(27, 31, fld, false);  /* TEX_ID */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->tex.swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(35, 42, fld, false);  /* TEX_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].use;
             const bitmask_t packed = pack_field(99, 99, fld, false);  /* SRC2_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}



static bitmask_t
snippet__instruction_10(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    if (__instruction_expr___instruction_has_src0(s, p, src)) {


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->tex.id;
             const bitmask_t packed = pack_field(27, 31, fld, false);  /* TEX_ID */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->tex.swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(35, 42, fld, false);  /* TEX_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].use;
             const bitmask_t packed = pack_field(70, 70, fld, false);  /* SRC1_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].use;
             const bitmask_t packed = pack_field(43, 43, fld, false);  /* SRC0_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].use;
             const bitmask_t packed = pack_field(99, 99, fld, false);  /* SRC2_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    if (__instruction_expr___instruction_has_src0_src1_src2(s, p, src)) {


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->tex.id;
             const bitmask_t packed = pack_field(27, 31, fld, false);  /* TEX_ID */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->tex.swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(35, 42, fld, false);  /* TEX_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[2].reg,  /* SRC2_REG */
    .SRC_AMODE = src->src[2].amode,  /* SRC2_AMODE */
    .SRC_RGROUP = src->src[2].rgroup,  /* SRC2_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[2]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(110, 119, fld, false);  /* SRC2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].use;
             const bitmask_t packed = pack_field(70, 70, fld, false);  /* SRC1_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].use;
             const bitmask_t packed = pack_field(43, 43, fld, false);  /* SRC0_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].use;
             const bitmask_t packed = pack_field(99, 99, fld, false);  /* SRC2_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].reg;
             const bitmask_t packed = pack_field(100, 108, fld, false);  /* SRC2_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].amode;
             const bitmask_t packed = pack_field(121, 123, fld, false);  /* SRC2_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].rgroup;
             const bitmask_t packed = pack_field(124, 126, fld, false);  /* SRC2_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    {


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->tex.id;
             const bitmask_t packed = pack_field(27, 31, fld, false);  /* TEX_ID */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->tex.swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(35, 42, fld, false);  /* TEX_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].use;
             const bitmask_t packed = pack_field(70, 70, fld, false);  /* SRC1_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].use;
             const bitmask_t packed = pack_field(43, 43, fld, false);  /* SRC0_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].use;
             const bitmask_t packed = pack_field(99, 99, fld, false);  /* SRC2_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}



static bitmask_t
snippet__instruction_11(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->tex.id;
             const bitmask_t packed = pack_field(27, 31, fld, false);  /* TEX_ID */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->tex.swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(35, 42, fld, false);  /* TEX_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[2].reg,  /* SRC2_REG */
    .SRC_AMODE = src->src[2].amode,  /* SRC2_AMODE */
    .SRC_RGROUP = src->src[2].rgroup,  /* SRC2_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[2]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(110, 119, fld, false);  /* SRC2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].reg;
             const bitmask_t packed = pack_field(100, 108, fld, false);  /* SRC2_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].amode;
             const bitmask_t packed = pack_field(121, 123, fld, false);  /* SRC2_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].rgroup;
             const bitmask_t packed = pack_field(124, 126, fld, false);  /* SRC2_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}














static bitmask_t
snippet__instruction_12(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->dst_full;
             const bitmask_t packed = pack_field(127, 127, fld, false);  /* DST_FULL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->sat;
             const bitmask_t packed = pack_field(11, 11, fld, false);  /* SAT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->skphp;
             const bitmask_t packed = pack_field(39, 39, fld, false);  /* SKPHP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->unk;
             const bitmask_t packed = pack_field(27, 27, fld, false);  /* UNK */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->rounding;
             const bitmask_t packed = pack_field(32, 33, fld, false);  /* RMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}






static bitmask_t
snippet__instruction_13(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->sat;
             const bitmask_t packed = pack_field(11, 11, fld, false);  /* SAT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->skphp;
             const bitmask_t packed = pack_field(39, 39, fld, false);  /* SKPHP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->denorm;
             const bitmask_t packed = pack_field(42, 42, fld, false);  /* DENORM */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->local;
             const bitmask_t packed = pack_field(40, 40, fld, false);  /* LOCAL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__left_shift(s, &bp, src->left_shift);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(35, 37, fld, false);  /* LEFT_SHIFT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}



static bitmask_t
snippet__instruction_14(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->sat;
             const bitmask_t packed = pack_field(11, 11, fld, false);  /* SAT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->skphp;
             const bitmask_t packed = pack_field(39, 39, fld, false);  /* SKPHP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->denorm;
             const bitmask_t packed = pack_field(42, 42, fld, false);  /* DENORM */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->local;
             const bitmask_t packed = pack_field(40, 40, fld, false);  /* LOCAL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__left_shift(s, &bp, src->left_shift);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(35, 37, fld, false);  /* LEFT_SHIFT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.write_mask;
             const bitmask_t packed = pack_field(23, 26, fld, false);  /* COMPS */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[2].reg,  /* SRC2_REG */
    .SRC_AMODE = src->src[2].amode,  /* SRC2_AMODE */
    .SRC_RGROUP = src->src[2].rgroup,  /* SRC2_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[2]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(110, 119, fld, false);  /* SRC2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].reg;
             const bitmask_t packed = pack_field(100, 108, fld, false);  /* SRC2_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].amode;
             const bitmask_t packed = pack_field(121, 123, fld, false);  /* SRC2_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[2].rgroup;
             const bitmask_t packed = pack_field(124, 126, fld, false);  /* SRC2_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}
























static bitmask_t
snippet__instruction_15(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = src->dst_full;
             const bitmask_t packed = pack_field(127, 127, fld, false);  /* DST_FULL */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->sat;
             const bitmask_t packed = pack_field(11, 11, fld, false);  /* SAT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->cond;
             const bitmask_t packed = pack_field(6, 10, fld, false);  /* COND */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->skphp;
             const bitmask_t packed = pack_field(39, 39, fld, false);  /* SKPHP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->unk;
             const bitmask_t packed = pack_field(27, 27, fld, false);  /* UNK */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = !src->pmode;
             const bitmask_t packed = pack_field(34, 34, fld, false);  /* PMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->rounding;
             const bitmask_t packed = pack_field(32, 33, fld, false);  /* RMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC2_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC2_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC2_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(110, 119, fld, false);  /* SRC2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->type & 0x4) > 2;
             const bitmask_t packed = pack_field(53, 53, fld, false);  /* TYPE_BIT2 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->type & 0x3;
             const bitmask_t packed = pack_field(94, 95, fld, true);  /* TYPE_BIT01 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(100, 108, fld, false);  /* SRC2_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(121, 123, fld, false);  /* SRC2_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(124, 126, fld, false);  /* SRC2_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}






















static bitmask_t
snippet__instruction_16(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             { 
 struct bitset_params bp = {
    .DST_USE = src->dst.use,  /* DST_USE */
    .COMPS = src->dst.write_mask,  /* COMPS */
 };

               bitmask_t tmp = encode__instruction_dst(s, &bp, &src->dst);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(13, 26, fld, false);  /* DST */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->tex.id;
             const bitmask_t packed = pack_field(27, 31, fld, false);  /* TEX_ID */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->tex.swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(35, 42, fld, false);  /* TEX_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[0].reg,  /* SRC0_REG */
    .SRC_AMODE = src->src[0].amode,  /* SRC0_AMODE */
    .SRC_RGROUP = src->src[0].rgroup,  /* SRC0_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[0]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(54, 63, fld, false);  /* SRC0 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
    .SRC_REG = src->src[1].reg,  /* SRC1_REG */
    .SRC_AMODE = src->src[1].amode,  /* SRC1_AMODE */
    .SRC_RGROUP = src->src[1].rgroup,  /* SRC1_RGROUP */
 };

               bitmask_t tmp = encode__instruction_src(s, &bp, &src->src[1]);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(81, 90, fld, false);  /* SRC1 */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x1);
             const bitmask_t packed = pack_field(109, 109, fld, false);  /* LOW_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src->thread & 0x2) > 1;
             const bitmask_t packed = pack_field(120, 120, fld, false);  /* HIGH_HALF */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->dst.use;
             const bitmask_t packed = pack_field(12, 12, fld, false);  /* DST_USE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].reg;
             const bitmask_t packed = pack_field(44, 52, fld, false);  /* SRC0_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].amode;
             const bitmask_t packed = pack_field(64, 66, fld, false);  /* SRC0_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[0].rgroup;
             const bitmask_t packed = pack_field(67, 69, fld, false);  /* SRC0_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].reg;
             const bitmask_t packed = pack_field(71, 79, fld, false);  /* SRC1_REG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].amode;
             const bitmask_t packed = pack_field(91, 93, fld, false);  /* SRC1_AMODE */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->src[1].rgroup;
             const bitmask_t packed = pack_field(96, 98, fld, false);  /* SRC1_RGROUP */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}




static bitmask_t
encode__instruction(struct encode_state *s, const struct bitset_params *p, const struct etna_inst * src)
{
   switch (__instruction_case(s, src)) {
   case ISA_OPC_NOP: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x0);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_0(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ADD: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000001);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_MAD: {


      bitmask_t val = uint64_t_to_bitmask(0x800000040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000002);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_MUL: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000003);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_DP3: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000005);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_DP4: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000006);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_DSX: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000007);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_DSY: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000008);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_MOV: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x9);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_MOVAR: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0xa);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_RCP: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0xc);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_RSQ: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0xd);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_SELECT: {


      bitmask_t val = uint64_t_to_bitmask(0x800000040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000000f);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_SET: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000010);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_EXP: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x11);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_LOG: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x12);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_FRC: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x13);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_CALL: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x14);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_5(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_RET: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x15);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_0(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_BRANCH: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x16);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_5(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_BRANCH_UNARY: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000016);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_6(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_BRANCH_BINARY: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000016);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_7(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_TEXKILL: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x17);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_8(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_TEXLD: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000018);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_9(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_TEXLDB: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x19);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_10(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_TEXLDD: {


      bitmask_t val = uint64_t_to_bitmask(0x800000040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000001a);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_11(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_TEXLDL: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x1b);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_10(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_SQRT: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x21);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_SIN: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x22);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_COS: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x23);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_BRANCH_ANY: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000024);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_7(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_FLOOR: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x25);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_CEIL: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x26);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_SIGN: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x27);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_BARRIER: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x2a);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_0(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_SWIZZLE: {


      bitmask_t val = uint64_t_to_bitmask(0x800000040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002b);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_I2I: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002c);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_I2F: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002d);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_12(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_F2I: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002e);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_12(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_F2IRND: {


      bitmask_t val = uint64_t_to_bitmask(0x0);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002f);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_12(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_CMP: {


      bitmask_t val = uint64_t_to_bitmask(0x800000040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000031);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_LOAD: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000032);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_13(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_STORE: {


      bitmask_t val = uint64_t_to_bitmask(0x800000040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000033);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_14(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_CLAMP0_MAX: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000036);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IADDSAT: {


      bitmask_t val = uint64_t_to_bitmask(0x800000000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000003b);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IMULLO0: {


      bitmask_t val = uint64_t_to_bitmask(0x40);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000003c);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IMULHI0: {


      bitmask_t val = uint64_t_to_bitmask(0x10040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000000);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IDIV0: {


      bitmask_t val = uint64_t_to_bitmask(0x10040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000004);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IMOD: {


      bitmask_t val = uint64_t_to_bitmask(0x10040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000008);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_TXF: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000009);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_11(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IMADLO0: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000000c);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IMADLOSAT0: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000000e);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_MOVAI: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x16);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IABS: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x17);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_LEADZERO: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x18);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_LSHIFT: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000019);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_RSHIFT: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000001a);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ROTATE: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000001b);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_OR: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000001c);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_AND: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000001d);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_XOR: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000001e);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_1(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_NOT: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x1f);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_BIT_EXTRACT: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000020);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_POPCOUNT: {


      bitmask_t val = uint64_t_to_bitmask(0x800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x21);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_4(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_DIV: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x24);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_15(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ATOMIC_ADD: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000025);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ATOMIC_XCHG: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000026);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ATOMIC_CMP_XCHG: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000027);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ATOMIC_MIN: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000028);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ATOMIC_MAX: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000029);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ATOMIC_OR: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002a);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ATOMIC_AND: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002b);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_ATOMIC_XOR: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002c);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_2(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_BIT_REV: {


      bitmask_t val = uint64_t_to_bitmask(0x10000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002d);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_12(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_TEXLDLPCF: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000002f);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_11(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_TEXLDGPCF: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000030);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_11(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_CONV: {


      bitmask_t val = uint64_t_to_bitmask(0x10040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000032);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_DP2: {


      bitmask_t val = uint64_t_to_bitmask(0x10040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000033);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_NORM_DP2: {


      bitmask_t val = uint64_t_to_bitmask(0x10000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000034);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_12(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_NORM_DP3: {


      bitmask_t val = uint64_t_to_bitmask(0x10000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000035);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_12(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_NORM_DP4: {


      bitmask_t val = uint64_t_to_bitmask(0x10000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000036);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_12(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_NORM_MUL: {


      bitmask_t val = uint64_t_to_bitmask(0x10040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000037);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_3(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IMG_LOAD: {


      bitmask_t val = uint64_t_to_bitmask(0x10040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x80000000039);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_13(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_IMG_STORE: {


      bitmask_t val = uint64_t_to_bitmask(0x800010040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000003a);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_14(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_TG4: {


      bitmask_t val = uint64_t_to_bitmask(0x10040);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000003d);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_16(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_BIT_FINDLSB: {


      bitmask_t val = uint64_t_to_bitmask(0x780000b800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000003f);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_12(s, p, src).bitset);
      return val;
    }
   case ISA_OPC_BIT_FINDMSB: {


      bitmask_t val = uint64_t_to_bitmask(0x780000c800010000);



      {
         bitmask_t word = uint64_t_to_bitmask(0x8000000003f);
         BITSET_SHL(val.bitset, 64);
         BITSET_OR(val.bitset, val.bitset, word.bitset);
      }

      BITSET_OR(val.bitset, val.bitset, snippet__instruction_12(s, p, src).bitset);
      return val;
    }
   default:
      /* Note that we need the default case, because there are
       * instructions which we never expect to be encoded, (ie.
       * meta/macro instructions) as they are removed/replace
       * in earlier stages of the compiler.
       */
      break;
   }
   mesa_loge("Unhandled #instruction encode case: 0x%x\n", __instruction_case(s, src));
   return uint64_t_to_bitmask(0);
}


static bitmask_t
snippet__src_swizzle_0(struct encode_state *s, const struct bitset_params *p, const uint8_t src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    {


           
    {

             fld = (src & 0x03) >> 0;
             const bitmask_t packed = pack_field(0, 1, fld, false);  /* SWIZ_X */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src & 0x0c) >> 2;
             const bitmask_t packed = pack_field(2, 3, fld, false);  /* SWIZ_Y */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src & 0x30) >> 4;
             const bitmask_t packed = pack_field(4, 5, fld, false);  /* SWIZ_Z */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = (src & 0xc0) >> 6;
             const bitmask_t packed = pack_field(6, 7, fld, false);  /* SWIZ_W */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}


static bitmask_t
encode__src_swizzle(struct encode_state *s, const struct bitset_params *p, const uint8_t src)
{

      bitmask_t val = uint64_t_to_bitmask(0x0);
      BITSET_OR(val.bitset, val.bitset, snippet__src_swizzle_0(s, p, src).bitset);
      return val;
}


static bitmask_t
snippet__instruction_src_0(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    if (__instruction_src_expr_anon_4(s, p, src)) {


           
    {

             fld = src->abs;
             const bitmask_t packed = pack_field(9, 9, fld, false);  /* SRC_ABS */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->neg;
             const bitmask_t packed = pack_field(8, 8, fld, false);  /* SRC_NEG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(0, 7, fld, false);  /* SRC_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    if (__instruction_src_expr_anon_6(s, p, src)) {


           
    {

             fld = src->abs;
             const bitmask_t packed = pack_field(9, 9, fld, false);  /* SRC_ABS */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->neg;
             const bitmask_t packed = pack_field(8, 8, fld, false);  /* SRC_NEG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(0, 7, fld, false);  /* SRC_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    if (__instruction_src_expr_anon_8(s, p, src)) {


           
    {

             fld = src->abs;
             const bitmask_t packed = pack_field(9, 9, fld, false);  /* SRC_ABS */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->neg;
             const bitmask_t packed = pack_field(8, 8, fld, false);  /* SRC_NEG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(0, 7, fld, false);  /* SRC_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    if (__instruction_src_expr_anon_10(s, p, src)) {


           
    {

             fld = src->abs;
             const bitmask_t packed = pack_field(9, 9, fld, false);  /* SRC_ABS */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->neg;
             const bitmask_t packed = pack_field(8, 8, fld, false);  /* SRC_NEG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(0, 7, fld, false);  /* SRC_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    {


           
    {

             fld = src->neg;
             const bitmask_t packed = pack_field(8, 8, fld, false);  /* SRC_NEG */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             fld = src->abs;
             const bitmask_t packed = pack_field(9, 9, fld, false);  /* SRC_ABS */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


           
    {

             { 
 struct bitset_params bp = {
 };

               bitmask_t tmp = encode__src_swizzle(s, &bp, src->swiz);
               fld = bitmask_to_uint64_t(tmp);
             }
             const bitmask_t packed = pack_field(0, 7, fld, false);  /* SRC_SWIZ */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}


static bitmask_t
encode__instruction_src(struct encode_state *s, const struct bitset_params *p, const struct etna_inst_src * src)
{

      bitmask_t val = uint64_t_to_bitmask(0x0);
      BITSET_OR(val.bitset, val.bitset, snippet__instruction_src_0(s, p, src).bitset);
      return val;
}


static bitmask_t
snippet__left_shift_0(struct encode_state *s, const struct bitset_params *p, const unsigned int src)
{
   bitmask_t val = uint64_t_to_bitmask(0);









      uint64_t fld;

      (void)fld;


    
    if (__left_shift_expr_anon_13(s, p, src)) {


           
    {

             fld = src;
             const bitmask_t packed = pack_field(0, 2, fld, false);  /* LEFT_SHIFT */
             BITSET_OR(val.bitset, val.bitset, packed.bitset);
             
    }


      {}  /* in case no unconditional field to close out last '} else' */
    
    } else


    
    {


      {}  /* in case no unconditional field to close out last '} else' */
    
    }


   return val;
}


static bitmask_t
encode__left_shift(struct encode_state *s, const struct bitset_params *p, const unsigned int src)
{

      bitmask_t val = uint64_t_to_bitmask(0x0);
      BITSET_OR(val.bitset, val.bitset, snippet__left_shift_0(s, p, src).bitset);
      return val;
}
