// (c) Copyright 2016 Hewlett Packard Enterprise Development LP
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package rules

import (
	"go/ast"
	"regexp"

	"github.com/securego/gosec/v2"
	"github.com/securego/gosec/v2/issue"
)

// Looks for net.Listen("0.0.0.0") or net.Listen(":8080")
type bindsToAllNetworkInterfaces struct {
	callListRule
	pattern *regexp.Regexp
}

func (r *bindsToAllNetworkInterfaces) Match(n ast.Node, c *gosec.Context) (*issue.Issue, error) {
	callExpr := r.calls.ContainsPkgCallExpr(n, c, false)
	if callExpr == nil {
		return nil, nil
	}
	if len(callExpr.Args) > 1 {
		arg := callExpr.Args[1]
		if bl, ok := arg.(*ast.BasicLit); ok {
			if arg, err := gosec.GetString(bl); err == nil {
				if gosec.RegexMatchWithCache(r.pattern, arg) {
					return c.NewIssue(n, r.ID(), r.What, r.Severity, r.Confidence), nil
				}
			}
		} else if ident, ok := arg.(*ast.Ident); ok {
			values := gosec.GetIdentStringValues(ident)
			for _, value := range values {
				if gosec.RegexMatchWithCache(r.pattern, value) {
					return c.NewIssue(n, r.ID(), r.What, r.Severity, r.Confidence), nil
				}
			}
		}
	} else if len(callExpr.Args) > 0 {
		values := gosec.GetCallStringArgsValues(callExpr.Args[0], c)
		for _, value := range values {
			if gosec.RegexMatchWithCache(r.pattern, value) {
				return c.NewIssue(n, r.ID(), r.What, r.Severity, r.Confidence), nil
			}
		}
	}
	return nil, nil
}

// NewBindsToAllNetworkInterfaces detects socket connections that are setup to
// listen on all network interfaces.
func NewBindsToAllNetworkInterfaces(id string, _ gosec.Config) (gosec.Rule, []ast.Node) {
	rule := &bindsToAllNetworkInterfaces{
		callListRule: newCallListRule(id, "Binds to all network interfaces", issue.Medium, issue.High),
		pattern:      regexp.MustCompile(`^(0.0.0.0|:).*$`),
	}
	rule.Add("net", "Listen").Add("crypto/tls", "Listen")
	return rule, []ast.Node{(*ast.CallExpr)(nil)}
}
