// Copyright 2025 The go-github AUTHORS. All rights reserved.
//
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package github

import (
	"context"
	"fmt"
)

// SCIMSchemasURINamespacesGroups is the SCIM schema URI namespace for group resources.
// This constant represents the standard SCIM core schema for group objects as defined by RFC 7643.
const SCIMSchemasURINamespacesGroups = "urn:ietf:params:scim:schemas:core:2.0:Group"

// SCIMSchemasURINamespacesUser is the SCIM schema URI namespace for user resources.
// This constant represents the standard SCIM core schema for user objects as defined by RFC 7643.
const SCIMSchemasURINamespacesUser = "urn:ietf:params:scim:schemas:core:2.0:User"

// SCIMSchemasURINamespacesListResponse is the SCIM schema URI namespace for list response resources.
// This constant represents the standard SCIM namespace for list responses used in paginated queries, as defined by RFC 7644.
const SCIMSchemasURINamespacesListResponse = "urn:ietf:params:scim:api:messages:2.0:ListResponse"

// SCIMSchemasURINamespacesPatchOp is the SCIM schema URI namespace for patch operations.
// This constant represents the standard SCIM namespace for patch operations as defined by RFC 7644.
const SCIMSchemasURINamespacesPatchOp = "urn:ietf:params:scim:api:messages:2.0:PatchOp"

// SCIMEnterpriseGroupAttributes represents supported SCIM Enterprise group attributes, and represents the result of calling UpdateSCIMGroupAttribute.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#supported-scim-group-attributes
type SCIMEnterpriseGroupAttributes struct {
	DisplayName *string                           `json:"displayName,omitempty"` // Human-readable name for a group.
	Members     []*SCIMEnterpriseDisplayReference `json:"members,omitempty"`     // List of members who are assigned to the group in SCIM provider
	ExternalID  *string                           `json:"externalId,omitempty"`  // This identifier is generated by a SCIM provider. Must be unique per user.
	// Bellow: Only populated as a result of calling UpdateSCIMGroupAttribute:
	Schemas []string            `json:"schemas,omitempty"` // The URIs that are used to indicate the namespaces of the SCIM schemas.
	ID      *string             `json:"id,omitempty"`      // The internally generated id for the group object.
	Meta    *SCIMEnterpriseMeta `json:"meta,omitempty"`    // The metadata associated with the creation/updates to the group.
}

// SCIMEnterpriseDisplayReference represents a JSON SCIM (System for Cross-domain Identity Management) resource reference.
type SCIMEnterpriseDisplayReference struct {
	Value   string  `json:"value"`             // The local unique identifier for the member (e.g., user ID or group ID).
	Ref     string  `json:"$ref"`              // The URI reference to the member resource (e.g., https://api.github.com/scim/v2/Users/{id}).
	Display *string `json:"display,omitempty"` // The display name associated with the member (e.g., user name or group name).
}

// SCIMEnterpriseMeta represents metadata about the SCIM resource.
type SCIMEnterpriseMeta struct {
	ResourceType string     `json:"resourceType"`           // A type of a resource (`User` or `Group`).
	Created      *Timestamp `json:"created,omitempty"`      // A date and time when the user was created.
	LastModified *Timestamp `json:"lastModified,omitempty"` // A date and time when the user was last modified.
	Location     *string    `json:"location,omitempty"`     // A URL location of an object
}

// SCIMEnterpriseGroups represents the result of calling ListProvisionedSCIMGroups.
type SCIMEnterpriseGroups struct {
	Schemas      []string                         `json:"schemas,omitempty"`
	TotalResults *int                             `json:"totalResults,omitempty"`
	Resources    []*SCIMEnterpriseGroupAttributes `json:"Resources,omitempty"`
	StartIndex   *int                             `json:"startIndex,omitempty"`
	ItemsPerPage *int                             `json:"itemsPerPage,omitempty"`
}

// ListProvisionedSCIMGroupsEnterpriseOptions represents query parameters for ListProvisionedSCIMGroups.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#list-provisioned-scim-groups-for-an-enterprise--parameters
type ListProvisionedSCIMGroupsEnterpriseOptions struct {
	// If specified, only results that match the specified filter will be returned.
	// Possible filters are `externalId`, `id`, and `displayName`. For example, `externalId eq "a123"`.
	Filter *string `url:"filter,omitempty"`
	// Excludes the specified attribute from being returned in the results.
	ExcludedAttributes *string `url:"excludedAttributes,omitempty"`
	// Used for pagination: the starting index of the first result to return when paginating through values.
	// Default: 1.
	StartIndex *int `url:"startIndex,omitempty"`
	// Used for pagination: the number of results to return per page.
	// Default: 30.
	Count *int `url:"count,omitempty"`
}

// SCIMEnterpriseUserAttributes represents supported SCIM enterprise user attributes, and represents the result of calling UpdateSCIMUserAttribute.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#supported-scim-user-attributes
type SCIMEnterpriseUserAttributes struct {
	DisplayName string                     `json:"displayName"`     // Human-readable name for a user
	Name        *SCIMEnterpriseUserName    `json:"name,omitempty"`  // The user's full name
	UserName    string                     `json:"userName"`        // The username for the user (GitHub Account after normalized), generated by the SCIM provider. Must be unique per user.
	Emails      []*SCIMEnterpriseUserEmail `json:"emails"`          // List of the user's emails. They all must be unique per user.
	Roles       []*SCIMEnterpriseUserRole  `json:"roles,omitempty"` // List of the user's roles.
	ExternalID  string                     `json:"externalId"`      // This identifier is generated by a SCIM provider. Must be unique per user.
	Active      bool                       `json:"active"`          // Indicates whether the identity is active (true) or should be suspended (false).
	Schemas     []string                   `json:"schemas"`         // The URIs that are used to indicate the namespaces of the SCIM schemas.
	// Bellow: Only populated as a result of calling UpdateSCIMUserAttribute:
	ID     *string                           `json:"id,omitempty"`     // Identifier generated by the GitHub's SCIM endpoint.
	Groups []*SCIMEnterpriseDisplayReference `json:"groups,omitempty"` // List of groups who are assigned to the user in SCIM provider
	Meta   *SCIMEnterpriseMeta               `json:"meta,omitempty"`   // The metadata associated with the creation/updates to the user.
}

// SCIMEnterpriseUserName represents SCIM enterprise user's name information.
type SCIMEnterpriseUserName struct {
	GivenName  string  `json:"givenName"`            // The first name of the user.
	FamilyName string  `json:"familyName"`           // The last name of the user.
	Formatted  *string `json:"formatted,omitempty"`  // The user's full name, including all middle names, titles, and suffixes, formatted for display.
	MiddleName *string `json:"middleName,omitempty"` // The middle name(s) of the user.
}

// SCIMEnterpriseUserEmail represents SCIM enterprise user's emails.
type SCIMEnterpriseUserEmail struct {
	Value   string `json:"value"`   // The email address.
	Primary bool   `json:"primary"` // Whether this email address is the primary address.
	Type    string `json:"type"`    // The type of email address
}

// SCIMEnterpriseUserRole is an enterprise-wide role granted to the user.
type SCIMEnterpriseUserRole struct {
	Value   string  `json:"value"` // The role value representing a user role in GitHub.
	Display *string `json:"display,omitempty"`
	Type    *string `json:"type,omitempty"`
	Primary *bool   `json:"primary,omitempty"` // Is the role a primary role for the user?
}

// SCIMEnterpriseUsers represents the result of calling ListProvisionedSCIMUsers.
type SCIMEnterpriseUsers struct {
	Schemas      []string                        `json:"schemas,omitempty"`
	TotalResults *int                            `json:"totalResults,omitempty"`
	ItemsPerPage *int                            `json:"itemsPerPage,omitempty"`
	StartIndex   *int                            `json:"startIndex,omitempty"`
	Resources    []*SCIMEnterpriseUserAttributes `json:"Resources,omitempty"`
}

// ListProvisionedSCIMUsersEnterpriseOptions represents query parameters for ListProvisionedSCIMUsers.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#list-scim-provisioned-identities-for-an-enterprise
type ListProvisionedSCIMUsersEnterpriseOptions struct {
	// If specified, only results that match the specified filter will be returned.
	// Possible filters are `userName`, `externalId`, `id`, and `displayName`. For example, `externalId eq "a123"`.
	Filter *string `url:"filter,omitempty"`
	// Used for pagination: the starting index of the first result to return when paginating through values.
	// Default: 1.
	StartIndex *int `url:"startIndex,omitempty"`
	// Used for pagination: the number of results to return per page.
	// Default: 30.
	Count *int `url:"count,omitempty"`
}

// SCIMEnterpriseAttribute represents attribute operations for UpdateSCIMGroupAttribute or UpdateSCIMUserAttribute.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#update-an-attribute-for-a-scim-enterprise-group
type SCIMEnterpriseAttribute struct {
	Schemas    []string                            `json:"schemas"`    // The URIs that are used to indicate the namespaces for a SCIM patch operation.
	Operations []*SCIMEnterpriseAttributeOperation `json:"Operations"` // Set of operations to be performed.
}

// SCIMEnterpriseAttributeOperation represents an operation for UpdateSCIMGroupAttribute or UpdateSCIMUserAttribute.
type SCIMEnterpriseAttributeOperation struct {
	Op    string  `json:"op"`              // Can be one of: `add`, `replace`, `remove`.
	Path  *string `json:"path,omitempty"`  // Path to the attribute being modified (Filters are not supported).
	Value *string `json:"value,omitempty"` // New value for the attribute being modified.
}

// ListProvisionedSCIMGroups lists provisioned SCIM groups in an enterprise.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#list-provisioned-scim-groups-for-an-enterprise
//
//meta:operation GET /scim/v2/enterprises/{enterprise}/Groups
func (s *EnterpriseService) ListProvisionedSCIMGroups(ctx context.Context, enterprise string, opts *ListProvisionedSCIMGroupsEnterpriseOptions) (*SCIMEnterpriseGroups, *Response, error) {
	u := fmt.Sprintf("scim/v2/enterprises/%v/Groups", enterprise)
	u, err := addOptions(u, opts)
	if err != nil {
		return nil, nil, err
	}

	req, err := s.client.NewRequest("GET", u, nil)
	if err != nil {
		return nil, nil, err
	}
	req.Header.Set("Accept", mediaTypeSCIM)

	groups := new(SCIMEnterpriseGroups)
	resp, err := s.client.Do(ctx, req, groups)
	if err != nil {
		return nil, resp, err
	}

	return groups, resp, nil
}

// ListProvisionedSCIMUsers lists provisioned SCIM enterprise users.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#list-scim-provisioned-identities-for-an-enterprise
//
//meta:operation GET /scim/v2/enterprises/{enterprise}/Users
func (s *EnterpriseService) ListProvisionedSCIMUsers(ctx context.Context, enterprise string, opts *ListProvisionedSCIMUsersEnterpriseOptions) (*SCIMEnterpriseUsers, *Response, error) {
	u := fmt.Sprintf("scim/v2/enterprises/%v/Users", enterprise)
	u, err := addOptions(u, opts)
	if err != nil {
		return nil, nil, err
	}

	req, err := s.client.NewRequest("GET", u, nil)
	if err != nil {
		return nil, nil, err
	}
	req.Header.Set("Accept", mediaTypeSCIM)

	users := new(SCIMEnterpriseUsers)
	resp, err := s.client.Do(ctx, req, users)
	if err != nil {
		return nil, resp, err
	}

	return users, resp, nil
}

// UpdateSCIMGroupAttribute updates a provisioned group’s individual attributes.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#update-an-attribute-for-a-scim-enterprise-group
//
//meta:operation PATCH /scim/v2/enterprises/{enterprise}/Groups/{scim_group_id}
func (s *EnterpriseService) UpdateSCIMGroupAttribute(ctx context.Context, enterprise, scimGroupID string, attribute SCIMEnterpriseAttribute) (*SCIMEnterpriseGroupAttributes, *Response, error) {
	u := fmt.Sprintf("scim/v2/enterprises/%v/Groups/%v", enterprise, scimGroupID)
	req, err := s.client.NewRequest("PATCH", u, attribute)
	if err != nil {
		return nil, nil, err
	}
	req.Header.Set("Accept", mediaTypeSCIM)

	group := new(SCIMEnterpriseGroupAttributes)
	resp, err := s.client.Do(ctx, req, group)
	if err != nil {
		return nil, resp, err
	}

	return group, resp, nil
}

// UpdateSCIMUserAttribute updates a provisioned user's individual attributes.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#update-an-attribute-for-a-scim-enterprise-user
//
//meta:operation PATCH /scim/v2/enterprises/{enterprise}/Users/{scim_user_id}
func (s *EnterpriseService) UpdateSCIMUserAttribute(ctx context.Context, enterprise, scimUserID string, attribute SCIMEnterpriseAttribute) (*SCIMEnterpriseUserAttributes, *Response, error) {
	u := fmt.Sprintf("scim/v2/enterprises/%v/Users/%v", enterprise, scimUserID)
	req, err := s.client.NewRequest("PATCH", u, attribute)
	if err != nil {
		return nil, nil, err
	}
	req.Header.Set("Accept", mediaTypeSCIM)

	user := new(SCIMEnterpriseUserAttributes)
	resp, err := s.client.Do(ctx, req, user)
	if err != nil {
		return nil, resp, err
	}

	return user, resp, nil
}
