# frozen_string_literal: true

module Banzai
  module Filter
    # Finds heading anchors and localises the accessible labels on them.
    #
    # Headings with anchors as generated by our Markdown parser look as follows.
    # Given the input:
    #
    # ```markdown
    # ## so _cool_ h2
    # ```
    #
    # The HTML generated is (newlines and spaces added for readability):
    #
    # ```html
    # <h2 id="user-content-so-cool-h2" data-sourcepos="3:1-3:15" dir="auto">
    #   so <em data-sourcepos="3:7-3:12">cool</em> h2
    #   <a href="#so-cool-h2" aria-label="Link to heading 'so cool h2'"
    #      data-heading-content="so cool h2" class="anchor"></a>
    # </h2>
    # ```
    #
    # We use [data-heading-content] to rewrite aria-label in a localisable way.
    class HeadingAccessibilityFilter < HTML::Pipeline::Filter
      prepend Concerns::PipelineTimingCheck

      SUBANCHOR_CSS = '> a:last-child.anchor[aria-label][data-heading-content]'
      ANCHOR_CSS    = %w[h1 h2 h3 h4 h5 h6].map { |h| "#{h} #{SUBANCHOR_CSS}" }.join(', ')
      ANCHOR_XPATH  = Gitlab::Utils::Nokogiri.css_to_xpath(ANCHOR_CSS).freeze

      def call
        doc.xpath(ANCHOR_XPATH).each do |node|
          node['aria-label'] = format(_("Link to heading '%{heading}'"), heading: node['data-heading-content'])
        end

        doc
      end
    end
  end
end
