/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */
use crate::MessageXml;
use serde::de::{self, Error, MapAccess, Visitor};
use serde::{Deserialize, Deserializer};
use std::fmt;
use std::marker::PhantomData;

/// A generic representation of a response message from a request.
///
/// See <https://learn.microsoft.com/en-us/exchange/client-developer/web-service-reference/responsemessages>
#[derive(Clone, Debug, Deserialize, PartialEq, Eq)]
#[serde(default)]
pub struct ResponseMessages<T> {
    #[serde(rename = "$value")]
    pub response_messages: Vec<ResponseClass<T>>,
}

impl<T> Default for ResponseMessages<T> {
    fn default() -> Self {
        Self {
            response_messages: vec![],
        }
    }
}

/// A fallible response to an operation.
///
/// For a list of upstream types that use this sort of structuring, you can see [`ResponseCode`]'s
/// [list of parent
/// types](https://learn.microsoft.com/en-us/exchange/client-developer/web-service-reference/responsecode#parent-elements).
#[derive(Clone, Debug, PartialEq, Eq)]
pub enum ResponseClass<T> {
    /// The operation was successful and returned the requested object.
    Success(T),
    /// The operation failed and returned an error.
    Error(ResponseError),
    /// The operation was only partially successful, and may have returned a usable object.  This
    /// should also include `ResponseError` information, but currently does not due to library
    /// limitations.
    // Flatten doesn't work with quick_xml, see https://github.com/tafia/quick-xml/issues/714
    Warning(T),
}

// Manually implemented because quick_xml's impl_deserialize_for_internally_tagged_enum doesn't
// currently support generics.
impl<'de, T: Deserialize<'de>> Deserialize<'de> for ResponseClass<T> {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        struct ResponseClassVisitor<T> {
            marker: PhantomData<fn() -> ResponseClass<T>>,
        }
        impl<'de, T: Deserialize<'de>> Visitor<'de> for ResponseClassVisitor<T> {
            type Value = ResponseClass<T>;

            fn expecting(&self, f: &mut fmt::Formatter) -> fmt::Result {
                f.write_str("a map with a ResponseClass attribute")
            }

            fn visit_map<A>(self, mut map: A) -> Result<Self::Value, A::Error>
            where
                A: MapAccess<'de>,
            {
                while let Some((attribute, value)) = map.next_entry::<String, String>()? {
                    if attribute == "@ResponseClass" {
                        let de = de::value::MapAccessDeserializer::new(map);
                        return match value.as_str() {
                            "Success" => Ok(Self::Value::Success(T::deserialize(de)?)),
                            "Error" => Ok(Self::Value::Error(ResponseError::deserialize(de)?)),
                            "Warning" => Ok(Self::Value::Warning(T::deserialize(de)?)),
                            var => Err(A::Error::unknown_variant(
                                var,
                                &["Success", "Warning", "Error"],
                            )),
                        };
                    }
                }

                Err(A::Error::missing_field("@ResponseClass"))
            }
        }
        deserializer.deserialize_map(ResponseClassVisitor {
            marker: PhantomData,
        })
    }
}

/// Information available when an operation responded with an Error.
#[derive(Clone, Debug, Deserialize, PartialEq, Eq)]
#[serde(rename_all = "PascalCase")]
pub struct ResponseError {
    pub message_text: String,
    pub response_code: ResponseCode,
    pub message_xml: Option<MessageXml>,
}

impl fmt::Display for ResponseError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> Result<(), std::fmt::Error> {
        if let Some(message_xml) = &self.message_xml {
            write!(
                f,
                "response code {:?}: {} ({message_xml:?})",
                self.response_code, self.message_text
            )
        } else {
            write!(
                f,
                "response code {:?}: {}",
                self.response_code, self.message_text
            )
        }
    }
}

impl std::error::Error for ResponseError {}

/// An error code describing the error encountered in processing a request, if
/// any.
///
/// See <https://learn.microsoft.com/en-us/exchange/client-developer/web-service-reference/responsecode>
#[derive(Clone, Copy, Debug, Default, Deserialize, PartialEq, Eq)]
pub enum ResponseCode {
    #[default]
    NoError,

    ErrorAccessDenied,
    ErrorAccessModeSpecified,
    ErrorAccountDisabled,
    ErrorAddDelegatesFailed,
    ErrorAddressSpaceNotFound,
    ErrorADOperation,
    ErrorADSessionFilter,
    ErrorADUnavailable,
    ErrorAutoDiscoverFailed,
    ErrorAffectedTaskOccurrencesRequired,
    ErrorAttachmentNestLevelLimitExceeded,
    ErrorAttachmentSizeLimitExceeded,
    ErrorArchiveFolderPathCreation,
    ErrorArchiveMailboxNotEnabled,
    ErrorArchiveMailboxServiceDiscoveryFailed,
    ErrorAvailabilityConfigNotFound,
    ErrorBatchProcessingStopped,
    ErrorCalendarCannotMoveOrCopyOccurrence,
    ErrorCalendarCannotUpdateDeletedItem,
    ErrorCalendarCannotUseIdForOccurrenceId,
    ErrorCalendarCannotUseIdForRecurringMasterId,
    ErrorCalendarDurationIsTooLong,
    ErrorCalendarEndDateIsEarlierThanStartDate,
    ErrorCalendarFolderIsInvalidForCalendarView,
    ErrorCalendarInvalidAttributeValue,
    ErrorCalendarInvalidDayForTimeChangePattern,
    ErrorCalendarInvalidDayForWeeklyRecurrence,
    ErrorCalendarInvalidPropertyState,
    ErrorCalendarInvalidPropertyValue,
    ErrorCalendarInvalidRecurrence,
    ErrorCalendarInvalidTimeZone,
    ErrorCalendarIsCancelledForAccept,
    ErrorCalendarIsCancelledForDecline,
    ErrorCalendarIsCancelledForRemove,
    ErrorCalendarIsCancelledForTentative,
    ErrorCalendarIsDelegatedForAccept,
    ErrorCalendarIsDelegatedForDecline,
    ErrorCalendarIsDelegatedForRemove,
    ErrorCalendarIsDelegatedForTentative,
    ErrorCalendarIsNotOrganizer,
    ErrorCalendarIsOrganizerForAccept,
    ErrorCalendarIsOrganizerForDecline,
    ErrorCalendarIsOrganizerForRemove,
    ErrorCalendarIsOrganizerForTentative,
    ErrorCalendarOccurrenceIndexIsOutOfRecurrenceRange,
    ErrorCalendarOccurrenceIsDeletedFromRecurrence,
    ErrorCalendarOutOfRange,
    ErrorCalendarMeetingRequestIsOutOfDate,
    ErrorCalendarViewRangeTooBig,
    ErrorCallerIsInvalidADAccount,
    ErrorCannotArchiveCalendarContactTaskFolderException,
    ErrorCannotArchiveItemsInPublicFolders,
    ErrorCannotArchiveItemsInArchiveMailbox,
    ErrorCannotCreateCalendarItemInNonCalendarFolder,
    ErrorCannotCreateContactInNonContactFolder,
    ErrorCannotCreatePostItemInNonMailFolder,
    ErrorCannotCreateTaskInNonTaskFolder,
    ErrorCannotDeleteObject,
    ErrorCannotDisableMandatoryExtension,
    ErrorCannotFindUser,
    ErrorCannotGetSourceFolderPath,
    ErrorCannotGetExternalEcpUrl,
    ErrorCannotOpenFileAttachment,
    ErrorCannotDeleteTaskOccurrence,
    ErrorCannotEmptyFolder,
    ErrorCannotSetCalendarPermissionOnNonCalendarFolder,
    ErrorCannotSetNonCalendarPermissionOnCalendarFolder,
    ErrorCannotSetPermissionUnknownEntries,
    ErrorCannotSpecifySearchFolderAsSourceFolder,
    ErrorCannotUseFolderIdForItemId,
    ErrorCannotUseItemIdForFolderId,
    ErrorChangeKeyRequired,
    ErrorChangeKeyRequiredForWriteOperations,
    ErrorClientDisconnected,
    ErrorClientIntentInvalidStateDefinition,
    ErrorClientIntentNotFound,
    ErrorConnectionFailed,
    ErrorContainsFilterWrongType,
    ErrorContentConversionFailed,
    ErrorContentIndexingNotEnabled,
    ErrorCorruptData,
    ErrorCreateItemAccessDenied,
    ErrorCreateManagedFolderPartialCompletion,
    ErrorCreateSubfolderAccessDenied,
    ErrorCrossMailboxMoveCopy,
    ErrorCrossSiteRequest,
    ErrorDataSizeLimitExceeded,
    ErrorDataSourceOperation,
    ErrorDelegateAlreadyExists,
    ErrorDelegateCannotAddOwner,
    ErrorDelegateMissingConfiguration,
    ErrorDelegateNoUser,
    ErrorDelegateValidationFailed,
    ErrorDeleteDistinguishedFolder,
    ErrorDeleteItemsFailed,
    ErrorDeleteUnifiedMessagingPromptFailed,
    ErrorDistinguishedUserNotSupported,
    ErrorDistributionListMemberNotExist,
    ErrorDuplicateInputFolderNames,
    ErrorDuplicateUserIdsSpecified,
    ErrorEmailAddressMismatch,
    ErrorEventNotFound,
    ErrorExceededConnectionCount,
    ErrorExceededSubscriptionCount,
    ErrorExceededFindCountLimit,
    ErrorExpiredSubscription,
    ErrorExtensionNotFound,
    ErrorFolderCorrupt,
    ErrorFolderNotFound,
    ErrorFolderPropertRequestFailed,
    ErrorFolderSave,
    ErrorFolderSaveFailed,
    ErrorFolderSavePropertyError,
    ErrorFolderExists,
    ErrorFreeBusyGenerationFailed,
    ErrorGetServerSecurityDescriptorFailed,
    ErrorImContactLimitReached,
    ErrorImGroupDisplayNameAlreadyExists,
    ErrorImGroupLimitReached,
    ErrorImpersonateUserDenied,
    ErrorImpersonationDenied,
    ErrorImpersonationFailed,
    ErrorIncorrectSchemaVersion,
    ErrorIncorrectUpdatePropertyCount,
    ErrorIndividualMailboxLimitReached,
    ErrorInsufficientResources,
    ErrorInternalServerError,
    ErrorInternalServerTransientError,
    ErrorInvalidAccessLevel,
    ErrorInvalidArgument,
    ErrorInvalidAttachmentId,
    ErrorInvalidAttachmentSubfilter,
    ErrorInvalidAttachmentSubfilterTextFilter,
    ErrorInvalidAuthorizationContext,
    ErrorInvalidChangeKey,
    ErrorInvalidClientSecurityContext,
    ErrorInvalidCompleteDate,
    ErrorInvalidContactEmailAddress,
    ErrorInvalidContactEmailIndex,
    ErrorInvalidCrossForestCredentials,
    ErrorInvalidDelegatePermission,
    ErrorInvalidDelegateUserId,
    ErrorInvalidExcludesRestriction,
    ErrorInvalidExpressionTypeForSubFilter,
    ErrorInvalidExtendedProperty,
    ErrorInvalidExtendedPropertyValue,
    ErrorInvalidFolderId,
    ErrorInvalidFolderTypeForOperation,
    ErrorInvalidFractionalPagingParameters,
    ErrorInvalidFreeBusyViewType,
    ErrorInvalidId,
    ErrorInvalidIdEmpty,
    ErrorInvalidIdMalformed,
    ErrorInvalidIdMalformedEwsLegacyIdFormat,
    ErrorInvalidIdMonikerTooLong,
    ErrorInvalidIdNotAnItemAttachmentId,
    ErrorInvalidIdReturnedByResolveNames,
    ErrorInvalidIdStoreObjectIdTooLong,
    ErrorInvalidIdTooManyAttachmentLevels,
    ErrorInvalidIdXml,
    ErrorInvalidImContactId,
    ErrorInvalidImDistributionGroupSmtpAddress,
    ErrorInvalidImGroupId,
    ErrorInvalidIndexedPagingParameters,
    ErrorInvalidInternetHeaderChildNodes,
    ErrorInvalidItemForOperationArchiveItem,
    ErrorInvalidItemForOperationCreateItemAttachment,
    ErrorInvalidItemForOperationCreateItem,
    ErrorInvalidItemForOperationAcceptItem,
    ErrorInvalidItemForOperationDeclineItem,
    ErrorInvalidItemForOperationCancelItem,
    ErrorInvalidItemForOperationExpandDL,
    ErrorInvalidItemForOperationRemoveItem,
    ErrorInvalidItemForOperationSendItem,
    ErrorInvalidItemForOperationTentative,
    ErrorInvalidLogonType,
    ErrorInvalidLikeRequest,
    ErrorInvalidMailbox,
    ErrorInvalidManagedFolderProperty,
    ErrorInvalidManagedFolderQuota,
    ErrorInvalidManagedFolderSize,
    ErrorInvalidMergedFreeBusyInterval,
    ErrorInvalidNameForNameResolution,
    ErrorInvalidOperation,
    ErrorInvalidNetworkServiceContext,
    ErrorInvalidOofParameter,
    ErrorInvalidPagingMaxRows,
    ErrorInvalidParentFolder,
    ErrorInvalidPercentCompleteValue,
    ErrorInvalidPermissionSettings,
    ErrorInvalidPhoneCallId,
    ErrorInvalidPhoneNumber,
    ErrorInvalidUserInfo,
    ErrorInvalidPropertyAppend,
    ErrorInvalidPropertyDelete,
    ErrorInvalidPropertyForExists,
    ErrorInvalidPropertyForOperation,
    ErrorInvalidPropertyRequest,
    ErrorInvalidPropertySet,
    ErrorInvalidPropertyUpdateSentMessage,
    ErrorInvalidProxySecurityContext,
    ErrorInvalidPullSubscriptionId,
    ErrorInvalidPushSubscriptionUrl,
    ErrorInvalidRecipients,
    ErrorInvalidRecipientSubfilter,
    ErrorInvalidRecipientSubfilterComparison,
    ErrorInvalidRecipientSubfilterOrder,
    ErrorInvalidRecipientSubfilterTextFilter,
    ErrorInvalidReferenceItem,
    ErrorInvalidRequest,
    ErrorInvalidRestriction,
    ErrorInvalidRetentionTagIdGuid,
    ErrorInvalidRetentionTagTypeMismatch,
    ErrorInvalidRetentionTagInvisible,
    ErrorInvalidRetentionTagInheritance,
    ErrorInvalidRoutingType,
    ErrorInvalidScheduledOofDuration,
    ErrorInvalidSchemaVersionForMailboxVersion,
    ErrorInvalidSecurityDescriptor,
    ErrorInvalidSendItemSaveSettings,
    ErrorInvalidSerializedAccessToken,
    ErrorInvalidServerVersion,
    ErrorInvalidSid,
    ErrorInvalidSIPUri,
    ErrorInvalidSmtpAddress,
    ErrorInvalidSubfilterType,
    ErrorInvalidSubfilterTypeNotAttendeeType,
    ErrorInvalidSubfilterTypeNotRecipientType,
    ErrorInvalidSubscription,
    ErrorInvalidSubscriptionRequest,
    ErrorInvalidSyncStateData,
    ErrorInvalidTimeInterval,
    ErrorInvalidUserOofSettings,
    ErrorInvalidUserPrincipalName,
    ErrorInvalidUserSid,
    ErrorInvalidUserSidMissingUPN,
    ErrorInvalidValueForProperty,
    ErrorInvalidWatermark,
    ErrorIPGatewayNotFound,
    ErrorIrresolvableConflict,
    ErrorItemCorrupt,
    ErrorItemNotFound,
    ErrorItemPropertyRequestFailed,
    ErrorItemSave,
    ErrorItemSavePropertyError,
    ErrorLegacyMailboxFreeBusyViewTypeNotMerged,
    ErrorLocalServerObjectNotFound,
    ErrorLogonAsNetworkServiceFailed,
    ErrorMailboxConfiguration,
    ErrorMailboxDataArrayEmpty,
    ErrorMailboxDataArrayTooBig,
    ErrorMailboxHoldNotFound,
    ErrorMailboxLogonFailed,
    ErrorMailboxMoveInProgress,
    ErrorMailboxStoreUnavailable,
    ErrorMailRecipientNotFound,
    ErrorMailTipsDisabled,
    ErrorManagedFolderAlreadyExists,
    ErrorManagedFolderNotFound,
    ErrorManagedFoldersRootFailure,
    ErrorMeetingSuggestionGenerationFailed,
    ErrorMessageDispositionRequired,
    ErrorMessageSizeExceeded,
    ErrorMimeContentConversionFailed,
    ErrorMimeContentInvalid,
    ErrorMimeContentInvalidBase64String,
    ErrorMissingArgument,
    ErrorMissingEmailAddress,
    ErrorMissingEmailAddressForManagedFolder,
    ErrorMissingInformationEmailAddress,
    ErrorMissingInformationReferenceItemId,
    ErrorMissingItemForCreateItemAttachment,
    ErrorMissingManagedFolderId,
    ErrorMissingRecipients,
    ErrorMissingUserIdInformation,
    ErrorMoreThanOneAccessModeSpecified,
    ErrorMoveCopyFailed,
    ErrorMoveDistinguishedFolder,
    ErrorMultiLegacyMailboxAccess,
    ErrorNameResolutionMultipleResults,
    ErrorNameResolutionNoMailbox,
    ErrorNameResolutionNoResults,
    ErrorNoApplicableProxyCASServersAvailable,
    ErrorNoCalendar,
    ErrorNoDestinationCASDueToKerberosRequirements,
    ErrorNoDestinationCASDueToSSLRequirements,
    ErrorNoDestinationCASDueToVersionMismatch,
    ErrorNoFolderClassOverride,
    ErrorNoFreeBusyAccess,
    ErrorNonExistentMailbox,
    ErrorNonPrimarySmtpAddress,
    ErrorNoPropertyTagForCustomProperties,
    ErrorNoPublicFolderReplicaAvailable,
    ErrorNoPublicFolderServerAvailable,
    ErrorNoRespondingCASInDestinationSite,
    ErrorNotDelegate,
    ErrorNotEnoughMemory,
    ErrorObjectTypeChanged,
    ErrorOccurrenceCrossingBoundary,
    ErrorOccurrenceTimeSpanTooBig,
    ErrorOperationNotAllowedWithPublicFolderRoot,
    ErrorParentFolderIdRequired,
    ErrorParentFolderNotFound,
    ErrorPasswordChangeRequired,
    ErrorPasswordExpired,
    ErrorPhoneNumberNotDialable,
    ErrorPropertyUpdate,
    ErrorPromptPublishingOperationFailed,
    ErrorPropertyValidationFailure,
    ErrorProxiedSubscriptionCallFailure,
    ErrorProxyCallFailed,
    ErrorProxyGroupSidLimitExceeded,
    ErrorProxyRequestNotAllowed,
    ErrorProxyRequestProcessingFailed,
    ErrorProxyServiceDiscoveryFailed,
    ErrorProxyTokenExpired,
    ErrorPublicFolderMailboxDiscoveryFailed,
    ErrorPublicFolderOperationFailed,
    ErrorPublicFolderRequestProcessingFailed,
    ErrorPublicFolderServerNotFound,
    ErrorPublicFolderSyncException,
    ErrorQueryFilterTooLong,
    ErrorQuotaExceeded,
    ErrorReadEventsFailed,
    ErrorReadReceiptNotPending,
    ErrorRecurrenceEndDateTooBig,
    ErrorRecurrenceHasNoOccurrence,
    ErrorRemoveDelegatesFailed,
    ErrorRequestAborted,
    ErrorRequestStreamTooBig,
    ErrorRequiredPropertyMissing,
    ErrorResolveNamesInvalidFolderType,
    ErrorResolveNamesOnlyOneContactsFolderAllowed,
    ErrorResponseSchemaValidation,
    ErrorRestrictionTooLong,
    ErrorRestrictionTooComplex,
    ErrorResultSetTooBig,
    ErrorInvalidExchangeImpersonationHeaderData,
    ErrorSavedItemFolderNotFound,
    ErrorSchemaValidation,
    ErrorSearchFolderNotInitialized,
    ErrorSendAsDenied,
    ErrorSendMeetingCancellationsRequired,
    ErrorSendMeetingInvitationsOrCancellationsRequired,
    ErrorSendMeetingInvitationsRequired,
    ErrorSentMeetingRequestUpdate,
    ErrorSentTaskRequestUpdate,
    ErrorServerBusy,
    ErrorServiceDiscoveryFailed,
    ErrorStaleObject,
    ErrorSubmissionQuotaExceeded,
    ErrorSubscriptionAccessDenied,
    ErrorSubscriptionDelegateAccessNotSupported,
    ErrorSubscriptionNotFound,
    ErrorSubscriptionUnsubscribed,
    ErrorSyncFolderNotFound,
    ErrorTeamMailboxNotFound,
    ErrorTeamMailboxNotLinkedToSharePoint,
    ErrorTeamMailboxUrlValidationFailed,
    ErrorTeamMailboxNotAuthorizedOwner,
    ErrorTeamMailboxActiveToPendingDelete,
    ErrorTeamMailboxFailedSendingNotifications,
    ErrorTeamMailboxErrorUnknown,
    ErrorTimeIntervalTooBig,
    ErrorTimeoutExpired,
    ErrorTimeZone,
    ErrorToFolderNotFound,
    ErrorTokenSerializationDenied,
    ErrorTooManyObjectsOpened,
    ErrorUpdatePropertyMismatch,
    ErrorUnifiedMessagingDialPlanNotFound,
    ErrorUnifiedMessagingReportDataNotFound,
    ErrorUnifiedMessagingPromptNotFound,
    ErrorUnifiedMessagingRequestFailed,
    ErrorUnifiedMessagingServerNotFound,
    ErrorUnableToGetUserOofSettings,
    ErrorUnableToRemoveImContactFromGroup,
    ErrorUnsupportedSubFilter,
    ErrorUnsupportedCulture,
    ErrorUnsupportedMapiPropertyType,
    ErrorUnsupportedMimeConversion,
    ErrorUnsupportedPathForQuery,
    ErrorUnsupportedPathForSortGroup,
    ErrorUnsupportedPropertyDefinition,
    ErrorUnsupportedQueryFilter,
    ErrorUnsupportedRecurrence,
    ErrorUnsupportedTypeForConversion,
    ErrorUpdateDelegatesFailed,
    ErrorUserNotUnifiedMessagingEnabled,
    ErrorVoiceMailNotImplemented,
    ErrorValueOutOfRange,
    ErrorVirusDetected,
    ErrorVirusMessageDeleted,
    ErrorWebRequestInInvalidState,
    ErrorWin32InteropError,
    ErrorWorkingHoursSaveFailed,
    ErrorWorkingHoursXmlMalformed,
    ErrorWrongServerVersion,
    ErrorWrongServerVersionDelegate,
    ErrorMissingInformationSharingFolderId,
    ErrorDuplicateSOAPHeader,
    ErrorSharingSynchronizationFailed,
    ErrorSharingNoExternalEwsAvailable,
    ErrorFreeBusyDLLimitReached,
    ErrorInvalidGetSharingFolderRequest,
    ErrorNotAllowedExternalSharingByPolicy,
    ErrorUserNotAllowedByPolicy,
    ErrorPermissionNotAllowedByPolicy,
    ErrorOrganizationNotFederated,
    ErrorMailboxFailover,
    ErrorInvalidExternalSharingInitiator,
    ErrorMessageTrackingPermanentError,
    ErrorMessageTrackingTransientError,
    ErrorMessageTrackingNoSuchDomain,
    ErrorUserWithoutFederatedProxyAddress,
    ErrorInvalidOrganizationRelationshipForFreeBusy,
    ErrorInvalidFederatedOrganizationId,
    ErrorInvalidExternalSharingSubscriber,
    ErrorInvalidSharingData,
    ErrorInvalidSharingMessage,
    ErrorNotSupportedSharingMessage,
    ErrorApplyConversationActionFailed,
    ErrorInboxRulesValidationError,
    ErrorOutlookRuleBlobExists,
    ErrorRulesOverQuota,
    ErrorNewEventStreamConnectionOpened,
    ErrorMissedNotificationEvents,
    ErrorDuplicateLegacyDistinguishedName,
    ErrorInvalidClientAccessTokenRequest,
    ErrorNoSpeechDetected,
    ErrorUMServerUnavailable,
    ErrorRecipientNotFound,
    ErrorRecognizerNotInstalled,
    ErrorSpeechGrammarError,
    ErrorInvalidManagementRoleHeader,
    ErrorLocationServicesDisabled,
    ErrorLocationServicesRequestTimedOut,
    ErrorLocationServicesRequestFailed,
    ErrorLocationServicesInvalidRequest,
    ErrorWeatherServiceDisabled,
    ErrorMailboxScopeNotAllowedWithoutQueryString,
    ErrorArchiveMailboxSearchFailed,
    ErrorGetRemoteArchiveFolderFailed,
    ErrorFindRemoteArchiveFolderFailed,
    ErrorGetRemoteArchiveItemFailed,
    ErrorExportRemoteArchiveItemsFailed,
    ErrorInvalidPhotoSize,
    ErrorSearchQueryHasTooManyKeywords,
    ErrorSearchTooManyMailboxes,
    ErrorDiscoverySearchesDisabled,
    ErrorCalendarSeekToConditionNotSupported,
    ErrorCalendarIsGroupMailboxForAccept,
    ErrorCalendarIsGroupMailboxForDecline,
    ErrorCalendarIsGroupMailboxForTentative,
    ErrorCalendarIsGroupMailboxForSuppressReadReceipt,
    ErrorOrganizationAccessBlocked,
    ErrorInvalidLicense,
    ErrorMessagePerFolderCountReceiveQuotaExceeded,
    ErrorInvalidBulkActionType,
    ErrorInvalidKeepNCount,
    ErrorInvalidKeepNType,
    ErrorNoOAuthServerAvailableForRequest,
    ErrorInstantSearchSessionExpired,
    ErrorInstantSearchTimeout,
    ErrorInstantSearchFailed,
    ErrorInvalidFederationOrganizationIdentifier,
    ErrorInvalidSweepRule,
    ErrorInvalidSweepRuleOperationType,
    ErrorTargetDomainNotSupported,
}
