// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F193

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_32_0_F_F_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x9f5a6cc9u, 0xca8566fau, 23, 109 }, // 9f5a6cc9ca8566fa = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x897d2a14u, 0xe6b20d3du, 23, 6 }, // 897d2a14e6b20d3d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x6a966848u, 0x1da7c4beu, 23, 69 }, // 6a9668481da7c4be = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x7a2eb3acu, 0xcb57ce03u, 23, 143 }, // 7a2eb3accb57ce03 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x99ffe615u, 0x4391df26u, 23, 86 }, // 99ffe6154391df26 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__32_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0x511fb268u, 0x477c7d9bu, 63, 143 }, // 511fb268477c7d9b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__32_32__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xcbd4ce21u, 0xc170c17cu, 103, 109 }, // cbd4ce21c170c17c = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__64_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x4de802e5u, 0x1e562e27u, 103, 6 }, // 4de802e51e562e27 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__64_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xf8c9d3cfu, 0x7cf258a1u, 103, 69 }, // f8c9d3cf7cf258a1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xf4af5dd7u, 0xf2c78cc6u, 103, 143 }, // f4af5dd7f2c78cc6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__64_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x568d2ee4u, 0x0bcd4ba5u, 103, 29 }, // 568d2ee40bcd4ba5 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__64_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xe8d15404u, 0x67d53ffdu, 103, 86 }, // e8d1540467d53ffd = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__64_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0xc00043cau, 0x2927d39au, 166, 6 }, // c00043ca2927d39a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_32_0_F_F_1___gfx950__P__64_64__CO__wave2_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{ 4, 1, 1,10, 9, 9, 9},
 { 3, 8, 4, 8, 9, 9, 9},
 { 1, 4, 3, 4, 9, 9, 9},
 { 7, 7, 4, 7, 1,12, 9},
 {11,11,11, 4, 9,11, 9},
 {11,11, 9, 4, 4, 9, 9},
 {11,11, 9,11, 9,12, 9}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F193 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 32
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 32 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

