/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hive.jdbc;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.withSettings;

import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Statement;

import org.apache.hive.service.rpc.thrift.TCLIService.Iface;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hive.service.rpc.thrift.TSessionHandle;
import org.junit.Test;

public class TestHiveStatement {

  /**
   * Gives the JDBC driver a hint as to the number of rows that should be
   * fetched from the database when more rows are needed for ResultSet objects
   * generated by this Statement. If the value specified is zero, then the hint
   * is ignored. Test a value greater than zero is accepted.
   */
  @Test
  public void testSetFetchSize() throws SQLException {
    final HiveConnection connection = mock(HiveConnection.class, withSettings().useConstructor());
    final Iface iface = mock(Iface.class);
    final TSessionHandle handle = mock(TSessionHandle.class);

    try (HiveStatement stmt = new HiveStatement(connection, iface, handle)) {
      stmt.setFetchSize(123);
      assertEquals(123, stmt.getFetchSize());
    }
  }

  /**
   * Gives the JDBC driver a hint as to the number of rows that should be
   * fetched from the database when more rows are needed for ResultSet objects
   * generated by this Statement. If the value specified is zero, then the hint
   * is ignored. Test for a value of zero (hint is ignored).
   */
  @Test
  public void testSetFetchSizeZero() throws SQLException {
    final HiveConnection connection = mock(HiveConnection.class);
    final Iface iface = mock(Iface.class);
    final TSessionHandle handle = mock(TSessionHandle.class);

    // No hint specified and no default value passed in through the constructor,
    // so it falls-back to the configuration default value
    int fetchSize = HiveConf.ConfVars.HIVE_SERVER2_THRIFT_RESULTSET_DEFAULT_FETCH_SIZE.defaultIntVal;
    connection.fetchSize = fetchSize;
    try (HiveStatement stmt = new HiveStatement(connection, iface, handle)) {
      stmt.setFetchSize(0);
      assertEquals(fetchSize, stmt.getFetchSize());
    }
  }

  /**
   * Gives the JDBC driver a hint as to the number of rows that should be
   * fetched from the database when more rows are needed for ResultSet objects
   * generated by this Statement. If the value specified is zero, then the hint
   * is ignored. Test for a value of zero (hint is ignored) and a default is
   * specified.
   */
  @Test
  public void testSetFetchSizeZeroWithDefault() throws SQLException {
    final HiveConnection connection = mock(HiveConnection.class);
    final Iface iface = mock(Iface.class);
    final TSessionHandle handle = mock(TSessionHandle.class);

    try (HiveStatement stmt = new HiveStatement(connection, iface, handle, false, 10)) {
      assertEquals(10, stmt.getFetchSize());
    }
  }

  /**
   * Gives the JDBC driver a hint as to the number of rows that should be
   * fetched from the database when more rows are needed for ResultSet objects
   * generated by this Statement. The fetch size can be passed in through the
   * JDBC connection string whereby every created {@code Statement} will start
   * with the fetch size specified if no explicit calls are made.
   *
   * @see Utils.JdbcConnectionParams#FETCH_SIZE
   */
  @Test
  public void testSetFetchSizeJdbcProperty() throws SQLException {
    final HiveConnection connection = mock(HiveConnection.class);
    final Iface iface = mock(Iface.class);
    final TSessionHandle handle = mock(TSessionHandle.class);

    try (HiveStatement stmt = new HiveStatement(connection, iface, handle, false, 4)) {
      assertEquals(4, stmt.getFetchSize());
    }
  }

  /**
   * Gives the JDBC driver a hint as to the number of rows that should be
   * fetched from the database when more rows are needed for ResultSet objects
   * generated by this Statement. If the value specified is zero, then the hint
   * is ignored. If the hint is less than zero, an error is thrown.
   *
   * @see Statement#setFetchSize(int)
   */
  @Test(expected = SQLException.class)
  public void testSetFetchSizeNegativeValue() throws SQLException {
    final HiveConnection connection = mock(HiveConnection.class, withSettings().useConstructor());
    final Iface iface = mock(Iface.class);
    final TSessionHandle handle = mock(TSessionHandle.class);

    try (HiveStatement stmt = new HiveStatement(connection, iface, handle)) {
      stmt.setFetchSize(-1);
    }
  }

  @Test(expected = SQLFeatureNotSupportedException.class)
  public void testaddBatch() throws SQLException {
    final HiveConnection connection = mock(HiveConnection.class, withSettings().useConstructor());
    final Iface iface = mock(Iface.class);
    final TSessionHandle handle = mock(TSessionHandle.class);

    try (HiveStatement stmt = new HiveStatement(connection, iface, handle)) {
      stmt.addBatch(null);
    }
  }
}
