---
title: Migration Guide
sidebar_position: 10
id: java_migration
license: |
  Licensed to the Apache Software Foundation (ASF) under one or more
  contributor license agreements.  See the NOTICE file distributed with
  this work for additional information regarding copyright ownership.
  The ASF licenses this file to You under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with
  the License.  You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
---

This page covers migration strategies from JDK serialization and upgrading between Fory versions.

## JDK Serialization Migration

If you use JDK serialization before and can't upgrade your client and server at the same time (which is common for online applications), Fory provides a utility method `org.apache.fory.serializer.JavaSerializer.serializedByJDK` to check whether the binary was generated by JDK serialization.

You can use the following pattern to make existing serialization protocol-aware, then upgrade serialization to Fory in an async rolling-up way:

```java
if (JavaSerializer.serializedByJDK(bytes)) {
  ObjectInputStream objectInputStream = xxx;
  return objectInputStream.readObject();
} else {
  return fory.deserialize(bytes);
}
```

This allows you to:

1. Deploy new code that can read both JDK and Fory serialized data
2. Gradually migrate serialization to Fory
3. Eventually remove JDK serialization support

## Upgrade Fory

### Minor Version Upgrades

Currently, binary compatibility is ensured for minor versions only. For example, if you are using Fory `v0.2.0`, binary compatibility will be provided if you upgrade to Fory `v0.2.1`.

If you upgrade by minor version, or you won't have data serialized by older Fory, you can upgrade Fory directly without any special handling.

### Major Version Upgrades

If you upgrade to Fory `v0.4.1` from `v0.2.x`, no binary compatibility is ensured.

Most of the time there is no need to upgrade Fory to a newer major version—the current version is fast and compact enough, and we provide some minor fixes for recent older versions.

### Versioning Serialized Data

If you do want to upgrade Fory for better performance and smaller size, you need to write the Fory version as a header to serialized data using code like the following to keep binary compatibility:

#### Serialization with Version Header

```java
MemoryBuffer buffer = xxx;
buffer.writeVarInt32(2);  // Write Fory version
fory.serialize(buffer, obj);
```

#### Deserialization with Version Header

```java
MemoryBuffer buffer = xxx;
int foryVersion = buffer.readVarInt32();
Fory fory = getFory(foryVersion);
fory.deserialize(buffer);
```

`getFory` is a method to load the corresponding Fory version. You can shade and relocate different versions of Fory to different packages, and load Fory by version.

### Shading Example

```xml
<!-- Maven Shade Plugin configuration for multiple Fory versions -->
<plugin>
  <groupId>org.apache.maven.plugins</groupId>
  <artifactId>maven-shade-plugin</artifactId>
  <executions>
    <execution>
      <id>shade-fory-v1</id>
      <phase>package</phase>
      <goals>
        <goal>shade</goal>
      </goals>
      <configuration>
        <relocations>
          <relocation>
            <pattern>org.apache.fory</pattern>
            <shadedPattern>com.myapp.fory.v1</shadedPattern>
          </relocation>
        </relocations>
      </configuration>
    </execution>
  </executions>
</plugin>
```

## Best Practices

1. **Test thoroughly before upgrading**: Ensure compatibility with existing serialized data
2. **Use version headers for long-term storage**: If data persists across Fory versions
3. **Maintain backward compatibility**: Keep old Fory versions available for reading legacy data
4. **Monitor after upgrade**: Watch for deserialization errors in production

## Related Topics

- [Configuration Options](configuration.md) - ForyBuilder options
- [Schema Evolution](schema-evolution.md) - Handling class changes
- [Troubleshooting](troubleshooting.md) - Common migration issues
