"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JwtAuthentication = exports.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS = void 0;
var _authentication_type = require("../authentication_type");
var _routes = require("./routes");
var _cookie_splitter = require("../../../session/cookie_splitter");
var _jwt_helper = require("./jwt_helper");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const JWT_DEFAULT_EXTRA_STORAGE_OPTIONS = exports.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS = {
  cookiePrefix: 'security_authentication_jwt',
  additionalCookies: 5
};
class JwtAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    var _this$config$jwt;
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "type", 'jwt');
    _defineProperty(this, "authHeaderName", void 0);
    this.authHeaderName = ((_this$config$jwt = this.config.jwt) === null || _this$config$jwt === void 0 ? void 0 : _this$config$jwt.header.toLowerCase()) || 'authorization';
  }
  async init() {
    this.createExtraStorage();
    const routes = new _routes.JwtAuthRoutes(this.router, this.sessionStorageFactory, this.config);
    routes.setupRoutes();
  }
  createExtraStorage() {
    // @ts-ignore
    const hapiServer = this.sessionStorageFactory.asScoped({}).server;
    const {
      cookiePrefix,
      additionalCookies
    } = this.getExtraAuthStorageOptions();
    const extraCookieSettings = {
      isSecure: this.config.cookie.secure,
      isSameSite: this.config.cookie.isSameSite,
      password: this.config.cookie.password,
      domain: this.config.cookie.domain,
      path: this.coreSetup.http.basePath.serverBasePath || '/',
      clearInvalid: false,
      isHttpOnly: true,
      ignoreErrors: true,
      encoding: 'iron' // Same as hapi auth cookie
    };

    for (let i = 1; i <= additionalCookies; i++) {
      hapiServer.states.add(cookiePrefix + i, extraCookieSettings);
    }
  }
  getExtraAuthStorageOptions() {
    var _this$config$jwt2, _this$config$jwt3;
    const extraAuthStorageOptions = {
      cookiePrefix: ((_this$config$jwt2 = this.config.jwt) === null || _this$config$jwt2 === void 0 ? void 0 : _this$config$jwt2.extra_storage.cookie_prefix) || JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.cookiePrefix,
      additionalCookies: ((_this$config$jwt3 = this.config.jwt) === null || _this$config$jwt3 === void 0 ? void 0 : _this$config$jwt3.extra_storage.additional_cookies) || JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.additionalCookies,
      logger: this.logger
    };
    return extraAuthStorageOptions;
  }
  getTokenFromUrlParam(request) {
    var _this$config$jwt4;
    const urlParamName = (_this$config$jwt4 = this.config.jwt) === null || _this$config$jwt4 === void 0 ? void 0 : _this$config$jwt4.url_param;
    if (urlParamName) {
      const token = request.url.searchParams.get(urlParamName);
      return token || undefined;
    }
    return undefined;
  }
  getBearerToken(request) {
    const token = this.getTokenFromUrlParam(request);
    if (token) {
      return `Bearer ${token}`;
    }

    // no token in url parameter, try to get token from header
    return request.headers[this.authHeaderName] || undefined;
  }
  requestIncludesAuthInfo(request) {
    var _this$config$jwt5;
    if (request.headers[this.authHeaderName]) {
      return true;
    }
    const urlParamName = (_this$config$jwt5 = this.config.jwt) === null || _this$config$jwt5 === void 0 ? void 0 : _this$config$jwt5.url_param;
    if (urlParamName && request.url.searchParams.get(urlParamName)) {
      return true;
    }
    return false;
  }
  async getAdditionalAuthHeader(request) {
    const header = {};
    const token = this.getTokenFromUrlParam(request);
    if (token) {
      header[this.authHeaderName] = `Bearer ${token}`;
    }
    return header;
  }
  getCookie(request, authInfo) {
    (0, _cookie_splitter.setExtraAuthStorage)(request, this.getBearerToken(request) || '', this.getExtraAuthStorageOptions());
    return {
      username: authInfo.user_name,
      credentials: {
        authHeaderValueExtra: true
      },
      authType: this.type,
      expiryTime: (0, _jwt_helper.getExpirationDate)(this.getBearerToken(request), Date.now() + this.config.session.ttl)
    };
  }
  async isValidCookie(cookie, request) {
    var _cookie$credentials;
    const hasAuthHeaderValue = ((_cookie$credentials = cookie.credentials) === null || _cookie$credentials === void 0 ? void 0 : _cookie$credentials.authHeaderValue) || this.getExtraAuthStorageValue(request, cookie);
    return cookie.authType === this.type && cookie.username && cookie.expiryTime && hasAuthHeaderValue;
  }
  getKeepAliveExpiry(cookie, request) {
    return (0, _jwt_helper.getExpirationDate)(this.buildAuthHeaderFromCookie(cookie, request)[this.authHeaderName], Date.now() + this.config.session.ttl);
  }
  handleUnauthedRequest(request, response, toolkit) {
    return response.unauthorized();
  }
  getExtraAuthStorageValue(request, cookie) {
    var _cookie$credentials2;
    let extraValue = '';
    if (!((_cookie$credentials2 = cookie.credentials) !== null && _cookie$credentials2 !== void 0 && _cookie$credentials2.authHeaderValueExtra)) {
      return extraValue;
    }
    try {
      extraValue = (0, _cookie_splitter.getExtraAuthStorageValue)(request, this.getExtraAuthStorageOptions());
    } catch (error) {
      this.logger.info(error);
    }
    return extraValue;
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$credentials3;
    const header = {};
    if (cookie.credentials.authHeaderValueExtra) {
      try {
        const extraAuthStorageValue = this.getExtraAuthStorageValue(request, cookie);
        header.authorization = extraAuthStorageValue;
        return header;
      } catch (error) {
        this.logger.error(error);
      }
    }
    const authHeaderValue = (_cookie$credentials3 = cookie.credentials) === null || _cookie$credentials3 === void 0 ? void 0 : _cookie$credentials3.authHeaderValue;
    if (authHeaderValue) {
      header[this.authHeaderName] = authHeaderValue;
    }
    return header;
  }
}
exports.JwtAuthentication = JwtAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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