"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDslRoute = registerDslRoute;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _metrics = require("../metrics");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const performance = require('perf_hooks').performance;
function registerDslRoute(router, openSearchServiceSetup, dataSourceEnabled) {
  router.post({
    path: _common.ServiceEndpoints.GetSearchResults,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    const {
      query1,
      dataSourceId1,
      query2,
      dataSourceId2
    } = request.body;
    const actionName = query1 && query2 ? _metrics.METRIC_ACTION.COMPARISON_SEARCH : _metrics.METRIC_ACTION.SINGLE_SEARCH;
    const resBody = {};
    if (query1) {
      const {
        index,
        pipeline,
        size,
        ...rest
      } = query1;
      const params = pipeline !== '' ? {
        index,
        size,
        body: rest,
        search_pipeline: pipeline
      } : {
        index,
        size,
        body: rest
      };
      const start = performance.now();
      try {
        let resp;
        if (dataSourceEnabled && dataSourceId1) {
          const client = context.dataSource.opensearch.legacy.getClient(dataSourceId1);
          resp = await client.callAPI('search', params);
        } else {
          resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
        }
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, 200, end - start);
        resBody.result1 = resp;
      } catch (error) {
        var _error$body, _error$body2;
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, error.statusCode, end - start);
        if (error.statusCode !== 404) console.error(error);

        // Template: Error: {{Error.type}} – {{Error.reason}}
        const errorMessage = `Error: ${(_error$body = error.body) === null || _error$body === void 0 || (_error$body = _error$body.error) === null || _error$body === void 0 ? void 0 : _error$body.type} - ${(_error$body2 = error.body) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.error) === null || _error$body2 === void 0 ? void 0 : _error$body2.reason}`;
        resBody.errorMessage1 = {
          statusCode: error.statusCode || 500,
          body: errorMessage
        };
      }
    }
    if (query2) {
      const {
        index,
        pipeline,
        size,
        ...rest
      } = query2;
      const params = pipeline !== '' ? {
        index,
        size,
        body: rest,
        search_pipeline: pipeline
      } : {
        index,
        size,
        body: rest
      };
      const start = performance.now();
      try {
        let resp;
        if (dataSourceEnabled && dataSourceId2) {
          const client = context.dataSource.opensearch.legacy.getClient(dataSourceId2);
          resp = await client.callAPI('search', params);
        } else {
          resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
        }
        const end = performance.now();
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, 200, end - start);
        resBody.result2 = resp;
      } catch (error) {
        var _error$body3, _error$body4;
        const end = performance.now();
        if (error.statusCode !== 404) console.error(error);
        context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, actionName, error.statusCode, end - start);

        // Template: Error: {{Error.type}} – {{Error.reason}}
        const errorMessage = `Error: ${(_error$body3 = error.body) === null || _error$body3 === void 0 || (_error$body3 = _error$body3.error) === null || _error$body3 === void 0 ? void 0 : _error$body3.type} - ${(_error$body4 = error.body) === null || _error$body4 === void 0 || (_error$body4 = _error$body4.error) === null || _error$body4 === void 0 ? void 0 : _error$body4.reason}`;
        resBody.errorMessage2 = {
          statusCode: error.statusCode || 500,
          body: errorMessage
        };
      }
    }
    return response.ok({
      body: resBody
    });
  });

  // Get Indices
  router.get({
    path: `${_common.ServiceEndpoints.GetIndexes}/{dataSourceId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      format: 'json'
    };
    const start = performance.now();
    try {
      const dataSourceId = request.params.dataSourceId;
      let callApi;
      if (dataSourceEnabled && dataSourceId) {
        callApi = context.dataSource.opensearch.legacy.getClient(dataSourceId).callAPI;
      } else {
        callApi = context.core.opensearch.legacy.client.callAsCurrentUser;
      }
      const resp = await callApi('cat.indices', params);
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_INDEX, 200, end - start);
      return response.ok({
        body: resp
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_INDEX, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });

  // Get Pipelines
  router.get({
    path: `${_common.ServiceEndpoints.GetPipelines}/{dataSourceId?}`,
    validate: {
      params: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      format: 'json'
    };
    const start = performance.now();
    let resBody = {};
    let resp;
    try {
      const dataSourceId = request.params.dataSourceId;
      if (dataSourceEnabled && dataSourceId) {
        resp = (await context.dataSource.opensearch.getClient(dataSourceId)).transport;
        resp = await resp.request({
          method: 'GET',
          path: `${_common.SEARCH_API}/pipeline`
        });
      } else {
        resp = await context.core.opensearch.client.asCurrentUser.transport.request({
          method: 'GET',
          path: `${_common.SEARCH_API}/pipeline`
        });
      }
      resBody = resp.body;
      console.log('inside pipleine', resBody);
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_PIPELINE, 200, end - start);
      return response.ok({
        body: resBody
      });
    } catch (error) {
      const end = performance.now();
      context.searchRelevance.metricsService.addMetric(_metrics.METRIC_NAME.SEARCH_RELEVANCE, _metrics.METRIC_ACTION.FETCH_PIPELINE, error.statusCode, end - start);
      if (error.statusCode !== 404) console.error(error);
      console.log(error, 'inside error');
      return response.customError({
        statusCode: 404,
        body: error
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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