"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateReportDefinition = exports.validateReport = exports.regexReportName = exports.regexRelativeUrl = exports.regexEmailAddress = exports.regexDuration = exports.isValidRelativeUrl = void 0;
var _path = _interopRequireDefault(require("path"));
var _model = require("../../server/model");
var _constants = require("../../server/routes/utils/constants");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const isValidRelativeUrl = relativeUrl => {
  let normalizedRelativeUrl = relativeUrl;
  if (!relativeUrl.includes('observability#/notebooks') && !relativeUrl.includes('notebooks-dashboards')) {
    normalizedRelativeUrl = _path.default.posix.normalize(relativeUrl);
  }

  // check pattern
  // ODFE pattern: /app/dashboards#/view/7adfa750-4c81-11e8-b3d7-01146121b73d?_g
  // AES pattern: /_plugin/kibana/app/dashboards#/view/7adfa750-4c81-11e8-b3d7-01146121b73d?_g
  const isValid = regexRelativeUrl.test(normalizedRelativeUrl);
  return isValid;
};

/**
 * moment.js isValid() API fails to validate time duration, so use regex
 * https://github.com/moment/moment/issues/1805
 **/
exports.isValidRelativeUrl = isValidRelativeUrl;
const regexDuration = exports.regexDuration = /^([-+]?)P(?=\d|T[-+]?\d)(?:([-+]?\d+)Y)?(?:([-+]?\d+)M)?(?:([-+]?\d+)([DW]))?(?:T(?:([-+]?\d+)H)?(?:([-+]?\d+)M)?(?:([-+]?\d+(?:\.\d+)?)S)?)?$/;
const regexEmailAddress = exports.regexEmailAddress = /\S+@\S+\.\S+/;
const regexReportName = exports.regexReportName = /^[\w\-\s\(\)\[\]\,\_\-+]+$/;
const regexRelativeUrl = exports.regexRelativeUrl = /^\/(_plugin\/kibana\/|_dashboards\/)?app\/(dashboards|visualize|discover|discoverLegacy|data-explorer\/discover\/?|observability-dashboards|observability-notebooks|notebooks-dashboards\?view=output_only(&security_tenant=.+)?)(\?security_tenant=.+)?#\/(notebooks\/|view\/|edit\/)?[^\/]+$/;
const validateReport = async (client, report, basePath) => {
  report.query_url = report.query_url.replace(basePath, '');
  report.report_definition.report_params.core_params.base_url = report.report_definition.report_params.core_params.base_url.replace(basePath, '');
  // validate basic schema
  report = _model.reportSchema.validate(report);
  // parse to retrieve data
  const {
    query_url: queryUrl,
    report_definition: {
      report_params: {
        report_source: reportSource
      }
    }
  } = report;
  // Check if saved object actually exists
  await validateSavedObject(client, queryUrl, reportSource);
  return report;
};
exports.validateReport = validateReport;
const validateReportDefinition = async (client, reportDefinition, basePath) => {
  reportDefinition.report_params.core_params.base_url = reportDefinition.report_params.core_params.base_url.replace(basePath, '');
  // validate basic schema
  reportDefinition = _model.reportDefinitionSchema.validate(reportDefinition);
  // parse to retrieve data
  const {
    report_params: {
      report_source: reportSource,
      core_params: {
        base_url: baseUrl
      }
    }
  } = reportDefinition;
  // Check if saved object actually exists
  await validateSavedObject(client, baseUrl, reportSource);
  return reportDefinition;
};
exports.validateReportDefinition = validateReportDefinition;
const validateSavedObject = async (client, url, source) => {
  const getId = url => {
    var _url$split$pop;
    return (_url$split$pop = url.split('/').pop()) === null || _url$split$pop === void 0 ? void 0 : _url$split$pop.replace(/\?\S+$/, '');
  };
  const getType = source => {
    switch (source) {
      case _constants.REPORT_TYPE.dashboard:
        return 'dashboard';
      case _constants.REPORT_TYPE.savedSearch:
        return 'search';
      case _constants.REPORT_TYPE.visualization:
        return 'visualization';
      case _constants.REPORT_TYPE.notebook:
        return 'notebook';
    }
  };
  let exist = false;
  let savedObjectId = '';
  if (getType(source) === 'notebook') {
    // no backend check for notebooks because we would just be checking against the notebooks api again
    exist = true;
  } else {
    savedObjectId = `${getType(source)}:${getId(url)}`;
    const params = {
      index: '.kibana',
      id: savedObjectId
    };
    exist = await client.callAsCurrentUser('exists', params);
  }
  if (!exist) {
    throw Error(`saved object with id ${savedObjectId} does not exist`);
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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