"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metaData = exports.getSelectedFields = exports.getOpenSearchData = exports.convertToExcel = exports.convertToCSV = exports.buildRequestBody = void 0;
var _elasticBuilder = _interopRequireDefault(require("elastic-builder"));
var _json2Csv = _interopRequireDefault(require("json-2-csv"));
var _lodash = _interopRequireDefault(require("lodash"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _common = require("../../../../../src/plugins/data/common");
var _excelBuilder = require("./excelBuilder");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

var metaData = exports.metaData = {
  saved_search_id: null,
  report_format: null,
  start: null,
  end: null,
  fields: null,
  type: null,
  timeFieldName: null,
  sorting: null,
  fields_exist: false,
  selectedFields: [],
  paternName: null,
  searchSourceJSON: [],
  dateFields: []
};

// Get the selected columns by the user.
const getSelectedFields = async columns => {
  const selectedFields = [];
  let fields_exist = false;
  for (let column of columns) {
    if (column !== '_source') {
      fields_exist = true;
      selectedFields.push(column);
    } else {
      fields_exist = false;
      selectedFields.push('_source');
    }
  }
  metaData.fields_exist = fields_exist;
  metaData.selectedFields = selectedFields;
};

// Build the OpenSearch query from the meta data
// is_count is set to 1 if we building the count query but 0 if we building the fetch data query
exports.getSelectedFields = getSelectedFields;
const buildRequestBody = (report, allowLeadingWildcards, is_count) => {
  let esbBoolQuery = _elasticBuilder.default.boolQuery();
  const searchSourceJSON = report._source.searchSourceJSON;
  const savedObjectQuery = JSON.parse(searchSourceJSON).query;
  const savedObjectFilter = JSON.parse(searchSourceJSON).filter;
  const savedObjectConfig = {
    allowLeadingWildcards: allowLeadingWildcards,
    queryStringOptions: {},
    ignoreFilterIfFieldNotInIndex: false
  };
  const QueryFromSavedObject = (0, _common.buildOpenSearchQuery)(undefined, savedObjectQuery, savedObjectFilter, savedObjectConfig);
  // Add time range
  if (report._source.timeFieldName && report._source.timeFieldName.length > 0) {
    esbBoolQuery.must(_elasticBuilder.default.rangeQuery(report._source.timeFieldName).format('epoch_millis').gte(report._source.start - 1).lte(report._source.end + 1));
  }
  if (is_count) {
    return _elasticBuilder.default.requestBodySearch().query(esbBoolQuery);
  }

  // Add sorting to the query
  let esbSearchQuery = _elasticBuilder.default.requestBodySearch().query(esbBoolQuery).version(true);
  if (report._source.sorting.length > 0) {
    const sortings = report._source.sorting.map(element => {
      return _elasticBuilder.default.sort(element[0], element[1]);
    });
    esbSearchQuery.sorts(sortings);
  }

  // add selected fields to query
  if (report._source.fields_exist) {
    esbSearchQuery.source({
      includes: report._source.selectedFields
    });
  }
  // Add a customizer to merge queries to generate request body
  let requestBody = _lodash.default.mergeWith({
    query: QueryFromSavedObject
  }, esbSearchQuery.toJSON(), (objValue, srcValue) => {
    if (_lodash.default.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  });
  requestBody = addDocValueFields(report, requestBody);
  return requestBody;
};

// Fetch the data from OpenSearch
exports.buildRequestBody = buildRequestBody;
const getOpenSearchData = (arrayHits, report, params, dateFormat, timezone) => {
  let hits = [];
  for (let valueRes of arrayHits) {
    for (let data of valueRes.hits) {
      const fields = data.fields;
      // get all the fields of type date and format them to excel format
      let tempKeyElement = [];
      for (let dateField of report._source.dateFields) {
        let keys;
        keys = dateField.split('.');
        const dateValue = data._source[dateField];
        const fieldDateValue = fields !== undefined ? fields[dateField] : undefined;
        const isDateFieldPresent = isKeyPresent(data._source, dateField);
        if (isDateFieldPresent) {
          // if its not a nested date field
          if (keys.length === 1) {
            // if conditions to determine if the date field's value is an array or a string
            if (typeof dateValue === 'string') {
              data._source[keys] = _momentTimezone.default.utc(dateValue).tz(timezone).format(dateFormat);
            } else if (dateValue.length !== 0 && dateValue instanceof Array && fieldDateValue !== undefined) {
              fieldDateValue.forEach((element, index) => {
                data._source[keys][index] = _momentTimezone.default.utc(element).tz(timezone).format(dateFormat);
              });
            } else {
              data._source[keys] = [];
            }
            // else to cover cases with nested date fields
          } else {
            let keyElement = keys.shift();
            // if conditions to determine if the date field's value is an array or a string
            if (fieldDateValue !== undefined && typeof fieldDateValue === 'string') {
              keys.push(_momentTimezone.default.utc(fieldDateValue).tz(timezone).format(dateFormat));
            } else if (dateValue.length !== 0 && dateValue instanceof Array && fieldDateValue !== undefined) {
              let tempArray = [];
              fieldDateValue.forEach(index => {
                tempArray.push(_momentTimezone.default.utc(index).tz(timezone).format(dateFormat));
              });
              keys.push(tempArray);
            } else {
              keys.push([]);
            }
            const nestedJSON = arrayToNestedJSON(keys);
            let keyLength = Object.keys(data._source);
            // to check if the nested field have anyother keys apart from date field
            if (tempKeyElement.includes(keyElement) || keyLength.length > 1) {
              data._source[keyElement] = {
                ...data._source[keyElement],
                ...nestedJSON
              };
            } else {
              data._source[keyElement] = nestedJSON;
              tempKeyElement.push(keyElement);
            }
          }
        }
      }
      delete data['fields'];
      if (report._source.fields_exist === true) {
        let result = traverse(data, report._source.selectedFields);
        hits.push(params.excel ? sanitize(result) : result);
      } else {
        hits.push(params.excel ? sanitize(data) : data);
      }
      // Truncate to expected limit size
      if (hits.length >= params.limit) {
        return hits;
      }
    }
  }
  return hits;
};

// Convert the data to Csv format
exports.getOpenSearchData = getOpenSearchData;
const convertToCSV = async (dataset, csvSeparator) => {
  let convertedData = [];
  const options = {
    delimiter: {
      field: csvSeparator,
      eol: '\n'
    },
    emptyFieldValue: ' '
  };
  await _json2Csv.default.json2csvAsync(dataset[0], options).then(csv => {
    convertedData = csv;
  });
  return convertedData;
};
exports.convertToCSV = convertToCSV;
function flattenHits(hits, result = {}, prefix = '') {
  for (const [key, value] of Object.entries(hits)) {
    if (!hits.hasOwnProperty(key)) continue;
    if (value != null && typeof value === 'object' && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenHits(value, result, prefix + key + '.');
    } else {
      result[prefix.replace(/^_source\./, '') + key] = value;
    }
  }
  return result;
}
function flattenObject(obj = {}, parentKey = '', result = {}) {
  for (const [key, value] of Object.entries(obj)) {
    const newKey = parentKey ? `${parentKey}.${key}` : key;
    if (typeof value == 'object' && value !== null && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenObject(value, newKey, result);
    } else if (Array.isArray(value)) {
      result[newKey] = JSON.stringify(value);
    } else {
      result[newKey] = value;
    }
  }
  return result;
}
function flattenArray(array = []) {
  return array.map(item => flattenObject(item));
}
const convertToExcel = async dataset => {
  const flatDataset = flattenArray(dataset[0]);
  const excelBuilder = new _excelBuilder.ExcelBuilder();
  const base64 = await excelBuilder.addHeaders(flatDataset).addRows(flatDataset).updateColumnWidths().applyHeaderStyles().applyAutoFilter().applyFreeze().getWorkbookAsBase64();
  return 'data:application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;base64,' + base64;
};

//Return only the selected fields
exports.convertToExcel = convertToExcel;
function traverse(data, keys, result = {}) {
  data = flattenHits(data);
  const sourceKeys = Object.keys(data);
  keys.forEach(key => {
    const value = _lodash.default.get(data, key, undefined);
    if (value !== undefined) result[key] = value;else {
      Object.keys(data).filter(sourceKey => sourceKey.startsWith(key + '.')).forEach(sourceKey => result[sourceKey] = data[sourceKey]);
    }
  });
  return result;
}

/**
 * Escape special characters if field value prefixed with.
 * This is intend to avoid CSV injection in Microsoft Excel.
 * @param doc   document
 */
function sanitize(doc) {
  for (const field in doc) {
    if (doc[field] == null) continue;
    if (doc[field].toString().startsWith('+') || doc[field].toString().startsWith('-') && typeof doc[field] !== 'number' || doc[field].toString().startsWith('=') || doc[field].toString().startsWith('@')) {
      doc[field] = "'" + doc[field];
    }
  }
  return doc;
}
function arrayToNestedJSON(arr) {
  if (arr.length === 0) {
    return null;
  } else if (arr.length === 1) {
    return arr[0];
  } else {
    const key = arr[0];
    const rest = arr.slice(1);
    return {
      [key]: arrayToNestedJSON(rest)
    };
  }
}
function isKeyPresent(data, key) {
  if (typeof data === 'object' && data !== null) {
    if (key in data) {
      return true;
    }
    for (const value of Object.values(data)) {
      if (isKeyPresent(value, key)) {
        return true;
      }
    }
  }
  return false;
}
const addDocValueFields = (report, requestBody) => {
  const docValues = [];
  for (const dateType of report._source.dateFields) {
    docValues.push({
      field: dateType,
      format: 'date_hour_minute_second_fraction'
    });
  }
  // elastic-builder doesn't provide function to build docvalue_fields with format,
  // this is a workaround which appends docvalues field to the request body.
  requestBody = {
    ...requestBody,
    docvalue_fields: docValues
  };
  return requestBody;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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