"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearParagraphs = clearParagraphs;
exports.createNotebook = createNotebook;
exports.createParagraphs = createParagraphs;
exports.deleteParagraphs = deleteParagraphs;
exports.fetchNotebook = fetchNotebook;
exports.runParagraph = runParagraph;
exports.updateFetchParagraph = updateFetchParagraph;
exports.updateParagraphs = updateParagraphs;
exports.updateRunFetchParagraph = updateRunFetchParagraph;
var _performanceNow = _interopRequireDefault(require("performance-now"));
var _uuid = require("uuid");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _query_helpers = require("../../common/helpers/notebooks/query_helpers");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function createNotebook(paragraphInput, inputType) {
  try {
    let paragraphType = 'MARKDOWN';
    if (inputType === 'VISUALIZATION') {
      paragraphType = 'VISUALIZATION';
    }
    if (inputType === 'OBSERVABILITY_VISUALIZATION') {
      paragraphType = 'OBSERVABILITY_VISUALIZATION';
    }
    if (paragraphInput.substring(0, 3) === '%sql' || paragraphInput.substring(0, 3) === '%ppl') {
      paragraphType = 'QUERY';
    }
    const inputObject = {
      inputType: paragraphType,
      inputText: paragraphInput
    };
    const outputObjects = [{
      outputType: paragraphType,
      result: '',
      execution_time: '0s'
    }];
    const newParagraph = {
      id: 'paragraph_' + (0, _uuid.v4)(),
      dateCreated: new Date().toISOString(),
      dateModified: new Date().toISOString(),
      input: inputObject,
      output: outputObjects
    };
    return newParagraph;
  } catch (error) {
    throw new Error('Create Paragraph Error:' + error);
  }
}
async function fetchNotebook(noteId, opensearchNotebooksClient) {
  try {
    const notebook = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId);
    return notebook;
  } catch (error) {
    throw new Error('update Paragraph Error:' + error);
  }
}
async function createParagraphs(params, opensearchNotebooksClient) {
  const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const paragraphs = notebookinfo.attributes.savedNotebook.paragraphs;
  const newParagraph = createNotebook(params.paragraphInput, params.inputType);
  paragraphs.splice(params.paragraphIndex, 0, newParagraph);
  const updateNotebook = {
    paragraphs,
    dateModified: new Date().toISOString()
  };
  await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
    savedNotebook: updateNotebook
  });
  await fetchNotebook(params.noteId, opensearchNotebooksClient);
  return newParagraph;
}
async function clearParagraphs(params, opensearchNotebooksClient) {
  const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const updatedparagraphs = [];
  notebookinfo.attributes.savedNotebook.paragraphs.map(paragraph => {
    const updatedParagraph = {
      ...paragraph
    };
    updatedParagraph.output = [];
    updatedparagraphs.push(updatedParagraph);
  });
  const updateNotebook = {
    paragraphs: updatedparagraphs,
    dateModified: new Date().toISOString()
  };
  try {
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    return {
      paragraphs: updatedparagraphs
    };
  } catch (error) {
    throw new Error('Clear Paragraph Error:' + error);
  }
}
async function deleteParagraphs(params, opensearchNotebooksClient) {
  const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const updatedparagraphs = [];
  if (params.paragraphId !== undefined) {
    notebookinfo.attributes.savedNotebook.paragraphs.map(paragraph => {
      if (paragraph.id !== params.paragraphId) {
        updatedparagraphs.push(paragraph);
      }
    });
  }
  const updateNotebook = {
    paragraphs: updatedparagraphs,
    dateModified: new Date().toISOString()
  };
  try {
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    return {
      paragraphs: updatedparagraphs
    };
  } catch (error) {
    throw new Error('update Paragraph Error:' + error);
  }
}
async function updateRunFetchParagraph(params, opensearchNotebooksClient) {
  try {
    const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
    const updatedInputParagraphs = updateParagraphs(notebookinfo.attributes.savedNotebook.paragraphs, params.paragraphId, params.paragraphInput, params.paragraphType, params.dataSourceMDSId, params.dataSourceMDSLabel);
    const updatedOutputParagraphs = await runParagraph(updatedInputParagraphs, params.paragraphId);
    const updateNotebook = {
      paragraphs: updatedOutputParagraphs,
      dateModified: new Date().toISOString()
    };
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    let resultParagraph = {};
    let index = 0;
    for (index = 0; index < updatedOutputParagraphs.length; ++index) {
      if (params.paragraphId === updatedOutputParagraphs[index].id) {
        resultParagraph = updatedOutputParagraphs[index];
      }
    }
    return resultParagraph;
  } catch (error) {
    throw new Error('Update/Run Paragraph Error:' + error);
  }
}
function runParagraph(paragraphs, paragraphId) {
  try {
    const updatedParagraphs = [];
    let index = 0;
    for (index = 0; index < paragraphs.length; ++index) {
      const startTime = (0, _performanceNow.default)();
      const updatedParagraph = {
        ...paragraphs[index]
      };
      if (paragraphs[index].id === paragraphId) {
        updatedParagraph.dateModified = new Date().toISOString();
        if ((0, _query_helpers.inputIsQuery)(paragraphs[index].input.inputText)) {
          updatedParagraph.output = [{
            outputType: 'QUERY',
            result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (paragraphs[index].input.inputText.substring(0, 3) === '%md') {
          updatedParagraph.output = [{
            outputType: 'MARKDOWN',
            result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (paragraphs[index].input.inputType === 'VISUALIZATION') {
          updatedParagraph.dateModified = new Date().toISOString();
          updatedParagraph.output = [{
            outputType: 'VISUALIZATION',
            result: '',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (paragraphs[index].input.inputType === 'OBSERVABILITY_VISUALIZATION') {
          updatedParagraph.dateModified = new Date().toISOString();
          updatedParagraph.output = [{
            outputType: 'OBSERVABILITY_VISUALIZATION',
            result: '',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if ((0, _query_helpers.formatNotRecognized)(paragraphs[index].input.inputText)) {
          updatedParagraph.output = [{
            outputType: 'MARKDOWN',
            result: 'Please select an input type (%sql, %ppl, or %md)',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        }
      }
      updatedParagraphs.push(updatedParagraph);
    }
    return updatedParagraphs;
  } catch (error) {
    throw new Error('Running Paragraph Error:' + error);
  }
}
function updateParagraphs(paragraphs, paragraphId, paragraphInput, paragraphType, dataSourceMDSId, dataSourceMDSLabel) {
  try {
    const updatedParagraphs = [];
    paragraphs.map(paragraph => {
      const updatedParagraph = {
        ...paragraph
      };
      if (paragraph.id === paragraphId) {
        updatedParagraph.dataSourceMDSId = dataSourceMDSId;
        updatedParagraph.dataSourceMDSLabel = dataSourceMDSLabel;
        updatedParagraph.dateModified = new Date().toISOString();
        updatedParagraph.input.inputText = paragraphInput;
        if (paragraphType.length > 0) {
          updatedParagraph.input.inputType = paragraphType;
        }
      }
      updatedParagraphs.push(updatedParagraph);
    });
    return updatedParagraphs;
  } catch (error) {
    throw new Error('Update Paragraph Error:' + error);
  }
}
async function updateFetchParagraph(params, opensearchNotebooksClient) {
  try {
    const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
    const updatedInputParagraphs = updateParagraphs(notebookinfo.attributes.savedNotebook.paragraphs, params.paragraphId, params.paragraphInput);
    const updateNotebook = {
      paragraphs: updatedInputParagraphs,
      dateModified: new Date().toISOString()
    };
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    let resultParagraph = {};
    updatedInputParagraphs.map(paragraph => {
      if (params.paragraphId === paragraph.id) {
        resultParagraph = paragraph;
      }
    });
    return resultParagraph;
  } catch (error) {
    throw new Error('update Paragraph Error:' + error);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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