"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FileSystemDataAdaptor = void 0;
var fs = _interopRequireWildcard(require("fs/promises"));
var _path = _interopRequireDefault(require("path"));
var _json = _interopRequireDefault(require("json5"));
var _utils = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
// Check if a location is a directory without an exception if location not found
const safeIsDirectory = async maybeDirectory => {
  try {
    return (await fs.lstat(maybeDirectory)).isDirectory();
  } catch (_err) {
    return false;
  }
};

/**
 * A CatalogDataAdaptor that reads from the local filesystem.
 * Used to read default Integrations shipped in the in-product catalog at `__data__`.
 */
class FileSystemDataAdaptor {
  /**
   * Creates a new FileSystemCatalogDataAdaptor instance.
   *
   * @param directory The base directory from which to read files. This is not sanitized.
   */
  constructor(directory) {
    _defineProperty(this, "isConfigLocalized", false);
    _defineProperty(this, "directory", void 0);
    this.directory = directory;
  }
  async readFile(filename, type) {
    let content;
    try {
      content = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename), {
        encoding: 'utf-8'
      });
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
    // First try to parse as JSON, then NDJSON, then fail.
    try {
      const parsed = _json.default.parse(content);
      return {
        ok: true,
        value: parsed
      };
    } catch (err) {
      const parsed = await (0, _utils.tryParseNDJson)(content);
      if (parsed) {
        return {
          ok: true,
          value: parsed
        };
      }
      return {
        ok: false,
        error: new Error('Unable to parse file as JSON or NDJson', {
          cause: err
        })
      };
    }
  }
  async readFileRaw(filename, type) {
    try {
      const buffer = await fs.readFile(_path.default.join(this.directory, type !== null && type !== void 0 ? type : '.', filename));
      return {
        ok: true,
        value: buffer
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }
  async findIntegrations(dirname = '.') {
    try {
      const integrations = [];
      await this.collectIntegrationsRecursive(dirname, integrations);
      return {
        ok: true,
        value: integrations
      };
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
  }
  async collectIntegrationsRecursive(dirname, integrations) {
    const entries = await fs.readdir(_path.default.join(this.directory, dirname));
    for (const entry of entries) {
      const fullPath = _path.default.join(dirname, entry);
      const isDirectory = (await this.getDirectoryType(fullPath)) === 'integration';
      if (isDirectory) {
        integrations.push(fullPath);
      } else if ((await this.getDirectoryType(fullPath)) === 'repository') {
        await this.collectIntegrationsRecursive(fullPath, integrations);
      }
    }
  }
  async findIntegrationVersions(dirname = '.') {
    let files;
    const integPath = _path.default.join(this.directory, dirname);
    try {
      files = await fs.readdir(integPath);
    } catch (err) {
      return {
        ok: false,
        error: err
      };
    }
    const versions = [];
    for (const file of files) {
      // TODO handle nested repositories -- assumes integrations are 1 level deep
      if (_path.default.extname(file) === '.json' && file.startsWith(`${_path.default.basename(integPath)}-`)) {
        const version = file.substring(_path.default.basename(integPath).length + 1, file.length - 5);
        if (!version.match(/^\d+(\.\d+)*$/)) {
          continue;
        }
        versions.push(version);
      }
    }
    return {
      ok: true,
      value: versions
    };
  }
  async getDirectoryType(dirname) {
    const isDir = await safeIsDirectory(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.'));
    if (!isDir) {
      return 'unknown';
    }
    // Sloppily just check for one mandatory integration directory to distinguish.
    // Improve if we need to distinguish a repository with an integration named "schemas".
    const hasSchemas = await safeIsDirectory(_path.default.join(this.directory, dirname !== null && dirname !== void 0 ? dirname : '.', 'schemas'));
    return hasSchemas ? 'integration' : 'repository';
  }
  join(filename) {
    return new FileSystemDataAdaptor(_path.default.join(this.directory, filename));
  }
}
exports.FileSystemDataAdaptor = FileSystemDataAdaptor;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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