"use strict";
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.realshortpathSync = exports.realShortPathSync = exports.realpathSync = exports.realPathSync = exports.resolveToShortNameSync = exports.resolveToShortPathSync = exports.resolveToFullNameSync = exports.resolveToFullPathSync = exports.shortNameSupportedSync = exports.shortNamesSupportedSync = exports.standardize = exports.NAMESPACE_PREFIX = void 0;
const child_process_1 = require("child_process");
const path_1 = require("path");
const fs_1 = require("fs");
exports.NAMESPACE_PREFIX = process.platform === 'win32' ? '\\\\?\\' : '';
/**
 * Get a standardized reference to a path
 * @param {string} path - the path to standardize
 * @param {boolean} [usePosix=true] - produce a posix reference
 * @param {boolean} [escapedBackslashes=true] - on Windows, double-backslash the reference
 * @param {boolean} [returnUNC=false] - produce an extended reference
 */
exports.standardize = (path, usePosix = true, escapedBackslashes = true, returnUNC = false) => {
    // Force os-dependant separators
    const normal = path_1.normalize(path);
    // Filter out in-browser executions as well as non-windows ones
    if (process?.platform !== 'win32')
        return normal;
    if (usePosix)
        return normal.replace(/\\/g, '/');
    else if (escapedBackslashes)
        return normal.replace(/\\/g, '\\\\');
    else if (returnUNC)
        return '\\\\?\\' + normal;
    return normal;
};
/**
 * Windows-only function that uses PowerShell to calculate the full path
 * @param {string} path
 * @private
 */
const getFullPathSync = (path) => {
    if (process.platform !== 'win32')
        return path;
    try {
        const fullName = child_process_1.execSync(`powershell "(Get-Item -LiteralPath '${path}').FullName"`, {
            encoding: 'utf8',
        })?.trim?.();
        // Make sure we got something back
        if (fullName?.length > 2)
            return fullName;
    }
    catch (ex) {
        // Do nothing
    }
    return path;
};
/**
 * Windows-only function that uses PowerShell and Com Object to calculate the 8.3 path
 * @param {string} path
 * @private
 */
const getShortPathSync = (path) => {
    if (process.platform !== 'win32')
        return path;
    try {
        const shortPath = child_process_1.execSync(`powershell "$FSO = New-Object -ComObject Scripting.FileSystemObject; $O = (Get-Item -LiteralPath '${path}'); if ($O.PSIsContainer) { $FSO.GetFolder($O.FullName).ShortPath } else { $FSO.GetFile($O.FullName).ShortPath }"`, {
            encoding: 'utf8',
        })?.trim?.();
        // Make sure we got something back
        if (shortPath?.length > 2)
            return shortPath;
    }
    catch (ex) {
        // Do nothing
    }
    return path;
};
/**
 * Checks if Windows 8.3 short names are supported on the volume of the given path
 * @param {string} [path='.'] - the path to examine
 */
exports.shortNamesSupportedSync = (path = '.') => {
    if (process.platform !== 'win32')
        return false;
    const testFileName = '.___osd-cross-platform-test.file';
    const file = path_1.resolve(path, testFileName);
    // Create a test file if it doesn't exist
    if (!fs_1.existsSync(file))
        fs_1.closeSync(fs_1.openSync(file, 'w'));
    // If the returned value's basename is not the same as the requested file name, it must be a short name
    const foundShortName = path_1.basename(getShortPathSync(file)) !== testFileName;
    // Cleanup
    fs_1.unlinkSync(file);
    return foundShortName;
};
/**
 * @borrows shortNamesSupportedSync
 */
exports.shortNameSupportedSync = exports.shortNamesSupportedSync;
/**
 * Get the full pathname
 * @param {string} path - the path to resolve
 */
exports.resolveToFullPathSync = (path) => getFullPathSync(path_1.resolve(path));
/**
 * @borrows resolveToFullPathSync
 */
exports.resolveToFullNameSync = exports.resolveToFullPathSync;
/**
 * Get the short pathname
 * @param {string} path - the path to resolve
 */
exports.resolveToShortPathSync = (path) => getShortPathSync(path_1.resolve(path));
/**
 * @borrows resolveToShortPathSync
 */
exports.resolveToShortNameSync = exports.resolveToShortPathSync;
/**
 * Get the canonical pathname
 * @param {string} path - the path to resolve
 */
exports.realPathSync = (path) => getFullPathSync(fs_1.realpathSync(path, 'utf8'));
/**
 * @borrows realPathSync
 */
exports.realpathSync = exports.realPathSync;
/**
 * Get the canonical pathname
 * @param {string} path - the path to resolve
 */
exports.realShortPathSync = (path) => getShortPathSync(fs_1.realpathSync(path, 'utf8'));
/**
 * @borrows realShortPathSync
 */
exports.realshortpathSync = exports.realShortPathSync;
