"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.cloneMaterial = void 0;
const cryptographic_material_1 = require("./cryptographic_material");
const node_algorithms_1 = require("./node_algorithms");
const needs_1 = require("./needs");
function cloneMaterial(source) {
    const { suite, encryptionContext } = source;
    const clone = (suite instanceof node_algorithms_1.NodeAlgorithmSuite
        ? source instanceof cryptographic_material_1.NodeEncryptionMaterial
            ? new cryptographic_material_1.NodeEncryptionMaterial(suite, encryptionContext)
            : new cryptographic_material_1.NodeDecryptionMaterial(suite, encryptionContext)
        : source instanceof cryptographic_material_1.WebCryptoEncryptionMaterial
            ? new cryptographic_material_1.WebCryptoEncryptionMaterial(suite, encryptionContext)
            : new cryptographic_material_1.WebCryptoDecryptionMaterial(suite, encryptionContext));
    /* The setTrace _must_ be the first trace,
     * If the material is an EncryptionMaterial
     * then the data key *must* have been generated.
     * If the material is DecryptionMaterial
     * then the data key *must* have been decrypted.
     * i.e. the required flags are:
     * WRAPPING_KEY_GENERATED_DATA_KEY, WRAPPING_KEY_DECRYPTED_DATA_KEY
     * These are controlled by the material itself.
     * Furthermore, subsequent trace entries,
     * *must* be in the same order as the added encrypted data keys.
     * See cryptographic_materials.ts `decorateCryptographicMaterial`, `decorateWebCryptoMaterial`.
     */
    const [setTrace, ...traces] = source.keyringTrace.slice();
    if (source.hasUnencryptedDataKey) {
        const udk = cloneUnencryptedDataKey(source.getUnencryptedDataKey());
        clone.setUnencryptedDataKey(udk, setTrace);
    }
    if (source.hasCryptoKey) {
        const cryptoKey = source.getCryptoKey();
        clone.setCryptoKey(cryptoKey, setTrace);
    }
    if ((0, cryptographic_material_1.isEncryptionMaterial)(source) && (0, cryptographic_material_1.isEncryptionMaterial)(clone)) {
        const encryptedDataKeys = source.encryptedDataKeys;
        /* Precondition: For each encrypted data key, there must be a trace. */
        (0, needs_1.needs)(encryptedDataKeys.length === traces.length, 'KeyringTrace length does not match encrypted data keys.');
        encryptedDataKeys.forEach((edk, i) => {
            const { providerInfo, providerId } = edk;
            const { keyNamespace, keyName, flags } = traces[i];
            /* Precondition: The traces must be in the same order as the encrypted data keys. */
            (0, needs_1.needs)(keyName === providerInfo && keyNamespace === providerId, 'Keyring trace does not match encrypted data key.');
            clone.addEncryptedDataKey(edk, flags);
        });
        if (source.suite.signatureCurve && source.signatureKey) {
            clone.setSignatureKey(source.signatureKey);
        }
    }
    else if ((0, cryptographic_material_1.isDecryptionMaterial)(source) && (0, cryptographic_material_1.isDecryptionMaterial)(clone)) {
        /* Precondition: On Decrypt there must not be any additional traces other than the setTrace. */
        (0, needs_1.needs)(!traces.length, 'Only 1 trace is valid on DecryptionMaterials.');
        if (source.suite.signatureCurve && source.verificationKey) {
            clone.setVerificationKey(source.verificationKey);
        }
    }
    else {
        throw new Error('Material mismatch');
    }
    return clone;
}
exports.cloneMaterial = cloneMaterial;
function cloneUnencryptedDataKey(dataKey) {
    if (dataKey instanceof Uint8Array) {
        return new Uint8Array(dataKey);
    }
    return dataKey;
}
//# sourceMappingURL=data:application/json;base64,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