"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.identifier2ClientBuilder = exports.getAwsKmsMrkAwareStrictMultiKeyringBuilder = void 0;
const material_management_1 = require("@aws-crypto/material-management");
const arn_parsing_1 = require("./arn_parsing");
const aws_kms_mrk_are_unique_1 = require("./aws_kms_mrk_are_unique");
function getAwsKmsMrkAwareStrictMultiKeyringBuilder(MrkAwareKeyring, MultiKeyring, defaultClientProvider) {
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
    //# The caller MUST provide:
    return function buildAwsKmsMrkAwareStrictMultiKeyring({ generatorKeyId, keyIds = [], 
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
    //# If
    //# a regional client supplier is not passed, then a default MUST be
    //# created that takes a region string and generates a default AWS SDK
    //# client for the given region.
    clientProvider = defaultClientProvider, grantTokens, } = {}) {
        const identifier2Client = identifier2ClientBuilder(clientProvider);
        const allIdentifiers = generatorKeyId ? [generatorKeyId, ...keyIds] : keyIds;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# At least one non-null or non-empty string AWS
        //# KMS key identifiers exists in the input this function MUST fail.
        //
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# If any of the AWS KMS key identifiers is null or an empty string this
        //# function MUST fail.
        (0, material_management_1.needs)(allIdentifiers.length &&
            allIdentifiers.every((key) => typeof key === 'string' && key !== ''), 'Noop keyring is not allowed: Set a generatorKeyId or at least one keyId.');
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# All
        //# AWS KMS identifiers are passed to Assert AWS KMS MRK are unique (aws-
        //# kms-mrk-are-unique.md#Implementation) and the function MUST return
        //# success otherwise this MUST fail.
        (0, aws_kms_mrk_are_unique_1.awsKmsMrkAreUnique)(allIdentifiers);
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# If there is a generator input then the generator keyring MUST be a
        //# AWS KMS MRK Aware Symmetric Keyring (aws-kms-mrk-aware-symmetric-
        //# keyring.md) initialized with
        const generator = generatorKeyId
            ? new MrkAwareKeyring({
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
                //# *  The AWS KMS client that MUST be created by the regional client
                //# supplier when called with the region part of the generator ARN or
                //# a signal for the AWS SDK to select the default region.
                client: identifier2Client(generatorKeyId),
                keyId: generatorKeyId,
                grantTokens,
            })
            : undefined;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# If there is a set of child identifiers then a set of AWS KMS MRK
        //# Aware Symmetric Keyring (aws-kms-mrk-aware-symmetric-keyring.md) MUST
        //# be created for each AWS KMS key identifier by initialized each
        //# keyring with
        const children = keyIds.map((keyId) => {
            return new MrkAwareKeyring({
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
                //# *  The AWS KMS client that MUST be created by the regional client
                //# supplier when called with the region part of the AWS KMS key
                //# identifier or a signal for the AWS SDK to select the default
                //# region.
                client: identifier2Client(keyId),
                keyId,
                grantTokens,
            });
        });
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# Then a Multi-Keyring (../multi-keyring.md#inputs) MUST be initialize
        //# by using this generator keyring as the generator keyring (../multi-
        //# keyring.md#generator-keyring) and this set of child keyrings as the
        //# child keyrings (../multi-keyring.md#child-keyrings).
        //
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# This Multi-
        //# Keyring MUST be this functions output.
        return new MultiKeyring({
            generator,
            children,
        });
    };
}
exports.getAwsKmsMrkAwareStrictMultiKeyringBuilder = getAwsKmsMrkAwareStrictMultiKeyringBuilder;
function identifier2ClientBuilder(clientProvider) {
    return function identifier2Client(identifier) {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.6
        //# NOTE: The AWS Encryption SDK SHOULD NOT attempt to evaluate its own
        //# default region.
        const region = (0, arn_parsing_1.getRegionFromIdentifier)(identifier);
        const client = clientProvider(region);
        /* Postcondition: If the configured clientProvider is not able to create a client for a defined generator key, throw an error. */
        (0, material_management_1.needs)(client, `Configured clientProvider is unable to create a client for configured ${identifier}.`);
        return client;
    };
}
exports.identifier2ClientBuilder = identifier2ClientBuilder;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia21zX21ya19zdHJpY3RfbXVsdGlfa2V5cmluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9rbXNfbXJrX3N0cmljdF9tdWx0aV9rZXlyaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxvRUFBb0U7QUFDcEUsc0NBQXNDOzs7QUFFdEMseUVBS3dDO0FBR3hDLCtDQUF1RDtBQUN2RCxxRUFBNkQ7QUFrQjdELFNBQWdCLDBDQUEwQyxDQUl4RCxlQUFvRSxFQUNwRSxZQUFzQyxFQUN0QyxxQkFBZ0Q7SUFFaEQseUVBQXlFO0lBQ3pFLDRCQUE0QjtJQUM1QixPQUFPLFNBQVMscUNBQXFDLENBQUMsRUFDcEQsY0FBYyxFQUNkLE1BQU0sR0FBRyxFQUFFO0lBQ1gseUVBQXlFO0lBQ3pFLE1BQU07SUFDTixvRUFBb0U7SUFDcEUsc0VBQXNFO0lBQ3RFLGdDQUFnQztJQUNoQyxjQUFjLEdBQUcscUJBQXFCLEVBQ3RDLFdBQVcsTUFDc0MsRUFBRTtRQUNuRCxNQUFNLGlCQUFpQixHQUFHLHdCQUF3QixDQUFDLGNBQWMsQ0FBQyxDQUFBO1FBRWxFLE1BQU0sY0FBYyxHQUFHLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLEVBQUUsR0FBRyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFBO1FBRTVFLHlFQUF5RTtRQUN6RSxpREFBaUQ7UUFDakQsb0VBQW9FO1FBQ3BFLEVBQUU7UUFDRix5RUFBeUU7UUFDekUseUVBQXlFO1FBQ3pFLHVCQUF1QjtRQUN2QixJQUFBLDJCQUFLLEVBQ0gsY0FBYyxDQUFDLE1BQU07WUFDbkIsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsRUFBRSxFQUFFLENBQUMsT0FBTyxHQUFHLEtBQUssUUFBUSxJQUFJLEdBQUcsS0FBSyxFQUFFLENBQUMsRUFDdEUsMEVBQTBFLENBQzNFLENBQUE7UUFFRCx5RUFBeUU7UUFDekUsT0FBTztRQUNQLHlFQUF5RTtRQUN6RSxzRUFBc0U7UUFDdEUscUNBQXFDO1FBQ3JDLElBQUEsMkNBQWtCLEVBQUMsY0FBYyxDQUFDLENBQUE7UUFFbEMseUVBQXlFO1FBQ3pFLHNFQUFzRTtRQUN0RSxxRUFBcUU7UUFDckUsZ0NBQWdDO1FBQ2hDLE1BQU0sU0FBUyxHQUFHLGNBQWM7WUFDOUIsQ0FBQyxDQUFDLElBQUksZUFBZSxDQUFDO2dCQUNsQix5RUFBeUU7Z0JBQ3pFLHFFQUFxRTtnQkFDckUscUVBQXFFO2dCQUNyRSwwREFBMEQ7Z0JBQzFELE1BQU0sRUFBRSxpQkFBaUIsQ0FBQyxjQUFjLENBQUM7Z0JBQ3pDLEtBQUssRUFBRSxjQUFjO2dCQUNyQixXQUFXO2FBQ1osQ0FBQztZQUNKLENBQUMsQ0FBQyxTQUFTLENBQUE7UUFFYix5RUFBeUU7UUFDekUsb0VBQW9FO1FBQ3BFLHlFQUF5RTtRQUN6RSxrRUFBa0U7UUFDbEUsZ0JBQWdCO1FBQ2hCLE1BQU0sUUFBUSxHQUFpRCxNQUFNLENBQUMsR0FBRyxDQUN2RSxDQUFDLEtBQUssRUFBRSxFQUFFO1lBQ1IsT0FBTyxJQUFJLGVBQWUsQ0FBQztnQkFDekIseUVBQXlFO2dCQUN6RSxxRUFBcUU7Z0JBQ3JFLGdFQUFnRTtnQkFDaEUsZ0VBQWdFO2dCQUNoRSxXQUFXO2dCQUNYLE1BQU0sRUFBRSxpQkFBaUIsQ0FBQyxLQUFLLENBQUM7Z0JBQ2hDLEtBQUs7Z0JBQ0wsV0FBVzthQUNaLENBQUMsQ0FBQTtRQUNKLENBQUMsQ0FDRixDQUFBO1FBRUQseUVBQXlFO1FBQ3pFLHdFQUF3RTtRQUN4RSx1RUFBdUU7UUFDdkUsdUVBQXVFO1FBQ3ZFLHdEQUF3RDtRQUN4RCxFQUFFO1FBQ0YseUVBQXlFO1FBQ3pFLGVBQWU7UUFDZiwwQ0FBMEM7UUFDMUMsT0FBTyxJQUFJLFlBQVksQ0FBQztZQUN0QixTQUFTO1lBQ1QsUUFBUTtTQUNULENBQUMsQ0FBQTtJQUNKLENBQUMsQ0FBQTtBQUNILENBQUM7QUEvRkQsZ0dBK0ZDO0FBRUQsU0FBZ0Isd0JBQXdCLENBQ3RDLGNBQXlDO0lBRXpDLE9BQU8sU0FBUyxpQkFBaUIsQ0FBQyxVQUFrQjtRQUNsRCx5RUFBeUU7UUFDekUsdUVBQXVFO1FBQ3ZFLG1CQUFtQjtRQUNuQixNQUFNLE1BQU0sR0FBRyxJQUFBLHFDQUF1QixFQUFDLFVBQVUsQ0FBQyxDQUFBO1FBQ2xELE1BQU0sTUFBTSxHQUFHLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQTtRQUNyQyxpSUFBaUk7UUFDakksSUFBQSwyQkFBSyxFQUNILE1BQU0sRUFDTix5RUFBeUUsVUFBVSxHQUFHLENBQ3ZGLENBQUE7UUFDRCxPQUFPLE1BQU0sQ0FBQTtJQUNmLENBQUMsQ0FBQTtBQUNILENBQUM7QUFoQkQsNERBZ0JDIn0=