"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _createReport = require("./lib/createReport");
var _helpers = require("./utils/helpers");
var _constants = require("./utils/constants");
var _backendToUi = require("./utils/converters/backendToUi");
var _metricHelper = require("./utils/metricHelper");
var _validationHelper = require("../../server/utils/validationHelper");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function _default(router, config) {
  const protocol = config.get('osd_server', 'protocol');
  const hostname = config.get('osd_server', 'hostname');
  const port = config.get('osd_server', 'port');
  const basePath = config.osdConfig.get('server', 'basePath');
  // generate report (with provided metadata)
  router.post({
    path: `${_common.API_PREFIX}/generateReport`,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        timezone: _configSchema.schema.maybe(_configSchema.schema.string()),
        dateFormat: _configSchema.schema.maybe(_configSchema.schema.string()),
        csvSeparator: _configSchema.schema.maybe(_configSchema.schema.string()),
        allowLeadingWildcards: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'create', 'count');
    //@ts-ignore
    const logger = context.reporting_plugin.logger;
    let report = request.body;
    // input validation
    try {
      report.report_definition.report_params.core_params.origin = `${protocol}://${hostname}:${port}${basePath}`;
      report = await (0, _validationHelper.validateReport)(context.core.opensearch.legacy.client, report, basePath);
    } catch (error) {
      logger.error(`Failed input validation for create report ${error}`);
      (0, _metricHelper.addToMetric)('report', 'create', 'user_error');
      return response.badRequest({
        body: error
      });
    }
    try {
      const reportData = await (0, _createReport.createReport)(request, context, report, config);

      // if not deliver to user himself , no need to send actual file data to client
      const delivery = report.report_definition.delivery;
      (0, _metricHelper.addToMetric)('report', 'create', 'count', report);
      return response.ok({
        body: {
          data: reportData.dataUrl,
          filename: reportData.fileName,
          reportId: reportData.reportId,
          queryUrl: reportData.queryUrl
        }
      });
    } catch (error) {
      // TODO: better error handling for delivery and stages in generating report, pass logger to deeper level
      logger.error(`Failed to generate report: ${error}`);
      logger.error(error);
      (0, _metricHelper.addToMetric)('report', 'create', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });

  // generate report from report id
  router.get({
    path: `${_common.API_PREFIX}/generateReport/{reportId}`,
    validate: {
      params: _configSchema.schema.object({
        reportId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        timezone: _configSchema.schema.string(),
        dateFormat: _configSchema.schema.string(),
        csvSeparator: _configSchema.schema.string(),
        allowLeadingWildcards: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'download', 'count');
    //@ts-ignore
    const logger = context.reporting_plugin.logger;
    try {
      const savedReportId = request.params.reportId;
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      // get report
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportById', {
        reportInstanceId: savedReportId
      });
      // convert report to use UI model
      const report = (0, _backendToUi.backendToUiReport)(opensearchResp.reportInstance, basePath);
      // generate report
      const reportData = await (0, _createReport.createReport)(request, context, report, config, savedReportId);
      (0, _metricHelper.addToMetric)('report', 'download', 'count', report);
      return response.ok({
        body: {
          data: reportData.dataUrl,
          filename: reportData.fileName,
          reportId: reportData.reportId,
          queryUrl: reportData.queryUrl
        }
      });
    } catch (error) {
      logger.error(`Failed to generate report by id: ${error}`);
      logger.error(error);
      (0, _metricHelper.addToMetric)('report', 'download', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });

  // create report from existing report definition
  router.post({
    path: `${_common.API_PREFIX}/generateReport/{reportDefinitionId}`,
    validate: {
      params: _configSchema.schema.object({
        reportDefinitionId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        timezone: _configSchema.schema.string(),
        dateFormat: _configSchema.schema.string(),
        csvSeparator: _configSchema.schema.string(),
        allowLeadingWildcards: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'create_from_definition', 'count');
    //@ts-ignore
    const logger = context.reporting_plugin.logger;
    const reportDefinitionId = request.params.reportDefinitionId;
    let report;
    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      // call OpenSearch API to create report from definition
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.createReportFromDefinition', {
        reportDefinitionId: reportDefinitionId,
        body: {
          reportDefinitionId: reportDefinitionId
        }
      });
      const reportId = opensearchResp.reportInstance.id;
      // convert report to use UI model
      const report = (0, _backendToUi.backendToUiReport)(opensearchResp.reportInstance, basePath);
      // generate report
      const reportData = await (0, _createReport.createReport)(request, context, report, config, reportId);
      (0, _metricHelper.addToMetric)('report', 'create_from_definition', 'count', report);
      return response.ok({
        body: {
          data: reportData.dataUrl,
          filename: reportData.fileName,
          reportId: reportData.reportId,
          queryUrl: reportData.queryUrl
        }
      });
    } catch (error) {
      logger.error(`Failed to generate report from reportDefinition id ${reportDefinitionId} : ${error}`);
      logger.error(error);
      (0, _metricHelper.addToMetric)('report', 'create_from_definition', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });

  // get all reports details
  router.get({
    path: `${_common.API_PREFIX}/reports`,
    validate: {
      query: _configSchema.schema.object({
        fromIndex: _configSchema.schema.maybe(_configSchema.schema.number()),
        maxItems: _configSchema.schema.maybe(_configSchema.schema.number())
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'list', 'count');
    const {
      fromIndex,
      maxItems
    } = request.query;
    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReports', {
        fromIndex: fromIndex,
        maxItems: maxItems || _constants.DEFAULT_MAX_SIZE
      });
      const reportsList = (0, _backendToUi.backendToUiReportsList)(opensearchResp.reportInstanceList, basePath);
      return response.ok({
        body: {
          data: reportsList
        }
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get reports details: ${error}`);
      (0, _metricHelper.addToMetric)('report', 'list', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });

  // get single report details by id
  router.get({
    path: `${_common.API_PREFIX}/reports/{reportId}`,
    validate: {
      params: _configSchema.schema.object({
        reportId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report', 'info', 'count');
    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportById', {
        reportInstanceId: request.params.reportId
      });
      const report = (0, _backendToUi.backendToUiReport)(opensearchResp.reportInstance, basePath);
      return response.ok({
        body: report
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get single report details: ${error}`);
      (0, _metricHelper.addToMetric)('report', 'info', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });
}
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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