"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDatasourcesRoute = registerDatasourcesRoute;
var _configSchema = require("@osd/config-schema");
var _shared = require("../../../common/constants/shared");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerDatasourcesRoute(router, dataSourceEnabled) {
  router.post({
    path: `${_shared.OBSERVABILITY_BASE}${_shared.JOBS_BASE}`,
    validate: {
      body: _configSchema.schema.object({
        query: _configSchema.schema.string(),
        lang: _configSchema.schema.string(),
        datasource: _configSchema.schema.string(),
        sessionId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const dataSourceMDSId = request.url.searchParams.get('dataSourceMDSId');
    const params = {
      body: {
        ...request.body
      }
    };
    try {
      let res;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = await context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        res = await client.callAPI('observability.runDirectQuery', params);
      } else {
        res = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('observability.runDirectQuery', params);
      }
      return response.ok({
        body: res
      });
    } catch (error) {
      console.error('Error in running direct query:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.body
      });
    }
  });
  router.get({
    path: `${_shared.OBSERVABILITY_BASE}${_shared.JOBS_BASE}/{queryId}/{dataSourceMDSId?}`,
    validate: {
      params: _configSchema.schema.object({
        queryId: _configSchema.schema.string(),
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      let res;
      if (dataSourceEnabled && request.params.dataSourceMDSId) {
        const client = await context.dataSource.opensearch.legacy.getClient(request.params.dataSourceMDSId);
        res = await client.callAPI('observability.getJobStatus', {
          queryId: request.params.queryId
        });
      } else {
        res = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('observability.getJobStatus', {
          queryId: request.params.queryId
        });
      }
      return response.ok({
        body: res
      });
    } catch (error) {
      console.error('Error in fetching job status:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_shared.OBSERVABILITY_BASE}${_shared.JOBS_BASE}/{queryId}`,
    validate: {
      params: _configSchema.schema.object({
        queryId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const res = await context.observability_plugin.observabilityClient.asScoped(request).callAsCurrentUser('observability.deleteJob', {
        queryId: request.params.queryId
      });
      return response.ok({
        body: res
      });
    } catch (error) {
      console.error('Error in deleting job:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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