// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/* Here I am reusing the Material implementation and interface from material-management.
 * This is because there are many security guarantees that this implementations offer
 * that map to the current implementation of raw AES keyrings.
 * The KeyringTrace is an unfortunate case because there is no mapping.
 * However the trade off seems worth it and the convolutions to make it work seem minimal.
 */
import { decorateCryptographicMaterial, decorateWebCryptoMaterial, frozenClass, NodeAlgorithmSuite, WebCryptoAlgorithmSuite, KeyringTraceFlag, needs, } from '@aws-crypto/material-management';
import { RawAesWrappingSuiteIdentifier, } from './raw_aes_algorithm_suite';
export class NodeRawAesMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    encryptionContext = Object.freeze({});
    constructor(suiteId) {
        /* Precondition: NodeRawAesMaterial suiteId must be RawAesWrappingSuiteIdentifier. */
        needs(RawAesWrappingSuiteIdentifier[suiteId], 'suiteId not supported.');
        this.suite = new NodeAlgorithmSuite(suiteId);
        /* NodeRawAesMaterial need to set a flag, this is an abuse of TraceFlags
         * because the material is not generated.
         * but CryptographicMaterial force a flag to be set.
         */
        const setFlags = KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlags);
        Object.setPrototypeOf(this, NodeRawAesMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey;
    }
}
frozenClass(NodeRawAesMaterial);
export class WebCryptoRawAesMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    setCryptoKey;
    getCryptoKey;
    hasCryptoKey;
    validUsages;
    encryptionContext = Object.freeze({});
    constructor(suiteId) {
        /* Precondition: WebCryptoAlgorithmSuite suiteId must be RawAesWrappingSuiteIdentifier. */
        needs(RawAesWrappingSuiteIdentifier[suiteId], 'suiteId not supported.');
        this.suite = new WebCryptoAlgorithmSuite(suiteId);
        this.validUsages = Object.freeze([
            'decrypt',
            'encrypt',
        ]);
        /* WebCryptoRawAesMaterial need to set a flag, this is an abuse of TraceFlags
         * because the material is not generated.
         * but CryptographicMaterial force a flag to be set.
         */
        const setFlag = KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlag);
        decorateWebCryptoMaterial(this, setFlag);
        Object.setPrototypeOf(this, WebCryptoRawAesMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey && this.hasCryptoKey;
    }
}
frozenClass(WebCryptoRawAesMaterial);
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmF3X2Flc19tYXRlcmlhbC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9yYXdfYWVzX21hdGVyaWFsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7QUFFdEM7Ozs7O0dBS0c7QUFFSCxPQUFPLEVBSUwsNkJBQTZCLEVBQzdCLHlCQUF5QixFQUN6QixXQUFXLEVBQ1gsa0JBQWtCLEVBQ2xCLHVCQUF1QixFQUt2QixnQkFBZ0IsRUFDaEIsS0FBSyxHQUVOLE1BQU0saUNBQWlDLENBQUE7QUFFeEMsT0FBTyxFQUVMLDZCQUE2QixHQUM5QixNQUFNLDJCQUEyQixDQUFBO0FBRWxDLE1BQU0sT0FBTyxrQkFBa0I7SUFHN0IsS0FBSyxDQUFvQjtJQUN6QixxQkFBcUIsQ0FHRTtJQUN2QixxQkFBcUIsQ0FBc0M7SUFDM0Qsc0JBQXNCLENBQTJCO0lBQ2pELHFCQUFxQixDQUFVO0lBQy9CLFlBQVksR0FBbUIsRUFBRSxDQUFBO0lBQ2pDLGlCQUFpQixHQUFzQixNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFBO0lBQ3hELFlBQVksT0FBZ0M7UUFDMUMscUZBQXFGO1FBQ3JGLEtBQUssQ0FBQyw2QkFBNkIsQ0FBQyxPQUFPLENBQUMsRUFBRSx3QkFBd0IsQ0FBQyxDQUFBO1FBQ3ZFLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQTtRQUM1Qzs7O1dBR0c7UUFDSCxNQUFNLFFBQVEsR0FBRyxnQkFBZ0IsQ0FBQywrQkFBK0IsQ0FBQTtRQUNqRSw2QkFBNkIsQ0FBcUIsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFBO1FBQ2pFLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxDQUFBO1FBQ3pELE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUE7SUFDckIsQ0FBQztJQUNELFdBQVc7UUFDVCxPQUFPLElBQUksQ0FBQyxxQkFBcUIsQ0FBQTtJQUNuQyxDQUFDO0NBQ0Y7QUFDRCxXQUFXLENBQUMsa0JBQWtCLENBQUMsQ0FBQTtBQUUvQixNQUFNLE9BQU8sdUJBQXVCO0lBS2xDLEtBQUssQ0FBeUI7SUFDOUIscUJBQXFCLENBR087SUFDNUIscUJBQXFCLENBQXNDO0lBQzNELHNCQUFzQixDQUFnQztJQUN0RCxxQkFBcUIsQ0FBVTtJQUMvQixZQUFZLEdBQW1CLEVBQUUsQ0FBQTtJQUNqQyxZQUFZLENBR2dCO0lBQzVCLFlBQVksQ0FBbUQ7SUFDL0QsWUFBWSxDQUFVO0lBQ3RCLFdBQVcsQ0FBa0M7SUFDN0MsaUJBQWlCLEdBQXNCLE1BQU0sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUE7SUFDeEQsWUFBWSxPQUFnQztRQUMxQywwRkFBMEY7UUFDMUYsS0FBSyxDQUFDLDZCQUE2QixDQUFDLE9BQU8sQ0FBQyxFQUFFLHdCQUF3QixDQUFDLENBQUE7UUFDdkUsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLHVCQUF1QixDQUFDLE9BQU8sQ0FBQyxDQUFBO1FBQ2pELElBQUksQ0FBQyxXQUFXLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQztZQUMvQixTQUFTO1lBQ1QsU0FBUztTQUNhLENBQUMsQ0FBQTtRQUN6Qjs7O1dBR0c7UUFDSCxNQUFNLE9BQU8sR0FBRyxnQkFBZ0IsQ0FBQywrQkFBK0IsQ0FBQTtRQUNoRSw2QkFBNkIsQ0FBMEIsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFBO1FBQ3JFLHlCQUF5QixDQUEwQixJQUFJLEVBQUUsT0FBTyxDQUFDLENBQUE7UUFDakUsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLENBQUMsU0FBUyxDQUFDLENBQUE7UUFDOUQsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQTtJQUNyQixDQUFDO0lBQ0QsV0FBVztRQUNULE9BQU8sSUFBSSxDQUFDLHFCQUFxQixJQUFJLElBQUksQ0FBQyxZQUFZLENBQUE7SUFDeEQsQ0FBQztDQUNGO0FBQ0QsV0FBVyxDQUFDLHVCQUF1QixDQUFDLENBQUEifQ==