// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { EncryptedDataKey } from './encrypted_data_key';
import { SignatureKey, VerificationKey } from './signature_key';
import { frozenClass, readOnlyProperty } from './immutable_class';
import { KeyringTraceFlag } from './keyring_trace';
import { NodeAlgorithmSuite } from './node_algorithms';
import { WebCryptoAlgorithmSuite } from './web_crypto_algorithms';
import { needs } from './needs';
export const supportsKeyObject = (function () {
    try {
        const { KeyObject, createSecretKey } = require('crypto'); // eslint-disable-line @typescript-eslint/no-var-requires
        if (!KeyObject || !createSecretKey)
            return false;
        return { KeyObject, createSecretKey };
    }
    catch (ex) {
        return false;
    }
})();
/*
 * This public interface to the CryptographicMaterial object is provided for
 * developers of CMMs and keyrings only. If you are a user of the AWS Encryption
 * SDK and you are not developing your own CMMs and/or keyrings, you do not
 * need to use it and you should not do so.
 *
 * The CryptographicMaterial's purpose is to bind together all the required elements for
 * encrypting or decrypting a payload.
 * The functional data key (unencrypted or CryptoKey) is the most sensitive data and needs to
 * be protected.  The longer this data persists in memory the
 * greater the opportunity to be invalidated.  Because
 * a Caching CMM exists it is important to ensure that the
 * unencrypted data key and its meta data can not be manipulated,
 * and that the unencrypted data key can be zeroed when
 * it is no longer needed.
 */
const timingSafeEqual = (function () {
    try {
        /* It is possible for `require` to return an empty object, or an object
         * that does not implement `timingSafeEqual`.
         * in this case I need a fallback
         */
        const { timingSafeEqual: nodeTimingSafeEqual } = require('crypto'); // eslint-disable-line @typescript-eslint/no-var-requires
        return nodeTimingSafeEqual || portableTimingSafeEqual;
    }
    catch (e) {
        return portableTimingSafeEqual;
    }
    /* https://codahale.com/a-lesson-in-timing-attacks/ */
    function portableTimingSafeEqual(a, b) {
        /* It is *possible* that a runtime could optimize this constant time function.
         * Adding `eval` could prevent the optimization, but this is no guarantee.
         * The eval below is commented out
         * because if a browser is using a Content Security Policy with `'unsafe-eval'`
         * it would fail on this eval.
         * The value in attempting to ensure that this function is not optimized
         * is not worth the cost of making customers allow `'unsafe-eval'`.
         * If you want to copy this function for your own use,
         * please review the timing-attack link above.
         * Side channel attacks are pernicious and subtle.
         */
        // eval('') // eslint-disable-line no-eval
        /* Check for early return (Postcondition) UNTESTED: Size is well-know information
         * and does not leak information about contents.
         */
        if (a.byteLength !== b.byteLength)
            return false;
        let diff = 0;
        for (let i = 0; i < b.length; i++) {
            diff |= a[i] ^ b[i];
        }
        return diff === 0;
    }
})();
export class NodeEncryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    encryptedDataKeys;
    addEncryptedDataKey;
    setSignatureKey;
    signatureKey;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: NodeEncryptionMaterial suite must be NodeAlgorithmSuite. */
        needs(suite instanceof NodeAlgorithmSuite, 'Suite must be a NodeAlgorithmSuite');
        this.suite = suite;
        /* Precondition: NodeEncryptionMaterial encryptionContext must be an object, even if it is empty. */
        needs(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // EncryptionMaterial have generated a data key on setUnencryptedDataKey
        const setFlags = KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlags);
        decorateEncryptionMaterial(this);
        Object.setPrototypeOf(this, NodeEncryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey;
    }
}
frozenClass(NodeEncryptionMaterial);
export class NodeDecryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    setVerificationKey;
    verificationKey;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: NodeDecryptionMaterial suite must be NodeAlgorithmSuite. */
        needs(suite instanceof NodeAlgorithmSuite, 'Suite must be a NodeAlgorithmSuite');
        this.suite = suite;
        /* Precondition: NodeDecryptionMaterial encryptionContext must be an object, even if it is empty. */
        needs(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // DecryptionMaterial have decrypted a data key on setUnencryptedDataKey
        const setFlags = KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlags);
        decorateDecryptionMaterial(this);
        Object.setPrototypeOf(this, NodeDecryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey;
    }
}
frozenClass(NodeDecryptionMaterial);
export class WebCryptoEncryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    encryptedDataKeys;
    addEncryptedDataKey;
    setSignatureKey;
    signatureKey;
    setCryptoKey;
    getCryptoKey;
    hasCryptoKey;
    validUsages;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: WebCryptoEncryptionMaterial suite must be WebCryptoAlgorithmSuite. */
        needs(suite instanceof WebCryptoAlgorithmSuite, 'Suite must be a WebCryptoAlgorithmSuite');
        this.suite = suite;
        this.validUsages = Object.freeze([
            'deriveKey',
            'encrypt',
        ]);
        /* Precondition: WebCryptoEncryptionMaterial encryptionContext must be an object, even if it is empty. */
        needs(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // EncryptionMaterial have generated a data key on setUnencryptedDataKey
        const setFlag = KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlag);
        decorateEncryptionMaterial(this);
        decorateWebCryptoMaterial(this, setFlag);
        Object.setPrototypeOf(this, WebCryptoEncryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasUnencryptedDataKey && this.hasCryptoKey;
    }
}
frozenClass(WebCryptoEncryptionMaterial);
export class WebCryptoDecryptionMaterial {
    suite;
    setUnencryptedDataKey;
    getUnencryptedDataKey;
    zeroUnencryptedDataKey;
    hasUnencryptedDataKey;
    keyringTrace = [];
    setVerificationKey;
    verificationKey;
    setCryptoKey;
    getCryptoKey;
    hasCryptoKey;
    validUsages;
    encryptionContext;
    constructor(suite, encryptionContext) {
        /* Precondition: WebCryptoDecryptionMaterial suite must be WebCryptoAlgorithmSuite. */
        needs(suite instanceof WebCryptoAlgorithmSuite, 'Suite must be a WebCryptoAlgorithmSuite');
        this.suite = suite;
        this.validUsages = Object.freeze([
            'deriveKey',
            'decrypt',
        ]);
        /* Precondition: WebCryptoDecryptionMaterial encryptionContext must be an object, even if it is empty. */
        needs(encryptionContext && typeof encryptionContext === 'object', 'Encryption context must be set');
        this.encryptionContext = Object.freeze({ ...encryptionContext });
        // DecryptionMaterial have decrypted a data key on setUnencryptedDataKey
        const setFlag = KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY;
        decorateCryptographicMaterial(this, setFlag);
        decorateDecryptionMaterial(this);
        decorateWebCryptoMaterial(this, setFlag);
        Object.setPrototypeOf(this, WebCryptoDecryptionMaterial.prototype);
        Object.freeze(this);
    }
    hasValidKey() {
        return this.hasCryptoKey;
    }
}
frozenClass(WebCryptoDecryptionMaterial);
export function isEncryptionMaterial(obj) {
    return (obj instanceof WebCryptoEncryptionMaterial ||
        obj instanceof NodeEncryptionMaterial);
}
export function isDecryptionMaterial(obj) {
    return (obj instanceof WebCryptoDecryptionMaterial ||
        obj instanceof NodeDecryptionMaterial);
}
export function decorateCryptographicMaterial(material, setFlag) {
    /* Precondition: setFlag must be in the set of KeyringTraceFlag.SET_FLAGS. */
    needs(setFlag & KeyringTraceFlag.SET_FLAGS, 'Invalid setFlag');
    /* When a KeyringTraceFlag is passed to setUnencryptedDataKey,
     * it must be valid for the type of material.
     * It is invalid to claim that EncryptionMaterial were decrypted.
     */
    const deniedSetFlags = (KeyringTraceFlag.SET_FLAGS ^ setFlag) |
        (setFlag === KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY
            ? KeyringTraceFlag.DECRYPT_FLAGS
            : setFlag === KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY
                ? KeyringTraceFlag.ENCRYPT_FLAGS
                : 0);
    let unencryptedDataKeyZeroed = false;
    let unencryptedDataKey;
    // This copy of the unencryptedDataKey is stored to insure that the
    // unencrypted data key is *never* modified.  Since the
    // unencryptedDataKey is returned by reference, any change
    // to it would be propagated to any cached versions.
    let udkForVerification;
    const setUnencryptedDataKey = (dataKey, trace) => {
        /* Avoid making unnecessary copies of the dataKey. */
        const tempUdk = dataKey instanceof Uint8Array ? dataKey : unwrapDataKey(dataKey);
        /* All security conditions are tested here and failures will throw. */
        verifyUnencryptedDataKeyForSet(tempUdk, trace);
        unencryptedDataKey = wrapWithKeyObjectIfSupported(dataKey);
        udkForVerification = new Uint8Array(tempUdk);
        material.keyringTrace.push(trace);
        return material;
    };
    const getUnencryptedDataKey = () => {
        /* Precondition: unencryptedDataKey must be set before we can return it. */
        needs(unencryptedDataKey, 'unencryptedDataKey has not been set');
        /* Precondition: unencryptedDataKey must not be Zeroed out.
         * Returning a null key would be incredibly bad.
         */
        needs(!unencryptedDataKeyZeroed, 'unencryptedDataKey has been zeroed.');
        /* Precondition: The unencryptedDataKey must not have been modified.
         * If the unencryptedDataKey is a KeyObject,
         * then the security around modification is handled in C.
         * Do not duplicate the secret just to check...
         */
        needs(!(unencryptedDataKey instanceof Uint8Array) ||
            timingSafeEqual(udkForVerification, unwrapDataKey(unencryptedDataKey)), 'unencryptedDataKey has been corrupted.');
        return unencryptedDataKey;
    };
    Object.defineProperty(material, 'hasUnencryptedDataKey', {
        // Check that we have both not zeroed AND that we have not set
        get: () => !!unencryptedDataKey && !unencryptedDataKeyZeroed,
        enumerable: true,
    });
    const zeroUnencryptedDataKey = () => {
        /* These checks are separated on purpose.  It should be impossible to have only one unset.
         * *But* if it was the case, I *must* make sure I zero out the set one, and not leave it up to GC.
         * If I only checked on say unencryptedDataKey, and udkForVerification was somehow set,
         * doing the simplest thing would be to set both to new Uint8Array.
         * Leaving udkForVerification to be garbage collected.
         * This level of insanity is due to the fact that we are dealing with the unencrypted data key.
         */
        let unsetCount = 0;
        /* Precondition: If the unencryptedDataKey has not been set, it should not be settable later. */
        if (!unencryptedDataKey) {
            unencryptedDataKey = new Uint8Array();
            unsetCount += 1;
        }
        /* Precondition: If the udkForVerification has not been set, it should not be settable later. */
        if (!udkForVerification) {
            udkForVerification = new Uint8Array();
            unsetCount += 1;
        }
        /* The KeyObject manages its own ref counter.
         * Once there are no more users, it will clean the memory.
         */
        if (!(unencryptedDataKey instanceof Uint8Array)) {
            unencryptedDataKey = new Uint8Array();
        }
        unencryptedDataKey.fill(0);
        udkForVerification.fill(0);
        unencryptedDataKeyZeroed = true;
        /* Postcondition UNTESTED: Both unencryptedDataKey and udkForVerification must be either set or unset.
         * If it is ever the case that only one was unset, then something is wrong in a profound way.
         * It is not clear how this could ever happen, unless someone is manipulating the OS...
         */
        needs(unsetCount === 0 || unsetCount === 2, 'Either unencryptedDataKey or udkForVerification was not set.');
        return material;
    };
    readOnlyProperty(material, 'setUnencryptedDataKey', setUnencryptedDataKey);
    readOnlyProperty(material, 'getUnencryptedDataKey', getUnencryptedDataKey);
    readOnlyProperty(material, 'zeroUnencryptedDataKey', zeroUnencryptedDataKey);
    return material;
    function verifyUnencryptedDataKeyForSet(dataKey, trace) {
        /* Precondition: unencryptedDataKey must not be set.  Modifying the unencryptedDataKey is denied */
        needs(!unencryptedDataKey, 'unencryptedDataKey has already been set');
        /* Precondition: dataKey must be Binary Data */
        needs(dataKey instanceof Uint8Array, 'dataKey must be a Uint8Array');
        /* Precondition: dataKey should have an ArrayBuffer that *only* stores the key.
         * This is a simple check to make sure that the key is not stored on
         * a large potentially shared ArrayBuffer.
         * If this was the case, it may be possible to find or manipulate.
         */
        needs(dataKey.byteOffset === 0, 'Unencrypted Master Key must be an isolated buffer.');
        /* Precondition: The data key length must agree with algorithm specification.
         * If this is not the case, it either means ciphertext was tampered
         * with or the keyring implementation is not setting the length properly.
         */
        needs(dataKey.byteLength === material.suite.keyLengthBytes, 'Key length does not agree with the algorithm specification.');
        /* Precondition: Trace must be set, and the flag must indicate that the data key was generated. */
        needs(trace && trace.keyName && trace.keyNamespace, 'Malformed KeyringTrace');
        /* Precondition: On set the required KeyringTraceFlag must be set. */
        needs(trace.flags & setFlag, 'Required KeyringTraceFlag not set');
        /* Precondition: Only valid flags are allowed.
         * An unencrypted data key can not be both generated and decrypted.
         */
        needs(!(trace.flags & deniedSetFlags), 'Invalid KeyringTraceFlags set.');
    }
}
export function decorateEncryptionMaterial(material) {
    const deniedEncryptFlags = KeyringTraceFlag.SET_FLAGS | KeyringTraceFlag.DECRYPT_FLAGS;
    const encryptedDataKeys = [];
    let signatureKey;
    const addEncryptedDataKey = (edk, flags) => {
        /* Precondition: If a data key has not already been generated, there must be no EDKs.
         * Pushing EDKs on the list before the data key has been generated may cause the list of
         * EDKs to be inconsistent. (i.e., they would decrypt to different data keys.)
         */
        needs(material.hasUnencryptedDataKey, 'Unencrypted data key not set.');
        /* Precondition: Edk must be EncryptedDataKey
         * Putting things onto the list that are not EncryptedDataKey
         * may cause the list of EDKs to be inconsistent. (i.e. they may not serialize, or be mutable)
         */
        needs(edk instanceof EncryptedDataKey, 'Unsupported instance of encryptedDataKey');
        /* Precondition: flags must indicate that the key was encrypted. */
        needs(flags & KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY, 'Encrypted data key flag must be set.');
        /* Precondition: flags must not include a setFlag or a decrypt flag.
         * The setFlag is reserved for setting the unencrypted data key
         * and must only occur once in the set of KeyringTrace flags.
         * The two setFlags in use are:
         * KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY
         * KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY
         *
         * KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX is reserved for the decrypt path
         */
        needs(!(flags & deniedEncryptFlags), 'Invalid flag for EncryptedDataKey.');
        material.keyringTrace.push({
            keyName: edk.providerInfo,
            keyNamespace: edk.providerId,
            flags,
        });
        encryptedDataKeys.push(edk);
        return material;
    };
    readOnlyProperty(material, 'addEncryptedDataKey', addEncryptedDataKey);
    Object.defineProperty(material, 'encryptedDataKeys', {
        // I only want EDKs added through addEncryptedDataKey
        // so I return a new array
        get: () => [...encryptedDataKeys],
        enumerable: true,
    });
    const setSignatureKey = (key) => {
        /* Precondition: The SignatureKey stored must agree with the algorithm specification.
         * If this is not the case it means the MaterialManager or Keyring is not setting
         * the SignatureKey correctly
         */
        needs(material.suite.signatureCurve, 'Algorithm specification does not support signatures.');
        /* Precondition: signatureKey must not be set.  Modifying the signatureKey is denied. */
        needs(!signatureKey, 'Signature key has already been set.');
        /* Precondition: key must be a SignatureKey. */
        needs(key instanceof SignatureKey, 'Unsupported instance of key');
        signatureKey = key;
        return material;
    };
    readOnlyProperty(material, 'setSignatureKey', setSignatureKey);
    Object.defineProperty(material, 'signatureKey', {
        get: () => {
            /* Precondition: The SignatureKey requested must agree with the algorithm specification.
             * If this is not the case it means the MaterialManager or Keyring is not setting
             * the SignatureKey correctly
             */
            needs(!!material.suite.signatureCurve === !!signatureKey, 'Algorithm specification not satisfied.');
            return signatureKey;
        },
        enumerable: true,
    });
    return material;
}
export function decorateDecryptionMaterial(material) {
    // Verification Key
    let verificationKey;
    const setVerificationKey = (key) => {
        /* Precondition: The VerificationKey stored must agree with the algorithm specification.
         * If this is not the case it means the MaterialManager or Keyring is not setting
         * the VerificationKey correctly
         */
        needs(material.suite.signatureCurve, 'Algorithm specification does not support signatures.');
        /* Precondition: verificationKey must not be set.  Modifying the verificationKey is denied. */
        needs(!verificationKey, 'Verification key has already been set.');
        /* Precondition: key must be a VerificationKey. */
        needs(key instanceof VerificationKey, 'Unsupported instance of key');
        verificationKey = key;
        return material;
    };
    readOnlyProperty(material, 'setVerificationKey', setVerificationKey);
    Object.defineProperty(material, 'verificationKey', {
        get: () => {
            /* Precondition: The VerificationKey requested must agree with the algorithm specification.
             * If this is not the case it means the MaterialManager or Keyring is not setting
             * the VerificationKey correctly
             */
            needs(!!material.suite.signatureCurve === !!verificationKey, 'Algorithm specification not satisfied.');
            return verificationKey;
        },
        enumerable: true,
    });
    return material;
}
export function decorateWebCryptoMaterial(material, setFlags) {
    let cryptoKey;
    const setCryptoKey = (dataKey, trace) => {
        /* Precondition: cryptoKey must not be set.  Modifying the cryptoKey is denied */
        needs(!cryptoKey, 'cryptoKey is already set.');
        /* Precondition: dataKey must be a supported type. */
        needs(isCryptoKey(dataKey) || isMixedBackendCryptoKey(dataKey), 'Unsupported dataKey type.');
        /* Precondition: The CryptoKey must match the algorithm suite specification. */
        needs(isValidCryptoKey(dataKey, material), 'CryptoKey settings not acceptable.');
        /* If the material does not have an unencrypted data key,
         * then we are setting the crypto key here and need a keyring trace .
         */
        if (!material.hasUnencryptedDataKey) {
            /* Precondition: If the CryptoKey is the only version, the trace information must be set here. */
            needs(trace && trace.keyName && trace.keyNamespace, 'Malformed KeyringTrace');
            /* Precondition: On setting the CryptoKey the required KeyringTraceFlag must be set. */
            needs(trace.flags & setFlags, 'Required KeyringTraceFlag not set');
            /* If I a setting a cryptoKey without an unencrypted data key,
             * an unencrypted data should never be set.
             * The expectation is if you are setting the cryptoKey *first* then
             * the unencrypted data key has already been "handled".
             * This ensures that a cryptoKey and an unencrypted data key always match.
             */
            material.zeroUnencryptedDataKey();
            material.keyringTrace.push(trace);
        }
        if (isCryptoKey(dataKey)) {
            cryptoKey = dataKey;
        }
        else {
            const { zeroByteCryptoKey, nonZeroByteCryptoKey } = dataKey;
            cryptoKey = Object.freeze({ zeroByteCryptoKey, nonZeroByteCryptoKey });
        }
        return material;
    };
    readOnlyProperty(material, 'setCryptoKey', setCryptoKey);
    const getCryptoKey = () => {
        /* Precondition: The cryptoKey must be set before we can return it. */
        needs(cryptoKey, 'Crypto key is not set.');
        // In the case of MixedBackendCryptoKey the object
        // has already been frozen above so it is safe to return
        return cryptoKey;
    };
    readOnlyProperty(material, 'getCryptoKey', getCryptoKey);
    Object.defineProperty(material, 'hasCryptoKey', {
        get: () => !!cryptoKey,
        enumerable: true,
    });
    return material;
}
export function isCryptoKey(dataKey) {
    return (dataKey &&
        'algorithm' in dataKey &&
        'type' in dataKey &&
        'usages' in dataKey &&
        'extractable' in dataKey);
}
export function isValidCryptoKey(dataKey, material) {
    if (!isCryptoKey(dataKey)) {
        const { zeroByteCryptoKey, nonZeroByteCryptoKey } = dataKey;
        return (isValidCryptoKey(zeroByteCryptoKey, material) &&
            isValidCryptoKey(nonZeroByteCryptoKey, material));
    }
    const { suite, validUsages } = material;
    const { encryption, keyLength, kdf } = suite;
    /* See:
     * https://developer.mozilla.org/en-US/docs/Web/API/CryptoKey
     * https://developer.mozilla.org/en-US/docs/Web/API/AesKeyGenParams
     */
    const { type, algorithm, usages, extractable } = dataKey;
    // @ts-ignore length is an optional value...
    const { name, length } = algorithm;
    /* MSRCrypto, for legacy reasons,
     * normalizes the algorithm name
     * to lower case.
     * https://github.com/microsoft/MSR-JavaScript-Crypto/issues/1
     * For now, I'm going to upper case the name.
     */
    // Only symmetric algorithms
    return (type === 'secret' &&
        // Must match the suite
        ((kdf && name.toUpperCase() === kdf) ||
            (name.toUpperCase() === encryption && length === keyLength)) &&
        /* Only valid usage are: encrypt|decrypt|deriveKey
         * The complexity between deriveKey and suite.kdf should be handled in the Material class.
         */
        usages.some((u) => validUsages.includes(u)) &&
        // Since CryptoKey can not be zeroized, not extractable is the next best thing
        !extractable);
}
function isMixedBackendCryptoKey(dataKey) {
    const { zeroByteCryptoKey, nonZeroByteCryptoKey } = dataKey;
    return isCryptoKey(zeroByteCryptoKey) && isCryptoKey(nonZeroByteCryptoKey);
}
export function keyUsageForMaterial(material) {
    const { suite } = material;
    if (suite.kdf)
        return 'deriveKey';
    return subtleFunctionForMaterial(material);
}
export function subtleFunctionForMaterial(material) {
    if (material instanceof WebCryptoEncryptionMaterial)
        return 'encrypt';
    if (material instanceof WebCryptoDecryptionMaterial)
        return 'decrypt';
    throw new Error('Unsupported material');
}
export function unwrapDataKey(dataKey) {
    if (dataKey instanceof Uint8Array)
        return dataKey;
    if (supportsKeyObject && dataKey instanceof supportsKeyObject.KeyObject)
        return dataKey.export();
    throw new Error('Unsupported dataKey type');
}
export function wrapWithKeyObjectIfSupported(dataKey) {
    if (supportsKeyObject) {
        if (dataKey instanceof Uint8Array) {
            const ko = supportsKeyObject.createSecretKey(dataKey);
            /* Postcondition: Zero the secret.  It is now inside the KeyObject. */
            dataKey.fill(0);
            return ko;
        }
        if (dataKey instanceof supportsKeyObject.KeyObject)
            return dataKey;
    }
    else if (dataKey instanceof Uint8Array) {
        return dataKey;
    }
    throw new Error('Unsupported dataKey type');
}
//# sourceMappingURL=data:application/json;base64,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