"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getEncryptionInfo = exports._encryptStream = void 0;
const material_management_node_1 = require("@aws-crypto/material-management-node");
const framed_encrypt_stream_1 = require("./framed_encrypt_stream");
const signature_stream_1 = require("./signature_stream");
const duplexify_1 = __importDefault(require("duplexify"));
const crypto_1 = require("crypto");
const serialize_1 = require("@aws-crypto/serialize");
// @ts-ignore
const readable_stream_1 = require("readable-stream");
const fromUtf8 = (input) => Buffer.from(input, 'utf8');
const { serializeMessageHeader, headerAuthIv, buildMessageHeader } = (0, serialize_1.serializeFactory)(fromUtf8);
/**
 * Takes a NodeDefaultCryptographicMaterialsManager or a KeyringNode that will
 * be wrapped in a NodeDefaultCryptographicMaterialsManager and returns a stream.
 *
 * @param commitmentPolicy
 * @param maxEncryptedDataKeys
 * @param cmm NodeMaterialsManager|KeyringNode
 * @param op EncryptStreamInput
 */
function _encryptStream({ commitmentPolicy, maxEncryptedDataKeys }, cmm, op = {}) {
    /* Precondition: encryptStream needs a valid commitmentPolicy. */
    (0, material_management_node_1.needs)(material_management_node_1.CommitmentPolicy[commitmentPolicy], 'Invalid commitment policy.');
    // buildEncrypt defaults this to false for backwards compatibility, so this is satisfied
    /* Precondition: encryptStream needs a valid maxEncryptedDataKeys. */
    (0, material_management_node_1.needs)(maxEncryptedDataKeys === false || maxEncryptedDataKeys >= 1, 'Invalid maxEncryptedDataKeys value.');
    const { suiteId, encryptionContext = {}, frameLength = serialize_1.FRAME_LENGTH, plaintextLength, } = op;
    /* Precondition: The frameLength must be less than the maximum frame size Node.js stream. */
    (0, material_management_node_1.needs)(frameLength > 0 && serialize_1.Maximum.FRAME_SIZE >= frameLength, `frameLength out of bounds: 0 > frameLength >= ${serialize_1.Maximum.FRAME_SIZE}`);
    /* If the cmm is a Keyring, wrap it with NodeDefaultCryptographicMaterialsManager. */
    cmm =
        cmm instanceof material_management_node_1.KeyringNode
            ? new material_management_node_1.NodeDefaultCryptographicMaterialsManager(cmm)
            : cmm;
    const suite = suiteId && new material_management_node_1.NodeAlgorithmSuite(suiteId);
    /* Precondition: Only request NodeEncryptionMaterial for algorithm suites supported in commitmentPolicy. */
    material_management_node_1.CommitmentPolicySuites.isEncryptEnabled(commitmentPolicy, suite);
    const wrappingStream = new duplexify_1.default();
    cmm
        .getEncryptionMaterials({
        suite,
        encryptionContext,
        plaintextLength,
        commitmentPolicy,
    })
        .then(async (material) => {
        /* Precondition: Only use NodeEncryptionMaterial for algorithm suites supported in commitmentPolicy. */
        material_management_node_1.CommitmentPolicySuites.isEncryptEnabled(commitmentPolicy, material.suite);
        /* Precondition: _encryptStream encryption materials must not exceed maxEncryptedDataKeys */
        (0, material_management_node_1.needs)(maxEncryptedDataKeys === false ||
            material.encryptedDataKeys.length <= maxEncryptedDataKeys, 'maxEncryptedDataKeys exceeded.');
        const { getCipher, messageHeader, rawHeader, dispose, getSigner } = getEncryptionInfo(material, frameLength);
        wrappingStream.emit('MessageHeader', messageHeader);
        const encryptStream = (0, framed_encrypt_stream_1.getFramedEncryptStream)(getCipher, messageHeader, dispose, { plaintextLength, suite: material.suite });
        const signatureStream = new signature_stream_1.SignatureStream(getSigner);
        (0, readable_stream_1.pipeline)(encryptStream, signatureStream);
        wrappingStream.setReadable(signatureStream);
        // Flush the rawHeader through the signatureStream
        rawHeader.forEach((buff) => signatureStream.write(buff));
        // @ts-ignore until readable-stream exports v3 types...
        wrappingStream.setWritable(encryptStream);
    })
        .catch((err) => wrappingStream.emit('error', err));
    return wrappingStream;
}
exports._encryptStream = _encryptStream;
function getEncryptionInfo(material, frameLength) {
    const { getCipherInfo, dispose, getSigner } = (0, material_management_node_1.getEncryptHelper)(material);
    const { suite, encryptionContext, encryptedDataKeys } = material;
    const { ivLength, messageFormat } = material.suite;
    const versionString = material_management_node_1.MessageFormat[messageFormat];
    const messageIdLength = parseInt(serialize_1.MessageIdLength[versionString], 10);
    /* Precondition UNTESTED: Node suites must result is some messageIdLength. */
    (0, material_management_node_1.needs)(messageIdLength > 0, 'Algorithm suite has unknown message format.');
    const messageId = (0, crypto_1.randomBytes)(messageIdLength);
    const { getCipher, keyCommitment } = getCipherInfo(messageId);
    const messageHeader = buildMessageHeader({
        suite: suite,
        encryptedDataKeys,
        encryptionContext,
        messageId,
        frameLength,
        suiteData: keyCommitment,
    });
    const { buffer, byteOffset, byteLength } = serializeMessageHeader(messageHeader);
    const headerBuffer = Buffer.from(buffer, byteOffset, byteLength);
    const headerIv = headerAuthIv(ivLength);
    const validateHeader = getCipher(headerIv);
    validateHeader.setAAD(headerBuffer);
    validateHeader.update(Buffer.alloc(0));
    validateHeader.final();
    const headerAuthTag = validateHeader.getAuthTag();
    return {
        getCipher,
        dispose,
        getSigner,
        messageHeader,
        rawHeader: [
            headerBuffer,
            (0, serialize_1.serializeMessageHeaderAuth)({ headerIv, headerAuthTag, messageHeader }),
        ],
    };
}
exports.getEncryptionInfo = getEncryptionInfo;
//# sourceMappingURL=data:application/json;base64,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