#' SET statement.
#'
#' This class represents a SQL SET statement, used in UPDATE queries to set
#' field values. It can hold one or more field/value pairs.
#' The factory function \code{\link{make_set}()} can be used to create
#' a SET statement more easily.
#'
#' @examples
#' # Create a SET statement with a single field/value pair:
#' set_stmt <- StmtSet$new()
#' set_stmt$add_field(ExprField$new("price"), ExprValue$new(9.50))
#'
#' # Use the created SET statement inside an UPDATE query:
#' query <- QueryUpdate$new(StmtUpdate$new("books"), set = set_stmt)
#'
#' @seealso \code{\link{make_set}()}
#' @import R6
#' @include Statement.R
#' @export
StmtSet <- R6::R6Class("StmtSet",
  inherit = Statement,
  public = list(

    #' @description
    #' Initializer.
    #' @return Nothing.
    initialize = function() {
    },

    #' @description
    #' Add a field/value pair.
    #' @param field The field, as an ExprField instance.
    #' @param value The value to set, as an Expr instance.
    #' @return Nothing.
    add_field = function(field, value) {
      chk::chk_is(field, "ExprField")
      chk::chk_is(value, "Expr")
      private$fields <- c(private$fields, field)
      private$values <- c(private$values, value)
    },

    #' @description
    #' Generates the list of tokens representing this statement.
    #' @return A list of Token objects.
    getTokens = function() {
      tokens <- list(.set, .spc)
      for (i in seq_along(private$fields)) {
        if (i > 1L)
          tokens <- c(tokens, .coma, optspace())
        tokens <- c(tokens,
                    ExprBinOp$new(private$fields[[i]], "=", private$values[[i]],
                                  paren=FALSE)$getTokens())
      }
    return(tokens)
    }
  ),

  private = list(
    fields = list(),
    values = list()
))
