% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/piFunctions.R
\name{pi.add}
\alias{pi.add}
\title{Position index addition}
\usage{
pi.add(
  attTbl,
  ngbList,
  rNumb = FALSE,
  RO,
  mainPI = NULL,
  secPI = NULL,
  add.mPI = NULL,
  add.sPI = NULL,
  cond.filter = NULL,
  min.N = NULL,
  plot = FALSE,
  r = NULL
)
}
\arguments{
\item{attTbl}{data.frame, the attribute table returned by the function
\code{\link{attTbl}}.}

\item{ngbList}{list, the list of neighborhoods returned by the function
\code{\link{ngbList}}.}

\item{rNumb}{logic, the neighborhoods of the argument \code{ngbList} are
identified by cell numbers (\code{rNumb=FALSE}) or by row numbers
(\code{rNumb=TRUE}) (see \code{\link{ngbList}}). It is advised to use row
numbers for large rasters.}

\item{RO}{column name, the name of the column with the raster object IDs.}

\item{mainPI}{column name, the name of the column with main position index
values.}

\item{secPI}{column name, the name of the column with secondary position
index values.}

\item{add.mPI}{numeric, threshold of main position index values. Cells with
values above the threshold are flagged as cells potentially being part of
new raster objects.}

\item{add.sPI}{numeric, threshold of secondary position index values. Cells
with values above the threshold flagged as cells potentially being part of
new raster objects.}

\item{cond.filter}{character string, defines what cells have to be considered
by the function the arguments. Test cell absolute conditions can be used
(see \code{\link{conditions}}).}

\item{min.N}{numeric, the minimum number of cells a raster object has to have
to be included in the function output.}

\item{plot}{logic, plot the results.}

\item{r}{single or multi-layer raster of the class \code{SpatRaster} (see
\code{help("rast", terra)}) used to compute the attribute table. Required
only if \code{plot = TRUE}.}
}
\value{
The function returns a class vector with raster objects IDs. The
vector has length equal to the number of rows of the attribute table. NA
values are assigned to cells that do not belong to any raster object.
}
\description{
Add new raster objects based on position index values.
}
\details{
New raster objects are added based on position index values. Two
different position indices can be passed to the function (\code{mainPI} and
\code{secPI}).
\itemize{
\item Input raster objects are assigned to the same class to flag cells that
are part of raster objects;
\item Cells with values above \code{mainPI} \strong{OR} above \code{mainPI} are
flagged as cells potentially being part of new raster objects;
\item If not connected to any of the existing raster objects, the groups of
cells above position index values are assigned to new raster objects.
\item Only raster objects with at least as many cells as specified by the
argument \code{min.N} are included in the function output.
\item If both \code{mainPI} and \code{secPI} are equal to \code{NULL}, the
function will exclusively filter raster objects based on their size
(\code{min.N}).
}
}
\note{
Raster objects are added only if they do not share any border with
input raster objects.
}
\examples{
# DUMMY DATA
######################################################################################
# LOAD LIBRARIES
library(scapesClassification)
library(terra)

# LOAD THE DUMMY RASTER
r <- list.files(system.file("extdata", package = "scapesClassification"),
                pattern = "dummy_raster\\\\.tif", full.names = TRUE)
r <- terra::rast(r)

# COMPUTE THE ATTRIBUTE TABLE
at <- attTbl(r, "dummy_var")

# COMPUTE THE LIST OF NEIGBORHOODS
nbs <- ngbList(r, attTbl=at)

################################################################################
# COMPUTE RASTER OBJECTS
################################################################################
at$RO[at$dummy_var==8] <- 1
at$RO <- cond.4.nofn(at, nbs, classVector = at$RO, class=1, nbs_of = 1,
                     cond = "dummy_var < dummy_var[] & dummy_var > 2")

# One raster object
unique(at$RO)

################################################################################
# POSITION INDEX
################################################################################
at$PI <- (at$dummy_var - mean(at$dummy_var))/stats::sd(at$dummy_var)

################################################################################
# POSITION INDEX ADDITION
################################################################################
RO1 <- pi.add(at, nbs,
              RO = "RO",     # Raster objects
              mainPI = "PI", # PI addition layer
              add.mPI = 1,   # add disjoint objects with PI values > 1
              plot = FALSE, r = r)

################################################################################
# PLOT
################################################################################
# Convert class vectors to raster
r_RO  <- cv.2.rast(r = r, classVector = at$RO)
r_RO1 <- cv.2.rast(r = r, classVector = RO1)

# Plot
oldpar <- par(mfrow = c(1,2))
m <- c(4.5, 0.5, 2, 3.2)

terra::plot(r_RO, type="classes", main="Raster objects - Input", mar=m,
            plg=list(x=1, y=1, cex=0.9))

terra::plot(r_RO1, type="classes", main="Raster objects - Output", mar=m,
            plg=list(x=1, y=1, cex=0.9))
text(xyFromCell(r,at$Cell), as.character(round(at$PI,2)),
cex = 0.8) # visualize relPI
text(0.01, 1, "Add on PI >= 1", adj=c(0,0), cex = 0.8)
par(oldpar)

# Two raster object
unique(RO1)
}
\seealso{
\code{\link[=attTbl]{attTbl()}}, \code{\link[=ngbList]{ngbList()}}, \code{\link[=rel.pi]{rel.pi()}}, \code{\link[=pi.sgm]{pi.sgm()}}, \code{\link[=conditions]{conditions()}}
}
