#' Akaike's Information Criterion
#'
#' @param object an object of type multiRec
#' @param ... not used
#' @param k numeric, the penalty per parameter to be used; the default k = 2 is
#'   the classical AIC.
#'
#' @return a number
#' @export
AIC.multiRec = function(object, ..., k=2) {
  p = nrow(object$coefficients)
  return(object$deviance + k*p)

}

#' Bayesian Information Criteron
#'
#' @param object an object of type multiRec
#' @param ... not used
#'
#' @return a number
#' @export
BIC.multiRec = function(object, ...) {
  p = nrow(object$coefficients)
  n = nrow(object$data)
  return(object$deviance + log(n)*p)
}

#' Variance covariance matrix of a fitted model
#'
#' If the model was fitted using robust=TRUE, this is the robust variance.
#' Otherwise it is the naive variance (i.e. the inverse of the information
#' matrix)
#'
#' @param object an object of type multiRec
#' @param ... not used
#'
#' @return a matrix
#' @export
vcov.multiRec = function(object, ...) {
  object$var
}

#' Deviance of a fitted model
#'
#' @param object an object of type multiRec
#' @param ... not used
#'
#' @return a number
#' @export
deviance.multiRec = function(object, ...) {
  object$deviance
}

#' Log likelihood of a fitted model
#'
#' @param object an object of type multiRec
#' @param ... not used
#'
#' @return a number
#' @export
logLik.multiRec = function(object, ...) {
  object$loglik
}

#' Confidence Intervals for Model Parameters
#'
#' @param object	a fitted model object.
#' @param parm	a specification of which parameters are to be given confidence
#'   intervals, either a vector of numbers or a vector of names. If missing, all
#'   parameters are considered.
#' @param level	the confidence level required.
#' @param ...	not used
#'
#' @importFrom stats qnorm
#' @return a matrix
#' @export
confint.multiRec = function(object, parm, level = 0.95, ...) {
  beta = object$coefficients
  est  = beta[,'estimate']
  se   = beta[,'se']
  alpha= (1 - level)/2
  fac  = c(alpha, 1 - alpha)
  CI   = est + se %o% qnorm(fac)
  dimnames(CI) = list(rownames(beta), format(100*fac))
  if (missing(parm)) return(CI)
  else return(CI[parm,])
}


